/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.generator;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.FileNames;
import jp.sourceforge.mergedoc.pleiades.resource.Files;
import jp.sourceforge.mergedoc.pleiades.resource.Property;
import jp.sourceforge.mergedoc.pleiades.resource.PropertySet;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.ArrayUtils;

/**
 * 複数のプロパティー・ファイルをマージし、1 つのプロパティー・ファイル
 * (translation.properties) を作成するための翻訳プロパティー・ジェネレーターです。
 * <p>
 * @author cypher256
 */
public class Generator implements FileNames {

	/** ロガー */
	private static final Logger log = Logger.getLogger(Generator.class);

	/** 正規表現キープレフィックス */
	private static final String REGEX_KEY_PREFIX = "%REGEX%";

	/** 翻訳除外キープレフィックス */
	private static final String EXCLUDE_KEY_PREFIX = "%EXCLUDE%";
	
	/** 校正＆ルール適用済み言語パック辞書の再構築を行う場合は true */
	private static boolean isClean;
	
	/**
	 * ジェネレーターを開始するための main メソッドです。
	 * <p>
	 * @param args 起動引数
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void main(String... args) throws IOException {
		
		String argStr = ArrayUtils.toString(args);
		log.info("起動オプション: " + argStr);
		isClean = argStr.contains("-clean");
		
		new Generator().run();
	}

	/**
	 * ジェネレーターを実行します。
	 * <p>
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void run() throws IOException {
		
		// ログ削除
		for (File file : Files.getResourceFile("props").listFiles()) {
			String name = file.getName();
			if (name.endsWith(".log")) {
				file.delete();
			}
		}
		
		//--------------------------------------------------------------------
		// 言語パック辞書のロード
		//--------------------------------------------------------------------
		
		// 校正＆ルール適用済み言語パック辞書プロパティー (既存訳)
		PropertySet nlsUiCustomRuleProp = null;
		
		if (isClean) {
			
			// 英文から & 除去。ニーモニックでなくても除去。&1 など。
			// これは実際に翻訳されるときに lookup メソッドで英文からまず & が除去されるため。
			// translation.properties の英文は & が付かない形式で格納されることに注意。
			// 日本語のほうはかっこがない限り除去されることはない。
			
			log.info("校正済み言語パック辞書プロパティーをロードします。");
			nlsUiCustomRuleProp = new PropertySet(NLS_UI_CUSTOM_PROP);
			
			log.info("校正済み言語パック辞書プロパティーに翻訳ルールを適用分割中...");
			nlsUiCustomRuleProp = new TranslationRule(NLS_UI_CUSTOM_PROP + "_rule.log")
				.apply(nlsUiCustomRuleProp);
			nlsUiCustomRuleProp.store(TEMP_NLS_UI_CUSTOM_PROP,
				"校正＆ルール適用済み言語パック・プロパティー (検証前一時ファイル)");

			// 現状、検証は分割後に最適化されているため、分割後に行わないと末尾 .. などでエラーとなる
			log.info("校正済み言語パック辞書プロパティーの問題除去中 (翻訳ルール適用分割後) ...");
			Validator v = new Validator(NLS_UI_CUSTOM_PROP + "_validate.log");
			v.validate(nlsUiCustomRuleProp);
			exitIfValidationError(v);
			
		} else {
			
			log.info("校正＆ルール適用済み言語パック辞書プロパティーをロードします。");
			nlsUiCustomRuleProp = new PropertySet(TEMP_NLS_UI_CUSTOM_PROP);
		}
		
		//--------------------------------------------------------------------
		// 追加辞書のロード
		//--------------------------------------------------------------------
		
		// 追加辞書プロパティー
		PropertySet tempAddAllProp = new PropertySet();
		// 正規表現辞書プロパティー
		PropertySet outRegexProp = new PropertySet();
		// 除外プロパティー
		PropertySet outExcludeProp = new PropertySet();
		
		// バリデーター
		Validator v = new Validator(TEMP_ADDITIONS_ALL_PROP + "_validate.log", nlsUiCustomRuleProp);

		// 追加辞書の振り分け
		log.info("追加辞書プロパティーをロードします。");
		File[] files = Files.getResourceFile("props/additions").listFiles();
		Arrays.sort(files); // 昇順

		for (File file : files) {
			if (file.isFile() && file.getName().endsWith(".properties")) {

				PropertySet addProp = new PropertySet(file);
				log.debug("Loaded " + file.getName() + " " + addProp.size());

				// % で始まる特殊プロパティを抽出し、別プロパティへ振り分け
				for (Property p : addProp) {

					if (p.key.startsWith(REGEX_KEY_PREFIX)) {

						// 正規表現プロパティ
						String newKey = p.key.replaceFirst("^" + REGEX_KEY_PREFIX, "").trim();
						String newVal = p.value.trim();
						outRegexProp.put(newKey, newVal);

					} else if (p.key.startsWith(EXCLUDE_KEY_PREFIX)) {

						// 翻訳除外プロパティ
						String newKey = p.key.replaceFirst("^" + EXCLUDE_KEY_PREFIX, "").trim();
						Object existsValue = outExcludeProp.get(newKey);
						if (existsValue != null) {
							p.value = existsValue + "," + p.value;
						}
						outExcludeProp.put(newKey, p.value);

					} else {

						// 訳語の検証
						v.validate(p, file.getName());

						// 翻訳辞書
						tempAddAllProp.put(p.key, p.value);
					}
				}
			}
		}
		v.logEndMessage("追加辞書プロパティー検証結果");
		exitIfValidationError(v);
		
		log.info("追加辞書プロパティーに翻訳ルールを適用分割中...");
		tempAddAllProp = new TranslationRule(TEMP_ADDITIONS_ALL_PROP + "_rule.log", nlsUiCustomRuleProp)
			.apply(tempAddAllProp);
		
		log.info("追加辞書プロパティーの問題除去中 (翻訳ルール適用分割後) ...");
		v = new Validator(TEMP_ADDITIONS_ALL_PROP + "_validate.log", nlsUiCustomRuleProp);
		tempAddAllProp = v.remove(tempAddAllProp);
		exitIfValidationError(v);
		
		//--------------------------------------------------------------------
		// ヘルプ辞書のロード
		//--------------------------------------------------------------------
		
		log.info("ヘルプ辞書プロパティーをロードします。");
		
		// ヘルプ・テキスト辞書 (ルール適用済み、検証済み)
		PropertySet tempNlsHelpTextProp = new PropertySet();
		File helpTextFile = Files.getResourceFile(TEMP_NLS_HELP_TEXT_PROP);
		if (helpTextFile.exists()) {
			tempNlsHelpTextProp.load(helpTextFile);
		}
		
		// ヘルプ HTML 辞書 (ルール適用済み、検証エラー含む)
		PropertySet outHelpHtmlProp = new PropertySet(TEMP_NLS_HELP_HTML_PROP);
		
		//--------------------------------------------------------------------
		// マージ
		//--------------------------------------------------------------------
		
		// ヘルプ HTML 辞書はマージしないで別ファイルとする
		PropertySet outTransProp = new PropertySet();
		outTransProp.putAll(tempNlsHelpTextProp);
		outTransProp.putAll(tempAddAllProp);
		outTransProp.putAll(nlsUiCustomRuleProp);
		
		// 分割なしプロパティー追加
		PropertySet noSplitProp = new PropertySet();
		noSplitProp.putAll(new TranslationRule().apply(noSplitProp));
		noSplitProp.load(NO_SPLIT_PROP);
		v = new Validator(NO_SPLIT_PROP + "_validate.log", noSplitProp);
		exitIfValidationError(v);
		outTransProp.putAll(noSplitProp);
		
		// マルチ・バイト・キー辞書
		PropertySet outMultibyteProp = new PropertySet();
		for (Property p : outTransProp) {
			if (p.key.length() != p.key.getBytes().length) {
				outMultibyteProp.put(p);
			}
		}
		for (Property p : outMultibyteProp) {
			outTransProp.remove(p.key);
		}		
		
		tempAddAllProp.store(TEMP_ADDITIONS_ALL_PROP,
			"追加辞書全プロパティー（テンポラリー）\n\n" +
			"  プラグイン別の追加辞書をマージしたものです。\n" +
			"  句点解析によりエントリーは可能な限り文単位に分割されています。\n" +
			"  入力元ファイル：props/additions/*.properties");
		
		storeHistory(outTransProp, TEMP_ALL_PROP,
			"辞書全プロパティー（テンポラリー）\n\n" +
			"  言語パック抽出辞書と追加辞書全プロパティー（テンポラリー）をマージしたものです。\n" +
			"  システム的には不要なファイルです。\n" +
			"  入力元ファイル：" + TEMP_NLS_HELP_TEXT_PROP + "、\n" +
			"                  " + TEMP_NLS_UI_CUSTOM_PROP + "、\n" +
			"                  " + TEMP_ADDITIONS_ALL_PROP);
		
		//--------------------------------------------------------------------
		// 保管
		//--------------------------------------------------------------------

		String commonComment =
			"  重複防止やリソース、メモリー消費量低減のため、次のような文字は\n" +
			"  除去されており、翻訳時は原文を元に自動的に補完されます。\n" +
			"  \n" +
			"  ・ニーモニック：英語の場合は & 1 文字、日本語の場合は (&x) 4 文字\n" +
			"  ・先頭・末尾の連続する空白：\\r、\\n、\\t、半角スペース\n" +
			"  ・前後の囲み文字 ()、[]、<>、\"\"、''、!! など (前後の組み合わせが一致する場合)\n" +
			"  ・先頭の IWAB0001E のような Eclipse 固有メッセージ・コード\n" +
			"  ・複数形を示す (s) (秒を示すものではない場合)" +
			"";

		String commonCommentSplited =
			"  \n" +
			"  句点解析によりエントリーは可能な限り文単位に分割されています。また、\n" +
			commonComment;
		
		if (isClean) {
			
			Files.getResourceFile(TEMP_NLS_UI_CUSTOM_PROP).delete();
			nlsUiCustomRuleProp.store(TEMP_NLS_UI_CUSTOM_PROP,
				"校正＆ルール適用済み言語パック・プロパティー\n\n" +
				"  校正済み言語パック・プロパティーに翻訳ルールを適用したものです。\n" +
				"  入力元ファイル：" + NLS_UI_CUSTOM_PROP + "\n" +
				commonCommentSplited);
		}
		
		outTransProp.store(TRANS_PROP,
			"翻訳辞書プロパティー\n\n" +
			"  校正＆ルール適用済み言語パック・プロパティーと全追加辞書をマージしたもので、\n" +
			"  Pleiades が実行時に参照します。\n" +
			"  \n" +
			"  入力元ファイル：" + TEMP_NLS_HELP_TEXT_PROP + "\n" +
			"                  " + TEMP_NLS_UI_CUSTOM_PROP + "\n" +
			"                  " + TEMP_ADDITIONS_ALL_PROP + "\n" +
			commonCommentSplited);
		
		outRegexProp.store(TRANS_REGEX_PROP,
			"正規表現辞書プロパティー\n\n" +
			"  正規表現で翻訳するための辞書で、Pleiades が実行時に参照します。\n" +
			"  入力元ファイル：props/additions/*.properties のキー先頭に " + REGEX_KEY_PREFIX + " がある項目\n" +
			"  \n" +
			commonComment);
		
		outExcludeProp.store(TRANS_EXCLUDE_PACKAGE_PROP,
			"翻訳除外パッケージ・プロパティー\n\n" +
			"  翻訳を Java パッケージ単位で除外訳するための辞書で、Pleiades が実行時に参照します。\n" +
			"  入力元ファイル：props/additions/*.properties のキー先頭に " + EXCLUDE_KEY_PREFIX + " がある項目");
		
		outMultibyteProp.store(TRANS_MULTIBYTES_KEY_PROP,
			"翻訳マルチ・バイト・キー・プロパティー\n\n" +
			"  翻訳プロパティーからキーにマルチ・バイトが含まれるものを抽出した辞書で、\n" +
			"  Pleiades が実行時に参照します。\n" +
			"  デフォルトの翻訳プロパティー・ロード抑止判定のために使用されます。");
		
		outHelpHtmlProp.store(TRANS_HELP_HTML_PROP,
			"ヘルプ HTML 辞書プロパティー (暫定、内容検証中)\n\n" +
			"  言語パックのヘルプから抽出したものです。\n" +
			"  入力元ファイル：" + TEMP_NLS_HELP_HTML_PROP);
	}
	
	/**
	 * バリデーターにエラーがある場合、強制終了します。
	 * (Ant からの呼び出しを終了させるため強制終了)
	 * <p>
	 * @param validator バリデーター
	 */
	private void exitIfValidationError(Validator validator) {
		
		if (!validator.isSuccess()) {
			log.error("検証エラーのため、強制終了します。");
			System.exit(-1);
		}
	}

	/**
	 * プロパティーを保管し、diff 用にテキスト・ファイルとして 2 世代保管します。
	 * <p>
	 * @param prop プロパティー
	 * @param path 保管パス
	 * @param comment コメント文字列
	 * @throws IOException 入出力k例外が発生した場合
	 */
	public static void storeHistory(PropertySet prop, String path, String comment) throws IOException {
		
		List<String> keyList = prop.store(path, comment);
		if (keyList == null) {
			return;
		}
		
		File textFile = Files.getResourceFile(Files.toVcIgnoreName(path + ".txt"));
		if (textFile.exists()) {
			File textOldFile = Files.getResourceFile(Files.toVcIgnoreName(path + "_old.txt"));
			textOldFile.delete();
			textFile.renameTo(textOldFile);
		}
		
		List<String> lines = new LinkedList<String>();
		lines.add("#" + prop.size() + " エントリー");
		for (String key : keyList) {
			String line = Property.toString(key, prop.get(key));
			lines.add(line);
		}
		FileUtils.writeLines(textFile, "UTF-8", lines);
	}
}
