/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.resource;


/**
 * 翻訳辞書クラスです。
 * これは静的な言語パック・ジェネレーターなどの外部からの利用を想定したクラスです。
 * Eclipse 実行時は DynamicTranslationDictionary が使用されます。
 * <p>
 * @author cypher256
 */
public class TranslationDictionary extends AbstractTranslationDictionary {

	/**
	 * 指定した英語リソース文字列から日本語リソースを探します。
	 * ニーモニックは日本向けに変換されます。
	 * <pre>
	 * ・Label		⇒ ラベル
	 * ・L&abel		⇒ ラベル(&A)	--- 辞書を参照する前にニーモニックを一旦除去
	 * ・CVS		⇒ CVS			--- 辞書に無い場合はそのまま
	 * ・C&VS		⇒ CVS(&V)		--- 辞書に無いがニーモニックは変換
	 * </pre>
	 * 対応する訳が辞書に存在しなかった場合は、戻り値 {@link TranslationResult}
	 * の isFound に false が設定されます。
	 * <p>
	 * <b>注）</b>辞書に存在しなくてもニーモニック変換は行われるため、
	 * 辞書存在有無の判定を行う場合は、引数の値と結果の値を比較するのではなく、
	 * 戻り値の isFound を使用するようにしてください。
	 * 
	 * @param enWithMnemonic 英語リソース文字列 (ニーモニック含む)
	 * @return 翻訳結果
	 */
	public TranslationResult lookup(String enWithMnemonic) {

		// 英語リソース文字列からニーモニック制御文字 & を除去
		String en = Mnemonics.removeEnMnemonic(enWithMnemonic);
		
		// 翻訳プロパティーから対訳を取得
		String ja = getValue(en);
		
		// ニーモニックを日本用に変換。
		// この変換は訳語が見つからなかった場合も行う。例：C&VS ⇒ CVS(&V)
		String jaWithMnemonic = Mnemonics.convertMnemonicEnToJa(enWithMnemonic, en, ja);
		
		// 翻訳結果を作成
		TranslationResult result = new TranslationResult();
		result.setValue(jaWithMnemonic);
		result.setFound(!ja.equals(en));
		return result;
	}

	/**
	 * 指定した英語リソース文字列から日本語リソースを探します。
	 * ニーモニックは処理されません。このメソッドは引数に含まれる & を
	 * ニーモニックとして処理したくない場合に使用します。
	 * <pre>
	 * ・Label		⇒ ラベル
	 * ・L&abel		⇒ L&abel		--- 辞書を参照する前にニーモニックは除去されない
	 * ・CVS		⇒ CVS			--- 辞書に無い場合はそのまま
	 * ・C&VS		⇒ C&VS			--- 辞書に無い。ニーモニックも無変換。
	 * </pre>
	 * @param en 英語リソース文字列
	 * @return 翻訳結果
	 */
	public TranslationResult lookupIgnoreMnemonic(String en) {
		
		// 翻訳プロパティーから対訳を取得
		String ja = getValue(en);
		
		// 翻訳結果を作成
		TranslationResult result = new TranslationResult();
		result.setValue(ja);
		result.setFound(!ja.equals(en));
		return result;
	}

	/**
	 * 指定した英語ヘルプ・リソース文字列から日本語ヘルプ・リソースを探します。
	 * <p>
	 * @param en 英語ヘルプ・リソース文字列
	 * @return 翻訳結果
	 */
	public TranslationResult lookupHelp(String en) {
		
		// 翻訳プロパティーから対訳を取得
		String ja = getValueForHelp(en);
		
		// 翻訳結果を作成
		TranslationResult result = new TranslationResult();
		result.setValue(ja);
		result.setFound(!ja.equals(en));
		return result;
	}
}
