/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.advice;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint.EditPoint;
import jp.sourceforge.mergedoc.pleiades.log.Logger;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * アスペクト・マッピングの設定ファイル・アセンブラです。<br>
 * ジョイント・ポイントとポイント・カットの関連付けを行います。
 * <p>
 * @author cypher256
 */
public class AspectMappingAssembler extends DefaultHandler {

	/** ロガー */
	private static final Logger log = Logger.getLogger(AspectMappingAssembler.class);

	/** プロパティーを保持するマップ */
	private final Set<Property> properties = new HashSet<Property>();

	/** ジョイント・ポイントとポイント・カットを保持するマップ */
	private final Map<JointPoint, PointCut> jointMap;

	/** ポイント・カット */
	private PointCut pointCut;

	/** アドバイス文字列バッファ */
	private StringBuilder adviceBuffer;

	/** ジョイント・ポイント */
	private JointPoint jointPoint;

	/** 編集ポイント */
	private String editPoint;

	/**
	 * アスペクト・マッピング・ハンドラを構築します。
	 * <p>
	 * @param jointMap ジョイント・ポイントとポイント・カットを保持するマップ
	 */
	public AspectMappingAssembler(Map<JointPoint, PointCut> jointMap) {
		this.jointMap = jointMap;
	}

	/**
	 * XML 要素開始時の処理です。
	 */
	@Override
	public void startElement(String uri, String localName, String qName,
			Attributes attributes) throws SAXException {

		if (qName.equals("pleiades")) {

		} else if (qName.equals("property")) {

			Property property = new Property();
			property.setName(attributes.getValue("name"));
			property.setValue(attributes.getValue("value"));
			properties.add(property);

		} else if (qName.equals("pointCut")) {

			pointCut = new PointCut();
			pointCut.setTiming(attributes.getValue("timing"));
			pointCut.setLevel(attributes.getValue("level"));
			editPoint = attributes.getValue("editPoint");

		} else if (qName.equals("advice")) {

			adviceBuffer = new StringBuilder();

		} else if (qName.equals("jointPoint")) {

			jointPoint = createJointPoint(new JointPoint(), attributes);

			if (jointMap.containsKey(jointPoint)) {
				log.warn("ジョイント・ポイント定義が重複しています。" + jointPoint);
			}
			PointCut curPointCut = new PointCut(pointCut);
			jointMap.put(jointPoint, curPointCut);
			curPointCut.setJointPoint(jointPoint);

		} else if (qName.endsWith(/* include～ or exclude～ */"cludeWhere")) {

			PointCut curPointCut = jointMap.get(jointPoint);
			if (curPointCut.getJointPoint().getEditPoint() == EditPoint.EXECUTION) {
				throw new IllegalStateException(
					"editPoint が execution の場合、*cludeWhere 属性を指定することはできません。" +
					curPointCut);

			}
			// *cludeWhere の descriptor は未サポート
			JointPoint where = createJointPoint(new JointPoint(), attributes);

			if (qName.equals("excludeWhere")) {
				curPointCut.getExcludeWheres().add(where);
			} else if (qName.equals("includeWhere")) {
				curPointCut.getIncludeWheres().add(where);
			} else {
				throw new IllegalStateException("不正な属性名です。" + qName);
			}
			if (curPointCut.getExcludeWheres().size() > 0 && curPointCut.getIncludeWheres().size() > 0) {
				throw new IllegalStateException(
					"同じ jointPoint 内に excludeWhere と includeWhere を同時に指定できません。" +
					curPointCut);
			}

		} else if (qName.endsWith(/* include～ or exclude～ */"cludeTrace")) {

			PointCut curPointCut = jointMap.get(jointPoint);
			if (curPointCut.getJointPoint().getEditPoint() == EditPoint.CALL) {
				throw new IllegalStateException(
					"editPoint が call の場合、*cludeTrace 属性を指定することはできません。" +
					curPointCut);

			}
			if (!curPointCut.getAdvice().contains("?{JOINT_POINT}")) {
				throw new IllegalStateException(
					"*cludeTrace を指定している場合は Advice に ?{JOINT_POINT} が" +
					"含まれている必要があります。" + curPointCut);
			}
			// *cludeTrace の descriptor は未サポート
			JointPoint where = createJointPoint(new JointPoint(), attributes);
			
			if (qName.equals("excludeTrace")) {
				curPointCut.getExcludeTrace().add(where);
			} else if (qName.equals("includeTrace")) {
				curPointCut.getIncludeTrace().add(where);
			} else {
				throw new IllegalStateException("不正な属性名です。" + qName);
			}
			if (curPointCut.getExcludeTrace().size() > 0 && curPointCut.getIncludeTrace().size() > 0) {
				throw new IllegalStateException(
					"同じ jointPoint 内に excludeTrace と includeTrace を同時に指定できません。" +
					curPointCut);
			}

		} else {
			throw new IllegalStateException("不正な属性名です。" + qName);
		}
	}
	
	/**
	 * 属性から JointPoint を作成します。
	 * @param jointPoint
	 * @param attributes 属性
	 * @return JointPoint
	 */
	private JointPoint createJointPoint(JointPoint jointPoint, Attributes attributes) {
		
		jointPoint.setClassName(attributes.getValue("className"));
		jointPoint.setMethodName(attributes.getValue("methodName"));
		jointPoint.setDescriptor(attributes.getValue("descriptor"));
		jointPoint.setEditPoint(editPoint);
		
		return jointPoint;
	}

	/**
	 * XML 要素終了時の処理です。
	 */
	@Override
	public void endElement(String uri, String localName, String qName)
			throws SAXException {

		if (qName.equals("advice")) {

			String advice = adviceBuffer.toString();
			for (Property property : properties) {
				advice = advice.replaceAll(
					"\\?\\{" + property.getName() + "\\}",
					property.getValue());
			}
			pointCut.setAdvice(advice);
		}
	}

	/**
	 * XML 要素ボディ部の処理です。
	 */
	@Override
	public void characters(char[] ch, int start, int length) throws SAXException {

		if (adviceBuffer != null) {
			adviceBuffer.append(ch, start, length);
		}
	}
}
