/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2007 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.metasearch.compass;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import jp.sourceforge.metasearch.compass.entity.Searchable;

import org.apache.oro.text.perl.Perl5Util;
import org.compass.core.CompassException;
import org.compass.core.CompassHits;
import org.compass.core.CompassQuery;
import org.compass.core.CompassQueryBuilder;
import org.compass.core.CompassQueryFilter;
import org.compass.core.CompassQueryFilterBuilder;
import org.compass.core.CompassSession;
import org.compass.core.CompassTransaction;
import org.compass.core.Property;
import org.compass.core.Resource;
import org.compass.core.CompassQueryBuilder.CompassBooleanQueryBuilder;
import org.compass.core.CompassTransaction.TransactionIsolation;

/**
 * IndexSearcher
 *
 * @author Yomei Komiya
 *
 * @since 0.9.2
 * @version $Id: IndexSearcher.java 5 2007-06-28 13:28:30Z whitestar $
 */
public class IndexSearcher {
	
	/**
	 * Context Object for Compass
	 * injected by Spring
	 */
	private CompassContext context = null;
	
	/**
	 * Role stored Index Field Name
	 * injected by Spring
	 */
	private String roleFieldName = "role";
	

	public IndexSearcher() {
		super();
	}

	
	public CompassContext getContext() {
		return context;
	}


	public void setContext(CompassContext context) {
		this.context = context;
	}


	public final void setRoleFieldName(String roleFieldName) {
		this.roleFieldName = roleFieldName;
	}


	/**
	 * Find Documents
	 * 
	 * @param query Query String
	 * @return Search Result Object
	 * @throws Exception
	 */
	public SearchResult find(String query) throws Exception {
		return this.find(query, 10, 0);
	}

	
	/**
	 * Find Documents with Role Filter
	 * 
	 * @param query Query String
	 * @param roles User's Roles
	 * @return Search Result Object
	 * @throws Exception
	 */
	public SearchResult find(String query, String[] roles) throws Exception {
		return this.find(query, 10, 0, roles);
	}

	
	/**
	 * Find Documents
	 * 
	 * @param query Query String
	 * @param num Display Number per Page
	 * @param start Display Start Position
	 * @return Search Result Object
	 * @throws SearchFailedException
	 */
	public SearchResult find(String query, int num, int start)
		throws SearchFailedException {
		
		return this.find(query, null, num, start, null);
	}
	
	
	/**
	 * Find documents
	 * 
	 * @param query New query String
	 * @param currentQuery Current query string 
	 * @param num Display Number per Page
	 * @param start Display Start Position
	 * @return Search Result Object
	 * @throws SearchFailedException
	 */
	public SearchResult find(
			String query,
			String currentQuery,
			int num,
			int start)
		throws SearchFailedException {
		
		return this.find(query, currentQuery, num, start, null);
	}
	
	
	/**
	 * Find Documents with Role Filter
	 * 
	 * @param query Query String
	 * @param num Display Number per Page
	 * @param start Display Start Position
	 * @param roles User's Roles
	 * @return Search Result Object
	 * @throws SearchFailedException
	 */
	public SearchResult find(String query, int num, int start, String[] roles)
		throws SearchFailedException {
		
		return this.find(query, null, num, start, roles);
	}

	
	/**
	 * Refine Search Documents with Role Filter
	 * 
	 * @param query New Query String
	 * @param currentQuery Current Parsed Query String
	 * @param num Display Number per Page
	 * @param start Display Start Position
	 * @param roles User's Roles
	 * @return Search Result Object
	 * @throws SearchFailedException
	 */
	public SearchResult find(
			String query,
			String currentQuery,
			int num,
			int start,
			String[] roles)
		throws SearchFailedException {

		CompassSession session = null;
		CompassTransaction tx = null;
		CompassHits hits = null;

		try {
			long elapsedTime = System.currentTimeMillis();
			
			session = this.context.getCompass().openSession();
			tx = session.beginTransaction(
					TransactionIsolation.READ_ONLY_READ_COMMITTED);
			
			CompassQueryBuilder queryBuilder = session.queryBuilder();
			CompassBooleanQueryBuilder
				mainBooleanQueryBuilder = queryBuilder.bool();
			
			// add current query
			if (currentQuery != null && !currentQuery.equals("")) {
				mainBooleanQueryBuilder
					.addMust(queryBuilder.queryString(currentQuery).toQuery());
			}
			
			// add new query
			mainBooleanQueryBuilder
				.addMust(queryBuilder.queryString(query).toQuery());
			
			// set role filter
			CompassQuery currentCompassQuery = null;
			if (roles != null && roles.length != 0) {
				CompassQueryFilterBuilder queryFilterBuilder
					= session.queryFilterBuilder();
				CompassQueryFilter roleFilter
					= this.buildRoleFilter(
							queryBuilder, queryFilterBuilder, roles);			
				currentCompassQuery
					= mainBooleanQueryBuilder.toQuery().setFilter(roleFilter);
				// non filter version
				//	= mainBooleanQueryBuilder
				//		.addMust(roleBooleanQueryBuilder.toQuery()).toQuery();
			}
			else {
				currentCompassQuery	= mainBooleanQueryBuilder.toQuery();
			}

			hits = currentCompassQuery.hits();
			
			SearchResult result
				= this.buildSearchResult(
						query,
						currentCompassQuery.toString(),
						num,
						start,
						hits);
			
			elapsedTime = System.currentTimeMillis() - elapsedTime;
			result.setSearchTime(elapsedTime);
			
			tx.commit();
			
			return result;
		}
		catch (CompassException ce) {
			if (tx != null) {
				tx.rollback();
			}
			ce.printStackTrace();
			throw new SearchFailedException(ce);
		}
		finally {
			if (hits != null) {
				hits.close();
			}
			if (session != null) {
				session.close();
			}
		}
	}

	
	/**
	 * Build Role Filter
	 * 
	 * @param queryBuilder
	 * @param queryFilterBuilder
	 * @param roles Role List
	 * @return Role Filter
	 */
	protected CompassQueryFilter buildRoleFilter(
			CompassQueryBuilder queryBuilder,
			CompassQueryFilterBuilder queryFilterBuilder,
			String[] roles) {

		CompassBooleanQueryBuilder roleBooleanQueryBuilder
			= queryBuilder.bool();

		for (int i = 0, legth = roles.length; i < legth; i++) {
			roleBooleanQueryBuilder
				.addShould(queryBuilder.term(this.roleFieldName, roles[i]));
		}

		CompassQueryFilter roleFilter
			= queryFilterBuilder.query(roleBooleanQueryBuilder.toQuery());
		
		return roleFilter;
	}
	
	
	/**
	 * Build Search Result Object
	 * 
	 * @param query New Raw Query String
	 * @param currentQuery Current Parsed Query String
	 * @param num Display Number per Page
	 * @param start Display Start Position
	 * @param hits CompassHits Object
	 * @return Search Result Object
	 */
	protected SearchResult buildSearchResult(
			String query,
			String currentQuery,
			int num,
			int start,
			CompassHits hits) {

		Map resourceContextMap = this.context.getResourceContextMap();
		SearchResult result = new SearchResult();
		List list = new ArrayList();
		for (int i = start, j = hits.getLength(); i < j; i++) {
			if (i >= start + num) {
				break;
			}
			else {
				Resource resource = hits.resource(i);
				Searchable data = (Searchable)hits.data(i);
				data.setResourceAlias(resource.getAlias());
				data.setResourceContext(
						(String)resourceContextMap.get(resource.getAlias()));
				data.buildIdentifier();
				data.setProvider(
						resource.getProperty("provider").getStringValue());
				data.setProviderUrl(
						resource.getProperty("providerUrl").getStringValue());
				data.setType(
						this.buildTypesString(resource.getProperties("type")));
				data.setRights(resource.getProperty("rights").getStringValue());
				String content = data.getContent();
				if (content != null) {
					data.setSize(content.length());
				}
				else {
					data.setSize(0);
				}
				data.setScore(hits.score(i));
				data.setFragment(hits.highlighter(i).fragment("all"));
				
				list.add(data);
			}
		}

		result.setProvider(this.context.getProvider());
		result.setProviderUrl(this.context.getProviderUrl());
		result.setQuery(trimExtraOperator(query));
		result.setCurrentQuery(trimExtraOperator(currentQuery));
		result.setTotalNum(hits.getLength());
		result.setNum(num);
		result.setStart(start);
		result.setList(list);
		
		return result;
	}
	
	
	/**
	 * Build Types String
	 * 
	 * @param typeProps types array
	 * @return types string (e.g. "type1,type2")
	 */
	protected String buildTypesString(Property[] typeProps) {
		String types = "";

		for (int i = 0, j = typeProps.length; i < j; i++) {
			if (types.equals("")) {
				types += typeProps[i].getStringValue();
			}
			else {
				types += "," + typeProps[i].getStringValue();
			}
		}
		
		return types;
	}

	
	/**
	 * Trim Extra Query Operator
	 * "+(keyword)" -> "keyword"
	 * 
	 * @param query fat query
	 * @return lean query
	 */
	static protected String trimExtraOperator(String query) {
		Perl5Util perl = new Perl5Util();

		boolean match = perl.match("/^\\+\\((.*)\\)$/", query);
		if (match){
			return trimExtraOperator(perl.group(1));
		}
		else{
			return query;
		}		
	}

}
