#include "common.h"
#include <algorithm>

extern "C"
{
#include <dirent.h>
#include <libgen.h>
#include <termios.h>
#include <sys/ioctl.h>
}

static list<string> gDirStack;

void cDirWnd::PushFrontDir()
{
    gDirStack.push_front(mPath);
}

void cDirWnd::PushBackDir()
{
    gDirStack.push_back(mPath);
}

void cDirWnd::PopDir()
{
    if(gDirStack.size() > 0) {
        string dir = gDirStack.front();
        gDirStack.pop_front();
        Move((char*)dir.c_str());
    }
}

cDirWnd* ActiveDir()
{
    if(gLDir->mActive)
        return gLDir;
    else
        return gRDir;
}

cDirWnd* SleepDir()
{
    if(!gLDir->mActive)
        return gLDir;
    else
        return gRDir;
}

cDirWnd::eSortKind cDirWnd::gSortKind = cDirWnd::kName;
cDirWnd::eViewOption cDirWnd::gViewOption = cDirWnd::kAll;

cDirWnd::cDirWnd(char* path, bool active)
{
TBEGIN();

    strcpy(mPath, path);

    read();

    mActive = active;
    mScrollTop = 0; 
    if(mFiles.size() <= 2)
        mCursor = 1;
    else
        mCursor = 2;
        
    Sort();

TEND();
}

cDirWnd::~cDirWnd()
{
}

bool cDirWnd::sort_name(sFile left, sFile right)
{
    if(strcmp(left.mName, ".") == 0) return true;
    if(strcmp(left.mName, "..") == 0) {
        if(strcmp(right.mName, ".") == 0) return false;

        return true;          
    }
    if(strcmp(right.mName, ".") == 0) return false;
    if(strcmp(right.mName, "..") == 0) return false;
     
    if(!S_ISDIR(left.mStat.st_mode) && !S_ISDIR(right.mStat.st_mode)
        || S_ISDIR(left.mStat.st_mode) && S_ISDIR(right.mStat.st_mode) )
        {
        return strcmp(left.mName, right.mName) < 0;
        }
          
    if(S_ISDIR(left.mStat.st_mode)) return true;
    return false;
}
    
bool cDirWnd::sort_ext(sFile left, sFile right)
{
     if(strcmp(left.mName, ".") == 0) return true;
     if(strcmp(left.mName, "..") == 0) {
          if(strcmp(right.mName, ".") == 0) return false;

          return true;          
     }
     if(strcmp(right.mName, ".") == 0) return false;
     if(strcmp(right.mName, "..") == 0) return false;
     
     char* lext = extname(left.mName);
     char* rext = extname(right.mName);
    
     bool result;
     if(S_ISDIR(left.mStat.st_mode)) {
          if(S_ISDIR(right.mStat.st_mode))
                result= strcmp(left.mName, right.mName) <0;
          else
                result = true;
     }
     else if(S_ISDIR(right.mStat.st_mode)) {
          result = false;
     }
     else {
          const int cmp = strcmp(lext, rext);
          if(cmp == 0) {
                result = strcmp(left.mName, right.mName) < 0;
          }
          else {
                result = cmp < 0;
          }
     }
    
     FREE(lext);
     FREE(rext);
    
     return result;
}
  
bool cDirWnd::sort_size(sFile left, sFile right)
{
     if(strcmp(left.mName, ".") == 0) return true;
     if(strcmp(left.mName, "..") == 0) {
          if(strcmp(right.mName, ".") == 0) return false;

          return true;          
     }
     if(strcmp(right.mName, ".") == 0) return false;
     if(strcmp(right.mName, "..") == 0) return false;

     if(S_ISDIR(left.mStat.st_mode)) {
          if(S_ISDIR(right.mStat.st_mode)) return strcmp(left.mName, right.mName) < 0;
          return true;
     }
     if(S_ISDIR(right.mStat.st_mode)) return false;

     const int lsize = S_ISDIR(left.mStat.st_mode) ? 0 : left.mStat.st_size; 
     const int rsize = S_ISDIR(right.mStat.st_mode) ? 0 : right.mStat.st_size;
    
     return lsize < rsize;
}
    
bool cDirWnd::sort_time(sFile left, sFile right)
{ 
     if(strcmp(left.mName, ".") == 0) return true;
     if(strcmp(left.mName, "..") == 0) {
          if(strcmp(right.mName, ".") == 0) return false;

          return true;          
     }
     if(strcmp(right.mName, ".") == 0) return false;
     if(strcmp(right.mName, "..") == 0) return false;
     
     return difftime(left.mStat.st_mtime, right.mStat.st_mtime) > 0;
}

void cDirWnd::Sort()
{
TBEGIN();

    switch(gSortKind)
    {
    case kName:
        sort(mFiles.begin(), mFiles.end(), sort_name);
        break;

    case kExt:
        sort(mFiles.begin(), mFiles.end(), sort_ext);
        break;

    case kSize:
        sort(mFiles.begin(), mFiles.end(), sort_size);
        break;

    case kTime:
        sort(mFiles.begin(), mFiles.end(), sort_time);
        break;
    }

TEND();     
}

void cDirWnd::read()
{
TBEGIN();

    if(access(Path(), F_OK) != 0) Move("/");

    DIR* dir = opendir(mPath);
    if(dir == NULL) {
        err_msg("cDirWnd::read(): opendir err");
        Move("/");
    }
    
    mFiles.clear();
    struct dirent* entry;
    while(entry = readdir(dir)) {
        char path[PATH_MAX];
        strcpy(path, mPath);
        strcat(path, entry->d_name);

        struct stat stat_;
        memset(&stat_, 0, sizeof(struct stat));
        stat(path, &stat_);
          
        struct stat lstat_;
        memset(&lstat_, 0, sizeof(struct stat));
        lstat(path, &lstat_);
          
        sFile file(entry->d_name, &stat_, &lstat_, false);
                
        mFiles.push_back(file);
    }
    closedir(dir);

TEND();     
}

void cDirWnd::Reread()
{
TBEGIN();

    read();
    MoveCursorIndex(mCursor);
    Sort();

TEND();
}

void cDirWnd::MoveCursor(int value)
{
TBEGIN();

    mCursor += value;

    if(mCursor < 0) mCursor = 0;
    if(mCursor >= mFiles.size()) mCursor = mFiles.size()-1;

    struct winsize ws;
    ioctl(STDOUT_FILENO,TIOCGWINSZ,&ws);

    const int maxy = ws.ws_row;
    
    if(mCursor > mScrollTop + maxy-kMaxYMinus-(gDirStack.size()>0?1:0)-(gMasters.size()>0?1:0)-1)
        mScrollTop += mCursor-mScrollTop - maxy+kMaxYMinus
                            +(gDirStack.size()>0?1:0)+(gMasters.size()>0?1:0)+1;
    if(mCursor < mScrollTop) 
        mScrollTop = mCursor;

TEND();          
}

void cDirWnd::MoveCursorIndex(int index)
{
TBEGIN();

    mCursor = index;
     
    if(mCursor < 0) mCursor = 0;
    if(mCursor >= mFiles.size()) mCursor = mFiles.size()-1;
     
    struct winsize ws;
    ioctl(STDOUT_FILENO,TIOCGWINSZ,&ws);

    const int maxy = ws.ws_row;

    if(mCursor > mScrollTop + maxy-kMaxYMinus-(gDirStack.size()>0?1:0)
                                                    -(gMasters.size()>0?1:0)-1)
        mScrollTop += mCursor-mScrollTop - maxy+kMaxYMinus
                          +(gDirStack.size()>0?1:0)+(gMasters.size()>0?1:0)+1;
    if(mCursor < mScrollTop) mScrollTop = mCursor;

TEND();     
}

void cDirWnd::Activate(cDirWnd* current)
{
TBEGIN();

    current->mActive = false;
    mActive = true;

    if(!gIndividualCursor)
        MoveCursorIndex(mScrollTop + current->mCursor - current->mScrollTop);

    chdir(mPath);

TEND();     
}

sFile cDirWnd::CursorFile()
{
    return mFiles[mCursor];
}

void cDirWnd::Mark()
{
TBEGIN();

    mFiles[mCursor].mMark = !mFiles[mCursor].mMark;

TEND();     
}

void cDirWnd::MarkAll()
{
TBEGIN();

    for(int i=0; i<mFiles.size(); i++) {
        sFile* file = &mFiles[i];
        if(strcmp(file->mName, ".") != 0 && strcmp(file->mName, "..") != 0) {
             file->mMark = !file->mMark;
        }
    }

TEND();     
}

bool cDirWnd::Marking()
{
    for(int i=0; i<mFiles.size(); i++)
        if(mFiles[i].mMark) return true;

    return false;
}

VALUE cDirWnd::MarkFiles()
{
TBEGIN();

    VALUE result = rb_ary_new();

    if(Marking()) {
        for(int i=0; i<mFiles.size(); i++) {
            if(mFiles[i].mMark) {
                rb_ary_push(result, rb_str_new2(mFiles[i].mName));
            }
        }
    }
    else {
        if(ActiveDir() == this) {
            rb_ary_push(result, rb_str_new2(ActiveDir()->CursorFile().mName));
        }
    }

    return result;

TEND();     
}

void cDirWnd::MarkFilesSQuote(char* result)
{
TBEGIN();

    strcpy(result, "");

    if(Marking()) {
        for(int i=0; i<mFiles.size(); i++) {
            if(mFiles[i].mMark) {
                strcat(result, "'");
                strcat(result, mFiles[i].mName);
                strcat(result, "'");
                strcat(result, " ");
            }
        }
    }
    else {
        if(ActiveDir() == this) {
            strcat(result, "'");
            strcat(result, ActiveDir()->CursorFile().mName);
            strcat(result, "'");
        }
    }

TEND();     
}

bool cDirWnd::check_path(char* new_path)
{
    bool result = false;

    DIR* dir = opendir(new_path);
    if(dir) {
        result = true;
        closedir(dir);
    }

    return result;   
}

void cDirWnd::Move(char* path)
{
TBEGIN();

    char buf[1024];

    if(strcmp(path, "..") == 0){
        if(strcmp(mPath, "/") != 0) {
             char* parentpath = parentname(mPath);

             if(check_path(parentpath)) {
                 mDirStack.push_back(mPath);
                 if(mDirStack.size() > kDirStackSize) mDirStack.pop_front();
                 
                 char* tmp = STRDUP(mPath);
                 char* fname = STRDUP(basename(tmp));
                 FREE(tmp);
                 
                 strcpy(mPath, parentpath); 
             
                 read();
                 Sort();

                 bool found = false;                
                 mScrollTop = 0;
                 for(int i=0; i<mFiles.size(); i++) {
                     if(strcmp(mFiles[i].mName, fname) == 0) {
                         MoveCursorIndex(i);
                         found = true;
                     }
                 }

                 if(!found) MoveCursorIndex(2);

                 FREE(fname);
             
                if(this == ActiveDir())  chdir(mPath);
            }
            
            FREE(parentpath);
        }
    }
    else if(path[0] == '/') {
        if(access(path, F_OK) == 0) {
             char path2[PATH_MAX];

             strcpy(path2, path);
             if(path[strlen(path) -1] != '/') {
                  strcat(path2, "/");
             }

             if(check_path(path2)) {
                 mDirStack.push_back(mPath);
                 if(mDirStack.size() > kDirStackSize) mDirStack.pop_front();
    
                 strcpy(mPath, path2);
                 read(); 
                 mScrollTop = 0;
                 MoveCursorIndex(2);
                 Sort();
                 
                if(this == ActiveDir())  chdir(mPath);
             }
        }
    }
    else if(strcmp(path, ".") != 0) {
        char path2[PATH_MAX];
        strcpy(path2, mPath);
        strcat(path2, path);
        strcat(path2, "/");
        
        if(access(path2, F_OK) == 0 && check_path(path2)) { 
             mDirStack.push_back(mPath);
             if(mDirStack.size() > kDirStackSize) mDirStack.pop_front();
        
             strcpy(mPath, path2);
             read();
             Sort();
             mScrollTop = 0;
             MoveCursorIndex(2);

            if(this == ActiveDir())  chdir(mPath);
        }
    }

TEND();    
}

void cDirWnd::MoveBack()
{
TBEGIN();

    if(mDirStack.size() > 0) { 
        string path = mDirStack.back();
        mDirStack.pop_back();

        if(access(path.c_str(), F_OK) == 0) {
             strcpy(mPath, (char*)path.c_str());
             read();

             Sort();
    
             mScrollTop = 0;
             MoveCursorIndex(2);
        }
    }

    if(ActiveDir() == this) {
        chdir(mPath);
    }

TEND();    
}

void cDirWnd::View()
{
TBEGIN();

    const int maxx = getmaxx(stdscr);
    const int maxy = getmaxy(stdscr);

    const int dstack_exist = gDirStack.size() > 0 ? 1 : 0;
    const int ptty_exist = gMasters.size() > 0 ? 1:0;

    /// view one dir ///
    if(cDirWnd::gViewOption == kOneDir) {
        if(mActive) {
            mybox(0, dstack_exist, maxx-1, maxy-2-dstack_exist-ptty_exist);

            char buf[256];

            int i;
            bool kanji = false;
            for(i=0; i<maxx-4 && mPath[i]!=0; i++) {
                if(kanji) {
                    kanji = false;
                }
                else if(is_kanji(mPath[i])) {
                    kanji = true;
                }

                buf[i] = mPath[i];
            }
            if(kanji) {
                buf[i-1] = ' ';
            }
            buf[i] = 0;

            attron(A_BOLD);
            mvprintw(0 + dstack_exist, 2, buf);
            attroff(A_BOLD);
            
            for(int i=mScrollTop;
                i-mScrollTop<maxy-kMaxYMinus -dstack_exist -ptty_exist
                            && i<mFiles.size();
                i++)
            {
                int attr = 0;
                sFile* file = &mFiles[i];
                 
                if(mCursor == i && mActive) {
                     attr |= A_REVERSE;
                }

                char fname[256];
                if(file->mMark) {
                     strcpy(fname, "*");

                     if(gColor)
                        attr |= COLOR_PAIR(kRed);
                     else 
                        attr |= A_BOLD;
                }
                else {
                     strcpy(fname, " ");
                }
        
                strcat(fname, file->mName);
            
                if(S_ISDIR(file->mStat.st_mode)) {
                     strcat(fname, "/");

                     if(gColor)
                         attr |= COLOR_PAIR(kBlue) | A_BOLD;
                     else
                         attr |= A_BOLD;
                }
                else if(file->mStat.st_mode&S_IXUSR
                            || file->mStat.st_mode&S_IXGRP
                            || file->mStat.st_mode&S_IXGRP)
                {
                     strcat(fname, "*");
             
                     if(gColor)
                         attr |= COLOR_PAIR(kGreen);
                     else
                         attr |= A_BOLD;
                }

                char permission[12];
                strcpy(permission, "----------");
                if(S_ISDIR(file->mStat.st_mode)) permission[0] = 'd';
                if(S_ISLNK(file->mStat.st_mode)) permission[0] = 'l';
            
                mode_t smode = file->mStat.st_mode;
                if(smode & S_IRUSR) permission[1] = 'r';
                if(smode & S_IWUSR) permission[2] = 'w';
                if(smode & S_IXUSR) permission[3] = 'x';
                if(smode & S_IRGRP) permission[4] = 'r';
                if(smode & S_IWGRP) permission[5] = 'w';
                if(smode & S_IXGRP) permission[6] = 'x';
                if(smode & S_IROTH) permission[7] = 'r';
                if(smode & S_IWOTH) permission[8] = 'w';
                if(smode & S_IXOTH) permission[9] = 'x';
        
                time_t t = file->mStat.st_mtime;
                struct tm* tm_ = (struct tm*)localtime(&t);
             
                cut(fname, buf,maxx-62-3);

                char owner[256];
                char* tmp = mygetpwuid(&file->mStat);
                if(tmp)
                    cut(tmp, owner, 9);
                else
                    sprintf(owner, "%d", file->mStat.st_uid);

                char group[256];
                tmp = mygetgrgid(&file->mStat);
                if(tmp)
                    cut(tmp, group, 10);
                else
                    sprintf(group, "%d", file->mStat.st_gid);
                    
                sprintf(buf + strlen(buf)
                       , " %s %3d %-9s %-10s%10lld %02d-%02d-%02d %02d:%02d "
                       , permission, file->mStat.st_nlink
                       , owner, group
                       
                       , file->mStat.st_size
                       
                       , tm_->tm_year-100, tm_->tm_mon+1
                       , tm_->tm_mday, tm_->tm_hour, tm_->tm_min
                );
                                        
                attron(attr);
                mvprintw(i-mScrollTop + 1 + dstack_exist, 1, buf);
                attroff(attr);
            }
        }
        
        mvprintw(maxy - kMaxYMinus+1 -ptty_exist, 2
                                    , "%dfiles", mFiles.size());
    }
    
    /// view_nameonly view_all ///
    else {
        int x;
        if(this == gLDir)
            x = 0;
        else
            x = maxx / 2; 

        int line = 0;
        mybox(x, dstack_exist, maxx/2-1, maxy-2-dstack_exist-ptty_exist);
    
        char buf[256];

        int i;
        bool kanji = false;
        for(i=0; i<maxx/2-5 && mPath[i]!=0; i++) {
            if(kanji) {
                kanji = false;
            }
            else if(is_kanji(mPath[i])) {
                kanji = true;
            }

            buf[i] = mPath[i];
        }
        if(kanji) {
            buf[i-1] = ' ';
        }
        buf[i] = 0;
        
        if(mActive) attron(A_BOLD);
        mvprintw(dstack_exist, 2+x, buf);
        if(mActive) attroff(A_BOLD);

        for(int i=mScrollTop;
                i-mScrollTop<maxy-kMaxYMinus-dstack_exist-ptty_exist
                        && i<mFiles.size();
                i++)
            {
            int attr = 0;
            sFile* file = &mFiles[i];
                 
            if(mCursor == i && mActive) attr |= A_REVERSE;

            char fname[256];
            if(file->mMark) {
                strcpy(fname, "*");
                if(gColor)
                    attr |= COLOR_PAIR(kRed);
                else
                    attr |= A_BOLD;
            }
            else {
                strcpy(fname, " ");
            }
        
            strcat(fname, file->mName);
        
            if(S_ISDIR(file->mStat.st_mode)) {
                strcat(fname, "/");

                if(gColor)
                    attr |= COLOR_PAIR(kBlue) | A_BOLD;
                else
                    attr |= A_BOLD;
            }
            else if(file->mStat.st_mode&S_IXUSR
                    || file->mStat.st_mode&S_IXGRP
                    || file->mStat.st_mode&S_IXGRP)
            {
                strcat(fname, "*");
             
                if(gColor) 
                    attr |= COLOR_PAIR(kGreen);
                else
                    attr |= A_BOLD;
            }
            else if(S_ISLNK(file->mStat.st_mode)) {
                strcat(fname, "@");
                if(gColor) attr |= COLOR_PAIR(kCyan);
            }

            /// view_all ///
            if(gViewOption == kAll) {
                char buf[256];

                char name[256];
                cut(fname, name, maxx/2-28);

                time_t t = file->mStat.st_mtime;
                struct tm* tm_ = (struct tm*)localtime(&t);
                
                sprintf(buf, "%s%9lld %02d-%02d-%02d %02d:%02d "
                         , name, file->mStat.st_size, tm_->tm_year-100
                         , tm_->tm_mon+1, tm_->tm_mday, tm_->tm_hour
                         , tm_->tm_min);
                                          
                attron(attr);
                mvprintw(i-mScrollTop + 1 + dstack_exist, x + 1, buf);
                attroff(attr);
            }
            /// view nameonly ///
            else {
                char name[256];
                cut(fname, name, maxx/2-3);
                 
                attron(attr);
                mvprintw(i-mScrollTop + 1 + dstack_exist, x + 1, name);
                attroff(attr);
            }
        }
          
        mvprintw(maxy - kMaxYMinus+1 -ptty_exist, x + 2
                                    , "%dfiles", mFiles.size());
    }

    /// draw gDirStack ///
    if(dstack_exist) {
        const int width = maxx / 3; //(gDirStack.size()+1);
        int j = 0;
        for(list<string>::iterator i = gDirStack.begin();
            i != gDirStack.end();
            i++)
            {
            if(j > 2) break;
            
            char buf[256];
            buf[0] = '[';
            buf[1] = '%';
            sprintf(buf + 2, "-%ds]", width-2);

            char buf2[256];
            const char* src = (*i).c_str();
            const int len = strlen(src);
            bool kanji = false;
            bool kanji_ato = false;
            int c = 0;
            for(int k=0; k<len; k++) {
                if(kanji) {
                    kanji = false;
                    kanji_ato = true;
                }
                else if(is_kanji(src[k])) {
                    kanji = true;
                    kanji_ato = false;
                }
                else {
                    kanji_ato = false;
                }

                if(len-k <= (width-2)) {
                    if(kanji_ato && c==0) {
                        buf2[c++] = ' ';
                    }
                    else {
                        buf2[c++] = src[k];
                    }
                }
            }
            buf2[c] = 0;
M(("[%s]",buf2));            
            
            mvprintw(0, width * j, buf, buf2);
            j++;                        
            }
    }

TEND(); 
}
