//******************************************************************************
//
// OGL Utility / OGLTransMatrix
//
// 変換行列クラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "OGLTransMatrix.h"
#import "OGLH.h"
#import <string.h>  //for memset
#import <math.h>


//******************************************************************************
// コンストラクタ
//******************************************************************************
OGLTransMatrix::OGLTransMatrix(void)
{
	Clear();
}

//******************************************************************************
// デストラクタ
//******************************************************************************
OGLTransMatrix::~OGLTransMatrix(void)
{
}

//******************************************************************************
// クリア
//******************************************************************************
void OGLTransMatrix::Clear()
{
	m_TransNum = 0;
	memset(&(m_TransInfo[0]), 0, sizeof(OGLTransInfo) * OGLTRANSMATRIX_MULTIPLY_MAX);
}

//******************************************************************************
// 拡大縮小
//******************************************************************************
void OGLTransMatrix::RegistScale(OGLfloat x, OGLfloat y, OGLfloat z)
{
	if (m_TransNum >= OGLTRANSMATRIX_MULTIPLY_MAX) goto EXIT;
	
	m_TransInfo[m_TransNum].type = OGLTransScale;
	m_TransInfo[m_TransNum].vector = OGLVECTOR3(x, y, z);
	m_TransNum++;
	
EXIT:;
	return;
}

//******************************************************************************
// 回転：X軸周り
//******************************************************************************
void OGLTransMatrix::RegistRotationX(OGLfloat angle)
{
	if (m_TransNum >= OGLTRANSMATRIX_MULTIPLY_MAX) goto EXIT;
	
	m_TransInfo[m_TransNum].type = OGLTransRotation;
	m_TransInfo[m_TransNum].vector = OGLVECTOR3(1.0f, 0.0f, 0.0f);
	m_TransInfo[m_TransNum].angle = angle;
	m_TransNum++;
	
EXIT:;
	return;
}

//******************************************************************************
// 回転：Y軸周り
//******************************************************************************
void OGLTransMatrix::RegistRotationY(OGLfloat angle)
{
	if (m_TransNum >= OGLTRANSMATRIX_MULTIPLY_MAX) goto EXIT;
	
	m_TransInfo[m_TransNum].type = OGLTransRotation;
	m_TransInfo[m_TransNum].vector = OGLVECTOR3(0.0f, 1.0f, 0.0f);
	m_TransInfo[m_TransNum].angle = angle;
	m_TransNum++;
	
EXIT:;
	return;
}

//******************************************************************************
// 回転：Z軸周り
//******************************************************************************
void OGLTransMatrix::RegistRotationZ(OGLfloat angle)
{
	if (m_TransNum >= OGLTRANSMATRIX_MULTIPLY_MAX) goto EXIT;
	
	m_TransInfo[m_TransNum].type = OGLTransRotation;
	m_TransInfo[m_TransNum].vector = OGLVECTOR3(0.0f, 0.0f, 1.0f);
	m_TransInfo[m_TransNum].angle = angle;
	m_TransNum++;
	
EXIT:;
	return;
}

//******************************************************************************
// 回転：任意軸周り
//******************************************************************************
void OGLTransMatrix::RegistRotationXYZ(
		OGLfloat angle,
		OGLVECTOR3 axisVector
	)
{
	if (m_TransNum >= OGLTRANSMATRIX_MULTIPLY_MAX) goto EXIT;
	
	m_TransInfo[m_TransNum].type = OGLTransRotation;
	m_TransInfo[m_TransNum].vector = axisVector;
	m_TransInfo[m_TransNum].angle = angle;
	m_TransNum++;
	
EXIT:;
	return;
}

//******************************************************************************
// 移動
//******************************************************************************
void OGLTransMatrix::RegistTranslation(OGLfloat x, OGLfloat y, OGLfloat z)
{
	if (m_TransNum >= OGLTRANSMATRIX_MULTIPLY_MAX) goto EXIT;
	
	m_TransInfo[m_TransNum].type = OGLTransToranslation;
	m_TransInfo[m_TransNum].vector = OGLVECTOR3(x, y, z);
	m_TransNum++;
	
EXIT:;
	return;
}

//******************************************************************************
// モデル行列取得
//******************************************************************************
matrix_float4x4 OGLTransMatrix::GetModelMatrix()
{
	OGLTransInfo transInfo;
	matrix_float4x4 matrix;
	matrix_float4x4 modelMatrix;
	int index = 0;
		
	modelMatrix = matrix_identity_float4x4;
	
	if (m_TransNum == 0) goto EXIT;
	
	//変換行列を適用
	for (index = (m_TransNum - 1); index >= 0; index--) {
		transInfo = m_TransInfo[index];
		switch (transInfo.type) {
			//回転
			case OGLTransRotation:
				matrix = _glRotatef(
								transInfo.angle,
								transInfo.vector.x,
								transInfo.vector.y,
								transInfo.vector.z
							);
				break;
			//移動
			case OGLTransToranslation:
				matrix = _glTranslatef(
								transInfo.vector.x,
								transInfo.vector.y,
								transInfo.vector.z
							);
				break;
			//拡大縮小
			case OGLTransScale:
				matrix = _glScalef(
							transInfo.vector.x,
							transInfo.vector.y,
							transInfo.vector.z
						);
				break;
			default:
				matrix = matrix_identity_float4x4;
				break;
		}
		//行列合成
		modelMatrix = matrix_multiply(modelMatrix, matrix);
	}
	
EXIT:;
	return modelMatrix;
}

//******************************************************************************
// 変換行列適用
//******************************************************************************
void OGLTransMatrix::push(OGLDevice* pDevice)
{
	pDevice->SetModelMatrix(GetModelMatrix());
}

//******************************************************************************
// 変換行列適用解除
//******************************************************************************
void OGLTransMatrix::pop(OGLDevice* pDevice)
{
	pDevice->SetModelMatrix(matrix_identity_float4x4);
}

//******************************************************************************
// コピー
//******************************************************************************
void OGLTransMatrix::CopyFrom(OGLTransMatrix* pTransMatrix)
{
	m_TransNum = pTransMatrix->m_TransNum;
	memcpy(
		&(m_TransInfo[0]),
		&(pTransMatrix->m_TransInfo[0]),
		sizeof(OGLTransInfo)*OGLTRANSMATRIX_MULTIPLY_MAX
	);
}

//******************************************************************************
// glRotatef 代替関数
//******************************************************************************
matrix_float4x4 OGLTransMatrix::_glRotatef(
		OGLfloat angle, //回転角度
		OGLfloat x,		//回転軸ベクトル
		OGLfloat y,		//
		OGLfloat z		//
	)
{
	OGLVECTOR3 n = OGLH::Vec3Normalize(OGLVECTOR3(x, y, z));
	OGLfloat s = sinf(OGLH::ToRadian(angle));
	OGLfloat c = cosf(OGLH::ToRadian(angle));
	OGLfloat c1 = 1.0f - c;
	
	//Metal,OpenGLは列優先(Column major order)
	matrix_float4x4 matrix = {{
		{ (n.x * n.x * c1) + c,         (n.y * n.x * c1) + (n.z * s), (n.x * n.z * c1) - (n.y * s), 0 },
		{ (n.x * n.y * c1) - (n.z * s), (n.y * n.y * c1) + c,         (n.y * n.z * c1) + (n.x * s), 0 },
		{ (n.x * n.z * c1) + (n.y * s), (n.y * n.z * c1) - (n.x * s), (n.z * n.z * c1) + c,         0 },
		{ 0,                            0,                            0,                            1 }
	}};
	
	return matrix;
}

//******************************************************************************
// glTranslatef 代替関数
//******************************************************************************
matrix_float4x4 OGLTransMatrix::_glTranslatef(
		OGLfloat x,		//移動ベクトル
		OGLfloat y,		//
		OGLfloat z		//
	)
{
	//Metal,OpenGLは列優先(Column major order)
	matrix_float4x4 matrix = {{
		{ 1,   0,  0,  0 },
		{ 0,   1,  0,  0 },
		{ 0,   0,  1,  0 },
		{ x,   y,  z,  1 }
	}};
	
	return matrix;
}

//******************************************************************************
// glScalef 代替関数
//******************************************************************************
matrix_float4x4 OGLTransMatrix::_glScalef(
		OGLfloat x,		//スケールファクター
		OGLfloat y,		//
		OGLfloat z		//
	)
{
	//Metal,OpenGLは列優先(Column major order)
	matrix_float4x4 matrix = {{
		{ x,   0,  0,  0 },
		{ 0,   y,  0,  0 },
		{ 0,   0,  z,  0 },
		{ 0,   0,  0,  1 }
	}};
	
	return matrix;
}

