//******************************************************************************
//
// MIDITrail / MTColorPaletteCfgDlg
//
// カラーパレット設定ダイアログ
//
// Copyright (C) 2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTColorPaletteCfgDlg.h"
#import "MTColorParamExportDlg.h"
#import "MTColorParamImportDlg.h"


//##############################################################################
// カラーパレット設定ダイアログクラス
//##############################################################################
@implementation MTColorPaletteCfgDlg

//******************************************************************************
// 生成
//******************************************************************************
- (id)init
{
	//パレット初期化
	m_ColorPalette.Initialize();
	m_DefaultColorPalette.Initialize();
	m_ColorPaletteNo = 0;
	
	//Nibファイルを指定してウィンドウコントローラを生成
	return [super initWithWindowNibName:@"ColorPaletteCfgDlg"];
}

//******************************************************************************
// 破棄
//******************************************************************************
- (void)dealloc
{
	[super dealloc];
}

//******************************************************************************
// ウィンドウ読み込み完了
//******************************************************************************
- (void)windowDidLoad
{
	int result = 0;
	
	//ウィンドウ表示項目初期化
	//  モーダル終了後に再度モーダル表示してもwindowDidLoadは呼び出されない
	result = [self initDlg];
	if (result != 0) goto EXIT;
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
}

//******************************************************************************
// モーダルウィンドウ表示
//******************************************************************************
- (void)showModalWindow
{
	m_isChanged = NO;
	
	//モーダルウィンドウ表示
	[NSApp runModalForWindow:[self window]];
	
	//モーダル表示終了後はウィンドウを非表示にする
	[[self window] orderOut:self];
}

//******************************************************************************
// OKボタン押下
//******************************************************************************
- (IBAction)onOK:(NSButton *)sender
{
	int result = 0;
	
	//カラーウェルの設定をパレットに反映
	result = [self updatePalette];
	if (result != 0) goto EXIT;
	
	//カラーパネルを閉じる
	[[NSColorPanel sharedColorPanel] close];
	
	//ダイアログを閉じる
	m_isChanged = YES;
	[NSApp stopModal];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// キャンセルボタン押下
//******************************************************************************
- (IBAction)onCancel:(NSButton *)sender
{
	//カラーパネルを閉じる
	[[NSColorPanel sharedColorPanel] close];
	
	//モーダル表示終了
	[NSApp stopModal];
}

//******************************************************************************
// クローズボタン押下
//******************************************************************************
- (void)windowWillClose:(NSNotification*)aNotification
{
	//カラーパネルを閉じる
	[[NSColorPanel sharedColorPanel] close];
	
	//モーダル表示終了
	[NSApp stopModal];
}

//******************************************************************************
// グラデーションツール：Set Colors ボタン押下
//******************************************************************************
- (IBAction)onBtnSetGradationColors:(NSButton *)sender
{
	unsigned int chNoStart = 0;
	unsigned int chNoEnd = 0;
	OGLCOLOR colorStart;
	OGLCOLOR colorEnd;
	NSColor* wellColor = nil;
	
	//選択されたチャンネル番号(Start/End)
	chNoStart = (unsigned int)[m_pPopUpBtnStart indexOfSelectedItem];
	chNoEnd = (unsigned int)[m_pPopUpBtnEnd indexOfSelectedItem];
	
	//選択された色(Start/End)
	wellColor = m_pColorWell_Start.color;
	colorStart = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	wellColor = m_pColorWell_End.color;
	colorEnd = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	
	//グラデーションカラー設定
	[self setGradationColorWithChNoStart:chNoStart chNoEnd:chNoEnd colorStart:colorStart colorEnd:colorEnd];
	
	return;
}

//******************************************************************************
// パラメータセットアップツール：Set Default Colors ボタン押下
//******************************************************************************
- (IBAction)onBtnSetDefaultColors:(NSButton *)sender
{
	//カラーウェル設定
	[self initColorWellsWithPalette:&m_DefaultColorPalette];
}

//******************************************************************************
// パラメータセットアップツール：Export Color Parameters ボタン押下
//******************************************************************************
- (IBAction)onBtnExportColorParameters:(NSButton *)sender
{
	int result = 0;
	MTColorParamExportDlg* pColorParamExportDlg = nil;
	NSString* pParamString = nil;
	
	//ウィンドウ生成
	pColorParamExportDlg = [[MTColorParamExportDlg alloc] init];
	if (pColorParamExportDlg == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//ダイアログにパラメータ文字列を設定
	pParamString = [self makeColorParamForExport];
	[pColorParamExportDlg setParamString:pParamString];
	
	//モーダルウィンドウ表示
	[pColorParamExportDlg showModalWindow];
	
EXIT:;
	[pColorParamExportDlg release];
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// パラメータセットアップツール：Import Color Parameters ボタン押下
//******************************************************************************
- (IBAction)onBtnImportColorParameters:(NSButton *)sender
{
	int result = 0;
	MTColorParamImportDlg* pColorParamImportDlg = nil;
	NSString* pParamString = nil;
	
	//ウィンドウ生成
	pColorParamImportDlg = [[MTColorParamImportDlg alloc] init];
	if (pColorParamImportDlg == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//モーダルウィンドウ表示
	[pColorParamImportDlg showModalWindow];
	
	////ダイアログに入力されたパラメータ文字列のインポート処理
	if ([pColorParamImportDlg isExecImport]) {
		pParamString = [pColorParamImportDlg getParamString];
		result = [self importColorParam:pParamString];
		if (result != 0) goto EXIT;
	}
	
EXIT:;
	[pColorParamImportDlg release];
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// カラーウェル更新通知 Ch.1-16
//******************************************************************************
- (IBAction)onColorWell_1:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_1 colorWell:m_pColorWell_1];
}

- (IBAction)onColorWell_2:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_2 colorWell:m_pColorWell_2];
}

- (IBAction)onColorWell_3:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_3 colorWell:m_pColorWell_3];
}

- (IBAction)onColorWell_4:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_4 colorWell:m_pColorWell_4];
}

- (IBAction)onColorWell_5:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_5 colorWell:m_pColorWell_5];
}

- (IBAction)onColorWell_6:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_6 colorWell:m_pColorWell_6];
}

- (IBAction)onColorWell_7:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_7 colorWell:m_pColorWell_7];
}

- (IBAction)onColorWell_8:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_8 colorWell:m_pColorWell_8];
}

- (IBAction)onColorWell_9:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_9 colorWell:m_pColorWell_9];
}

- (IBAction)onColorWell_10:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_10 colorWell:m_pColorWell_10];
}

- (IBAction)onColorWell_11:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_11 colorWell:m_pColorWell_11];
}

- (IBAction)onColorWell_12:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_12 colorWell:m_pColorWell_12];
}

- (IBAction)onColorWell_13:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_13 colorWell:m_pColorWell_13];
}

- (IBAction)onColorWell_14:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_14 colorWell:m_pColorWell_14];
}

- (IBAction)onColorWell_15:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_15 colorWell:m_pColorWell_15];
}

- (IBAction)onColorWell_16:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_16 colorWell:m_pColorWell_16];
}

//******************************************************************************
// カラーウェル更新通知 背景
//******************************************************************************
- (IBAction)onColorWell_BG:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_BG colorWell:m_pColorWell_BG];
}

//******************************************************************************
// カラーウェル更新通知 グリッドライン
//******************************************************************************
- (IBAction)onColorWell_GL:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_GL colorWell:m_pColorWell_GL];
}

//******************************************************************************
// カラーウェル更新通知 カウンター
//******************************************************************************
- (IBAction)onColorWell_CT:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_CT colorWell:m_pColorWell_CT];
}

//******************************************************************************
// カラーウェル更新通知 Start/End
//******************************************************************************
- (IBAction)onColorWell_Start:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_Start colorWell:m_pColorWell_Start];
}

- (IBAction)onColorWell_End:(NSColorWell *)sender
{
	[self updateColorText:m_pColorText_End colorWell:m_pColorWell_End];
}

//******************************************************************************
// 変更確認
//******************************************************************************
- (BOOL)isCahnged
{
	return m_isChanged;
}

//******************************************************************************
// カラーパレット設定
//******************************************************************************
- (void)setColorPalette:(MTColorPalette*)pColorPalette
	defaultColorPalette:(MTColorPalette*)pDefaultColorPalette
		 colorPaletteNo:(unsigned int)colorPaletteNo
{
	m_ColorPalette.CopyFrom(pColorPalette);
	m_DefaultColorPalette.CopyFrom(pDefaultColorPalette);
	m_ColorPaletteNo = colorPaletteNo;
}

//******************************************************************************
// カラーパレット取得
//******************************************************************************
- (void)getColorPalette:(MTColorPalette*)pColorPalette
{
	pColorPalette->CopyFrom(&m_ColorPalette);
}

//******************************************************************************
// ダイアログ初期化
//******************************************************************************
- (int)initDlg
{
	int result = 0;
	
	//カラーウェルリスト初期化
	[self initColorWellList];
	
	//カラーウェル設定
	[self initColorWellsWithPalette:&m_ColorPalette];
	[self initColorWellsStartEnd];
	
	//ポップアップボタン初期化
	[self initPopUpButton:m_pPopUpBtnStart selectedIndex:0];
	[self initPopUpButton:m_pPopUpBtnEnd selectedIndex:15];
	
	//ウィンドウタイトル
	self.window.title = [NSString stringWithFormat:@"Color Palette %u", m_ColorPaletteNo];
	
EXIT:;
	return result;
}

//******************************************************************************
// カラーウェルリスト初期化
//******************************************************************************
- (void)initColorWellList
{
	//カラーウェルリスト
	m_pColorWellList[0] = m_pColorWell_1;
	m_pColorWellList[1] = m_pColorWell_2;
	m_pColorWellList[2] = m_pColorWell_3;
	m_pColorWellList[3] = m_pColorWell_4;
	m_pColorWellList[4] = m_pColorWell_5;
	m_pColorWellList[5] = m_pColorWell_6;
	m_pColorWellList[6] = m_pColorWell_7;
	m_pColorWellList[7] = m_pColorWell_8;
	m_pColorWellList[8] = m_pColorWell_9;
	m_pColorWellList[9] = m_pColorWell_10;
	m_pColorWellList[10] = m_pColorWell_11;
	m_pColorWellList[11] = m_pColorWell_12;
	m_pColorWellList[12] = m_pColorWell_13;
	m_pColorWellList[13] = m_pColorWell_14;
	m_pColorWellList[14] = m_pColorWell_15;
	m_pColorWellList[15] = m_pColorWell_16;
	m_pColorWellList[16] = m_pColorWell_BG;
	m_pColorWellList[17] = m_pColorWell_GL;
	m_pColorWellList[18] = m_pColorWell_CT;
	//カラーテキストリスト
	m_pColorTextList[0] = m_pColorText_1;
	m_pColorTextList[1] = m_pColorText_2;
	m_pColorTextList[2] = m_pColorText_3;
	m_pColorTextList[3] = m_pColorText_4;
	m_pColorTextList[4] = m_pColorText_5;
	m_pColorTextList[5] = m_pColorText_6;
	m_pColorTextList[6] = m_pColorText_7;
	m_pColorTextList[7] = m_pColorText_8;
	m_pColorTextList[8] = m_pColorText_9;
	m_pColorTextList[9] = m_pColorText_10;
	m_pColorTextList[10] = m_pColorText_11;
	m_pColorTextList[11] = m_pColorText_12;
	m_pColorTextList[12] = m_pColorText_13;
	m_pColorTextList[13] = m_pColorText_14;
	m_pColorTextList[14] = m_pColorText_15;
	m_pColorTextList[15] = m_pColorText_16;
	m_pColorTextList[16] = m_pColorText_BG;
	m_pColorTextList[17] = m_pColorText_GL;
	m_pColorTextList[18] = m_pColorText_CT;
}

//******************************************************************************
// カラーウェル設定：パレット
//******************************************************************************
- (int)initColorWellsWithPalette:(MTColorPalette*)pColorPalette
{
	int result = 0;
	unsigned int chNo = 0;
	OGLCOLOR color;
	
	//カラーウェル設定 Ch.1-16
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		//パレットから色を取得
		result = pColorPalette->GetChColor(chNo, &color);
		if (result != 0) goto EXIT;
		//カラーウェル設定
		m_pColorWellList[chNo].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
		//カラーテキスト設定
		[self updateColorText:m_pColorTextList[chNo] colorWell:m_pColorWellList[chNo]];
	}
	
	//カラーウェル設定 背景
	pColorPalette->GetBackgroundColor(&color);
	m_pColorWellList[SM_MAX_CH_NUM + 0].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
	//カラーテキスト設定
	[self updateColorText:m_pColorTextList[SM_MAX_CH_NUM + 0] colorWell:m_pColorWellList[SM_MAX_CH_NUM + 0]];
	
	//カラーウェル設定 グリッドライン
	pColorPalette->GetGridLineColor(&color);
	m_pColorWellList[SM_MAX_CH_NUM + 1].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
	//カラーテキスト設定
	[self updateColorText:m_pColorTextList[SM_MAX_CH_NUM + 1] colorWell:m_pColorWellList[SM_MAX_CH_NUM + 1]];
	
	//カラーウェル設定 カウンター
	pColorPalette->GetCounterColor(&color);
	m_pColorWellList[SM_MAX_CH_NUM + 2].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
	//カラーテキスト設定
	[self updateColorText:m_pColorTextList[SM_MAX_CH_NUM + 2] colorWell:m_pColorWellList[SM_MAX_CH_NUM + 2]];
	
EXIT:;
	return result;
}

//******************************************************************************
// カラーウェル設定：Start/End
//******************************************************************************
- (int)initColorWellsStartEnd
{
	int result = 0;
	OGLCOLOR color;
	
	//カラーウェル設定 Start/End
	color = OGLCOLOR(1.0f, 1.0f, 1.0f, 1.0f);
	m_pColorWell_Start.color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
	m_pColorWell_End.color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
	//カラーテキスト設定
	[self updateColorText:m_pColorText_Start colorWell:m_pColorWell_Start];
	[self updateColorText:m_pColorText_End colorWell:m_pColorWell_End];
	
EXIT:;
	return result;
}
	
//******************************************************************************
// ポップアップボタン初期化
//******************************************************************************
- (void)initPopUpButton:(NSPopUpButton*)pPopUpButton selectedIndex:(NSInteger)selectedIndex
{
	unsigned int index = 0;
	NSInteger itemIndex = 0;
	NSMenu* pMenu = nil;
	NSMenuItem* pItem = nil;
	NSString* pTitle = nil;
	
	//メニュー生成
	pMenu = [[[NSMenu alloc] initWithTitle:@"Ch.No"] autorelease];
	
	//項目の生成とメニュー登録
	for (itemIndex = 0; itemIndex < SM_MAX_CH_NUM; itemIndex++) {
		pTitle = [NSString stringWithFormat:@"Ch.%d", ((int)itemIndex)+1];
		pItem = [[NSMenuItem alloc] initWithTitle:pTitle action:nil keyEquivalent:@""];
		[pItem setTag:index];
		[pMenu addItem:pItem];
		[pItem release];
	}
	
	//メニューをポップアップボタンに登録
	[pPopUpButton setMenu:pMenu];
	//	[pMenu release];
	//リリースするとdeallocで落ちる
	
	//選択状態設定
	[pPopUpButton selectItemAtIndex:selectedIndex];
	
	//同期
	[pPopUpButton synchronizeTitleAndSelectedItem];
	
	return;
}

//******************************************************************************
// カラーテキスト更新
//******************************************************************************
- (void)updateColorText:(NSTextField*)pTextField colorWell:(NSColorWell*)pColorWell
{
	NSColor* wellColor;
	OGLCOLOR color;
	
	//更新された色をラベルに反映
	wellColor = pColorWell.color;
	color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	[pTextField setStringValue:OGLColorUtil::MakeHexRGBAFromColor(color)];
	
	return;
}

//******************************************************************************
// カラーパレット更新
//******************************************************************************
- (int)updatePalette
{
	int result = 0;
	unsigned int chNo = 0;
	NSColor* wellColor;
	OGLCOLOR color;
	
	//カラーウェル Ch.1-16 の設定をパレットに反映
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		wellColor = m_pColorWellList[chNo].color;
		color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
		result = m_ColorPalette.SetChColor(chNo, color);
		if (result != 0) goto EXIT;
	}
	
	//カラーウェル 背景 の設定をパレットに反映
	wellColor = m_pColorWellList[SM_MAX_CH_NUM + 0].color;
	color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	m_ColorPalette.SetBackgroundColor(color);
	
	//カラーウェル グリッドライン の設定をパレットに反映
	wellColor = m_pColorWellList[SM_MAX_CH_NUM + 1].color;
	color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	m_ColorPalette.SetGridLineColor(color);
	
	//カラーウェル カウンター の設定をパレットに反映
	wellColor = m_pColorWellList[SM_MAX_CH_NUM + 2].color;
	color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	m_ColorPalette.SetCounterColor(color);
	
EXIT:;
	return result;
}

//******************************************************************************
// グラデーションカラー設定
//******************************************************************************
- (void)setGradationColorWithChNoStart:(unsigned int)chNoStart
							   chNoEnd:(unsigned int)chNoEnd
							colorStart:(OGLCOLOR)colorStart
							  colorEnd:(OGLCOLOR)colorEnd
{
	unsigned int chNo = 0;
	OGLCOLOR color;
	float ratio = 0.0f;
	
	if ((chNoStart >= SM_MAX_CH_NUM) || (chNoEnd >= SM_MAX_CH_NUM)) {
		//何もしない
	}
	
	if (chNoStart == chNoEnd) {
		//何もしない
	}
	else if (chNoStart < chNoEnd) {
		for (chNo = chNoStart; chNo <= chNoEnd; chNo++) {
			ratio = (float)(chNo - chNoStart) / (float)(chNoEnd - chNoStart);
			color = OGLCOLOR((colorEnd.r - colorStart.r) * ratio + colorStart.r,
							 (colorEnd.g - colorStart.g) * ratio + colorStart.g,
							 (colorEnd.b - colorStart.b) * ratio + colorStart.b,
							 1.0f);
			//カラーウェル設定
			m_pColorWellList[chNo].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
			//カラーテキスト設定
			[self updateColorText:m_pColorTextList[chNo] colorWell:m_pColorWellList[chNo]];
		}
	}
	else {
		for (chNo = chNoEnd; chNo <= chNoStart; chNo++) {
			ratio = (float)(chNo - chNoEnd) / (float)(chNoStart - chNoEnd);
			color = OGLCOLOR((colorStart.r - colorEnd.r) * ratio + colorEnd.r,
							 (colorStart.g - colorEnd.g) * ratio + colorEnd.g,
							 (colorStart.b - colorEnd.b) * ratio + colorEnd.b,
							 1.0f);
			//カラーウェル設定
			m_pColorWellList[chNo].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
			//カラーテキスト設定
			[self updateColorText:m_pColorTextList[chNo] colorWell:m_pColorWellList[chNo]];
		}
	}
	
	return;
}

//******************************************************************************
// 出力用パラメータ文字列生成
//******************************************************************************
- (NSString*)makeColorParamForExport
{
	unsigned int chNo = 0;
	NSColor* wellColor;
	OGLCOLOR color;
	NSMutableString* pLines = nil;
	NSString* pLine = nil;
	NSString* pExportString = nil;
	
	pLines = [[NSMutableString alloc] initWithString:@""];
	
	//パラメータ文字列生成：Ch.1-16
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		wellColor = m_pColorWellList[chNo].color;
		color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
		pLine = [NSString stringWithFormat:@"Ch-%02u-NoteRGBA=%@\n", chNo+1, OGLColorUtil::MakeHexRGBAFromColor(color)];
		[pLines appendString:pLine];
	}
	
	//パラメータ文字列生成：背景
	wellColor = m_pColorWellList[SM_MAX_CH_NUM + 0].color;
	color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	pLine = [NSString stringWithFormat:@"BackGroundRGBA=%@\n", OGLColorUtil::MakeHexRGBAFromColor(color)];
	[pLines appendString:pLine];
	
	//パラメータ文字列生成：グリッドライン
	wellColor = m_pColorWellList[SM_MAX_CH_NUM + 1].color;
	color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	pLine = [NSString stringWithFormat:@"GridLineRGBA=%@\n", OGLColorUtil::MakeHexRGBAFromColor(color)];
	[pLines appendString:pLine];
	
	//パラメータ文字列生成：カウンター
	wellColor = m_pColorWellList[SM_MAX_CH_NUM + 2].color;
	color = OGLCOLOR(wellColor.redComponent, wellColor.greenComponent, wellColor.blueComponent, wellColor.alphaComponent);
	pLine = [NSString stringWithFormat:@"CounterRGBA=%@\n", OGLColorUtil::MakeHexRGBAFromColor(color)];
	[pLines appendString:pLine];
	
	pExportString = [NSString stringWithString:pLines];
	
	[pLines release];
	return pExportString;
}

//******************************************************************************
// パラメータ入力処理
//******************************************************************************
- (int)importColorParam:(NSString*)pString
{
	int result = 0;
	YNConfFile* pConfFile = nil;
	unsigned int chNo = 0;
	NSString* initext = nil;
	NSString* pKey = nil;
	NSString* pHexColor = nil;
	OGLCOLOR color;
	
	//設定ファイル初期化
	pConfFile = [[YNConfFile alloc] init];
	if (pConfFile == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//テキスト読み込み
	initext = [NSString stringWithFormat:@"[Color]\n%@", pString];
	result = [pConfFile loadText:initext];
	if (result != 0) goto EXIT;
	
	//セクション設定
	[pConfFile setCurSection:@"Color"];
	
	//カラーウェル反映：Ch.1-16（パラメータが取得できた場合のみ）
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		pKey = [NSString stringWithFormat:@"Ch-%02u-NoteRGBA", chNo+1];
		pHexColor = [pConfFile strValueForKey:pKey defaultValue:@"NOT_FOUND"];
		if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
			//カラーウェル設定
			color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
			m_pColorWellList[chNo].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
			//カラーテキスト設定
			[self updateColorText:m_pColorTextList[chNo] colorWell:m_pColorWellList[chNo]];
		}
	}
	
	//カラーウェル反映：背景（パラメータが取得できた場合のみ）
	pHexColor = [pConfFile strValueForKey:@"BackGroundRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		//カラーウェル設定
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_pColorWellList[SM_MAX_CH_NUM + 0].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
		//カラーテキスト設定
		[self updateColorText:m_pColorTextList[SM_MAX_CH_NUM + 0] colorWell:m_pColorWellList[SM_MAX_CH_NUM + 0]];
	}
	
	//カラーウェル反映：グリッドライン（パラメータが取得できた場合のみ）
	pHexColor = [pConfFile strValueForKey:@"GridLineRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		//カラーウェル設定
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_pColorWellList[SM_MAX_CH_NUM + 1].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
		//カラーテキスト設定
		[self updateColorText:m_pColorTextList[SM_MAX_CH_NUM + 1] colorWell:m_pColorWellList[SM_MAX_CH_NUM + 1]];
	}
	
	//カラーウェル反映：カウンター（パラメータが取得できた場合のみ）
	pHexColor = [pConfFile strValueForKey:@"CounterRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		//カラーウェル設定
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_pColorWellList[SM_MAX_CH_NUM + 2].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
		//カラーテキスト設定
		[self updateColorText:m_pColorTextList[SM_MAX_CH_NUM + 2] colorWell:m_pColorWellList[SM_MAX_CH_NUM + 2]];
	}
	
	//カラーウェル反映：カウンター（パラメータが取得できた場合のみ）
	// 互換性を保つため ini ファイルに定義されている"CaptionRGBA"をインポート可能とする
	pHexColor = [pConfFile strValueForKey:@"CaptionRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		//カラーウェル設定
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_pColorWellList[SM_MAX_CH_NUM + 2].color = [NSColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
		//カラーテキスト設定
		[self updateColorText:m_pColorTextList[SM_MAX_CH_NUM + 2] colorWell:m_pColorWellList[SM_MAX_CH_NUM + 2]];
	}
	
EXIT:;
	[pConfFile release];
	return result;
}

@end

