//******************************************************************************
//
// MIDITrail / MTFileOpenPanel
//
// ファイル選択パネルクラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "MTFileOpenPanel.h"
#import "YNBaseLib.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTFileOpenPanel::MTFileOpenPanel(void)
{
	m_pPanel = nil;
	m_pAccessoryView = nil;
	m_pLabel = nil;
	m_pEncodingPopUp = nil;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTFileOpenPanel::~MTFileOpenPanel(void)
{
	_Release();
}

//******************************************************************************
// メンバオブジェクト解放
//******************************************************************************
void MTFileOpenPanel::_Release()
{
	[m_pPanel release];
	[m_pAccessoryView release];
	[m_pLabel release];
	[m_pEncodingPopUp release];
	
	m_pPanel = nil;
	m_pAccessoryView = nil;
	m_pLabel = nil;
	m_pEncodingPopUp = nil;
}

//******************************************************************************
// 初期化
//******************************************************************************
int MTFileOpenPanel::Initialize()
{
	int result = 0;
	
	//メンバオブジェクト解放
	_Release();
	
	//エンコーディングリスト初期化
	result = m_EncodingList.Initialize();
	if (result != 0) goto EXIT;
	
	//パネル生成
	m_pPanel = [NSOpenPanel openPanel];
	if (m_pPanel == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	[m_pPanel retain];
	
	//ファイルとフォルダの選択可否
	m_pPanel.canChooseFiles = YES;
	m_pPanel.canChooseDirectories = NO;
	
	//アクセサリビュー生成
	result = _CreateAccessoryView();
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// エンコーディング選択
//******************************************************************************
int MTFileOpenPanel::SetSelectedEncodingName(
		NSString* pEncodingName
	)
{
	int result = 0;
	unsigned int index = 0;
	NSInteger selectedIndex = 0;
	NSStringEncoding targetId = 0;
	BOOL isFound = NO;
	
	//エンコーディングリストから指定されたエンコーディング名を検索
	for (index = 0; index < m_EncodingList.GetSize(); index++) {
		if ([pEncodingName isEqualToString:m_EncodingList.GetEncodingName(index)]) {
			selectedIndex = index;
			isFound = YES;
			break;
		}
	}
	
	//指定されたエンコーディング名が見つからなかった場合
	//必ず存在するはずのエンコーディングを選択状態とする
	if (!isFound) {
		targetId = GetDefaultEncodingId();
		for (index = 0; index < m_EncodingList.GetSize(); index++) {
			if (m_EncodingList.GetEncodingId(index) == targetId) {
				selectedIndex = index;
				isFound = YES;
				break;
			}
		}
	}
	
	//それでも見つからない場合はリスト先頭を選択状態とする
	if (!isFound) {
		selectedIndex = 0;
	}
	
	//エンコーディング選択ポップアップボタンの選択状態を設定
	[m_pEncodingPopUp selectItemAtIndex:selectedIndex];	
	
	return result;
}

//******************************************************************************
// ファイル選択パネル表示
//******************************************************************************
NSInteger MTFileOpenPanel::showModalWindowForSelectingFile()
{
	NSInteger btn = 0;
	
	//ファイルのみ選択可能
	m_pPanel.canChooseFiles = YES;
	m_pPanel.canChooseDirectories = NO;
	
	//ファイル選択パネル表示
	btn = [m_pPanel runModal];
	
	//ファイル選択時の処理
	if (btn == NSModalResponseOK) {
		//ここでは何もしない
	}
	
	return btn;
}

//******************************************************************************
// フォルダ選択パネル表示
//******************************************************************************
NSInteger MTFileOpenPanel::showModalWindowForSelectingFolder()
{
	NSInteger btn = 0;
	
	//フォルダのみ選択可能
	m_pPanel.canChooseFiles = NO;
	m_pPanel.canChooseDirectories = YES;
	
	//フォルダ選択パネル表示
	btn = [m_pPanel runModal];
	
	//フォルダ選択時の処理
	if (btn == NSModalResponseOK) {
		//ここでは何もしない
	}
	
	return btn;
}

//******************************************************************************
// 選択ファイルパス取得
//******************************************************************************
NSString* MTFileOpenPanel::GetSelectedFilePath()
{
	return [[m_pPanel URL] path];
}

//******************************************************************************
// 選択エンコーディング名取得
//******************************************************************************
NSString* MTFileOpenPanel::GetSelectedEncodingName()
{
	return [[m_pEncodingPopUp selectedItem] title];
}

//******************************************************************************
// 選択エンコーディングID取得
//******************************************************************************
NSStringEncoding MTFileOpenPanel::GetSelectedEncodingId()
{
	return [[m_pEncodingPopUp selectedItem] tag];
}

//******************************************************************************
// アクセサリビュー生成
//******************************************************************************
int MTFileOpenPanel::_CreateAccessoryView()
{
	int result = 0;
	NSMenu* pMenu = nil;
	NSMenuItem* pItem = nil;
	unsigned int index = 0;
	
	//  アクセサリビュー
	//  +---------------------------------+
	//  | +--------+ +------------------+ |
	//  | | ラベル  | | ポップアップボタン  | |
	//  | +--------+ +------------------+ |
	//  +---------------------------------+
	
	//アクセサリビュー生成
	m_pAccessoryView = [[NSView alloc] initWithFrame:NSMakeRect(0, 0, 406, 30)];
	
	//ラベル生成
	m_pLabel = [[NSTextField alloc] initWithFrame:NSMakeRect(2,8,120,17)];
	[m_pLabel setEditable:NO];
	[m_pLabel setBezeled:NO];
	[m_pLabel setBackgroundColor:[NSColor controlColor]];
	[m_pLabel setStringValue:@"Default encoding"];
	
	//ポップアップボタン生成
	m_pEncodingPopUp = [[NSPopUpButton alloc] initWithFrame:NSMakeRect(124, 2, 280, 26) pullsDown:NO];
	
	//メニュー生成
	pMenu = [[[NSMenu alloc] initWithTitle:@"Encoding"] autorelease];
	
	//エンコーディング一覧をメニューに登録
	for (index = 0; index < m_EncodingList.GetSize(); index++) {
		//項目生成
		pItem = [[NSMenuItem alloc] initWithTitle:(m_EncodingList.GetEncodingName(index))
										   action:nil
									keyEquivalent:@""];
		[pItem setTag:(m_EncodingList.GetEncodingId(index))];
		//メニューに項目を登録
		[pMenu addItem:pItem];
		[pItem release];
	}
	
	//ポップアップボタンにメニューを登録
	[m_pEncodingPopUp setMenu:pMenu];
	
	//エンコーディング選択ポップアップボタンの選択状態を設定
	[m_pEncodingPopUp selectItemAtIndex:0];	
	
	//アクセサリビューにラベルとポップアップボタンを登録
	[m_pAccessoryView addSubview:m_pLabel];
	[m_pAccessoryView addSubview:m_pEncodingPopUp];
	
	//ファイル選択パネルにアクセサリビューを登録
	[m_pPanel setAccessoryView:m_pAccessoryView];
	
EXIT:;
	return result;
}

//******************************************************************************
// デフォルトエンコーディング名取得
//******************************************************************************
NSString* MTFileOpenPanel::GetDefaultEncodingName()
{
	unsigned int index = 0;
	NSStringEncoding defaultEncodingId = 0;
	NSString* pDefaultEncodingName = @"";
	
	//デフォルトエンコーディングID取得
	defaultEncodingId = GetDefaultEncodingId();
	
	//エンコーディングIDを一覧から検索
	for (index = 0; index < m_EncodingList.GetSize(); index++) {
		if (defaultEncodingId == m_EncodingList.GetEncodingId(index)) {
			pDefaultEncodingName = m_EncodingList.GetEncodingName(index);
			break;
		}
	}
	
	//ロケールが日本なら ShiftJIS  @"Japanese (Windows, DOS)"
	//それ以外は Windows-1252    @"Western (Windows Latin 1)"
	
	return pDefaultEncodingName;
}

//******************************************************************************
// デフォルトエンコーディングID取得
//******************************************************************************
NSStringEncoding MTFileOpenPanel::GetDefaultEncodingId()
{
	NSStringEncoding encodingId = 0;
	NSArray* pLanguages = nil;
	NSString* pLanguage = nil;
	
	//選択されている言語を取得
	pLanguages = [NSLocale preferredLanguages];
	pLanguage = [pLanguages objectAtIndex:0];
	
	//言語によってエンコーディングを切り替える
	//iOS9以降はpreferredLanguagesの返却値が"ja"から"ja-JP"に変わっている
	if ([pLanguage hasPrefix:@"ja"]) {
		//日本語の場合 ShiftJIS
		encodingId = NSShiftJISStringEncoding;
	}
	else {
		//それ以外は Windows-1252 （ISO-8859-1の方が適切？）
		encodingId = NSWindowsCP1252StringEncoding;
	}
	
	return encodingId;
}


