//******************************************************************************
//
// OGL Utility / OGLDirLight
//
// ディレクショナルライトクラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "OGLH.h"
#import "OGLDirLight.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
OGLDirLight::OGLDirLight(void)
{
	memset(m_Diffuse, 0, sizeof(OGLCOLOR));
	memset(m_Specular, 0, sizeof(OGLCOLOR));
	memset(m_Ambient, 0, sizeof(OGLCOLOR));
	memset(m_Direction, 0, sizeof(OGLVECTOR3));
}

//******************************************************************************
// デストラクタ
//******************************************************************************
OGLDirLight::~OGLDirLight(void)
{
}

//******************************************************************************
// 初期化
//******************************************************************************
int OGLDirLight::Initialize()
{
	int result = 0;
	
	//拡散光
	m_Diffuse.r = 1.0f;
	m_Diffuse.g = 1.0f;
	m_Diffuse.b = 1.0f;
	m_Diffuse.a = 1.0f;
	
	//スペキュラ光
	m_Specular.r = 1.0f;
	m_Specular.g = 1.0f;
	m_Specular.b = 1.0f;
	m_Specular.a = 1.0f;
	
	//環境光
	m_Ambient.r = 0.1f;
	m_Ambient.g = 0.1f;
	m_Ambient.b = 0.1f;
	m_Ambient.a = 1.0f;
	
	//方向：ベクトルは正規化されていなければならない
	m_Direction = OGLVECTOR3(0.0f, 0.0f, 1.0f);
	
	return result;
}

//******************************************************************************
// ライト色設定
//******************************************************************************
void OGLDirLight::SetColor(
		OGLCOLOR diffuse,
		OGLCOLOR specular,
		OGLCOLOR ambient
	)
{
	m_Diffuse  = diffuse;
	m_Specular = specular;
	m_Ambient  = ambient;
}

//******************************************************************************
// ライト方向設定
//******************************************************************************
void OGLDirLight::SetDirection(
		OGLVECTOR3 dirVector
	)
{
	OGLVECTOR3 normalizedVector;
	
	//ベクトル正規化
	normalizedVector = OGLH::Vec3Normalize(dirVector);
	
	//ライト情報構造体に登録
	m_Direction = normalizedVector;
}

//******************************************************************************
// デバイス登録：インデックス0
//******************************************************************************
int OGLDirLight::SetDevice(
		OGLDevice* pOGLDevice,
		bool isLightON
	)
{
	return SetDevice(pOGLDevice, 0, isLightON);
}

//******************************************************************************
// デバイス登録：インデックス指定
//******************************************************************************
int OGLDirLight::SetDevice(
		OGLDevice* pOGLDevice,
		unsigned int index,
		bool isLightON
	)
{
	int result = 0;
	OGLLight light;
	
	//ライト有効
	light.enable = 0;
	if (isLightON) {
		light.enable = 1;
	}
	
	//ライト方向
	//  DirectXは原点を光源としてその方向をベクトルで表現するが
	//  OpenGLは光源の位置をベクトルで示して原点に光を向ける
	//  ここではDirectXの形式をOpenGLの形式に変換する
	float3 direction = {-(m_Direction.x), -(m_Direction.y), -(m_Direction.z)};
	light.direction = direction;
	
	//拡散光
	float3 diffuse = {m_Diffuse.r, m_Diffuse.g, m_Diffuse.b};
	light.diffuseColor = diffuse;
	
	//環境光
	float3 ambient = {m_Ambient.r, m_Ambient.g, m_Ambient.b};
	light.ambientColor= ambient;
	
	//スペキュラ光
	float3 specular = {m_Specular.r, m_Specular.g, m_Specular.b};
	light.specularColor = specular;
	
	//ライト登録
	switch (index) {
		case 0:
			pOGLDevice->SetLight0(light);
			break;
		case 1:
			pOGLDevice->SetLight1(light);
			break;
		default:
			result = YN_SET_ERR(@"Program error.", index, 0);
			goto EXIT;
	}
	
EXIT:;
	return result;
}


