/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2008-2009  Kouhei Sutou <kou@clear-code.com>
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "../../config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <stdlib.h>

#include "milter-agent.h"
#include "milter-marshalers.h"
#include "milter-enum-types.h"
#include "milter-utils.h"
#include "milter-logger.h"

#define MILTER_AGENT_GET_PRIVATE(obj)                          \
    (G_TYPE_INSTANCE_GET_PRIVATE((obj),                        \
                                 MILTER_TYPE_AGENT,            \
                                 MilterAgentPrivate))

typedef struct _MilterAgentPrivate	MilterAgentPrivate;
struct _MilterAgentPrivate
{
    MilterEncoder *encoder;
    MilterDecoder *decoder;
    MilterReader *reader;
    MilterWriter *writer;
    gboolean finished;
    guint tag;
};

enum
{
    PROP_0,
    PROP_READER,
    PROP_DECODER,
    PROP_TAG
};

static GStaticMutex auto_tag_mutex = G_STATIC_MUTEX_INIT;
static guint auto_tag = 0;

static void         finished           (MilterFinishedEmittable *emittable);

MILTER_IMPLEMENT_ERROR_EMITTABLE(error_emittable_init);
MILTER_IMPLEMENT_FINISHED_EMITTABLE_WITH_CODE(finished_emittable_init,
                                              iface->finished = finished)
G_DEFINE_ABSTRACT_TYPE_WITH_CODE(MilterAgent, milter_agent, G_TYPE_OBJECT,
    G_IMPLEMENT_INTERFACE(MILTER_TYPE_ERROR_EMITTABLE, error_emittable_init)
    G_IMPLEMENT_INTERFACE(MILTER_TYPE_FINISHED_EMITTABLE, finished_emittable_init))


static GObject *constructor  (GType                  type,
                              guint                  n_props,
                              GObjectConstructParam *props);

static void dispose        (GObject         *object);
static void set_property   (GObject         *object,
                            guint            prop_id,
                            const GValue    *value,
                            GParamSpec      *pspec);
static void get_property   (GObject         *object,
                            guint            prop_id,
                            GValue          *value,
                            GParamSpec      *pspec);

static void
milter_agent_class_init (MilterAgentClass *klass)
{
    GObjectClass *gobject_class;
    GParamSpec *spec;

    gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->constructor  = constructor;
    gobject_class->dispose      = dispose;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;

    klass->decoder_new = NULL;
    klass->encoder_new = NULL;

    spec = g_param_spec_object("reader",
                               "Reader",
                               "The reader of the agent",
                               MILTER_TYPE_READER,
                               G_PARAM_READWRITE);
    g_object_class_install_property(gobject_class, PROP_READER, spec);

    spec = g_param_spec_object("decoder",
                               "Decoder",
                               "The decoder of the agent",
                               MILTER_TYPE_DECODER,
                               G_PARAM_READABLE);
    g_object_class_install_property(gobject_class, PROP_DECODER, spec);

    spec = g_param_spec_uint("tag",
                             "Tag",
                             "The tag of the agent",
                             0, G_MAXUINT, 0,
                             G_PARAM_READABLE);
    g_object_class_install_property(gobject_class, PROP_TAG, spec);

    g_type_class_add_private(gobject_class, sizeof(MilterAgentPrivate));
}

static void
apply_tag (MilterAgentPrivate *priv)
{
    if (priv->encoder)
        milter_encoder_set_tag(priv->encoder, priv->tag);
    if (priv->decoder)
        milter_decoder_set_tag(priv->decoder, priv->tag);
    if (priv->writer)
        milter_writer_set_tag(priv->writer, priv->tag);
    if (priv->reader)
        milter_reader_set_tag(priv->reader, priv->tag);
}

static GObject *
constructor (GType type, guint n_props, GObjectConstructParam *props)
{
    GObject *object;
    MilterAgent *agent;
    GObjectClass *klass;
    MilterAgentClass *agent_class;
    MilterAgentPrivate *priv;

    klass = G_OBJECT_CLASS(milter_agent_parent_class);
    object = klass->constructor(type, n_props, props);

    priv = MILTER_AGENT_GET_PRIVATE(object);

    agent = MILTER_AGENT(object);
    agent_class = MILTER_AGENT_GET_CLASS(object);
    if (agent_class->decoder_new)
        priv->decoder = agent_class->decoder_new(agent);
    if (agent_class->encoder_new)
        priv->encoder = agent_class->encoder_new(agent);
    if (priv->tag == 0) {
        g_static_mutex_lock(&auto_tag_mutex);
        priv->tag = auto_tag++;
        if (priv->tag == 0)
            priv->tag = auto_tag++;
        g_static_mutex_unlock(&auto_tag_mutex);
    }
    apply_tag(priv);

    return object;
}

static void
milter_agent_init (MilterAgent *agent)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);
    priv->encoder = NULL;
    priv->decoder = NULL;
    priv->writer = NULL;
    priv->reader = NULL;
    priv->tag = 0;
    priv->finished = FALSE;
}

static void
dispose (GObject *object)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(object);

    if (priv->reader) {
        g_object_unref(priv->reader);
        priv->reader = NULL;
    }

    if (priv->writer) {
        g_object_unref(priv->writer);
        priv->writer = NULL;
    }

    if (priv->decoder) {
        g_object_unref(priv->decoder);
        priv->decoder = NULL;
    }

    if (priv->encoder) {
        g_object_unref(priv->encoder);
        priv->encoder = NULL;
    }

    G_OBJECT_CLASS(milter_agent_parent_class)->dispose(object);
}

static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(object);
    switch (prop_id) {
    case PROP_READER:
        milter_agent_set_reader(MILTER_AGENT(object), g_value_get_object(value));
        break;
    case PROP_TAG:
        milter_agent_set_tag(MILTER_AGENT(object), g_value_get_uint(value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject    *object,
              guint       prop_id,
              GValue     *value,
              GParamSpec *pspec)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(object);
    switch (prop_id) {
    case PROP_READER:
        g_value_set_object(value, priv->reader);
        break;
    case PROP_DECODER:
        g_value_set_object(value, priv->decoder);
        break;
    case PROP_TAG:
        g_value_set_uint(value, priv->tag);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
finished (MilterFinishedEmittable *emittable)
{
    MilterAgent *agent;
    MilterAgentPrivate *priv;

    agent = MILTER_AGENT(emittable);
    priv = MILTER_AGENT_GET_PRIVATE(agent);

    priv->finished = TRUE;
    if (priv->reader)
        milter_reader_shutdown(priv->reader);
    if (priv->writer)
        milter_writer_shutdown(priv->writer);
}

static void
cb_reader_flow (MilterReader *reader,
                const gchar *data, gsize data_size,
                gpointer user_data)
{
    MilterAgentPrivate *priv;
    GError *decoder_error = NULL;

    priv = MILTER_AGENT_GET_PRIVATE(user_data);

    milter_decoder_decode(priv->decoder, data, data_size, &decoder_error);

    if (decoder_error) {
        GError *error = NULL;

        milter_utils_set_error_with_sub_error(&error,
                                              MILTER_AGENT_ERROR,
                                              MILTER_AGENT_ERROR_DECODE_ERROR,
                                              decoder_error,
                                              "Decode error");
        milter_error("[%u] [agent][error][decode] %s",
                     priv->tag, error->message);
        milter_error_emittable_emit(MILTER_ERROR_EMITTABLE(user_data),
                                    error);
        g_error_free(error);
    }
}

static void
cb_reader_error (MilterReader *reader,
                 GError *reader_error,
                 gpointer user_data)
{
    GError *error = NULL;
    MilterAgent *agent = user_data;
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);
    milter_utils_set_error_with_sub_error(&error,
                                          MILTER_AGENT_ERROR,
                                          MILTER_AGENT_ERROR_IO_ERROR,
                                          g_error_copy(reader_error),
                                          "Input error");
    milter_error("[%u] [agent][error][reader] %s", priv->tag, error->message);
    milter_error_emittable_emit(MILTER_ERROR_EMITTABLE(user_data),
                                error);
    g_error_free(error);
}

static void
cb_reader_finished (MilterFinishedEmittable *emittable, gpointer user_data)
{
    MilterAgent *agent = user_data;
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);
    milter_agent_set_reader(agent, NULL);
    if (!priv->finished)
        milter_finished_emittable_emit(MILTER_FINISHED_EMITTABLE(agent));
}

void
milter_agent_set_reader (MilterAgent *agent, MilterReader *reader)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);

    if (priv->reader) {
        milter_reader_set_tag(priv->reader, 0);

#define DISCONNECT(name)                                                \
        g_signal_handlers_disconnect_by_func(priv->reader,              \
                                             G_CALLBACK(cb_reader_ ## name), \
                                             agent)

        DISCONNECT(flow);
        DISCONNECT(error);
        DISCONNECT(finished);
#undef DISCONNECT

        g_object_unref(priv->reader);
    }

    priv->reader = reader;
    if (priv->reader) {
        g_object_ref(priv->reader);

#define CONNECT(name)                                                   \
        g_signal_connect(priv->reader, #name, G_CALLBACK(cb_reader_ ## name), \
                         agent)
        CONNECT(flow);
        CONNECT(error);
        CONNECT(finished);
#undef CONNECT

        milter_reader_set_tag(priv->reader, priv->tag);
    }
}

GQuark
milter_agent_error_quark (void)
{
    return g_quark_from_static_string("milter-agent-error-quark");
}

static void
cb_writer_error (MilterWriter *writer,
                 GError *writer_error,
                 gpointer user_data)
{
    GError *error = NULL;
    MilterAgent *agent = user_data;
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);
    milter_utils_set_error_with_sub_error(&error,
                                          MILTER_AGENT_ERROR,
                                          MILTER_AGENT_ERROR_IO_ERROR,
                                          g_error_copy(writer_error),
                                          "Output error");
    milter_error("[%u] [agent][error][writer] %s",
                 priv->tag, error->message);
    milter_error_emittable_emit(MILTER_ERROR_EMITTABLE(user_data),
                                error);
    g_error_free(error);
}

static void
cb_writer_finished (MilterFinishedEmittable *emittable, gpointer user_data)
{
    MilterAgent *agent = user_data;

    milter_agent_set_writer(agent, NULL);
}

gboolean
milter_agent_write_packet (MilterAgent *agent,
                           const gchar *packet, gsize packet_size,
                           GError **error)
{
    MilterAgentPrivate *priv;
    gboolean success;

    priv = MILTER_AGENT_GET_PRIVATE(agent);

    if (!priv->writer)
        return TRUE;

    success = milter_writer_write(priv->writer, packet, packet_size,
                                  NULL, error);

    return success;
}

gboolean
milter_agent_flush (MilterAgent *agent, GError **error)
{
    MilterAgentPrivate *priv;
    gboolean success;

    priv = MILTER_AGENT_GET_PRIVATE(agent);

    if (!priv->writer)
        return TRUE;

    success = milter_writer_flush(priv->writer, error);

    return success;
}

void
milter_agent_set_writer (MilterAgent *agent, MilterWriter *writer)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);

    if (priv->writer) {
        milter_writer_set_tag(priv->writer, 0);

#define DISCONNECT(name)                                                \
        g_signal_handlers_disconnect_by_func(priv->writer,              \
                                             G_CALLBACK(cb_writer_ ## name), \
                                             agent)
        DISCONNECT(error);
        DISCONNECT(finished);
#undef DISCONNECT

        g_object_unref(priv->writer);
    }

    priv->writer = writer;
    if (priv->writer) {
        g_object_ref(priv->writer);

#define CONNECT(name)                                                   \
        g_signal_connect(priv->writer, #name,                           \
                         G_CALLBACK(cb_writer_ ## name),                \
                         agent)
        CONNECT(error);
        CONNECT(finished);
#undef CONNECT

        milter_writer_set_tag(priv->writer, priv->tag);
    }
}

MilterEncoder *
milter_agent_get_encoder (MilterAgent *agent)
{
    return MILTER_AGENT_GET_PRIVATE(agent)->encoder;
}

MilterDecoder *
milter_agent_get_decoder (MilterAgent *agent)
{
    return MILTER_AGENT_GET_PRIVATE(agent)->decoder;
}

void
milter_agent_start (MilterAgent *agent)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);

    if (priv->reader)
        milter_reader_start(priv->reader);
    if (priv->writer)
        milter_writer_start(priv->writer);
}

void
milter_agent_shutdown (MilterAgent *agent)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);

    milter_debug("[%u] [agent][shutdown]", priv->tag);
    if (priv->reader) {
        milter_debug("[%u] [agent][shutdown][reader]", priv->tag);
        milter_reader_shutdown(priv->reader);
    } else {
        if (!priv->finished)
            milter_finished_emittable_emit(MILTER_FINISHED_EMITTABLE(agent));
    }
}

guint
milter_agent_get_tag (MilterAgent *agent)
{
    return MILTER_AGENT_GET_PRIVATE(agent)->tag;
}

void
milter_agent_set_tag (MilterAgent *agent, guint tag)
{
    MilterAgentPrivate *priv;

    priv = MILTER_AGENT_GET_PRIVATE(agent);
    priv->tag = tag;
    apply_tag(priv);
}

/*
vi:ts=4:nowrap:ai:expandtab:sw=4
*/
