/*
 * gdbexec.c
 *
 * Loader stub to invoke GDB; checks for availability of python27.dll,
 * before invoking python-dependent GDB, falling back to invocation of
 * a python-independent GDB variant, if Python-2.7 is not installed.
 *
 *
 * $Id$
 *
 * Written by Keith Marshall <keith@users.osdn.me>
 * Copyright (C) 2020, MinGW.org Project
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */
#include <libgen.h>
#include <process.h>
#include <winbase.h>
#include <stdlib.h>
#include <stdio.h>

static int gdb_spawn( const char *const *argv, const char *mode )
{
  /* Helper function to invoke either the python-enabled GDB, which is
   * designated by the "main" mode argument, or the python-independent
   * variant, which is designated as "stub".
   */
  intptr_t status;
  const char *fmt = "%s";

  /* Insert the mode designator into the gdb.exe name, (for which the
   * full path name is passed in argv[0]), so identifying the full path
   * to the GDB variant which is to be invoked.
   */
  char cmd[8 + snprintf( NULL, 0, fmt, *argv )]; sprintf( cmd, fmt, *argv );
  sprintf( cmd, "%s\\gdb%s.exe", dirname( cmd ), mode );

  /* Delegate all further process activity to the selected GDB variant.
   */
  if( (status = _spawnv( P_WAIT, cmd, argv )) < 0 )
    fprintf( stderr, "Failed to execute \"%s\"; status = %d\n", cmd, status );

  /* Ultimately, return the exit status from GDB itself, (or a failure
   * status, if GDB cannot be invoked).
   */
  return status;
}

int main( int argc, char **argv )
{
  /* GDB loader entry point: first attempt to bind (dynamically)
   * to the python27.dll...
   */
  HMODULE python_dll;
  putenv( "TERM=unknown" ); *argv = _pgmptr;
  if( (python_dll = LoadLibrary( "python27.dll" )) == NULL )

    /* ...immediately delegating all further process activity to
     * the python-independent GDB, when binding fails...
     */
    return gdb_spawn( (const char *const *)(argv), "stub" );

  else
    /* ...otherwise, release the dynamically bound DLL...
     */
    FreeLibrary( python_dll );

  /* ...before handing off to the python-enabled GDB variant.
   */
  return gdb_spawn( (const char *const *)(argv), "main" );
}

/* $RCSfile$: end of file */
