#pragma once

#include <windows.h>
#include <stdio.h>
#include <memory.h>
#include <stdlib.h>
#include <stdio.h>
#include "../hed/hed_picturelib/faststream.h"


typedef unsigned int MEMORY_ACCESS_HISTORY;

struct DATA_INFO
{
	enum E_DATA_TYPE
	{
		IN_MEMORY,
		IN_FILE,
		IN_LOCKED_MEMORY,
		IN_LOCKED_FILE,
		IN_FILE_TO_LOCK
	};
	E_DATA_TYPE m_eType;
	unsigned int m_nIndex;
};

struct MEMORY_CACHE_TABLE
{
	unsigned int							m_nBlockPosition;
	list<MEMORY_ACCESS_HISTORY>::iterator	m_itHistory;
};

class CBlockCacheArray
{
public:
	static char SYSTEM_TEMP_DIR[MAX_PATH];
	static char SYSTEM_TEMP_PREFIX[16];

	CBlockCacheArray( )
	{
		m_pMemoryCacheTable = NULL;
		m_pMemoryCache = NULL;
		m_pTempBuffer = NULL;
		m_nBlockSize = 0;
		m_nMemoryCacheBlockCount = 0;
	}
	
	~CBlockCacheArray( )
	{
		clean();
	}

	bool init( unsigned int nBlockCount, unsigned int nBlockSize, unsigned int nOnMemoryBlockCount )
	{
		if( m_BlockData.size() > 0 )
			return false;
		if( nBlockCount == 0 || nBlockSize == 0 )
			return false;

		char chTempFile[MAX_PATH];
		CreateTempFile( chTempFile );
		m_file.open( chTempFile, CFastStream::OPEN_READ|CFastStream::OPEN_WRITE|CFastStream::OPEN_TRUNC|CFastStream::OPEN_TEMP );

		for( unsigned int cnt = 0; cnt < nBlockCount; cnt ++ )
		{
			DATA_INFO di;
			di.m_eType = ( cnt < nOnMemoryBlockCount ) ? DATA_INFO::IN_MEMORY : DATA_INFO::IN_FILE;
			di.m_nIndex = cnt;
			m_BlockData.push_back( di );
		}

		m_pMemoryCacheTable = (MEMORY_CACHE_TABLE*)malloc( sizeof( MEMORY_CACHE_TABLE ) * nOnMemoryBlockCount );
		memset( m_pMemoryCacheTable, 0x00, sizeof( MEMORY_CACHE_TABLE ) * nOnMemoryBlockCount );

		for( unsigned int cnt2 = 0; cnt2 < nOnMemoryBlockCount; cnt2 ++ )
		{
			MEMORY_ACCESS_HISTORY mah = cnt2;
			m_MemoryAccessHistory.push_back( mah );
			list<MEMORY_ACCESS_HISTORY>::iterator it = m_MemoryAccessHistory.end();
			it --;
			m_pMemoryCacheTable[cnt2].m_itHistory = it;
			m_pMemoryCacheTable[cnt2].m_nBlockPosition = cnt2;
		}

		m_pMemoryCache = (unsigned char*) malloc( nBlockSize * nOnMemoryBlockCount );
		memset( m_pMemoryCache, 0x00, nBlockSize * nOnMemoryBlockCount );

		m_pTempBuffer = (unsigned char*)malloc( nBlockSize );
		memset( m_pTempBuffer, 0x00, nBlockSize );

		m_pNonLockedMemoryCache = m_MemoryAccessHistory.begin();
	
		m_nBlockSize = nBlockSize;
		m_nMemoryCacheBlockCount = nOnMemoryBlockCount;

		return true;
	}

	bool re_init( unsigned int nBlockCount, unsigned int nBlockSize, unsigned int nOnMemoryBlockCount )
	{
		clean( );
		return init( nBlockCount, nBlockSize, nOnMemoryBlockCount );
	}

	bool clean( )
	{
		m_BlockData.clear();
		m_MemoryAccessHistory.clear();
		free( m_pMemoryCacheTable );
		m_pMemoryCacheTable = NULL;
		free( m_pMemoryCache );
		m_pMemoryCache = NULL;
		free( m_pTempBuffer );
		m_pTempBuffer = NULL;
		m_nBlockSize = 0;
		m_nMemoryCacheBlockCount = 0;
		m_file.close();
		return true;
	}

	unsigned int get_block_count( )
	{
		return (unsigned int)m_BlockData.size();
	}

	bool set_block( unsigned int nIndex, unsigned char *pBlock )
	{
		if( !pBlock )
			return false;
		if( nIndex < 0 || nIndex >= m_BlockData.size() )
			return false;

		DATA_INFO &bi = m_BlockData[nIndex];
		switch( bi.m_eType )
		{
		case DATA_INFO::IN_FILE:
		case DATA_INFO::IN_FILE_TO_LOCK:
			{
				bool bToLock = false;
				if( bi.m_eType == DATA_INFO::IN_FILE_TO_LOCK )
					bToLock = true;
				if( move_to_memory_cache( bi.m_nIndex ) )
				{
					setup_memory_cache( nIndex, bi.m_nIndex, pBlock );
					set_latest_memory_access( bi.m_nIndex );
					if( bToLock )
						lock_memory( nIndex, nIndex );
				}
				else
				{
					setup_file_cache( nIndex, pBlock );
				}
			}
			break;
		case DATA_INFO::IN_LOCKED_FILE:
			setup_file_cache( nIndex, pBlock );
			break;
		case DATA_INFO::IN_MEMORY:
		case DATA_INFO::IN_LOCKED_MEMORY:
			setup_memory_cache( nIndex, bi.m_nIndex, pBlock );
			set_latest_memory_access( bi.m_nIndex );
			break;
		}
		return true;
	}
	
	bool get_block( unsigned int nIndex, unsigned char *pBlock )
	{
		if( !pBlock )
			return false;
		if( nIndex < 0 || nIndex >= m_BlockData.size() )
			return false;

		DATA_INFO &bi = m_BlockData[nIndex];
		switch( bi.m_eType )
		{
		case DATA_INFO::IN_MEMORY:
		case DATA_INFO::IN_LOCKED_MEMORY:
			memcpy( pBlock, m_pMemoryCache + m_nBlockSize * bi.m_nIndex, m_nBlockSize );
			if( bi.m_eType == DATA_INFO::IN_MEMORY )
				set_latest_memory_access( bi.m_nIndex );
			break;
		case DATA_INFO::IN_FILE:
		case DATA_INFO::IN_FILE_TO_LOCK:
			{
				bool bToLock = false;
				if( bi.m_eType == DATA_INFO::IN_FILE_TO_LOCK )
					bToLock = true;
				if( move_to_memory_cache( nIndex) )
				{
					memcpy( pBlock, m_pMemoryCache + m_nBlockSize * bi.m_nIndex, m_nBlockSize );
					set_latest_memory_access( bi.m_nIndex );
					if( bToLock )
						lock_memory( nIndex, nIndex );
				}
				else
				{
					m_file.read( pBlock, m_nBlockSize * bi.m_nIndex, m_nBlockSize );
				}
			}
			break;
		case DATA_INFO::IN_LOCKED_FILE:
			m_file.read( pBlock, m_nBlockSize * bi.m_nIndex, m_nBlockSize );
			break;
		}
		return true;
	}

	bool get_block( unsigned int nIndex, const unsigned char ** const ppBlock )
	{
		if( !ppBlock )
			return false;
		*ppBlock = NULL;
		if( nIndex < 0 || nIndex >= m_BlockData.size() )
			return false;

		DATA_INFO &bi = m_BlockData[nIndex];
		switch( bi.m_eType )
		{
		case DATA_INFO::IN_MEMORY:
		case DATA_INFO::IN_LOCKED_MEMORY:
			*ppBlock = m_pMemoryCache + m_nBlockSize * bi.m_nIndex;
			if( bi.m_eType == DATA_INFO::IN_MEMORY )
				set_latest_memory_access( bi.m_nIndex );
			break;
		case DATA_INFO::IN_FILE:
		case DATA_INFO::IN_FILE_TO_LOCK:
			{
				bool bToLock = false;
				if( bi.m_eType == DATA_INFO::IN_FILE_TO_LOCK )
					bToLock = true;
				if( move_to_memory_cache( nIndex) )
				{
					*ppBlock = m_pMemoryCache + m_nBlockSize * bi.m_nIndex;
					set_latest_memory_access( bi.m_nIndex );
					if( bToLock )
						lock_memory( nIndex, nIndex );
				}
				else
				{
					m_file.direct_read( ppBlock, m_nBlockSize * bi.m_nIndex, m_nBlockSize );
				}
			}
			break;
		case DATA_INFO::IN_LOCKED_FILE:
			m_file.direct_read( ppBlock, m_nBlockSize * bi.m_nIndex, m_nBlockSize );
			break;

		}
		return true;
	}

	bool lock_memory( unsigned int nFirst, unsigned int nEnd )
	{
		if( nFirst < 0 || nEnd < 0 )
			return false;
		if( nFirst >= m_BlockData.size() || nEnd >= m_BlockData.size() )
			return false;
		if( nFirst > nEnd )
			return false;

		int nLockMemoryCount = 0;
		for( unsigned int cnt = nFirst; cnt <= nEnd; cnt ++ )
		{
			DATA_INFO &di = m_BlockData[cnt];
			switch( di.m_eType )
			{
			case DATA_INFO::IN_MEMORY:
				{
					di.m_eType = DATA_INFO::IN_LOCKED_MEMORY;
					set_latest_memory_access( di.m_nIndex );
					m_pNonLockedMemoryCache ++;
				}
				break;
			case DATA_INFO::IN_FILE:
				{
					di.m_eType = DATA_INFO::IN_FILE_TO_LOCK;
				}
				break;
			}
			nLockMemoryCount ++;
		}


		return true;
	}

	bool unlock_memory( )
	{
		unsigned int cnt = 0;
		for( cnt = 0; cnt < m_BlockData.size(); cnt ++ )
		{
			DATA_INFO &di = m_BlockData[cnt];
			if( di.m_eType == DATA_INFO::IN_LOCKED_MEMORY )
			{
				di.m_eType = DATA_INFO::IN_MEMORY;
			}
			else if( di.m_eType == DATA_INFO::IN_LOCKED_FILE ||
				di.m_eType == DATA_INFO::IN_FILE_TO_LOCK )
			{
				di.m_eType = DATA_INFO::IN_FILE;
			}
		}
		m_pNonLockedMemoryCache = m_MemoryAccessHistory.begin();
		return true;
	}

public:
#ifdef _CPP_UNITTEST_
	// debug API
	DATA_INFO::E_DATA_TYPE get_cache_type( unsigned int nBlockNumber )
	{
		return m_BlockData[nBlockNumber].m_eType;
	}

	unsigned int get_cache_position( unsigned int nBlockNumber )
	{
		return m_BlockData[nBlockNumber].m_nIndex;
	}

	void dump_cache_state( ostream &out )
	{
		for( unsigned int i = 0; i < m_BlockData.size(); i++ )
		{
			DATA_INFO &bi = m_BlockData[i];
			switch( bi.m_eType )
			{
			case DATA_INFO::IN_FILE:
				out << "f";
				break;
			case DATA_INFO::IN_LOCKED_FILE:
				out << "F";
				break;
			case DATA_INFO::IN_FILE_TO_LOCK:
				out << "L";
				break;
			case DATA_INFO::IN_MEMORY:
				out << "m";
				break;
			case DATA_INFO::IN_LOCKED_MEMORY:
				out << "M";
				break;
			default:
				out << "X";
			}
			out << bi.m_nIndex << " ";
		}
		out << endl;
	}

	void dump_memory_access_history( ostream &out )
	{
		list<MEMORY_ACCESS_HISTORY>::iterator it;
		for( it = m_MemoryAccessHistory.begin(); it != m_MemoryAccessHistory.end(); it ++ )
		{
			out << *it;
			out << " ";
		}
		out << endl;
		out << *(m_pNonLockedMemoryCache) << endl;
	}

	void dump_memory_access_history_link( ostream &out )
	{
		out << "[MEMORY_INDEX:BLOCK_INDEX:CACHE_INDEX]" << endl;
		for( unsigned int cnt = 0; cnt < m_nMemoryCacheBlockCount; cnt ++ )
		{
			MEMORY_CACHE_TABLE mct = m_pMemoryCacheTable[cnt];
			out << cnt << ":" << mct.m_nBlockPosition << ":" << *(mct.m_itHistory) << " ";
		}
		out << endl;
	}

	void dump_block_top_charactor( ostream &out )
	{
		unsigned char pBuffer[16];
		char pWriteBuffer[16];

		for( unsigned int cnt = 0; cnt < m_BlockData.size(); cnt ++ )
		{
			switch( m_BlockData[cnt].m_eType )
			{
			case DATA_INFO::IN_FILE:
				m_file.read( pBuffer, cnt * m_nBlockSize, 1 );
				break;
			case DATA_INFO::IN_MEMORY:
			case DATA_INFO::IN_LOCKED_MEMORY:
				memcpy( pBuffer, m_pMemoryCache + m_BlockData[cnt].m_nIndex * m_nBlockSize, 1 );
				break;
			}
			sprintf( pWriteBuffer, "% 2x", pBuffer[0] );
			out << pWriteBuffer << " ";
		}
		out << endl;
	}
#endif // _CPP_UNITTEST_

protected:
	vector<DATA_INFO>				m_BlockData;
	list<MEMORY_ACCESS_HISTORY>		m_MemoryAccessHistory;
	MEMORY_CACHE_TABLE*				m_pMemoryCacheTable;
	unsigned char*					m_pMemoryCache;
	unsigned char*					m_pTempBuffer;
	list<MEMORY_ACCESS_HISTORY>::iterator	m_pNonLockedMemoryCache;
	
	unsigned int					m_nBlockSize;
	unsigned int					m_nMemoryCacheBlockCount;

	CFastStream						m_file;
	
	void __fastcall CreateTempFile( char *chTempFileName )
	{
		char chTempPath[MAX_PATH];
		char drive[_MAX_DRIVE];
		char dir[_MAX_DIR];
		char fname[_MAX_FNAME];
		char ext[_MAX_EXT];
		GetModuleFileName( NULL, chTempPath, MAX_PATH );
		_splitpath( chTempPath, drive, dir, fname, ext );
		sprintf( chTempPath, "%s%s%s", drive, dir, SYSTEM_TEMP_DIR );
		CreateDirectory( chTempPath, NULL );
		
		GetTempFileName( chTempPath, SYSTEM_TEMP_PREFIX, 0, chTempFileName );
	}

	bool set_latest_memory_access( unsigned int nMemoryPosition )
	{
		m_MemoryAccessHistory.splice( 
				m_pNonLockedMemoryCache, 
				m_MemoryAccessHistory, 
				m_pMemoryCacheTable[nMemoryPosition].m_itHistory );

		m_pNonLockedMemoryCache = m_pMemoryCacheTable[nMemoryPosition].m_itHistory;
		return true;
	}
	
	unsigned int find_removable_memory_cache( )
	{
		if( m_MemoryAccessHistory.size() > 0 )
		{
			list<MEMORY_ACCESS_HISTORY>::iterator it = m_MemoryAccessHistory.end();
			it --;
			return *it;
		}
		else
		{
			return 0;
		}
	}
	
	bool move_to_file_cache( unsigned int nMemoryCachePosition )
	{
		if( nMemoryCachePosition >= m_nMemoryCacheBlockCount )
			return false;

		MEMORY_CACHE_TABLE &mt = m_pMemoryCacheTable[nMemoryCachePosition];
		DATA_INFO &di = m_BlockData[ mt.m_nBlockPosition ];
		if( di.m_eType == DATA_INFO::IN_LOCKED_MEMORY )
			return false;
		setup_file_cache( mt.m_nBlockPosition, m_pMemoryCache + m_nBlockSize * nMemoryCachePosition );
		return true;
	}
	
	bool move_to_memory_cache( unsigned int nFileCachePosition )
	{
		unsigned int nTrashMemory = find_removable_memory_cache( );
		if( !move_to_file_cache( nTrashMemory ) )
		{
			return false;
		}
		else
		{
			DATA_INFO &di = m_BlockData[nFileCachePosition];
			MEMORY_CACHE_TABLE &mt = m_pMemoryCacheTable[nTrashMemory];
			if( di.m_eType != DATA_INFO::IN_LOCKED_MEMORY )
				di.m_eType = DATA_INFO::IN_MEMORY;
			di.m_nIndex = nTrashMemory;
			mt.m_nBlockPosition = nFileCachePosition;
			m_file.read( m_pMemoryCache + m_nBlockSize * di.m_nIndex, m_nBlockSize * nFileCachePosition, m_nBlockSize );
			return true;
		}
	}
	
	bool setup_memory_cache( unsigned int nBlockPosition, unsigned int nMemoryCachePosition, unsigned char *pData )
	{
		DATA_INFO &di = m_BlockData[nBlockPosition];
		MEMORY_CACHE_TABLE &mt = m_pMemoryCacheTable[nMemoryCachePosition];
		if( di.m_eType != DATA_INFO::IN_LOCKED_MEMORY )
			di.m_eType = DATA_INFO::IN_MEMORY;
		di.m_nIndex = nMemoryCachePosition;
		mt.m_nBlockPosition = nBlockPosition;
		memcpy( m_pMemoryCache + m_nBlockSize * di.m_nIndex, pData, m_nBlockSize );
		return true;
	}
	
	bool setup_file_cache( unsigned int nBlockPosition, unsigned char *pData )
	{
		DATA_INFO &di = m_BlockData[nBlockPosition];
		di.m_eType = DATA_INFO::IN_FILE;
		di.m_nIndex = nBlockPosition;
		m_file.write( pData, 
					m_nBlockSize * di.m_nIndex, 
					m_nBlockSize );
		return true;
	}
};
