#include "mk.h"
#include <stdlib.h>
#include <memory.h>
#include <string.h>

#define MK_TYPE_EXPR_IF_CONDITION		1		// if expr then
#define MK_TYPE_EXPR_WHILE_CONDITION	2		// while expr \n
#define MK_TYPE_EXPR_DOWHILE_CONDITION	2		// end while expr \n
#define MK_TYPE_EXPR_FUNCTION_ARG		3		// function( expr, expr, ... )
#define MK_TYPE_EXPR_ROOT_SEGMENT		4		// expr \n
#define MK_TYPE_EXPR_INNER_EXPR			5		// ... op ( expr ) op ...
#define MK_TYPE_EXPR_NONAME_BLOCK		6		// { expr }

#define MK_TYPE_SEGMENT_ROOT			1		// top level segment
#define MK_TYPE_SEGMENT_BLOCK			2		// function top level segment
#define MK_TYPE_SEGMENT_IF				3		// if block segmnet
#define MK_TYPE_SEGMENT_WHILE			4		// while block segment
#define MK_TYPE_SEGMENT_DOWHILE			5		// do while block segment
#define MK_TYPE_SEGMENT_TRY				6		// try .. segment
#define MK_TYPE_SEGMENT_CATCH			7		// catch arg ... segment
#define MK_TYPE_SEGMENT_CATCH_ELSE		8		// catch ... else ... segment
#define MK_TYPE_SEGMENT_FINALLY			9		// finally .. segment

void *mk_open( void*value );
int mk_getc( void*stream );
void mk_close( void*stream );
void mk_trace( unsigned int code, MK_CHAR *message );

static
int do_compile_check_inhevior( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *target, MK_HASHTABLE *classest );
static
MK_NODE_BLOCK *create_mk_node_block( unsigned int type );
static
MK_NODE_EXPR *create_numeric_expr( unsigned int value, MK_CHAR *text, unsigned int isNeg );
static
MK_NODE_EXPR *create_string_expr( MK_CHAR *text, unsigned int length );
static
MK_NODE_EXPR *create_operation_expr( unsigned int value );
static
MK_NODE_EXPR *create_member_variable_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *create_static_member_variable_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *create_symbol_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *create_new_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *create_this_expr( );
static
MK_NODE_EXPR *create_super_expr( );
static
MK_NODE_EXPR *create_owner_expr( );
static
MK_NODE_EXPR *do_compile_break_state( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *do_compile_continue_state( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *do_compile_raise_state( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *create_nil_expr( );
static
MK_NODE_EXPR *create_true_expr( );
static
MK_NODE_EXPR *create_false_expr( );
static
MK_NODE_EXPR *create_multiple_top_expr( );

static
MK_NODE_EXPR *do_compile_return_state( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_IF *do_compile_if_block( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_WHILE *do_compile_while_block( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_WHILE *do_compile_do_while_block( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
int do_compile_function_argment( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_FUNCTION *newFunction );
static
MK_NODE_WITH_PARAM *do_compile_node_widh_param( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_TRY_BLOCK *do_compile_try_catch_finally_block( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *do_compile_function_call( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *do_compile_function_call( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
MK_NODE_EXPR *do_compile_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer,  unsigned int state );
static
int do_compile_segment( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_NODE_BLOCK *block, unsigned int typeSegment );
static
int do_compile_function( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *targetClass );
static
int do_compile_property( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *targetClass );
static
int do_compile_inner_class( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *targetClass );
static
MK_CLASS *do_compile_class_definition( MK_VM_STRUCT *vm, MK_LEXER *lexer );
static
int do_compile_root( MK_VM_STRUCT *vm, MK_LEXER *lexer );


int mk_register_class( MK_VM_STRUCT *vm, MK_CLASS *target )
{
	if( mk_is_key_hashtable( vm->code->classes, target->nameThis ) != 0 )
		return 0;

	return mk_insert_item_hashtable( 
		vm->code->classes, 
		mk_get_symbol_name_ptr( vm, target->nameThis ), 
		target ) != 0xffffffff;
}

MK_VARIABLE *mk_create_variable( MK_VM_STRUCT *vm, MK_CHAR *mk_name, unsigned int type, MK_NODE_EXPR *defaultValue )
{
	MK_VARIABLE *result = NULL;
	if( MK_OBJECT_TYPE( type ) != MK_TYPE_VARIABLE )
		return NULL;
	result = mk_create_object( type );
	result->name = mk_get_symbol_name_ptr( vm, mk_name );
	result->defaultValue = defaultValue;
	return result;
}

int mk_register_variable( MK_VM_STRUCT *vm, MK_VARIABLE *targetVariable, MK_CLASS *targetClass )
{
	MK_CHAR *name = 
		mk_get_symbol_name_ptr( vm, targetVariable->name );
	if( targetClass->variables == NULL )
		targetClass->variables = mk_create_hashtable( MK_SIZEOF_HASH_DEFAULT, MK_SIZEOF_EXTEND_HASH_DEFAULT );
	else if( mk_is_key_hashtable( targetClass->variables, name ) != 0 )
		return 0;
	
	return mk_insert_item_hashtable( targetClass->variables, name, targetVariable ) != 0xffffffff;
}

MK_FUNCTION *mk_create_native_function( MK_VM_STRUCT *vm, MK_CHAR *mk_name, unsigned int sizeArgs, unsigned int type, void *fp )
{
	MK_FUNCTION *result = NULL;

	if( MK_OBJECT_TYPE( type ) != MK_TYPE_FUNCTION )
		return NULL;
	type |= 
		( MK_TYPE_ATTRIBUTE_FUNCTION_NATIVE );
	result = mk_create_object( type );
	result->name = mk_get_symbol_name_ptr( vm, mk_name );
	result->entryPoint = fp;
	result->sizeArgs = sizeArgs;
	return result;
}

int mk_register_function( MK_VM_STRUCT *vm, MK_FUNCTION *targetFunction, MK_CLASS *targetClass )
{
	MK_CHAR *name = 
		mk_get_symbol_name_ptr( vm, targetFunction->name );
	if( ( MK_TYPE_ATTRIBUTE( targetFunction->flags ) & MK_TYPE_ATTRIBUTE_FUNCTION_OPERATOR ) == 
			MK_TYPE_ATTRIBUTE_FUNCTION_OPERATOR )
	{
		unsigned int index = MK_RESERVED_MARK_INDEX(targetFunction->flags);
		if( index >= MK_SIZEOF_ENABELD_OPELATORS )
			return 0;
		if( targetClass->operatorFunctions == NULL )
		{
			int insert = 0;
			targetClass->operatorFunctions =
				mk_create_vector( MK_SIZEOF_ENABELD_OPELATORS, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
			for( insert = 0; insert < MK_SIZEOF_ENABELD_OPELATORS; insert ++ )
				mk_insert_at_vector( &targetClass->operatorFunctions, insert, NULL );
		}
		if( mk_get_at_vector( targetClass->operatorFunctions, index ) != NULL )
			return 0;
		mk_set_at_vector( targetClass->operatorFunctions, index, targetFunction );
		return 1;
	}
	else
	{
		if( targetClass->functions == NULL )
			targetClass->functions = mk_create_hashtable( MK_SIZEOF_HASH_DEFAULT, MK_SIZEOF_EXTEND_HASH_DEFAULT );
		else if( mk_is_key_hashtable( targetClass->functions, name ) != 0 )
			return 0;
		return mk_insert_item_hashtable( targetClass->functions, name, targetFunction ) != 0xffffffff;
	}
}

int mk_register_function_arg( MK_VM_STRUCT *vm, MK_VECTOR *targetArgs, MK_VARIABLE *arg )
{
	unsigned int size = 
		mk_size_vector( targetArgs );
	unsigned int index = 0;

	for( index = 0; index < size; index ++ )
	{
		MK_VARIABLE *value = 
			mk_get_at_vector( targetArgs, index );
		if( strcmp( value->name, arg->name ) == 0 )
			break;
	}
	if( index < size )
	{
		return 0;
	}
	else
	{
		if( targetArgs == NULL )
			targetArgs = 
			mk_create_vector( MK_SIZEOF_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
		mk_push_vector( targetArgs, arg );
		return 1;
	}
}

static
MK_NODE_BLOCK *create_mk_node_block( unsigned int type )
{
	MK_NODE_BLOCK *result = 
		mk_create_object( MK_TYPE_NODE_BLOCK );
	result->flags = MK_TYPE_SET_ATTRIBUTE( result->flags, type );
	return result;
}

static
MK_NODE_EXPR *create_numeric_expr( unsigned int value, MK_CHAR *text, unsigned int isNeg )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	if( value == MK_LEX_TYPE_INT_VALUE )
	{
		unsigned int num = 
			atoi( text );
		if( isNeg )
			num *= -1;
		result->flags = 
			MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_INT32;
		result->u1.constantValue = 0;
		result->u2.constantValue = 
			num;
	}
	else if( value == MK_LEX_TYPE_FLOAT_VALUE )
	{
		MK_FLOAT num =
			atof( text );
		unsigned int *pIntNum = (unsigned int*)&num;
		if( isNeg )
			num = num * -1.0;
		result->flags = 
			MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_FLOAT;
		result->u1.constantValue = *pIntNum;
		result->u2.constantValue = *(++pIntNum);
	}
	else
	{
		free( result );
		result = NULL;
	}
	return result;
}

static 
MK_NODE_EXPR *create_string_expr( MK_CHAR *text, unsigned int length )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = 
		MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_STRING;

	result->u1.length = length;
	result->u2.value = malloc( sizeof(MK_CHAR) * (length + 1) );
	memset( result->u2.value, 0x00, sizeof(MK_CHAR) * (length + 1) );
	memcpy( result->u2.value, text, sizeof(MK_CHAR) * (length) );

	return result;
}

static
MK_NODE_EXPR *create_operation_expr( unsigned int value )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = 
		MK_TYPE_NODE_EXPR | 
		MK_TYPE_NODE_EXPR_OPERATION |
		( value & ( MK_TYPE_NODE_EXPR_OPERATIONTYPE_MASK | 
				   MK_TYPE_NODE_EXPR_OPERATIONPRIORITY_MASK | 
				   MK_TYPE_NODE_EXPR_OPERATION_LEFT_CONST | 
				   MK_TYPE_NODE_EXPR_OPERATION_RIGHT_CONST ) );
	return result;
}

static
MK_NODE_EXPR *create_static_member_variable_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	void *keyPtr = NULL;

	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = 
		MK_TYPE_NODE_EXPR |
		MK_TYPE_NODE_EXPR_DBLATSYMBOL;
	result->u1.symbolName = mk_get_symbol_name_ptr( vm, lexer->text );
	result->u2.constantValue = 0;

	return result;
}

static
MK_NODE_EXPR *create_member_variable_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	void *keyPtr = NULL;
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = 
		MK_TYPE_NODE_EXPR |
		MK_TYPE_NODE_EXPR_ATSYMBOL;
	result->u1.symbolName = mk_get_symbol_name_ptr( vm, lexer->text );
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *create_symbol_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	void *keyPtr = NULL;
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = 
		MK_TYPE_NODE_EXPR |
		MK_TYPE_NODE_EXPR_SYMBOL;
	result->u1.symbolName = mk_get_symbol_name_ptr( vm, lexer->text );
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *create_new_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	MK_NODE_EXPR *result =
		do_compile_function_call( vm, lexer );
	if( result != NULL )
		result->flags = 
			MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_NEW;
	return result;
}

static
MK_NODE_EXPR *create_this_expr( )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_SYMBOL_THIS;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *create_super_expr( )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_SYMBOL_SUPER;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	return result;
}


static
MK_NODE_EXPR *create_owner_expr( )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_SYMBOL_OWNER;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *create_nil_expr( )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_SYMBOL_NIL;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *create_true_expr( )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_SYMBOL_TRUE;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *create_false_expr( )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_SYMBOL_FALSE;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *create_multiple_top_expr( )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_MULTIPLESYMBOL;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	return result;
}

static
MK_NODE_EXPR *do_compile_break_state( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_BREAK;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	
	mk_get_token( lexer );
	if( lexer->value != MK_LEX_TYPE_EOL &&
		lexer->value != MK_LEX_TYPE_EOF )
	{
		if( vm->exceptionObject == NULL )
			mk_raise_internal_error( vm, 
				lexer->stream.name, 
				lexer->stream.line, 
				MK_ERROR_TYPE_COMILE_ERROR | 3, 
				"break", 
				lexer->text, 
				"<CR>",
				NULL );
		mk_destroy_node( vm->memoryPool, result );
		result = NULL;
	}

	return result;
}

static
MK_NODE_EXPR *do_compile_continue_state( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_CONTINUE;
	result->u1.constantValue = 0;
	result->u2.constantValue = 0;
	mk_get_token( lexer );
	if( lexer->value != MK_LEX_TYPE_EOL &&
		lexer->value != MK_LEX_TYPE_EOF )
	{
		if( vm->exceptionObject == NULL )
			mk_raise_internal_error( vm, 
				lexer->stream.name, 
				lexer->stream.line, 
				MK_ERROR_TYPE_COMILE_ERROR | 3, 
				"continue", 
				lexer->text, 
				"<CR>",
				NULL );
		mk_destroy_node( vm->memoryPool, result );
		result = NULL;
	}

	return result;
}

static
MK_NODE_EXPR *do_compile_raise_state( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	int isSuccess = 0;
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	do
	{
		if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_RAISE )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, 
					lexer->stream.name, 
					lexer->stream.line, 
					MK_ERROR_TYPE_COMILE_ERROR | 17, 
					"raise <expr>", 
					NULL );
			break;
		}
		result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_RAISE;
		mk_get_token( lexer );
		if( lexer->value != MK_LEX_TYPE_EOL )
		{
			result->u1.left = do_compile_expr( vm, lexer, MK_TYPE_EXPR_ROOT_SEGMENT );
			if( result->u1.left == NULL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 18, "raise", "<expr>", NULL );
				break;
			}
			if( lexer->value != MK_LEX_TYPE_EOL &&
				lexer->value != MK_LEX_TYPE_EOF )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<CR>", NULL );
				break;
			}
		}
		isSuccess = 1;
	}while( 0 );
	if( lexer->hasError != 0 )
		isSuccess = 0;
	if( isSuccess == 0 )
	{
		mk_destroy_node( vm->memoryPool, result );
		result = NULL;
	}
	return result;
}

static
MK_NODE_EXPR *do_compile_return_state( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	int isSuccess = 0;
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	do
	{
		if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_RETURN )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 17, "return <expr>", NULL );
			break;
		}
		result->flags = MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_RETURN;
		mk_get_token( lexer );
		if( lexer->value != MK_LEX_TYPE_EOL )
		{
			result->u1.left = do_compile_expr( vm, lexer, MK_TYPE_EXPR_ROOT_SEGMENT );
			if( result->u1.left == NULL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 18, "return", "<expr>", NULL );
				break;
			}
			if( lexer->value != MK_LEX_TYPE_EOL &&
				lexer->value != MK_LEX_TYPE_EOF )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<CR>", NULL );
				break;
			}
		}
		isSuccess = 1;
	}while( 0 );
	if( lexer->hasError != 0 )
		isSuccess = 0;
	if( isSuccess == 0 )
	{
		mk_destroy_node( vm->memoryPool, result );
		result = NULL;
	}
	return result;
}

static
MK_NODE_IF *do_compile_if_block( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	unsigned int status = 0;
	unsigned int isSuccess = 0;
	MK_NODE_IF *result = NULL, 
		*current = NULL, 
		*previous = NULL;

	while( lexer->value != MK_LEX_TYPE_EOF )
	{
		if( status == 0 )
		{
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_IF )
			{
				status = 1;
			}
			else
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "if", NULL );
				break;
			}
		}
		else if( status == 1 )
		{
			if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_ELSEIF ||
				lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_ELSE )
			{
				if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_ELSE )
					status = 2;
			}
			else
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "[elseif|else]", NULL );
				break;
			}
		}
		else	// status == 2
		{
			if( vm->exceptionObject == NULL )
			{
				if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_ELSE )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 15, "else", NULL );
				else
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "end", NULL );
			}
			break;
		}

		current = 
			( MK_NODE_IF * )mk_create_object( MK_TYPE_NODE_IF );
		if( previous != NULL )
			previous->next = current;
		else
			result = current;
		previous = current;

		mk_get_token( lexer );
		if( status != 2 )
		{
			current->expr = do_compile_expr( vm, lexer, MK_TYPE_EXPR_IF_CONDITION );
			if( current->expr == NULL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "<if condition>", NULL );
				break;
			}
			else if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_THEN )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "then", NULL );
				break;
			}
			mk_get_token( lexer );
		}

		current->block = create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_IF_BLOCK );
		if( do_compile_segment( vm, lexer, current->block, MK_TYPE_SEGMENT_IF ) == 0 )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "<if segment>", NULL );
			break;
		}
		if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
		{
			isSuccess = 1;
			mk_get_token( lexer );
			break;
		}
	}
	if( lexer->hasError != 0 )
		isSuccess = 0;
	if( isSuccess == 0 )
	{
		mk_destroy_node( vm->memoryPool, result );
		result = NULL;
	}
	return ( isSuccess != 0 ) ? result : NULL;
}

static
MK_NODE_WHILE *do_compile_while_block( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	unsigned int isSuccess = 0;
	MK_NODE_WHILE *result = 
			( MK_NODE_WHILE * )mk_create_object( MK_TYPE_NODE_WHILE );
	result->flags = MK_TYPE_NODE_WHILE | MK_TYPE_ATTRIBUTE_NODE_WHILE_FRONT;
	do
	{
		// lexer->text = "while"
		mk_get_token( lexer );
		result->expr = do_compile_expr( vm, lexer, MK_TYPE_EXPR_WHILE_CONDITION );
		if( result->expr == NULL ||
			lexer->value != MK_LEX_TYPE_EOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "<while condition>", NULL );
			break;
		}
		mk_get_token( lexer );
		result->block = create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_WHILE_BLOCK );
		if( do_compile_segment( vm, lexer, result->block, MK_TYPE_SEGMENT_WHILE ) == 0 )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "<while segment>", NULL );
		}
	}
	while( 0 );
	if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
	{
		isSuccess = 1;
		mk_get_token( lexer );
	}
	else
	{
		if( vm->exceptionObject == NULL )
			mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "end", NULL );
	}
	if( lexer->hasError != 0 )
		isSuccess = 0;
	if( isSuccess == 0 )
		mk_destroy_node( vm->memoryPool, result );
	return ( isSuccess != 0 ) ? result : NULL;
}

static
MK_NODE_WHILE *do_compile_do_while_block( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	unsigned int isSuccess = 0;
	MK_NODE_WHILE *result = 
		( MK_NODE_WHILE * )mk_create_object( MK_TYPE_NODE_WHILE );
	result->flags = MK_TYPE_NODE_WHILE | MK_TYPE_ATTRIBUTE_NODE_WHILE_BACK;
	do
	{
		// lexer->text == "do"
		mk_get_token( lexer );

		if( lexer->value != MK_LEX_TYPE_EOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<CR>", NULL );
			break;
		}

		result->block = create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_WHILE_BLOCK );
		if( do_compile_segment( vm, lexer, result->block, MK_TYPE_SEGMENT_DOWHILE ) == 0 )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "<do...while segment>", NULL );
			break;
		}
		// lexer->text == "end"
		mk_get_token( lexer );
		if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_WHILE )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "while", NULL );
			break;
		}
		mk_get_token( lexer );
		result->expr = do_compile_expr( vm, lexer, MK_TYPE_EXPR_WHILE_CONDITION );
		if( result->expr == NULL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "do...while condition", NULL );
			break;
		}

		if( lexer->value != MK_LEX_TYPE_EOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<CR>", NULL );
			break;
		}
		isSuccess = 1;
	}
	while( 0 );
	if( lexer->hasError != 0 )
		isSuccess = 0;
	return ( isSuccess != 0 ) ? result : NULL;
}

static
MK_NODE_WITH_PARAM *do_compile_node_widh_param( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	// lexer->value = '|' or '{'
	int argCount = 0;
	int isError = 0;

	MK_NODE_WITH_PARAM *newNode = 
		mk_create_object( MK_TYPE_NODE_WITH_PARAM );
	// do compile_parameter
	if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_OR )
	{
		while( 1 )
		{
			MK_VARIABLE *arg = NULL;
			mk_get_token( lexer );
			if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_OR )
			{
				mk_get_token( lexer );
				break;	// success
			}
			newNode->param = 
				mk_create_vector( MK_SIZEOF_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
			if( argCount > 0 )
			{
				if( lexer->value != MK_LEX_TYPE_RESERVED_MARK_CONMA )
				{
					if( vm->exceptionObject == NULL )
						mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, ",", NULL );
					isError = 1;
					break;
				}
				mk_get_token( lexer );
			}
			if( ( lexer->value & MK_LEX_TYPE_MASK ) != MK_LEX_TYPE_SYMBOL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<symbol name>", NULL );
				isError = 1;
				break;
			}
			arg = 
				( MK_VARIABLE * )mk_create_object( MK_TYPE_VARIABLE );
			arg->name = 
				mk_get_symbol_name_ptr( vm, lexer->text );
			if( mk_register_function_arg( vm, newNode->param, arg ) == 0 )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 15, arg->name, NULL );
				mk_destroy_node( vm->memoryPool, arg );
				isError = 1;
				break;
			}
			argCount ++;
		}
	}
	
	// do compile block
	if( isError == 0 && 
		lexer->value == MK_LEX_TYPE_RESERVED_MARK_BRACE )
	{
		MK_NODE_EXPR * expr = NULL;
		mk_get_token( lexer );
		expr = do_compile_expr( vm, lexer, MK_TYPE_EXPR_NONAME_BLOCK );
		if( expr == NULL )
		{
			isError = 1;
		}
		else
		{
			newNode->node = (unsigned int *)expr;
		}
	}
	if( isError != 0 )
	{
		mk_destroy_node( vm->memoryPool, newNode );
		newNode = NULL;
	}
	return newNode;
}

static
MK_TRY_BLOCK *do_compile_try_catch_finally_block( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	unsigned int isFail = 0;
	MK_TRY_BLOCK *tryBlock = 
		mk_create_object( MK_TYPE_TRY_BLOCK );

	// try block
	do
	{
		// lexer->text = "try"
		mk_get_token( lexer );
		if( lexer->value != MK_LEX_TYPE_EOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, 
					lexer->stream.line, 
					MK_ERROR_TYPE_COMILE_ERROR | 2, 
					lexer->text, 
					"<CR>", NULL );
			isFail = 1;
			break;
		}
		tryBlock->blockTry = 
			create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_TRY_BLOCK );
		if( do_compile_segment( vm, lexer, tryBlock->blockTry, MK_TYPE_SEGMENT_TRY ) == 0 )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, 
					lexer->stream.name, lexer->stream.line, 
					MK_ERROR_TYPE_COMILE_ERROR | 11, 
					"<try...catch...finally...segment>", NULL );
			isFail = 1;
			break;
		}
	}
	while( 0 );

	if( isFail == 0 )
	{
		// catch block
		tryBlock->blockCatch = 
			mk_create_vector( 
				MK_SIZEOF_VECTOR_DEFAULT, 
				MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
		do
		{
			MK_NODE_WITH_PARAM *param = NULL;
			MK_NODE_BLOCK *block = NULL;
			MK_CATCH_BLOCK *catchBlock = NULL;

			// lexer->text = "catch"
			mk_get_token( lexer );

			param = do_compile_node_widh_param( vm, lexer );
			if( param == NULL )
			{
				isFail = 1;
				break;
			}
			mk_get_token( lexer );

			block =
				create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_CATCH_BLOCK );
			if( do_compile_segment( vm, lexer, block, MK_TYPE_SEGMENT_CATCH ) == 0 )
				isFail = 1;	// error
			if( isFail != 0 )
			{
				mk_destroy_node( vm->memoryPool, param );
				mk_destroy_node( vm->memoryPool, block );
				break;
			}
			else
			{
				catchBlock = 
					malloc( sizeof( MK_CATCH_BLOCK ) );
				catchBlock->paramCatch = param;
				catchBlock->blockCatch = block;
				mk_push_vector( tryBlock->blockCatch, catchBlock );
			}
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_ELSE ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_FINALLY ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
				break;
			else if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_CATCH )
				isFail = 1;
		}
		while( 1 );
	}

	// else block ( no exception )
	if( isFail == 0 && lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_ELSE )
	{
		// lexer->text == "else" 
		mk_get_token( lexer );
		tryBlock->blockNoException =
			create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_CATCH_BLOCK );
		if( do_compile_segment( vm, lexer, tryBlock->blockNoException, MK_TYPE_SEGMENT_CATCH_ELSE ) == 0 )
			isFail = 1;
	}

	// finally block
	if( isFail == 0 && lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_FINALLY )
	{
		// lexer-> text == "finally"
		mk_get_token( lexer );
		
		tryBlock->blockFinally =
			create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_CATCH_BLOCK );
		if( do_compile_segment( vm, lexer, tryBlock->blockFinally, MK_TYPE_SEGMENT_FINALLY ) == 0 )
			isFail = 1;
	}

	if( isFail == 0 )
	{
		// lexer->text == "end"
		mk_get_token( lexer );
	}
	else
	{
		mk_destroy_node( vm->memoryPool, tryBlock );
		tryBlock = NULL;
	}
	return tryBlock;
}

static
MK_NODE_EXPR *do_compile_function_call( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	void *keyPtr = NULL;
	unsigned int isSuccess = 0;
	MK_NODE_EXPR *result = 
		( MK_NODE_EXPR * )mk_create_object( MK_TYPE_NODE_EXPR );
	result->flags = 
		MK_TYPE_NODE_EXPR | 
		MK_TYPE_NODE_EXPR_FUNCTION_CALL;
	do
	{
		result->u1.symbolName = mk_get_symbol_name_ptr( vm, lexer->text );
		mk_get_token( lexer );
		if( lexer->value != MK_LEX_TYPE_RESERVED_MARK_PARENTHESIS )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "(", NULL );
			break;
		}

		mk_get_token( lexer );
		if( lexer->value != MK_LEX_TYPE_RESERVED_MARK_END_PARENTHESIS )
		{
			result->u2.args = 
				mk_create_vector( MK_SIZEOF_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
			do
			{
				MK_NODE_EXPR *arg = 
					do_compile_expr( vm, lexer, MK_TYPE_EXPR_FUNCTION_ARG );
				if( arg == NULL )
				{
					if( vm->exceptionObject == NULL )
						mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "<method argument>", NULL );
					break;
				}
				if( result->u2.args == NULL )
					result->u2.args = 
					mk_create_vector( MK_SIZEOF_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
				mk_push_vector( result->u2.args, arg );
				if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_END_PARENTHESIS )
					break;
				else if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_CONMA )
					mk_get_token( lexer );
				else
					break;;
			}
			while( lexer->value != MK_LEX_TYPE_EOF );
			if( mk_size_vector( result->u2.args ) == 0 )
			{
				mk_destroy_vector_node( vm->memoryPool, result->u2.args );
				result->u2.args = NULL;
			}
		}
		if( lexer->value != MK_LEX_TYPE_RESERVED_MARK_END_PARENTHESIS )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, ")", NULL );
			break;
		}
		isSuccess = 1;
	}
	while( 0 );

	if( lexer->hasError != 0 )
		isSuccess = 0;

	if( isSuccess == 0 && result != NULL )
		mk_destroy_node( vm->memoryPool, result );

	return ( isSuccess != 0 ) ? result : NULL;
}

static
MK_NODE_EXPR *do_compile_symbol( MK_VM_STRUCT *vm, MK_LEXER *lexer, unsigned int state )
{
	MK_NODE_EXPR *multipleTop = NULL;
	MK_NODE_EXPR *current = NULL;
	int neg = 0;
	int isError = 0;
	do
	{
		current = NULL;
		if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_PARENTHESIS )
		{
			mk_get_token( lexer );
			current = 
				do_compile_expr( vm, lexer, MK_TYPE_EXPR_INNER_EXPR );
		}
		else
		{
			if( ( lexer->value == MK_LEX_TYPE_RESERVED_MARK_PLUS ||
					lexer->value == MK_LEX_TYPE_RESERVED_MARK_MINUS ) )
			{
				if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_MINUS )
					neg = 1;
				else if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_PLUS )
					neg = 2;
				mk_get_token( lexer );
			}
			if( lexer->value != MK_LEX_TYPE_INT_VALUE &&
				lexer->value != MK_LEX_TYPE_FLOAT_VALUE &&
				neg != 0 )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, 
											lexer->stream.name, 
											lexer->stream.line, 
											MK_ERROR_TYPE_COMILE_ERROR | 20, 
											lexer->text, 
											NULL );
				isError = 1;
			}

			switch( lexer->value & MK_LEX_TYPE_MASK )
			{
			case MK_LEX_TYPE_INT_VALUE:
			case MK_LEX_TYPE_FLOAT_VALUE:
				if( multipleTop != NULL )
					mk_raise_internal_error( vm, 
											lexer->stream.name,
											lexer->stream.line,
											MK_ERROR_TYPE_COMILE_ERROR | 3,
											".",
											lexer->text,
											"<symbol>",
											NULL );	// error
				else
					current = create_numeric_expr( lexer->value, lexer->text, neg );
				break;

			case MK_LEX_TYPE_STRING:
			case MK_LEX_TYPE_CHARCTER:
				current = create_string_expr( lexer->text, lexer->length );
				break;

			case MK_LEX_TYPE_SYMBOL:
				if( lexer->nextValue == MK_LEX_TYPE_RESERVED_MARK_PARENTHESIS )
					current = do_compile_function_call( vm, lexer );
				else
					current = create_symbol_expr( vm, lexer );
				break;

			case MK_LEX_TYPE_RESERVED_MARK:
				switch( lexer->value )
				{
				case MK_LEX_TYPE_RESERVED_MARK_DBLATMARK:
					if( ( lexer->nextValue & MK_LEX_TYPE_MASK ) == MK_LEX_TYPE_SYMBOL )
					{
						mk_get_token( lexer );
						current = create_static_member_variable_expr( vm, lexer );
					}
					break;
				case MK_LEX_TYPE_RESERVED_MARK_ATMARK:
					if( ( lexer->nextValue & MK_LEX_TYPE_MASK ) == MK_LEX_TYPE_SYMBOL )
					{
						mk_get_token( lexer );
						current = create_member_variable_expr( vm, lexer );
					}
					break;
				}
				break;
			case MK_LEX_TYPE_RESERVED_SYMBOL:
				switch( lexer->value )
				{
				case MK_LEX_TYPE_RESERVED_SYMBOL_NEW:
					current = create_new_expr( vm, lexer );
					break;

				case MK_LEX_TYPE_RESERVED_SYMBOL_THIS:
					current = create_this_expr( );
					break;

				case MK_LEX_TYPE_RESERVED_SYMBOL_SUPER:
					if( lexer->nextValue == MK_LEX_TYPE_RESERVED_MARK_PARENTHESIS )
					{
						current = do_compile_function_call( vm, lexer );
						if( current != NULL )
							current->flags = 
								MK_TYPE_NODE_EXPR | MK_TYPE_NODE_EXPR_SUPER_CALL;
					}
					else
					{
						current = create_super_expr( );
					}
					break;

				case MK_LEX_TYPE_RESERVED_SYMBOL_OWNER:
					current = create_owner_expr( );
					break;

				case MK_LEX_TYPE_RESERVED_SYMBOL_NIL:
					current = create_nil_expr( );
					break;
				case MK_LEX_TYPE_RESERVED_SYMBOL_TRUE:
					current = create_true_expr( );
					break;
				case MK_LEX_TYPE_RESERVED_SYMBOL_FALSE:
					current = create_false_expr( );
					break;
				}
				break;
			default:
				break;
			}
		}
		if( current == NULL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, 
										MK_ERROR_TYPE_COMILE_ERROR | 19, "<symbol name>", lexer->text, lexer->nextText, NULL );
			isError = 1;
			break;
		}

		mk_get_token( lexer );
		if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_DOT &&
			multipleTop == NULL )
		{
			multipleTop = create_multiple_top_expr( );
			multipleTop->u1.multipleSymbols = 
				mk_create_vector( MK_SIZEOF_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
		}
		if( multipleTop != NULL )
		{
			mk_push_vector( multipleTop->u1.multipleSymbols, current );
		}
		if( lexer->value != MK_LEX_TYPE_RESERVED_MARK_DOT )
			break;

		if( lexer->nextValue != MK_LEX_TYPE_SYMBOL &&
			lexer->nextValue != MK_LEX_TYPE_RESERVED_MARK &&
			lexer->nextValue != MK_LEX_TYPE_RESERVED_SYMBOL_NEW &&
			lexer->nextValue != MK_LEX_TYPE_RESERVED_SYMBOL_THIS &&
			lexer->nextValue != MK_LEX_TYPE_RESERVED_SYMBOL_SUPER &&
			lexer->nextValue != MK_LEX_TYPE_RESERVED_SYMBOL_OWNER &&
			lexer->nextValue != MK_LEX_TYPE_RESERVED_MARK_ATMARK &&
			lexer->nextValue != MK_LEX_TYPE_RESERVED_MARK_DBLATMARK )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, 
									MK_ERROR_TYPE_COMILE_ERROR | 3,
									lexer->text,
									lexer->nextText,
									"<symbol>",
									NULL );
			isError = 1;
			break;
		}
		mk_get_token( lexer );
	}while( lexer->value != MK_LEX_TYPE_EOF );

	if( lexer->hasError != 0 )
		isError = 1;

	if( isError != 0 )
	{
		if( multipleTop != NULL )
			mk_destroy_node( vm->memoryPool, multipleTop );
		else if( current != NULL )
			mk_destroy_node( vm->memoryPool, current );
		multipleTop = NULL;
		current = NULL;
	}
	return ( multipleTop != NULL ) ? multipleTop : current;
}

static
void do_compile_set_right_const_flag( unsigned int flagsParentOperator, MK_NODE_EXPR *right )
{
	if( ( right->flags & MK_TYPE_NODE_EXPR_CONSTANT ) == MK_TYPE_NODE_EXPR_CONSTANT &&
	   ( flagsParentOperator & MK_TYPE_NODE_EXPR_OPERATION_RIGHT_CONST ) )
		right->flags |= MK_TYPE_NODE_EXPR_CONST_FINAL;
}

static
void do_compile_set_left_const_flag( unsigned int flagsParentOperator, MK_NODE_EXPR *left )
{
	if( ( left->flags & MK_TYPE_NODE_EXPR_CONSTANT ) == MK_TYPE_NODE_EXPR_CONSTANT &&
	   ( flagsParentOperator & MK_TYPE_NODE_EXPR_OPERATION_LEFT_CONST ) )
		left->flags |= MK_TYPE_NODE_EXPR_CONST_FINAL;
}

static
MK_NODE_EXPR *do_compile_expr( MK_VM_STRUCT *vm, MK_LEXER *lexer,  unsigned int state )
{
	int isError = 0;
	int isEndExpr = 0;
	unsigned int type = 0;
	MK_VECTOR *stack = 
		mk_create_vector( MK_SIZEOF_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
	MK_NODE_EXPR *result = NULL;
	MK_NODE_EXPR *current = NULL;
	do
	{
		unsigned int sizeStack = 
			mk_size_vector( stack );
		current = do_compile_symbol( vm, lexer, state );
		if( current == NULL )
		{
			isError = 1;
			break;
		}

		// check expr end.
		switch( state )
		{
		case MK_TYPE_EXPR_IF_CONDITION:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_THEN )
				isEndExpr = 1;
			break;
		case MK_TYPE_EXPR_WHILE_CONDITION:
			if( lexer->value == MK_LEX_TYPE_EOL ||
				lexer->value == MK_LEX_TYPE_EOF )
				isEndExpr = 1;
			break;
		case MK_TYPE_EXPR_FUNCTION_ARG:
			if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_END_PARENTHESIS ||
				lexer->value == MK_LEX_TYPE_RESERVED_MARK_CONMA )
				isEndExpr = 1;
			break;
		case MK_TYPE_EXPR_ROOT_SEGMENT:
			if( lexer->value == MK_LEX_TYPE_EOL ||
				lexer->value == MK_LEX_TYPE_EOF )
				isEndExpr = 1;
			break;
		case MK_TYPE_EXPR_INNER_EXPR:
			if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_END_PARENTHESIS )
				isEndExpr = 1;
			break;
		case MK_TYPE_EXPR_NONAME_BLOCK:
			if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_END_BRACE )
				isEndExpr = 1;
			break;
		default:
			isError = 1;
			break;
		}
		if( isEndExpr != 0 )
		{
			if( sizeStack != 0 )
			{
				MK_NODE_EXPR *top = 
					mk_get_at_vector( stack, 0 );
				MK_NODE_EXPR *last = 
					mk_get_at_vector( stack, sizeStack - 1 );
				last->u2.right = current;
				do_compile_set_right_const_flag( last->flags, current );
				result = top;
			}
			else
			{
				result = current;
			}
			current = NULL;
			break;
		}
		if( ( lexer->value & MK_LEX_TYPE_MASK ) == MK_LEX_TYPE_RESERVED_MARK )
		{
			int index = 0;
			MK_NODE_EXPR *pExpr = NULL;
			MK_NODE_EXPR *pOperator = 
				create_operation_expr( lexer->value );
			int currentPriority = MK_LEX_RESERVED_MARK_PRIORITY( lexer->value );
			for( index = sizeStack - 1; index >= 0; index -- )
			{
				int targetPriority = 0;
				pExpr = 
					mk_get_at_vector( stack, index );
				targetPriority = MK_LEX_RESERVED_MARK_PRIORITY( pExpr->flags );
				if( targetPriority < currentPriority )
					break;
			}
			if( index == -1 )
				pExpr = NULL;

			if( pExpr == NULL )
			{
				if( sizeStack == 0 )
				{
					pOperator->u1.left = current;
					do_compile_set_left_const_flag( pOperator->flags, current);
				}
				else
				{
					MK_NODE_EXPR *last = 
						mk_get_at_vector( stack, sizeStack - 1 );
					last->u2.right = current;
					do_compile_set_right_const_flag( last->flags, current );
					pOperator->u1.left = mk_get_at_vector( stack, 0 );
					do_compile_set_left_const_flag( pOperator->flags, pOperator->u1.left );
					stack->used = 0;
				}
				mk_insert_at_vector( &stack, 0, pOperator );
			}
			else
			{
				MK_NODE_EXPR *last = 
					mk_get_at_vector( stack, sizeStack - 1 );
				last->u2.right = current;
				do_compile_set_right_const_flag( last->flags, current );
				pOperator->u1.left = pExpr->u2.right;
				do_compile_set_left_const_flag( pOperator->flags, pExpr->u2.right);
				pExpr->u2.right = pOperator;
				do_compile_set_right_const_flag( pExpr->flags, pOperator );
				stack->used = index + 1;
				mk_push_vector( stack, pOperator );
			}
			current = NULL;
		}
		else
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<operator>", NULL );
			isError = 1;
			break;
		}
		mk_get_token( lexer );
	}
	while ( lexer->value != MK_LEX_TYPE_EOF );

	if( lexer->hasError != 0 )
		isError = 1;

	if( isError != 0 && result != NULL )
	{
		mk_destroy_node( vm->memoryPool, result );
		result = NULL;
	}
	if( result == NULL )
	{
		if( current != NULL )
			mk_destroy_node( vm->memoryPool, current );
	}
	else
	{
		mk_trim_size_vector( stack, 0 );
	}
	mk_destroy_vector_node( vm->memoryPool, stack );
	return ( isEndExpr != 0 ) ? result : NULL;
}

static
int do_compile_segment( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_NODE_BLOCK *block, unsigned int typeSegment )
{
	unsigned int isSuccess = 0;
	MK_VECTOR *target = block->exprs;
	if( target == NULL )
		target = 
			mk_create_vector( MK_SIZEOF_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
	do
	{
		void *statement = NULL;

		switch( typeSegment )
		{
		case MK_TYPE_SEGMENT_IF:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_ELSEIF ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_ELSE )
				isSuccess = 1;
			break;
		case MK_TYPE_SEGMENT_WHILE:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
				isSuccess = 1;
			break;
		case MK_TYPE_SEGMENT_BLOCK:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
				isSuccess = 1;
			break;
		case MK_TYPE_SEGMENT_ROOT:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_CLASS ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_DEF ||
				lexer->value == MK_LEX_TYPE_EOF )
				isSuccess = 1;
			break;
		case MK_TYPE_SEGMENT_DOWHILE:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
				isSuccess = 1;
			break;
		case MK_TYPE_SEGMENT_TRY:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_CATCH )
				isSuccess = 1;
			break;
		case MK_TYPE_SEGMENT_CATCH:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_CATCH ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_ELSE ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_FINALLY ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
				isSuccess = 1;
		case MK_TYPE_SEGMENT_CATCH_ELSE:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_FINALLY ||
				lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
				isSuccess = 1;
			break;
		case MK_TYPE_SEGMENT_FINALLY:
			if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
				isSuccess = 1;
		default:
			break;
		}
		if( isSuccess != 0 )
			break;

		if( lexer->value == MK_LEX_TYPE_EOF )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 12, "<segment>", lexer->text, NULL );
			break;
		}

		switch( lexer->value )
		{
		case MK_LEX_TYPE_RESERVED_SYMBOL_IF:
			statement = do_compile_if_block( vm, lexer );
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_DO:
			statement = do_compile_do_while_block( vm, lexer );
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_WHILE:
			statement = do_compile_while_block( vm, lexer );
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_RETURN:
			statement = do_compile_return_state( vm, lexer );
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_BREAK:
			statement = do_compile_break_state( vm, lexer );
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_CONTINUE:
			statement = do_compile_continue_state( vm, lexer );
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_TRY:
			statement = do_compile_try_catch_finally_block( vm, lexer );
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_RAISE:
			statement = do_compile_raise_state( vm, lexer );
			break;
		case MK_LEX_TYPE_EOL:
			mk_get_token( lexer );
			continue;
		default:
			statement = do_compile_expr( vm, lexer, MK_TYPE_EXPR_ROOT_SEGMENT );
			break;
		}
		if( statement == NULL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "<segment>", NULL );
			break;
		}
		mk_push_vector( target, statement );
	}
	while( 1 );

	if( lexer->hasError != 0 )
		isSuccess = 0;
	block->exprs = target;

	return ( isSuccess != 0 ) ? 1 : 0;
}

static
int do_compile_function_argment( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_FUNCTION *newFunction )
{
	int argCount = 0;
	int isError = 0;
	while( 1 )
	{
		MK_VARIABLE *arg = NULL;
		mk_get_token( lexer );
		if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_END_PARENTHESIS )
			break;
		if( argCount > 0 )
		{
			if( lexer->value != MK_LEX_TYPE_RESERVED_MARK_CONMA )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, ",", NULL );
				isError = 1;
				break;
			}
			mk_get_token( lexer );
		}
		if( ( lexer->value & MK_LEX_TYPE_MASK ) != MK_LEX_TYPE_SYMBOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<symbol name>", NULL );
			isError = 1;
			break;
		}
		arg = 
			( MK_VARIABLE * )mk_create_object( MK_TYPE_VARIABLE );
		arg->name = 
			mk_get_symbol_name_ptr( vm, lexer->text );
		if( lexer->nextValue == MK_LEX_TYPE_RESERVED_MARK_EXCLAMATION )
		{
			arg->flags |= MK_TYPE_ATTRIBUTE_VARIABLE_CALL_BY_VALUE;
			mk_get_token( lexer );
		}
		if( newFunction->args == NULL )
			newFunction->args = 
				mk_create_vector( MK_SIZEOF_EXTEND_VECTOR_DEFAULT, MK_SIZEOF_EXTEND_VECTOR_DEFAULT );
		if( mk_register_function_arg( vm, newFunction->args, arg ) == 0 )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 15, arg->name, NULL );
			mk_destroy_node( vm->memoryPool, arg );
			isError = 1;
			break;
		}
		argCount ++;
	}
	return isError == 0 ;
}

static
int do_compile_function( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *targetClass )
{
	int isSuccess = 0;
	int isError = 0;
	MK_FUNCTION *newFunction = 
		(MK_FUNCTION*)mk_create_object( MK_TYPE_FUNCTION );
	do
	{
		if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_DEF )
			break;
		mk_get_token( lexer );
		switch( lexer->value )
		{
		case MK_LEX_TYPE_RESERVED_SYMBOL_CLASS:
			newFunction->flags |= MK_TYPE_ATTRIBUTE_FUNCTION_STATIC;
			mk_get_token( lexer );
			if( lexer->value != MK_LEX_TYPE_RESERVED_MARK_DOT )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, ".", NULL );
				isError = 1;
				break;
			}
			mk_get_token( lexer );
			break;

		case MK_LEX_TYPE_RESERVED_SYMBOL_INITIALIZE:
			newFunction->flags |= MK_TYPE_ATTRIBUTE_FUNCTION_CONSTRUCTER;
			break;

		case MK_LEX_TYPE_SYMBOL:
			break;

		default:
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, 
					lexer->stream.name, lexer->stream.line, 
					MK_ERROR_TYPE_COMILE_ERROR | 2, 
					lexer->text, "[class|initialize|<symbol name>]", NULL );
			isError = 1;
			break;
		}
		if( isError == 1 )
			break;
		newFunction->name = 
				mk_get_symbol_name_ptr( vm, lexer->text );

		mk_get_token( lexer );
		
		switch( lexer->value )
		{
		case MK_LEX_TYPE_RESERVED_MARK_PARENTHESIS:
			if( do_compile_function_argment( vm, lexer, newFunction ) == 0 )
				isError = 1;
			break;
		case MK_LEX_TYPE_EOL:
			break;
		
		default:
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "[(|<CR>]", NULL );
			isError = 1;
			break;
		}
		if( isError != 0 &&
			lexer->value != MK_LEX_TYPE_EOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<CR>", NULL );
			isError = 1;
		}
		if( isError != 0 )
			break;
		
		mk_get_token( lexer );
		newFunction->impl =
			create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_FUNCTION_ROOT );
		if( do_compile_segment( vm, lexer, newFunction->impl, MK_TYPE_SEGMENT_BLOCK ) == 0 )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 10, newFunction->name, "method", NULL );
			break;
		}
		if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_END )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "end", NULL );
			break;
		}
		mk_get_token( lexer );
		if( lexer->value != MK_LEX_TYPE_EOL &&
			lexer->value != MK_LEX_TYPE_EOF )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 3, "end", lexer->text, "<CR>", NULL );
			break;
		}
		isSuccess = 1;
	}
	while( 0 );

	if( lexer->hasError != 0 )
		isSuccess = 0;

	if( isSuccess == 0 )
	{
		mk_destroy_node( vm->memoryPool, newFunction );
	}
	else if( mk_register_function( vm, newFunction, targetClass ) == 0 )
	{
		if( vm->exceptionObject == NULL )
			mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 4, newFunction->name, "method", targetClass->nameThis, NULL );
		mk_destroy_node( vm->memoryPool, newFunction );
		isSuccess = 0;
	}
	return isSuccess;
}

static
int do_compile_property( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *targetClass )
{
	int isError = 0;
	int isDefinitionEnd = 0;
	unsigned int flags = MK_TYPE_VARIABLE;
	MK_VARIABLE *newVariable =
		( MK_VARIABLE * )mk_create_object( MK_TYPE_VARIABLE );
	do
	{
		if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_DBLATMARK )
		{
			flags |= MK_TYPE_ATTRIBUTE_VARIABLE_STATIC_MEMBER;
		}
		else if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_ATMARK )
		{
			flags |= MK_TYPE_ATTRIBUTE_VARIABLE_MEMBER;
		}
		else
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "property", NULL );
			isError = 1;
		}
		if( isError == 1 )
			break;

		 mk_get_token( lexer );
		if( ( lexer->value & MK_LEX_TYPE_MASK ) != MK_LEX_TYPE_SYMBOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<symbol name>", NULL ); 
			isError = 1;
		}
		if( isError == 1 )
			break;
		newVariable->name = 
			mk_get_symbol_name_ptr( vm, lexer->text );

		mk_get_token( lexer );
		switch( lexer->value )
		{
		case MK_LEX_TYPE_RESERVED_MARK_EXCLAMATION:
			flags |= MK_TYPE_ATTRIBUTE_VARIABLE_READ_PUBLIC;
			break;
		case MK_LEX_TYPE_RESERVED_MARK_QUESTION:
			flags |= MK_TYPE_ATTRIBUTE_VARIABLE_READ_PROTECTED;
			break;
		case MK_LEX_TYPE_RESERVED_MARK_DOT:
			flags |= MK_TYPE_ATTRIBUTE_VARIABLE_READ_PRIVATE;
			break;
		case MK_LEX_TYPE_EOL:
		case MK_LEX_TYPE_RESERVED_MARK_EQUAL:
		case MK_LEX_TYPE_EOF:
			isDefinitionEnd = 1;
			break;
		default:
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "[!|?|.|=|<CR>]", NULL );
			isError = 1;
			break;
		}
		if( isError != 0 )
			break;
		if( isDefinitionEnd == 0 )
		{
			mk_get_token( lexer );
			switch( lexer->value )
			{
			case MK_LEX_TYPE_RESERVED_MARK_EXCLAMATION:
				flags |= MK_TYPE_ATTRIBUTE_VARIABLE_WRITE_PUBLIC;
				break;
			case MK_LEX_TYPE_RESERVED_MARK_QUESTION:
				flags |= MK_TYPE_ATTRIBUTE_VARIABLE_WRITE_PROTECTED;
				break;
			case MK_LEX_TYPE_RESERVED_MARK_DOT:
				flags |= MK_TYPE_ATTRIBUTE_VARIABLE_WRITE_PRIVATE;
				break;
			case MK_LEX_TYPE_EOL:
			case MK_LEX_TYPE_RESERVED_MARK_EQUAL:
			case MK_LEX_TYPE_EOF:
				isDefinitionEnd = 1;
				break;
			default:
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "[!|?|.|=|<CR>]", NULL );
				isError = 1;
				break;
			}
		}
		if( isError != 0 )
			break;
		if( isDefinitionEnd == 0 )
			mk_get_token( lexer );
		
		if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_EQUAL )
		{
			// compile default value
			mk_get_token( lexer );
			newVariable->defaultValue = do_compile_expr( vm, lexer, MK_TYPE_EXPR_ROOT_SEGMENT );
			if( newVariable->defaultValue == NULL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 13, "=", NULL );
				isError = 1;
			}
		}
		else if( lexer->value != MK_LEX_TYPE_EOL && lexer->value != MK_LEX_TYPE_EOF )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "[=|<CR>]", NULL );
			isError = 1;
		}
		else
		{
			// lexer->value == MK_LEX_TYPE_EOL
		}
	}while( 0 );
	newVariable->flags = flags;

	if( lexer->hasError != 0 )
		isError = 1;

	if( isError != 0 )
	{
		mk_destroy_node( vm->memoryPool, newVariable );
	}
	else
	{
		if( mk_register_variable( vm, newVariable, targetClass ) == 0 )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 4, newVariable->name, "variable", targetClass->nameThis, NULL );
			mk_destroy_node( vm->memoryPool, newVariable );
			isError = 1;
		}
	}
	return !isError;
}

static
int do_compile_inner_class( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *targetClass )
{
	int isError = 0;
	do
	{
		switch( lexer->value )
		{
		case MK_LEX_TYPE_RESERVED_SYMBOL_DEF:
			if( do_compile_function( vm, lexer, targetClass ) == 0 )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "method", NULL );
				isError = 1;
				break;
			}
			if( lexer->value != MK_LEX_TYPE_EOL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 3, "end", lexer->text, "<CR>", NULL );
				isError = 1;
				break;
			}
			break;
		case MK_LEX_TYPE_RESERVED_MARK_ATMARK:
		case MK_LEX_TYPE_RESERVED_MARK_DBLATMARK:
			if( do_compile_property( vm, lexer, targetClass ) == 0 )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "property", NULL );
				isError = 1;
				break;
			}
			if( lexer->value != MK_LEX_TYPE_EOL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<CR>", NULL );
				isError = 1;
				break;
			}
			break;
		case MK_LEX_TYPE_EOL:
			mk_get_token( lexer );
			continue;

		case MK_LEX_TYPE_RESERVED_SYMBOL_END:
			break;

		default:
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "[end|def|@|@@]", NULL );
			isError = 1;
			break;
		}
		if( isError != 0 )
			break;
		if( lexer->value == MK_LEX_TYPE_RESERVED_SYMBOL_END )
		{
			mk_get_token( lexer );
			break;
		}
	}
	while( lexer->value != MK_LEX_TYPE_EOF );
	if( lexer->hasError != 0 )
		isError = 1;

	return !isError;
}

static
int do_compile_check_inhevior( MK_VM_STRUCT *vm, MK_LEXER *lexer, MK_CLASS *target, MK_HASHTABLE *classes )
{
	int isSuccess = 0;
	MK_CLASS *source = target, *previous = target;
	MK_HASHTABLE *check = 
		mk_create_hashtable( MK_SIZEOF_HASH_DEFAULT, MK_SIZEOF_EXTEND_HASH_DEFAULT );

	while( target != NULL )
	{
		if( mk_is_key_hashtable( check, target->nameThis ) != 0 )
			break;
		mk_insert_item_hashtable( check, target->nameThis, NULL );
		if( target->nameSuper == NULL || 
			target->nameSuper[0] == '\0' )
			break;
		previous = target;
		mk_find_item_hashtable( classes, target->nameSuper, (void**)&target );
	}
	if( target == NULL )
	{
		// super class not found.
		if( vm->exceptionObject == NULL )
			mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 6, source->nameThis, previous->nameSuper, NULL );
	}
	else if( target->nameSuper == NULL || 
		target->nameSuper[0] == '\0' )
	{
		isSuccess = 1;
	}
	else
	{
		// invalid super class
		if( vm->exceptionObject == NULL )
			mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 5, source->nameThis, NULL );
	}

	mk_destroy_hashtable_node( vm->memoryPool, check, 0 );

	return isSuccess;
}

static
MK_CLASS *do_compile_class_definition( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	int isSuccess = 0;
	MK_CLASS *newClass = mk_create_object( MK_TYPE_CLASS );
	do
	{
		if( lexer->value != MK_LEX_TYPE_RESERVED_SYMBOL_CLASS )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "class", NULL );
			break;	// $error	1
		}

		mk_get_token( lexer );
		if( ( lexer->value & MK_LEX_TYPE_MASK ) != MK_LEX_TYPE_SYMBOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "<class name>", NULL );
			break;	// $error	1
		}
		newClass->nameThis = 
			mk_get_symbol_name_ptr( vm, lexer->text );

		mk_get_token( lexer );
		if( lexer->value == MK_LEX_TYPE_RESERVED_MARK_CORON )
		{
			mk_get_token( lexer );
			if( ( lexer->value & MK_LEX_TYPE_MASK ) != MK_LEX_TYPE_SYMBOL )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 3, ":", lexer->text, "<super class name>", NULL );
				break;
			}
			newClass->nameSuper = 
				mk_get_symbol_name_ptr( vm, lexer->text );

			mk_get_token( lexer );
		}
		else if( lexer->value != MK_LEX_TYPE_EOL )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 2, lexer->text, "[:|<CR>]", NULL );
			break;
		}
		else
		{
			// super class not defined.
			newClass->nameSuper = 
				mk_get_symbol_name_ptr( vm, CLASS_INTERNAL_OBJECT );
		}
		isSuccess = 1;
	}while( 0 );

	if( isSuccess != 0 )
		isSuccess = do_compile_check_inhevior( vm, lexer, newClass, vm->code->classes );

	mk_get_token(lexer);
	if( isSuccess != 0 )
		isSuccess = 
			do_compile_inner_class( vm, lexer, newClass );

	if( isSuccess != 0 )
	{
		if( lexer->value != MK_LEX_TYPE_EOL &&
			lexer->value != MK_LEX_TYPE_EOF )
		{
			if( vm->exceptionObject == NULL )
				mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 3, "end", lexer->text, "<CR>", NULL );
			isSuccess = 0;
		}
		if( lexer->hasError != 0 )
			isSuccess = 0;
	}

	if( isSuccess == 0 )
	{
		mk_destroy_node( vm->memoryPool, newClass );
		newClass = NULL;
	}
	return newClass;
}

static
int do_compile_root( MK_VM_STRUCT *vm, MK_LEXER *lexer )
{
	int isError = 0;
	MK_CLASS *rootClass = NULL;
	MK_FUNCTION *entryPoint = NULL;
	MK_NODE_BLOCK *block = NULL;

	mk_find_item_hashtable( vm->code->classes, 
		mk_get_symbol_name_ptr( vm, mk_get_symbol_name_ptr( vm, CLASS_INTERNAL_KERNEL ) ), 
		(void**)&rootClass );
	mk_find_item_hashtable( rootClass->functions, 
		mk_get_symbol_name_ptr( vm, mk_get_symbol_name_ptr( vm, FUNCTIONNAME_ENTRYPOINT ) ), 
		(void**)&entryPoint );
	block = 
		create_mk_node_block( MK_TYPE_ATTRIBUTE_BLOCK_FUNCTION_ROOT );
	entryPoint->impl = block;

	mk_get_token( lexer );
	mk_get_token( lexer );
	do
	{
		switch( lexer->value )
		{
		case MK_LEX_TYPE_RESERVED_SYMBOL_CLASS:
			{
				MK_CLASS *newClass = 
					do_compile_class_definition( vm, lexer );
				if( newClass == NULL )
				{
					if( vm->exceptionObject == NULL )
						mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "class", NULL );
					isError = 1;
				}
				else
				{
					isError = !mk_register_class( vm, newClass );
					if( isError != 0 )
					{
						if( vm->exceptionObject == NULL )
							mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 4, newClass->nameThis, "class", "<root>", NULL );
						mk_destroy_node( vm->memoryPool, newClass );
					}
				}
			}
			break;
		case MK_LEX_TYPE_RESERVED_SYMBOL_DEF:
			if( do_compile_function( vm, lexer, rootClass ) == 0 )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 11, "method", NULL );
				isError = 1;
				break;
			}
			if( lexer->value != MK_LEX_TYPE_EOL &&
				lexer->value != MK_LEX_TYPE_EOF )
			{
				if( vm->exceptionObject == NULL )
					mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 3, "end", lexer->text, "<CR>", NULL );
				isError = 1;
				break;
			}
			break;
		case MK_LEX_TYPE_EOL:
			{
				mk_get_token( lexer );
			}
			continue;
		default:
			{
				int isSuccess = 
					do_compile_segment( vm, lexer, block, MK_TYPE_SEGMENT_ROOT );
				if( isSuccess == 0 )
				{
					if( vm->exceptionObject == NULL )
						mk_raise_internal_error( vm, 
							lexer->stream.name, 
							lexer->stream.line, 
							MK_ERROR_TYPE_COMILE_ERROR | 10, FUNCTIONNAME_ENTRYPOINT, 
							mk_get_symbol_name_ptr( vm, CLASS_INTERNAL_KERNEL ), NULL );
					isError = 1;
				}
			}
			break;
		}
		if( isError != 0 )
			break;
	}
	while( ( lexer->value & MK_LEX_TYPE_MASK ) != MK_LEX_TYPE_EOF );

	return !isError && !lexer->hasError;
}

int do_compile( MK_VM_STRUCT *vm, void*target )
{
	int result = 0;
	MK_STREAM *stream = NULL;
	MK_LEXER *lexer = NULL;
	MK_VM_FRAME_ITEM *exceptionObjectOld = vm->exceptionObject;

	if( vm == NULL || target == NULL )
		return 0;

	lexer =
		mk_create_lexer_for_file( );

	vm->exceptionObject = NULL;
	stream = 
		&( lexer->stream );
	stream->name = mk_allocate_memory_pool( 
		&vm->memoryPool, 
		target, 
		strlen( (MK_CHAR*)target) + 1 );
	stream->stream = stream->open( target );
	if( stream->stream != NULL )
	{
		result = do_compile_root( vm, lexer );
		stream->close( stream->stream );
	}
	else
	{
		if( vm->exceptionObject == NULL )
			mk_raise_internal_error( vm, lexer->stream.name, lexer->stream.line, MK_ERROR_TYPE_COMILE_ERROR | 22, target, NULL );
	}

	if( vm->exceptionObject != NULL )
	{
		MK_VM_FRAME_ITEM *pErrorDescription = NULL;
		mk_find_item_hashtable( vm->exceptionObject->classTypeValue.variables, 
							   mk_get_symbol_name_ptr(vm, "description" ), 
							   (void**)&pErrorDescription );
		lexer->stream.trace( MK_TRACE_TYPE_ERROR, pErrorDescription->stringTypeValue );
	}
	else
	{
		vm->exceptionObject = exceptionObjectOld;
	}
	mk_clear_lexer( lexer );

	return result;
}

MK_LEXER * mk_create_lexer_for_file( )
{
	MK_LEXER *lexer = 
		malloc( sizeof(MK_LEXER) );
	memset( lexer, 0x00, sizeof(MK_LEXER) );

	// initialize lexer default parameter
	lexer->defaultBufferLength = SIZEOF_TOKENBUFFER;
	lexer->maxBufferLength = 0;
	lexer->sizeExtendBufferLength = SIZEOF_TOKENBUFFER;

	// initialize stream for file.
	lexer->stream.open = mk_open;
	lexer->stream.getc = mk_getc;
	lexer->stream.close = mk_close;
	lexer->stream.trace = mk_trace;

	return lexer;
}

void mk_clear_lexer( MK_LEXER *lexer )
{
	free( lexer->text );
	free( lexer->nextText );
	free( lexer );
}
