/* vi: sw=8 ts=8 : */
/**
        libmkimgfs - support library for creating filesystem images

	Copyright (C) 2002-2003 Masuichi Noritoshi <nor@users.sourceforge.jp>

        This program is free software; you can redistribute it and/or
        modify it under the terms of the GNU General Public License
        as published by the Free Software Foundation; either version
        2 of the License, or (at your option) any later version.
	
	This code is derived from:

	genext2fs.c

	ext2 filesystem generator for embedded systems
	Copyright (C) 2000 Xavier Bestel <xavier.bestel@free.fr>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; version
	2 of the License.
**/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#ifdef HAVE_STRING_H
#       include <string.h>
#else
#       ifdef HAVE_STRINGS_H
#               include <strings.h>
#       endif
#endif
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>

#include <util.h>
#include <srcinode.h>

#include <mkimgfs.h>

#define XIS_SPECIAL_DOT(s)	((!strcmp((s), ".")) || (!strcmp((s), "..")))

uint32 TVfs_AddEntry_TSrcInode(TVfs *fs, uint32 this_nod, TSrcInode* si);

/** **************************************************************************
	adds entries to the filesystem from a text file
************************************************************************** **/
void
TVfs_AddListedEntriesInFile(TVfs *fs, uint32 this_nod, FILE * fh)
{
	uint32 mode;
	uint32 nod;
	uint32 nod2;
	char cmod[11], path[1024], *name, *dir;
	int major, minor;
	char buf[2048];
	int n;

	for (;;) {
		if (feof(fh)) {
			break;
		}
		if (!fgets(buf, 2048, fh)) {
			break;
		}
		n = strlen(buf);
		buf[n-1] = '\0';
		if (n > 0 && buf[n-1] == '\n') {
			buf[n-1] = '\0';
		}
		TVfs_AddEntryByString(fs, this_nod, buf);
	}
}

/** **************************************************************************
************************************************************************** **/
uint32
TVfs_AddEntryByString(TVfs *fs, uint32 this_nod, const char* line)
{
	uint32 nod;

	TSrcInode* si = TSrcInode_New_String(line);

	nod = TVfs_AddEntry_TSrcInode(fs, this_nod, si);
	
	SafeDel(TSrcInode_Fini, si);

	return nod;
}

/** **************************************************************************
************************************************************************** **/
uint32
TVfs_AddEntry_TSrcInode(TVfs *fs, uint32 this_nod, TSrcInode* si)
{
	uint32 nod;
	uint32 nod2 = 0;
	TSplitPath* sph;
	
	sph = si->mTargets;
	if (!sph) {
		/* FIX ME ! : ??? */
		return 0;
	}

	nod = fs->mFsops->mfFindInodeOf(fs, this_nod, sph->mDir);
	if (!nod) {
		errexit("can't find directory '%s' to create '%s'",
			sph->mDir, sph->mBase
		);
	}
	if ((!strcmp(sph->mBase, ".")) || (!strcmp(sph->mBase, ".."))) {
		/* FIX ME ! : skip... */
		return 0;
	}

	if ((si->mMode & VFS_IFMT) != VFS_IFHLNK) {
		nod2 = fs->mFsops->mfAllocateBlockForInode(fs, nod, si);
	}
	switch (si->mMode & VFS_IFMT) {
	case VFS_IFDIR:
		nod2 = fs->mFsops->mfAddEntryToDirectory(
				fs, nod, nod2, sph->mBase
		       );
		fs->mFsops->mfAddEntryToDirectory(fs, nod2, nod2, ".");
		fs->mFsops->mfAddEntryToDirectory(fs, nod2, nod, "..");
		break;
	case VFS_IFBLK:
	case VFS_IFCHR:
		nod2 = fs->mFsops->mfAddEntryToDirectory(
				fs, nod, nod2, sph->mBase
		       );
		break;
	case VFS_IFLNK:
		nod2 = fs->mFsops->mfAddEntryToDirectory(
				fs, nod, nod2, sph->mBase
		       );
		if (fs->mFsops->mfAddSymlink) {
			fs->mFsops->mfAddSymlink(fs, nod2, si);
		} else {
			/* FIX ME ! : Oops!! Cannot add symlink */
		}
		break;
	case VFS_IFREG:
		nod2 = fs->mFsops->mfAddEntryToDirectory(
				fs, nod, nod2, sph->mBase
		       );
		if (fs->mFsops->mfAddRegulerFile) {
			fs->mFsops->mfAddRegulerFile(fs, nod2, si);
		} else {
			/* FIX ME ! : Oops!! Cannot add reguler file */
		}
		/* fail into the next case */
	case VFS_IFHLNK:
		if (!nod2) {
			nod2 = fs->mFsops->mfFindInodeOf(fs, nod, sph->mBase);
		}
		while (sph->mNext) {
			sph = sph->mNext;
			if (XIS_SPECIAL_DOT(sph->mBase)) {
				/* FIX ME ! : */
				continue;
			}
			nod = fs->mFsops->mfFindInodeOf(
					fs, this_nod, sph->mDir
			      );
			if (!nod) {
				errexit("can't find directory '%s'"
					" to create '%s'",
					sph->mDir, sph->mBase
				);
			}
			fs->mFsops->mfAddEntryToDirectory(
				fs, nod, nod2, sph->mBase
			);
		}
		break;
	default:
		/* FIX ME ! : */
		break;
	}
	return nod2;
}

/** **************************************************************************
************************************************************************** **/
int
TVfs_MakeImage(TVfsOps* fsops, TVfsParam* vfsbp)
{
	FILE* fh;
	TVfs *fs;

	if (vfsbp->fsin) {
		if (!(fh = fopen(vfsbp->fsin, "r"))) {
			pexit(vfsbp->fsin);
		}
		fs = fsops->mfLoadFromFile(fsops, vfsbp, fh);
		fclose(fh);
	} else {
		fs = fsops->mfCreateFs(fsops, vfsbp);
	}

	/*
	 * put files into the filesystem image.
	 */
	if (vfsbp->fsspec) {
		if (!(fh = fopen(vfsbp->fsspec, "r"))) {
			pexit(vfsbp->fsspec);
		}
		TVfs_AddListedEntriesInFile(
			fs, fsops->mfGetRootInodeNo(fs), fh
		);
		fclose(fh);
	} else {
		TVfs_AddListedEntriesInFile(
			fs, fsops->mfGetRootInodeNo(fs), stdin
		);
	}
		
	if (vfsbp->emptyval) {
		fsops->mfFillUnallocatedBlock(fs, vfsbp->emptyval);
	}

	/*
	 * print the resulting report.
	 */
	if (vfsbp->fsrep) {
		if (!(fh = fopen(vfsbp->fsrep, "w"))) {
			pexit(vfsbp->fsrep);
		}
		fsops->mfPrint(fs, stderr);
		fclose(fh);
	} else if (vfsbp->verbose) {
		fsops->mfPrint(fs, stderr);
	}

	if (vfsbp->fsout) {
		if (!(fh = fopen(vfsbp->fsout, "w"))) {
			pexit(vfsbp->fsout);
		}
		fsops->mfSaveToFile(fs, fh);
		fclose(fh);
	} else {
		fsops->mfSaveToFile(fs, stdout);
	}

	return 0;
}


/* END-OF-FILE */
