/* vi: sw=8 ts=8 : */
/**
	mkimge2fs - create an ext2fs image like as mkisofs

	Copyright (c) 2002-2003 Masuichi Noritoshi <nor@users.sourceforge.jp>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version
	2 of the License, or (at your option) any later version.
	
	This code is derived from:

	genext2fs.c

	ext2 filesystem generator for embedded systems
	Copyright (C) 2000 Xavier Bestel <xavier.bestel@free.fr>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; version
	2 of the License.
**/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#ifdef HAVE_STRING_H
#       include <string.h>
#else
#       ifdef HAVE_STRINGS_H
#               include <strings.h>
#       endif
#endif
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>

#include <util.h>
#include <srcinode.h>
#include <mkimgfs.h>
#include <mkimge2fs.h>

typedef struct {
	TVfsParam mCommon;
	int nbblocks;
	int nbinodes;
	int nbresrvd;
} TExt2fsParam;

TVfs* TExt2fs_LoadFromFile(TVfsOps*, TVfsParam*, FILE * fh);
TVfs* TExt2fs_CreateFs(TVfsOps*, TVfsParam*);
void TExt2fs_FillUnallocatedBlock(TVfs* fs, char c);
void TExt2fs_Print(TVfs *fs, FILE* fp);
void TExt2fs_SaveToFile(TVfs *fs, FILE * fh);
uint32 TExt2fs_FindInodeOf(TVfs *fs, uint32 nod, const char * name);
uint32 TExt2fs_AddEntryToDirectory(TVfs *fs, uint32 dnod, uint32 nod, const char* name);
uint32 TExt2fs_AllocateBlockForInode(TVfs* fs, uint32 nod, TSrcInode* si);
uint32 TExt2fs_GetRootInodeNo(TVfs* fs);
void TExt2fs_AddSymlink(TVfs* vfs, uint32 nod2, TSrcInode* si);
void TExt2fs_AddRegulerFile(TVfs* vfs, uint32 nod2, TSrcInode* si);


/** **************************************************************************
TBitmap_IsSet

RETURN: 
	0    : item-bit is 0
	o.w. : item-bit is 1

************************************************************************** **/
inline uint32
TBitmap_IsSet(TBitmap* b, uint32 item)
{
	return b->bm[(item-1) / 8] & (1 << ((item-1) % 8));
}

/** **************************************************************************
TBitmap_FindFreeBit

************************************************************************** **/
uint32
TBitmap_FindFreeBit(const TBitmap* b)
{
	int i;
	uint8 m;

	for(i = 0; i < BLOCKSIZE; i++) {
		m = b->bm[i];
		if (m == 0xff) continue;
		if (!(m & 0x01)) return i * 8 + 1;
		if (!(m & 0x02)) return i * 8 + 2;
		if (!(m & 0x04)) return i * 8 + 3;
		if (!(m & 0x08)) return i * 8 + 4;
		if (!(m & 0x10)) return i * 8 + 5;
		if (!(m & 0x20)) return i * 8 + 6;
		if (!(m & 0x40)) return i * 8 + 7;
		return i * 8 + 8;
	}
	return 0;
}

/** **************************************************************************
TBitmap_SetBit
************************************************************************** **/
void
TBitmap_SetBit(TBitmap* b, uint32 item)
{
	b->bm[(item-1) / 8] |= (1 << ((item-1) % 8));
}

/** **************************************************************************
TBitmap_ResetBit
************************************************************************** **/
#if 0
void
TBitmap_ResetBit(TBitmap* b, uint32 item)
{
	b->bm[(item-1) / 8] &= ~(1 << ((item-1) % 8));
}
#endif

/** **************************************************************************
TBitmap_FindFreeBitAndSetIt
************************************************************************** **/
uint32
TBitmap_FindFreeBitAndSetIt(TBitmap* b)
{
	uint32 item = TBitmap_FindFreeBit(b);

	if (item) {
		TBitmap_SetBit(b, item);
	}
	return item;
}

/** **************************************************************************
************************************************************************** **/
void
TBitmap_Print(TBitmap* b, uint32 max, FILE* fp)
{
	uint32 i;

	fprintf(fp, "----+----1----+----2----+----3----+----4----+----5\n");
	for (i = 1 ; i <= max; i++) {
		fputc((TBitmap_IsSet(b, i) ? '*' : '.'), fp);
		if (!(i % 50)) {
			fprintf(fp, "\n");
		}
	}
	if ((i-1) % 50) {
		fprintf(fp, "\n");
	}
}

/** **************************************************************************
TExt2fs_GetBlockPtr

	returns pointer to given block

************************************************************************** **/
inline uint8*
TExt2fs_GetBlockPtr(TExt2fs *fs, uint32 blk)
{
	return fs->rawimg + blk*BLOCKSIZE;
}

/** **************************************************************************
TExt2fs_GetInodePtr

	returns pointer to given inode

************************************************************************** **/
inline inode*
TExt2fs_GetInodePtr(TExt2fs *fs, uint32 nod)
{
	if (!nod) {
		fprintf(stderr, "nod == 0\n"); /* FIX ME ! : */
	}
	return &fs->itab[nod-1];
}

/** **************************************************************************
TExt2fs_AllocateBlock

return:
	0	hole.
	!0	block no.
************************************************************************** **/
uint32
TExt2fs_AllocateBlock(TExt2fs *fs, uint32 hole)
{
	uint32 bn;

	if (hole) {
		return 0;
	}
		
	bn = TBitmap_FindFreeBitAndSetIt(fs->bbm);
	if (!bn) {
		errexit("No free space (couldn't allocate a block)");
	}
	if (!(fs->gd->bg_free_blocks_count--)) {
		errexit("Corrupted FS (group descr. free blocks count == 0)");
	}
	if (!(fs->sb->s_free_blocks_count--)) {
		errexit("Corrupted FS (superblock free blocks count == 0)");
	}
	return bn;
}

/** **************************************************************************
TExt2fs_AllocateBlockForIndirect

	allocate block for indirect block
************************************************************************** **/
inline uint32
TExt2fs_AllocateBlockForIndirect(TExt2fs *fs)
{
	return TExt2fs_AllocateBlock(fs, 0);
}

/** **************************************************************************
TExt2fs_AllocateBlockForInode
	allocate a inode
************************************************************************** **/
uint32
TExt2fs_AllocateBlockForInode(TVfs* vfs, uint32 nod__unused, TSrcInode* si)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	uint32 nod2;
	inode* ip;

	nod2 = TBitmap_FindFreeBitAndSetIt(fs->ibm);

	if (!nod2) {
		errexit("No free inode (couldn't allocate a inode)");
	}
	if (!(fs->gd->bg_free_inodes_count--)) {
		errexit("Corrupted FS (group descr. free inode count == 0)");
	}
	if (!(fs->sb->s_free_inodes_count--)) {
		errexit("Corrupted FS (superblock free inode count == 0)");
	}

	ip = TExt2fs_GetInodePtr(fs, nod2);

	ip->i_mode = si->mMode;
	ip->i_uid = si->mUid;
	ip->i_gid = si->mGid;
	ip->i_atime = si->mAtime;
	ip->i_ctime = si->mCtime;
	ip->i_mtime = si->mMtime;

	switch (si->mMode & VFS_IFMT) {
	case VFS_IFDIR:
		fs->gd->bg_used_dirs_count++;
		break;
	case VFS_IFBLK:
	case VFS_IFCHR:
		((uint8*)ip->i_block)[0] = si->mMinor;
		((uint8*)ip->i_block)[1] = si->mMajor;
		break;
	default:
		/* nothing to do. */
		break;
	}
	
	return nod2;
}

/** **************************************************************************
	initalize a TBpFs (iterator for blocks list)
************************************************************************** **/
void
TBpFs_Init(TBpFs* bp, TExt2fs* fs, uint32 nod)
{
	bp->asnfs = fs;
	bp->asnod = nod;
	bp->asnode = TExt2fs_GetInodePtr(bp->asnfs, bp->asnod);
	bp->bnum = 0;
	bp->bpdir = EXT2_BP_INIT;
	bp->state = EXT2_BP_STATE_INI;
}

/** **************************************************************************
	finalize a TBpFs
************************************************************************** **/
#if 0
void
TBpFs_Fini(TBpFs* bp)
{
	bp->asnfs = NULL;
}
#endif

/** **************************************************************************
************************************************************************** **/
void
TBpFs_ExtendIndirectBlock(TBpFs* bp)
{
	uint32* b;
	inode* node = bp->asnode;
	TExt2fs* fs = bp->asnfs;

	switch (bp->state) {
	case EXT2_BP_STATE_IND:
		node->i_block[EXT2_BP_IND] =
			TExt2fs_AllocateBlockForIndirect(fs);
		break;
	case EXT2_BP_STATE_DIND:
		if (!(node->i_block[EXT2_BP_DIND])) {
			node->i_block[EXT2_BP_DIND] =
				TExt2fs_AllocateBlockForIndirect(fs);
		}
		b = (uint32*)TExt2fs_GetBlockPtr(fs,
			node->i_block[EXT2_BP_DIND]
		    );
		b[bp->bpind] = TExt2fs_AllocateBlockForIndirect(fs);
		break;
	case EXT2_BP_STATE_TIND:
		errexit("file too big ! blocks list for "
			"inode %d extends past double "
			"indirect blocks!", bp->asnod);
		break;
	}
}

/** **************************************************************************
************************************************************************** **/
uint32*
TBpFs_GetRefToCurrentPos(TBpFs* bp)
{
	uint32* b;
	uint32* ref;
	inode* node = bp->asnode;
	TExt2fs* fs = bp->asnfs;

	switch (bp->state) {
	case EXT2_BP_STATE_DIR:
	case EXT2_BP_STATE_DIR_LAST:
		ref = &(node->i_block[bp->bpdir]);
		break;
	case EXT2_BP_STATE_IND:
	case EXT2_BP_STATE_IND_LAST:
		b = (uint32*)TExt2fs_GetBlockPtr(fs,node->i_block[EXT2_BP_IND]);
		ref = &b[bp->bpind];
		break;
	case EXT2_BP_STATE_DIND:
	case EXT2_BP_STATE_DIND_LAST:
		b = (uint32*)TExt2fs_GetBlockPtr(fs,
			node->i_block[EXT2_BP_DIND]
		    );
		b = (uint32*)TExt2fs_GetBlockPtr(fs, b[bp->bpind]);
		ref = &b[bp->bpdind];
		break;
	case EXT2_BP_STATE_TIND:
	case EXT2_BP_STATE_TIND_LAST:
		b = (uint32*)TExt2fs_GetBlockPtr(fs,
			node->i_block[EXT2_BP_TIND]
		    );
		b = (uint32*)TExt2fs_GetBlockPtr(fs, b[bp->bpind]);
		b = (uint32*)TExt2fs_GetBlockPtr(fs, b[bp->bpdind]);
		ref = &b[bp->bptind];
		break;
	}
	return ref;
}

/** **************************************************************************
************************************************************************** **/
uint32
TBpFs_AllocateBlock(TBpFs* bp, uint32 hole)
{
	return TExt2fs_AllocateBlock(bp->asnfs, hole);
}

/** **************************************************************************
RETURN:
	next block of inode (WALK_END for end)
	if create>0, append a newly allocated block at the end
	if hole!=0, create a hole in the file
************************************************************************** **/
uint32
TBpFs_Next(TBpFs* bp, uint32 *create, uint32 hole)
{
	uint32 *bkref = NULL;
	uint32 *b;
	uint32 extend = 0;
	uint32 ipb = BLOCKSIZE/sizeof(uint32);

	if (bp->bnum * B512PERBLOCK >= bp->asnode->i_blocks) {
		if (create && (*create)--) {
			extend = 1;
		} else {
			return WALK_END;
		}
	}

	 /* first direct block */
	switch (bp->state) {
	case EXT2_BP_STATE_INI:
		bp->bpdir = 0;
		bp->state = EXT2_BP_STATE_DIR;
		break;

	case EXT2_BP_STATE_DIR:
		bp->bpdir++;
		if (bp->bpdir == EXT2_BP_IND-1) {
			bp->state = EXT2_BP_STATE_DIR_LAST;
		}
		break;

	case EXT2_BP_STATE_DIR_LAST:
		bp->bpdir = EXT2_BP_IND;
		bp->bpind = 0;
		bp->state = EXT2_BP_STATE_IND;
		bp->bnum++;
		if (extend) {
			TBpFs_ExtendIndirectBlock(bp);
		}
		break;

	case EXT2_BP_STATE_IND:
		bp->bpind++;
		if (bp->bpind == ipb-1) {
			bp->state = EXT2_BP_STATE_IND_LAST;
		}
		break;

	case EXT2_BP_STATE_IND_LAST:
		bp->bpdir = EXT2_BP_DIND;
		bp->bpind = 0;
		bp->bpdind = 0;
		bp->state = EXT2_BP_STATE_DIND;
		bp->bnum += 2;
		if (extend) {
			TBpFs_ExtendIndirectBlock(bp);
		}
		break;

	case EXT2_BP_STATE_DIND:
		bp->bpdind++;
		if (bp->bpdind == ipb-1) {
			bp->state = EXT2_BP_STATE_DIND_LAST;
		}
		break;

	case EXT2_BP_STATE_DIND_LAST:
		bp->bpdind = 0;
		bp->bpind++;
		if (bp->bpind == ipb-1) {
			bp->bnum += 3;
			bp->state = EXT2_BP_STATE_TIND;
		} else {
			bp->bnum++;
			bp->state = EXT2_BP_STATE_DIND;
		}
		if (extend) {
			TBpFs_ExtendIndirectBlock(bp);
		}
		break;
	default:
		errexit("file too big ! blocks list for "
			"inode %d extends past double indirect blocks!",
			bp->asnod
		);
		break;
	}

	bkref = TBpFs_GetRefToCurrentPos(bp);
	if (extend) {
		*bkref = TBpFs_AllocateBlock(bp, hole);
	}

	if (*bkref) {
		bp->bnum++;
		if (!TBitmap_IsSet(bp->asnfs->bbm, *bkref)) {
			errexit("[block %d of inode %d is unallocated !]",
				*bkref, bp->asnod
			);
		}
	}

	if (extend) {
		bp->asnode->i_blocks = bp->bnum * B512PERBLOCK;
	}

	return *bkref;
}

/** **************************************************************************
	add blocks to an inode (file/dir/etc...)
************************************************************************** **/
void
TExt2fs_ExtendBlock(TExt2fs *fs, uint32 nod, uint8 b[BLOCKSIZE], int amount)
{
	int create = amount;
	TBpFs bp;
	TBpFs lbw;
	uint32 bk;

	TBpFs_Init(&bp, fs, nod);

	lbw = bp;
	while ((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		lbw = bp;
	}
	bp = lbw;

	while (create) {
		int i;
		int hf = 1;

		if (!(fs->mCommon.holes)) {
			hf = 0;
		} else {
			/* check that block is filled with zero */
			uint32* d = (int32*)(b + BLOCKSIZE * (amount - create));
			for (i = 0; i < BLOCKSIZE / sizeof(uint32); i++) {
				if (d[i]) {
					hf = 0;
					break;
				}
			}
		}
		if ((bk = TBpFs_Next(&bp, &create, hf)) == WALK_END) {
			break;
		}
		if (!hf) {
			memcpy(TExt2fs_GetBlockPtr(fs, bk),
				b + BLOCKSIZE * (amount - (create+1)),
				BLOCKSIZE);
		}
	}
}

/** **************************************************************************
************************************************************************** **/
uint32
TExt2fs_AddEntryToDirectory(TVfs *vfs, uint32 dnod, uint32 nod, const char* name)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	TBpFs bp;
	uint32 bk;
	directory *d;
	int elen, nlen;
	uint8 *b;
	inode *node;
	inode *pnode;

	pnode = TExt2fs_GetInodePtr(fs, dnod);

	if (!S_ISDIR(pnode->i_mode)) {
		/* FIX ME ! : using S_ISDIR is not good idea ! */
		errexit("can't add '%s' to a non-directory", name);
	}
	if (!*name) {
		errexit("bad name '%s' (not meaningful)", name);
	}
	if (strchr(name, '/')) {
		errexit("bad name '%s' (contains a slash)", name);
	}

	nlen = strlen(name) + 1;
	elen = sizeof(directory) + ut_RoundUp(nlen, 4);
	if(elen > BLOCKSIZE) {
		errexit("bad name '%s' (too long)", name);
	}

	TBpFs_Init(&bp, fs, dnod);

	while ((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		b = TExt2fs_GetBlockPtr(fs, bk);

#define XNEXT_DENTRY(d)		\
		((directory*)((int8*)(d) + (d)->d_rec_len))
#define XISVALID_DENTRY(b,d,bs)	\
		(((uint8*)(b) + (bs)) - (uint8*)(d) > sizeof(directory))

		// for all dir entries in block
		for (d = (directory*)b; XISVALID_DENTRY(b, d, BLOCKSIZE) ;
		     d = XNEXT_DENTRY(d)) {
			if ((!d->d_inode) && (d->d_rec_len >= elen)) {
				/* Update link count */
				d->d_inode = nod;
				node = TExt2fs_GetInodePtr(fs, nod);
				node->i_links_count++;

				d->d_name_len = nlen -1;
				strcpy(d->d_name, name); /* XXX USE memcpy ? */
				return nod;
			}
			if (d->d_rec_len >= (sizeof(directory) + ut_RoundUp(d->d_name_len, 4) + elen)) {
				elen = d->d_rec_len;
				d->d_rec_len = sizeof(directory) + ut_RoundUp(d->d_name_len, 4);
				elen -= d->d_rec_len;
				d = (directory*) (((int8*)d) + d->d_rec_len);
				d->d_rec_len = elen;

				d->d_inode = nod;
				node = TExt2fs_GetInodePtr(fs, nod);
				node->i_links_count++;

				d->d_name_len = nlen -1;
				strcpy(d->d_name, name);

fflush(stderr);
				return nod;
			}
		}
	}
	// we found no free entry in the directory, so we add a block
	{
		uint8 b[BLOCKSIZE];

		memset(b, 0, sizeof(b));
		d = (directory*)b;

		d->d_inode = nod;
		node = TExt2fs_GetInodePtr(fs, nod);
		node->i_links_count++;

		d->d_rec_len = BLOCKSIZE;
		d->d_name_len = nlen -1;
		strcpy(d->d_name, name);

		TExt2fs_ExtendBlock(fs, dnod, b, 1);
		TExt2fs_GetInodePtr(fs, dnod)->i_size += BLOCKSIZE;
	}
	return nod;
}

/** **************************************************************************
************************************************************************** **/
void
TExt2fs_AddSymlink(TVfs* vfs, uint32 nod2, TSrcInode* si)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	inode* ip;
	uint32 nlen;
	uint8* blk;
	uint32 rlen;

	nlen = strlen(si->mSourcePath);
	ip = TExt2fs_GetInodePtr(fs, nod2);
	ip->i_size = nlen;

	if (nlen <= 4 * (EXT2_TIND_BLOCK+1)) {	/* SHORT SYMLINK */
		strncpy((char*)ip->i_block, si->mSourcePath, nlen);
	} else {				/* LONG SYMLINK */
		rlen = ut_RoundUp(nlen, BLOCKSIZE);
		blk = XMALLOC(uint8, rlen);
		if (rlen > nlen) {
			memset(blk + nlen, 0, rlen - nlen);
		}
		strncpy(blk, si->mSourcePath, nlen);

		TExt2fs_ExtendBlock(fs, nod2, blk, rlen / BLOCKSIZE);
		free(blk);
	}
}

/** **************************************************************************
************************************************************************** **/
void
TExt2fs_AddRegulerFile(TVfs* vfs, uint32 nod2, TSrcInode* si)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	inode* ip;
	uint8* blk;
	struct stat st;
	FILE *fh;
	uint32 rlen;

	ip = TExt2fs_GetInodePtr(fs, nod2);
	lstat(si->mSourcePath, &st);
	if ((st.st_mode & S_IFMT) != S_IFREG) {
		pexit(si->mSourcePath);
	}
	if (!(fh = fopen(si->mSourcePath, "r"))) {
		pexit(si->mSourcePath);
	}
	ip->i_size = st.st_size;

	rlen = ut_RoundUp(ip->i_size, BLOCKSIZE);
	blk = XMALLOC(uint8, rlen);
	if (rlen > st.st_size) {
		memset(blk + st.st_size, 0, rlen - st.st_size);
	}

	if (fh) {
		fread(blk, ip->i_size, 1, fh);
	} else {
		memset(blk, 0, ip->i_size);
	}
	TExt2fs_ExtendBlock(fs, nod2, blk, rlen / BLOCKSIZE);
	free(blk);
	fclose(fh);
}

/** **************************************************************************
************************************************************************** **/
uint32
TExt2fs_FindDir(TExt2fs *fs, uint32 nod, const char * name)
{
	TBpFs bp;
	uint32 bk;
	int nlen = strlen(name);
	TBpFs_Init(&bp, fs, nod);

	while((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END)
	{
		directory *d;
		uint8 *b;
		b = TExt2fs_GetBlockPtr(fs, bk);
		for (d = (directory*)b;
			(int8*)d + sizeof(*d) < (int8*)b + BLOCKSIZE;
			d = (directory*)((int8*)d + d->d_rec_len)) {
			if (d->d_inode && (nlen == d->d_name_len) && !strncmp(d->d_name, name, nlen)) {
				return d->d_inode;
			}
		}
	}
	return 0;
}

/** **************************************************************************
************************************************************************** **/
uint32
TExt2fs_FindInodeOf(TVfs *vfs, uint32 nod, const char * name)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	char *p, *n, *n2 = strdup(name);
	n = n2;

	while(*n == '/') {
		nod = EXT2_ROOT_INO;
		n++;
	}

	while(*n) {
		if((p = strchr(n, '/')))
			(*p) = 0;
		if (!(nod = TExt2fs_FindDir(fs, nod, n)))
			break;
		if (p)
			n = p + 1;
		else
			break;
	}

	free(n2);
	return nod;
}

/** **************************************************************************
************************************************************************** **/
uint32
TExt2fs_GetRootInodeNo(TVfs* fs)
{
	return EXT2_ROOT_INO;
}

/** **************************************************************************
	Create new filesystem
************************************************************************** **/
TVfs*
TExt2fs_CreateFs(TVfsOps* vfsops, TVfsParam* param)
{
	TExt2fsParam* p = (TExt2fsParam*)param;
	int i;
	TExt2fs *fs;
	directory *d;
	uint8 b[BLOCKSIZE];
	uint32 nod;
	
	
	if(p->nbblocks == -1)
		errexit("filesystem size unspecified");
	if(p->nbinodes == -1)
		p->nbinodes = p->nbblocks * BLOCKSIZE / ut_RoundUp(BYTES_PER_INODE, BLOCKSIZE);
	if(p->nbresrvd == -1)
		p->nbresrvd = p->nbblocks * RESERVED_INODES;

	if (p->nbblocks < 16) // totally arbitrary
		errexit("too small filesystem");
	if (p->nbblocks >BLOCKS_PER_GROUP) // I build only one group
		errexit("too big filesystem");

	if (!(fs = (TExt2fs*)calloc(1, sizeof(TExt2fs)))) {
		errexit("not enough memory for filesystem");
	}
	if (!(fs->rawimg = (uint8*)calloc(p->nbblocks, BLOCKSIZE))) {
		errexit("not enough memory for filesystem");
	}

	fs->mCommon.mFsops = vfsops;

	fs->sb = (superblock*)(fs->rawimg + BLOCKSIZE * 1);
	fs->gd = (groupdescriptor*)(fs->rawimg + BLOCKSIZE * 2);
	fs->bbm = (TBitmap*)(fs->rawimg + BLOCKSIZE * 3);
	fs->ibm = (TBitmap*)(fs->rawimg + BLOCKSIZE * 4);
	fs->itab = (inode*)(fs->rawimg + BLOCKSIZE * 5);

	// create the superblock for an empty filesystem
	fs->sb->s_inodes_count = ut_RoundUp(p->nbinodes, BLOCKSIZE/sizeof(inode));
	fs->sb->s_blocks_count = p->nbblocks;
	fs->sb->s_r_blocks_count = p->nbresrvd;
	fs->sb->s_free_blocks_count = p->nbblocks;
	fs->sb->s_free_inodes_count = fs->sb->s_inodes_count - EXT2_FIRST_INO + 1;
	fs->sb->s_first_data_block = (BLOCKSIZE == 1024);
	fs->sb->s_log_block_size = BLOCKSIZE >> 11;
	fs->sb->s_log_frag_size = BLOCKSIZE >> 11;
	fs->sb->s_blocks_per_group = BLOCKS_PER_GROUP;
	fs->sb->s_frags_per_group = BLOCKS_PER_GROUP;
	fs->sb->s_inodes_per_group = fs->sb->s_inodes_count;
	fs->sb->s_magic = EXT2_MAGIC_NUMBER;

	// set up groupdescriptors
	fs->sb->s_free_blocks_count -= 5 + fs->sb->s_inodes_count * sizeof(inode) / BLOCKSIZE;
	fs->gd->bg_free_blocks_count = fs->sb->s_free_blocks_count;
	fs->gd->bg_free_inodes_count = fs->sb->s_free_inodes_count;
	fs->gd->bg_used_dirs_count = 1;
	fs->gd->bg_block_bitmap = 3;
	fs->gd->bg_inode_bitmap = 4;
	fs->gd->bg_inode_table = 5;

	// mark non-filesystem blocks and inodes as TBitmap_IsSet
	for (i = fs->sb->s_blocks_count; i <= BLOCKSIZE * 8; i++) {
		TBitmap_SetBit(fs->bbm, i);
	}
	for (i = fs->sb->s_inodes_count + 1; i <= BLOCKSIZE * 8; i++) {
		TBitmap_SetBit(fs->ibm, i);
	}

	// mark system blocsk and inodes as TBitmap_IsSet
	for (i = 1; i <= 4 + fs->sb->s_inodes_count * sizeof(inode) / BLOCKSIZE; i++) {
		TBitmap_SetBit(fs->bbm, i);
	}
	for (i = 1; i < EXT2_FIRST_INO; i++) {
		TBitmap_SetBit(fs->ibm, i);
	}

	// make root inode and directory
	fs->itab[EXT2_ROOT_INO-1].i_mode = EXT2_IFDIR | EXT2_IRWXU | EXT2_IRWXG | EXT2_IRWXO;
	TExt2fs_AddEntryToDirectory((TVfs*)fs, EXT2_ROOT_INO, EXT2_ROOT_INO, ".");
	TExt2fs_AddEntryToDirectory((TVfs*)fs, EXT2_ROOT_INO, EXT2_ROOT_INO, "..");

	/* make "lost+found" */
	if (fs->sb->s_r_blocks_count) {
		nod = TVfs_AddEntryByString((TVfs*)fs, EXT2_ROOT_INO,
					 "drwxr-xr-x 0 0 /lost+found");
		memset(b, 0, BLOCKSIZE);
		((directory*)b)->d_rec_len = BLOCKSIZE;
		for(i = 1; i < fs->sb->s_r_blocks_count; i++)
			TExt2fs_ExtendBlock(fs, nod, b, 1);
		TExt2fs_GetInodePtr(fs, nod)->i_size = fs->sb->s_r_blocks_count * BLOCKSIZE;
	}

	// administrative info
	fs->sb->s_state = 1;
	fs->sb->s_max_mnt_count = 20;

	/* internal working flags */
	if (p->mCommon.holes) {
		fs->mCommon.holes = 1;
	}
	
	return (TVfs*)fs;
}

/** **************************************************************************
	load filesystem image
************************************************************************** **/
TVfs*
TExt2fs_LoadFromFile(TVfsOps* vfsops, TVfsParam* vfsprm__unused, FILE * fh)
{
	size_t fssize = 0;
	TExt2fs *fs;

	if ((fseek(fh, 0, SEEK_END) < 0) || ((fssize = ftell(fh)) < 0)) {
		pexit("input filesystem image");
	}
	rewind(fh);

	fssize = (fssize + BLOCKSIZE - 1) / BLOCKSIZE;
	if(fssize < 16) // totally arbitrary
		errexit("too small filesystem");
	if(fssize > BLOCKS_PER_GROUP) // I build only one group
		errexit("too big filesystem");

	if (!(fs = (TExt2fs*)calloc(1, sizeof(TExt2fs)))) {
		errexit("not enough memory for filesystem");
	}
	if (!(fs->rawimg = (uint8*)calloc(fssize, BLOCKSIZE))) {
		errexit("not enough memory for filesystem");
	}

	if(fread(fs->rawimg, BLOCKSIZE, fssize, fh) != fssize)
		pexit("input filesystem image");

	fs->mCommon.mFsops = vfsops;
	fs->sb = (superblock*)(fs->rawimg + BLOCKSIZE * 1);
	fs->gd = (groupdescriptor*)(fs->rawimg + BLOCKSIZE * 2);
	fs->bbm = (TBitmap*)(fs->rawimg + BLOCKSIZE * 3);
	fs->ibm = (TBitmap*)(fs->rawimg + BLOCKSIZE * 4);
	fs->itab = (inode*)(fs->rawimg + BLOCKSIZE * 5);

	if(fs->sb->s_rev_level || (fs->sb->s_magic != EXT2_MAGIC_NUMBER))
		errexit("not a suitable ext2 filesystem");

	return (TVfs*)fs;
}

/** **************************************************************************
	finalize filesystem image object
************************************************************************** **/
void
TExt2fs_Fini(TExt2fs *fs)
{
	free(fs->rawimg);
	free(fs);
}

/** **************************************************************************
	print list of data blocks
************************************************************************** **/
void
TExt2fs_PrintListDataBlockNo(TExt2fs *fs, uint32 nod, FILE* fp)
{
	int i = 0;
	TBpFs bp;
	uint32 bn;
	TBpFs_Init(&bp, fs, nod);

	fprintf(fp, "[");

	while ((bn = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		fprintf(fp, "%d ", bn), i++;
	}

	fprintf(fp, "] %d(%d)\n", i, i * BLOCKSIZE);
}

/** **************************************************************************
	print block/char device minor and major
************************************************************************** **/
void
TExt2fs_PrintInodeDev(TExt2fs *fs, uint32 nod, FILE* fp)
{
	int minor = ((uint8*)TExt2fs_GetInodePtr(fs, nod)->i_block)[0];
	int major = ((uint8*)TExt2fs_GetInodePtr(fs, nod)->i_block)[1];
	fprintf(fp, "%d %d\n", major, minor);
}

/** **************************************************************************
	print an inode as a directory
************************************************************************** **/
void
TExt2fs_PrintInodeDir(TExt2fs *fs, uint32 nod, FILE* fp)
{
	TBpFs bp;
	uint32 bn;
	int i;

	TBpFs_Init(&bp, fs, nod);
	fprintf(fp, "directory for inode %d:\n", nod);

	while ((bn = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		uint8* b = TExt2fs_GetBlockPtr(fs, bn);
		directory* d = (directory*)b;
		while ((uint8*)d + sizeof(*d) < b + BLOCKSIZE) {
			if (d->d_inode) {
				fprintf(fp, "[\"");
				for(i = 0; i < d->d_name_len; i++) {
					fputc(d->d_name[i], fp);
				}
				fprintf(fp, "\" %d %d(%d)]\n",
					d->d_inode, d->d_rec_len, d->d_name_len);
			}
			d = (directory*)((uint8*)d + d->d_rec_len);
		}
	}
}

/** **************************************************************************
	saves blocks to FILE*
************************************************************************** **/
void
TExt2fs_WriteBlockImage(TExt2fs *fs, uint32 nod, FILE* f)
{
	TBpFs bp;
	uint32 bk;
	int32 fsize = TExt2fs_GetInodePtr(fs, nod)->i_size;

	TBpFs_Init(&bp, fs, nod);
	while ((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		if (fsize <= 0) {
			errexit("wrong size while saving inode %d", nod);
		}
		if (fwrite(TExt2fs_GetBlockPtr(fs, bk), (fsize > BLOCKSIZE) ? BLOCKSIZE : fsize, 1, f) != 1) {
			errexit("error while saving inode %d", nod);
		}
		fsize -= BLOCKSIZE;
	}
}

/** **************************************************************************
	print a symbolic link
************************************************************************** **/
void
TExt2fs_PrintInodeLink(TExt2fs *fs, uint32 nod, FILE* fp)
{
	if (!TExt2fs_GetInodePtr(fs, nod)->i_blocks) {
		fprintf(fp, "(S:%s)\n", (char*)TExt2fs_GetInodePtr(fs, nod)->i_block);
	} else {
		fprintf(fp, "(L:");
		TExt2fs_WriteBlockImage(fs, nod, fp);
		fprintf(fp, ")\n");
	}
}

/** **************************************************************************
	make a ls-like printout of permissions
************************************************************************** **/
void
make_perms(uint32 mode, char perms[11])
{
	strcpy(perms, "----------");
	if (mode & EXT2_IRUSR) perms[1] = 'r';
	if (mode & EXT2_IWUSR) perms[2] = 'w';
	if (mode & EXT2_IXUSR) perms[3] = 'x';
	if (mode & EXT2_IRGRP) perms[4] = 'r';
	if (mode & EXT2_IWGRP) perms[5] = 'w';
	if (mode & EXT2_IXGRP) perms[6] = 'x';
	if (mode & EXT2_IROTH) perms[7] = 'r';
	if (mode & EXT2_IWOTH) perms[8] = 'w';
	if (mode & EXT2_IXOTH) perms[9] = 'x';
	if (mode & EXT2_ISUID) perms[3] = 's';
	if (mode & EXT2_ISGID) perms[6] = 's';
	if (mode & EXT2_ISVTX) perms[9] = 't';

	switch (mode & EXT2_IFMT) {
	case EXT2_IFSOCK:*perms = 's'; break;
	case EXT2_IFLNK: *perms = 'l'; break;
	case EXT2_IFREG: *perms = '-'; break;
	case EXT2_IFBLK: *perms = 'b'; break;
	case EXT2_IFDIR: *perms = 'd'; break;
	case EXT2_IFCHR: *perms = 'c'; break;
	case EXT2_IFIFO: *perms = 'p'; break;
	default:	 *perms = '?'; break;
	}
}

/** **************************************************************************
	print an inode
************************************************************************** **/
void
print_inode(TExt2fs *fs, uint32 nod, FILE* fp)
{
	char *s;
	char perms[11];
	static char* s_StringRepOfInodeType[17] = {
		"zero", "badB", "root", "ACLi",
		"ACLd", "boot", "udel", "rezv",
		"rezv", "rezv", "rezv", "rezv",
		"rezv", "rezv", "rezv", "rezv",
		"nrml",
	};

	inode* in;

	in = TExt2fs_GetInodePtr(fs, nod);

	if(!in->i_mode) {
		return;
	}

	s = s_StringRepOfInodeType[nod < EXT2_FIRST_INO ? nod : 16];
	make_perms(TExt2fs_GetInodePtr(fs, nod)->i_mode, perms);
/*              --inode-- type -link- -uid- -gid- */
	fprintf(fp, "%09d %s:%s:------- %6d %5d %5d %10d(%10d) ",
		nod,
		s, perms,
		in->i_links_count,
		in->i_uid, in->i_gid,
		in->i_size, (in->i_blocks /B512PERBLOCK)
	);

	if (!TBitmap_IsSet(fs->ibm, nod)) {
		fprintf(fp, "(unallocated)\n");
		return;
	}

	switch (TExt2fs_GetInodePtr(fs, nod)->i_mode & EXT2_IFMT) {
	case EXT2_IFLNK:
		TExt2fs_PrintInodeLink(fs, nod, fp);
		break;
	case EXT2_IFBLK:
	case EXT2_IFCHR:
		TExt2fs_PrintInodeDev(fs, nod, fp);
		break;
	case EXT2_IFDIR:
		TExt2fs_PrintListDataBlockNo(fs, nod, fp);
		TExt2fs_PrintInodeDir(fs, nod, fp);
		break;
	case EXT2_IFSOCK:
	case EXT2_IFREG:
	case EXT2_IFIFO:
	default:
		TExt2fs_PrintListDataBlockNo(fs, nod, fp);
		break;
	}
}

/** **************************************************************************
	printout fields in the filesystem
************************************************************************** **/
void
TExt2fs_Print(TVfs *vfs, FILE* fp)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	int i;

	fprintf(fp, "%d blocks (%d free, %d reserved), first data block: %d\n",
			fs->sb->s_blocks_count,
			fs->sb->s_free_blocks_count,
			fs->sb->s_r_blocks_count,
			fs->sb->s_first_data_block
	);
	fprintf(fp, "%d inodes (%d free)\n",
			fs->sb->s_inodes_count,
			fs->sb->s_free_inodes_count
	);
	fprintf(fp, "block size = %d, frag size = %d\n",
			fs->sb->s_log_block_size ? (fs->sb->s_log_block_size << 11) : 1024,
			fs->sb->s_log_frag_size ? (fs->sb->s_log_frag_size << 11) : 1024
	);
	fprintf(fp, "%d blocks per group, %d frags per group, %d inodes per group\n",
			fs->sb->s_blocks_per_group,
			fs->sb->s_frags_per_group,
			fs->sb->s_inodes_per_group
	);
	fprintf(fp, "block bitmap: block %d, inode bitmap: block %d, inode table: block %d\n",
			fs->gd->bg_block_bitmap,
			fs->gd->bg_inode_bitmap,
			fs->gd->bg_inode_table
	);
	fprintf(fp, "block bitmap allocation:\n");
	TBitmap_Print(fs->bbm, fs->sb->s_blocks_count, stdout);
	fprintf(fp, "inode bitmap allocation:\n");
	TBitmap_Print(fs->ibm, fs->sb->s_inodes_count, stdout);
	for(i=1; i<=fs->sb->s_inodes_count; i++) {
		if(TBitmap_IsSet(fs->ibm, i)) {
			print_inode(fs, i, fp);
		}
	}
}

/** **************************************************************************
************************************************************************** **/
void
TExt2fs_SaveToFile(TVfs *vfs, FILE * fh)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	int n = fs->sb->s_blocks_count;

	if (fwrite(fs->rawimg, BLOCKSIZE, n, fh) < n) {
		pexit("output filesystem image");
	}
}

/** **************************************************************************
************************************************************************** **/
void
TExt2fs_FillUnallocatedBlock(TVfs* vfs, char c)
{
	TExt2fs* fs = (TExt2fs*)vfs;
	int i;

	for (i = 1; i < fs->sb->s_blocks_count; i++) {
		if (!TBitmap_IsSet(fs->bbm, i)) {
			memset(TExt2fs_GetBlockPtr(fs, i), c, BLOCKSIZE);
		}
	}
}

/** **************************************************************************
************************************************************************** **/
TVfsOps st_ext2fsops = {
	.mFilesystemName = "ext2fs",
	.mfLoadFromFile = TExt2fs_LoadFromFile,
	.mfCreateFs = TExt2fs_CreateFs,
	.mfGetRootInodeNo = TExt2fs_GetRootInodeNo,
	.mfFillUnallocatedBlock = TExt2fs_FillUnallocatedBlock,
	.mfPrint = TExt2fs_Print,
	.mfSaveToFile = TExt2fs_SaveToFile,
	.mfFindInodeOf = TExt2fs_FindInodeOf,
	.mfAllocateBlockForInode = TExt2fs_AllocateBlockForInode,
	.mfAddEntryToDirectory = TExt2fs_AddEntryToDirectory,
	.mfAddSymlink = TExt2fs_AddSymlink,
	.mfAddRegulerFile = TExt2fs_AddRegulerFile,
};

/** **************************************************************************
************************************************************************** **/
void showhelp(void)
{
	/* mkimage.ext2fs */
	fprintf(stderr, "Usage: %s [options]\n"
	/* --------1---------2---------3---------4---------5---------6---------7---------8 */
	"mkimge2fs " VERSION " (c) 2002-2005 N.Masuichi <nor at users.sourceforge.jp>\n"
	"  derived from: genext2fs (C) 2000 Xavier Bestel <xavier.bestel at free.fr>\n"
	"\n"
	"Base image:\n"
	"  -s image	Use this image as a starting point\n"
	"               (default: Create NEW filesystem image.)\n"
	"Filesystem option:\n"
	"  -b blocks    size in blocks\n"
	"  -i inodes    Number of inodes\n"
	"  -r reserved  Number of reserved blocks\n"
	"  -e value     Fill unallocated blocks with value\n"
	"  -H           Make files with holes\n"
	"Specify entries to add:\n"
	"  -f file      Add nodes from this spec file [default:stdin]\n"
#if 0
	"  -S dirctory  A starting directory to search file(s) with relative path\n"
	"  -p file	Use this file instead of /etc/passwd\n"
	/* ex.)
	 * -p /path/foo/bar/passwd (The file is in your real fs.)
	 * -p foo/bar/passwd (The file is in your real fs.)
	 * -p /_fsimage_/path/passwd (The file is in your image fs.)
	 *	Keyword : /_fsimag_
	 */
	"  -g file	Use this file instead of /etc/group\n"
#endif
	"Output image:\n"
	"  -o file      put result image to this file\n"
	"Listing:\n"
	"  -l file      Print resulting filesystem structure\n"
	"Help:\n"
	"  -h           Show this help\n"
	, g_ThisExecName);
}

/** **************************************************************************
************************************************************************** **/
int
main(int argc, char **argv)
{
	TExt2fsParam param = {
		.mCommon = {
			.fsin = NULL,
			.fsspec = NULL,
			.fsout = NULL,
			.fsrep = NULL,
			.fspasswd = NULL,
			.verbose = 0,
			.holes = 0,
			.emptyval = 0,
			.nosquash = 0,
		},
		.nbblocks = -1,
		.nbinodes = -1,
		.nbresrvd = -1,
	};

	char c;

	g_ThisExecName = argv[0];
	while ((c = getopt(argc, argv, "s:f:o:l:b:i:r:e:zvnhp:")) != EOF) {
		switch (c) {
/** NOT IMPELMENTED **/
#if 0
		case 'S':
			g_PrefixOfRelatvePath = optarg;
			break;
#endif
		case 's':
			param.mCommon.fsin = optarg;
			break;
		case 'f':
			param.mCommon.fsspec = optarg;
			break;
		case 'l':
			param.mCommon.fsrep = optarg;
			break;
		case 'o':
			param.mCommon.fsout = optarg;
			break;
		case 'b':
			param.nbblocks = atoi(optarg);
			break;
		case 'i':
			param.nbinodes = atoi(optarg);
			break;
		case 'r':
			param.nbresrvd = atoi(optarg);
			break;
		case 'e':
			param.mCommon.emptyval = atoi(optarg);
			break;
		case 'z':
			param.mCommon.holes = 1;
			break;
		case 'n':
			param.mCommon.nosquash = 1;
			break;
		case 'v':
			param.mCommon.verbose = 1;
			break;
		case 'h':
		case '?':
			showhelp();
			exit(0);
		default:
			exit(1);
		}
	}
	if (optind < (argc - 1)) {
		errexit("too many arguments");
	}

	return TVfs_MakeImage(&st_ext2fsops, (TVfsParam*)&param);
}

/* END-OF-FILE */
