/* vi: ts=8 sw=8 : */
/**
	mkimge2fs - create an ext2fs image like as mkisofs

	Copyright (c) 2002-2003 Masuichi Noritoshi <nor@users.sourceforge.jp>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version
	2 of the License, or (at your option) any later version.

	This file was derived from:

		linux/include/linux/ext2_fs.h

		Copyright (C) 1992, 1993, 1994, 1995
		Remy Card (card@masi.ibp.fr)
		Laboratoire MASI - Institut Blaise Pascal
		Universite Pierre et Marie Curie (Paris VI)

	from:

		linux/include/linux/minix_fs.h

		Copyright (C) 1991, 1992  Linus Torvalds
**/

#ifndef INCLUDE_MKIMGEXT2FS_H
#define INCLUDE_MKIMGEXT2FS_H

#include <util.h>

/* block size */

#define BLOCKSIZE         1024
#define BLOCKS_PER_GROUP  8192
#define BYTES_PER_INODE   (8*BLOCKSIZE)
#define RESERVED_INODES   5/100


/* inode block size (why is it != BLOCKSIZE ?!?) */

#define INODE_BLOCKSIZE   512
#define INOBLK            (BLOCKSIZE / INODE_BLOCKSIZE)

/**
	number of 512byte-blocks in ext2fs-block
**/
#define B512PERBLOCK      (BLOCKSIZE / 512)

/* reserved inodes */

#define EXT2_BAD_INO         1     // Bad blocks inode
#define EXT2_ROOT_INO        2     // Root inode
#define EXT2_ACL_IDX_INO     3     // ACL inode
#define EXT2_ACL_DATA_INO    4     // ACL inode
#define EXT2_BOOT_LOADER_INO 5     // Boot loader inode
#define EXT2_UNDEL_DIR_INO   6     // Undelete directory inode
#define EXT2_FIRST_INO       11    // First non reserved inode

/* magic number for ext2 */

#define EXT2_MAGIC_NUMBER  0xEF53

#define EXT2_INIT_BLOCK    0xFFFFFFFF
#define WALK_END           0xFFFFFFFE

typedef struct {
	uint8 bm[BLOCKSIZE];
} TBitmap;

#if BLOCKSIZE != 1024
	/* FIX ME! */
#error TExt2fs can handle only block which size is 1024.
#endif


/*
 * Constants relative to the data blocks
 */
#define	EXT2_NDIR_BLOCKS		12
#define	EXT2_IND_BLOCK			EXT2_NDIR_BLOCKS
#define	EXT2_DIND_BLOCK			(EXT2_IND_BLOCK + 1)
#define	EXT2_TIND_BLOCK			(EXT2_DIND_BLOCK + 1)
#define	EXT2_N_BLOCKS			(EXT2_TIND_BLOCK + 1)

#define	EXT2_MDMSK_FL			0x00FF /* modifiable attribute mask */
#define	EXT2_SECRM_FL			0x0001 /* Secure deletion */
#define	EXT2_UNRM_FL			0x0002 /* Undelete */
#define	EXT2_COMPR_FL			0x0004 /* Compress file */
#define EXT2_SYNC_FL			0x0008 /* Synchronous updates */
#define EXT2_IMMUTABLE_FL		0x0010 /* Immutable file */
#define EXT2_APPEND_FL			0x0020 /* writes to file may only append */
#define EXT2_NODUMP_FL			0x0040 /* do not dump file */
#define EXT2_NOATIME_FL			0x0080 /* do not update atime */

#define EXT2_IFMT  			0xF000	/* format mask */
#define EXT2_IFHLNK 			0xE000	/* hardlink (only for internal use) */
#define EXT2_IFLNK 			0xA000	/* symlink */
#define EXT2_IFSOCK 			0xC000	/* socket */
#define EXT2_IFREG  			0x8000	/* regular file */
#define EXT2_IFBLK			0x6000	/* block device */
#define EXT2_IFDIR			0x4000	/* directory */
#define EXT2_IFCHR			0x2000	/* character device */
#define EXT2_IFIFO			0x1000	/* fifo */

#define EXT2_ISUID			0x0800	/* SUID */
#define EXT2_ISGID			0x0400	/* SGID */
#define EXT2_ISVTX			0x0200	/* sticky bit */

#define EXT2_IRWXU			0x01C0	/* user mask */
#define EXT2_IRUSR			0x0100	/* read */
#define EXT2_IWUSR			0x0080	/* write */
#define EXT2_IXUSR			0x0040	/* execute */

#define EXT2_IRWXG			0x0038	/* group mask */
#define EXT2_IRGRP			0x0020	/* read */
#define EXT2_IWGRP			0x0010	/* write */
#define EXT2_IXGRP			0x0008	/* execute */

#define EXT2_IRWXO			0x0007	/* other mask */
#define EXT2_IROTH			0x0004	/* read */
#define EXT2_IWOTH			0x0002	/* write */
#define EXT2_IXOTH			0x0001	/* execute */

typedef struct {
	uint16	i_mode;		/* File mode */
	uint16	i_uid;		/* Low 16 bits of Owner Uid */
	uint32	i_size;		/* Size in bytes */
	uint32	i_atime;	/* Access time */
	uint32	i_ctime;	/* Creation time */
	uint32	i_mtime;	/* Modification time */
	uint32	i_dtime;	/* Deletion Time */
	uint16	i_gid;		/* Low 16 bits of Group Id */
	uint16	i_links_count;	/* Links count */
	uint32	i_blocks;	/* Blocks count */
	uint32	i_flags;	/* File flags */
	uint32	i_reserved1;	/* Reserved 1 */
	uint32	i_block[EXT2_N_BLOCKS];/* Pointers to blocks */
	uint32	i_generation;	/* File version (for NFS) */
	uint32	i_file_acl;	/* File ACL */
	uint32	i_dir_acl;	/* Directory ACL */
	uint32	i_faddr;	/* Fragment address */
	uint8	i_frag;	/* Fragment number */
	uint8	i_fsize;	/* Fragment size */
	uint16	i_pad1;
	uint16	i_uid_high;	/* these 2 fields    */
	uint16	i_gid_high;	/* were reserved2[0] */
	uint32	i_reserved2;
} inode;

/*
 * Structure of the super block
 */
typedef struct {
	uint32	s_inodes_count;		/* Inodes count */
	uint32	s_blocks_count;		/* Blocks count */
	uint32	s_r_blocks_count;	/* Reserved blocks count */
	uint32	s_free_blocks_count;	/* Free blocks count */
	uint32	s_free_inodes_count;	/* Free inodes count */
	uint32	s_first_data_block;	/* First Data Block */
	uint32	s_log_block_size;	/* Block size */
	int32	s_log_frag_size;	/* Fragment size */
	uint32	s_blocks_per_group;	/* # Blocks per group */
	uint32	s_frags_per_group;	/* # Fragments per group */
	uint32	s_inodes_per_group;	/* # Inodes per group */
	uint32	s_mtime;		/* Mount time */
	uint32	s_wtime;		/* Write time */
	uint16	s_mnt_count;		/* Mount count */
	int16	s_max_mnt_count;	/* Maximal mount count */
	uint16	s_magic;		/* Magic signature */
	uint16	s_state;		/* File system state */
	uint16	s_errors;		/* Behaviour when detecting errors */
	uint16	s_minor_rev_level; 	/* minor revision level */
	uint32	s_lastcheck;		/* time of last check */
	uint32	s_checkinterval;	/* max. time between checks */
	uint32	s_creator_os;		/* OS */
	uint32	s_rev_level;		/* Revision level */
	uint16	s_def_resuid;		/* Default uid for reserved blocks */
	uint16	s_def_resgid;		/* Default gid for reserved blocks */
	/*
	 * These fields are for EXT2_DYNAMIC_REV superblocks only.
	 *
	 * Note: the difference between the compatible feature set and
	 * the incompatible feature set is that if there is a bit set
	 * in the incompatible feature set that the kernel doesn't
	 * know about, it should refuse to mount the filesystem.
	 * 
	 * e2fsck's requirements are more strict; if it doesn't know
	 * about a feature in either the compatible or incompatible
	 * feature set, it must abort and not try to meddle with
	 * things it doesn't understand...
	 */
	uint32	s_first_ino; 		/* First non-reserved inode */
	uint16  s_inode_size; 		/* size of inode structure */
	uint16	s_block_group_nr; 	/* block group # of this superblock */
	uint32	s_feature_compat; 	/* compatible feature set */
	uint32	s_feature_incompat; 	/* incompatible feature set */
	uint32	s_feature_ro_compat; 	/* readonly-compatible feature set */
	uint8	s_uuid[16];		/* 128-bit uuid for volume */
	char	s_volume_name[16]; 	/* volume name */
	char	s_last_mounted[64]; 	/* directory where last mounted */
	uint32	s_algorithm_usage_bitmap; /* For compression */
	/*
	 * Performance hints.  Directory preallocation should only
	 * happen if the EXT2_COMPAT_PREALLOC flag is on.
	 */
	uint8	s_prealloc_blocks;	/* Nr of blocks to try to preallocate*/
	uint8	s_prealloc_dir_blocks;	/* Nr to preallocate for dirs */
	uint16	s_padding1;
	uint32	s_reserved[204];	/* Padding to the end of the block */
} superblock;

/*
 * Structure of a blocks group descriptor
 */
typedef struct {
	uint32	bg_block_bitmap;		/* Blocks bitmap block */
	uint32	bg_inode_bitmap;		/* Inodes bitmap block */
	uint32	bg_inode_table;		/* Inodes table block */
	uint16	bg_free_blocks_count;	/* Free blocks count */
	uint16	bg_free_inodes_count;	/* Free inodes count */
	uint16	bg_used_dirs_count;	/* Directories count */
	uint16	bg_pad;
	uint32	bg_reserved[3];
} groupdescriptor;

/*
 * Structure of a directory entry
 */
#define EXT2_NAME_LEN 255

typedef struct {
	uint32	d_inode;			/* Inode number */
	uint16	d_rec_len;		/* Directory entry length */
	uint16	d_name_len;		/* Name length */
	char	d_name[0];		/* File name */
} directory;

typedef struct
{
	TVfs mCommon;		/* Virtual Filesystem infomation */
	uint32 holef;		/* internal working infomation */
				/* [flag] allow to make a file with hole */
				/*	0  : Not allowed. */
				/*	!0 : allowed. */
	uint8* rawimg;		/* The raw image of filesystem */
	superblock* sb;		/* The superblock */
	groupdescriptor* gd;	/* The group desciptor */
	TBitmap* bbm;		/* The block bitmap */
	TBitmap* ibm;		/* The inode bitmap */
	inode* itab;		/* The inode table */
} TExt2fs;

typedef struct
{
	TExt2fs* asnfs;		/* link to filesystem */
	uint32 asnod;
	inode* asnode;
	uint32 bnum;		/* number of allocated blocks */
	uint32 state;		/* direct/indirect -state */
	uint32 bpdir;		/* index of direct block */
	uint32 bpind;		/* index of indirect block */
	uint32 bpdind;		/* index of double indirect block */
	uint32 bptind;		/* index of triple indirect block */
				/* Not support tri-ind : FIX ME ! */
} TBpFs;

#define EXT2_BP_STATE_INI		0
#define EXT2_BP_STATE_DIR		1
#define EXT2_BP_STATE_DIR_LAST		2
#define EXT2_BP_STATE_IND		3
#define EXT2_BP_STATE_IND_LAST		4
#define EXT2_BP_STATE_DIND		5
#define EXT2_BP_STATE_DIND_LAST		6
#define EXT2_BP_STATE_TIND		7
#define EXT2_BP_STATE_TIND_LAST		8
#define EXT2_BP_STATE_TOO_DEEP		9

#define	EXT2_BP_FINI	0xFFFFFFFE
#define	EXT2_BP_INIT	0xFFFFFFFF
#define	EXT2_BP_IND	EXT2_IND_BLOCK
#define	EXT2_BP_DIND	EXT2_DIND_BLOCK
#define	EXT2_BP_TIND	EXT2_TIND_BLOCK
/**

bpdir			bpind	bpdind	BLOCK NO in INODE
------------------------------------------------------------------
EXT2_BP_INIT(-1)	*	*	(initial state)
0			*	*	block 0.
...					...
11			*	*	block 11.
EXT2_BP_IND(12)		0	*	block 11+1.
			...		...
EXT2_BP_IND(12)		IPB-1	*	block 11+IPB		
EXT2_BP_DIND(13)	0	0	block 11+IPB+1
				...	...
EXT2_BP_DIND(13)	0	IPB-1	block 11+IPB+IPB
			...	...	...
EXT2_BP_DIND(13)	IPB-1	IPB-1	block 11+IPB+IPB*IPB
EXT2_BP_TIND(14)	*	*	(PANIC !)

**/

#endif /* INCLUDE_USER_EXT2FS_H */

/* END-OF-FILE */
