/* vi: sw=8 ts=8 : */
/**
	mkimgfatfs - create a FAT image like as mkisofs

	Copyright (c) 2003-2005 Masuichi Noritoshi <nor@users.sourceforge.jp>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version
	2 of the License, or (at your option) any later version.

	This code is derived from:

	genext2fs.c

	ext2 filesystem generator for embedded systems
	Copyright (C) 2000 Xavier Bestel <xavier.bestel@free.fr>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; version
	2 of the License.
**/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#ifdef HAVE_STRING_H
#       include <string.h>
#else
#       ifdef HAVE_STRINGS_H
#               include <strings.h>
#       endif
#endif
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>

#include <util.h>
#include <srcinode.h>
#include <mkimgfs.h>
#include <mkimgfatfs.h>

typedef struct {
	int mSectors;
	unsigned short mRootDirEntries;
	char mMedia;
	char mClusterSize;
	char mCyl;
	char mHead;
	char mSect;
} TFat_DiskGeom;

TFat_DiskGeom g_fat_disk_geom_table[] = {
	{  720, 112, 0xFD, 2, 40, 2,  9}, /* 5.25" -  360K */
	{ 1440, 112, 0xF9, 2, 80, 2,  9}, /* 3.5"  -  720K */
	{ 2400, 224, 0xF9, 1, 80, 2, 15}, /* 5.25" - 1200K */
	{ 5760, 224, 0xF0, 2, 80, 2, 36}, /* 3.5"  - 2880K */
	{ 2880, 224, 0xF0, 1, 80, 2, 18}, /* 3.5"  - 1440K */
#if 0
	{    0, 512, 0xF8, 4,  0, 0,  0}, /* HDD image */
#endif
	{    0,   0,    0, 0,  0, 0,  0},
};

#define SECTORSIZE	512

#define GET_INODE_NO(pfs, pointer)	\
	((((uint8*)(pointer)) - (pfs)->mRawImage) / sizeof(TFatDirEnt))
#define GET_INODE_PTR(pfs, ino)	\
	(((TFatDirEnt*)((pfs)->mRawImage)) + (ino))
#define FAT_ROOT_INO	1

#define FAT_GET_FAT12(fs, i)	\
	((i % 2) ? \
	  (((fs)->mFAT[((i)/2)*3+2] << 4) + ((fs)->mFAT[((i)/2)*3+1] >> 4)) : \
	  ((fs)->mFAT[((i)/2)*3] + (((fs)->mFAT[((i)/2)*3+1] &0x0f) << 8)) \
	)
#define FAT_SET_FAT12(fs, i, n)	\
	do { \
		if ((i) % 2) { \
			(fs)->mFAT[((i)/2)*3+2] = ((n)>>4); \
			(fs)->mFAT[((i)/2)*3+1] = ((fs)->mFAT[((i)/2)*3+1] & 0x0F) | ((n)<<4);   \
		} else { \
			(fs)->mFAT[((i)/2)*3] = ((n) & 0xFF); \
			(fs)->mFAT[((i)/2)*3+1] = ((fs)->mFAT[((i)/2)*3+1] & 0xF0) | ((n)>>8);   \
		} \
	} while (0)

#define WALK_END           0xFFFFFFFE

/** **************************************************************************
************************************************************************** **/
uint8*
TFatfs_GetBlockPtr(TFatfs* fs, uint32 bk)
{
	return fs->mData + fs->s_cluster_size * (bk - 2);
}

/** **************************************************************************
	initalize a TBpFs (iterator for blocks list)
************************************************************************** **/
void
TBpFs_Init(TBpFs* bp, TFatfs* fs, uint32 nod)
{
	bp->asnfs = fs;
	bp->asnod = nod;
	bp->asnode = GET_INODE_PTR(bp->asnfs, bp->asnod);
	bp->bpdir = 0;
}

/** **************************************************************************
	finalize a TBpFs
************************************************************************** **/
#if 0
void
TBpFs_Fini(TBpFs* bp)
{
	bp->asnfs = NULL;
	bp->asnode = NULL;
}
#endif

uint32 TFatfs_AllocateCluster(TFatfs* fs); /* FIXME : PROTTYPE BAD LOCATION */

/** **************************************************************************
************************************************************************** **/
uint32
TBpFs_AllocateCluster(TBpFs* bp)
{
	return TFatfs_AllocateCluster(bp->asnfs);
}

/** **************************************************************************
RETURN:
	next cluster of inode (WALK_END for end)
	if create>0, append a newly allocated block at the end
************************************************************************** **/
uint32
TBpFs_Next(TBpFs* bp, uint32 *create, uint32 hole__unused)
{
	uint32 od;

	od = bp->bpdir;
	if (od == 0) {
		bp->bpdir = bp->asnode->start;
		if (bp->bpdir == 0) {
			if (create && (*create)--) {
				bp->bpdir = TBpFs_AllocateCluster(bp);
				bp->asnode->start = bp->bpdir;
			} else {
				return WALK_END;
			}
		}
	} else if (FAT_GET_FAT12(bp->asnfs, od) == 0x0FF8) {
		if (create && (*create)--) {
			bp->bpdir = TBpFs_AllocateCluster(bp);
			FAT_SET_FAT12(bp->asnfs, od, bp->bpdir);
		} else {
			return WALK_END;
		}
	} else {
		bp->bpdir = FAT_GET_FAT12(bp->asnfs, bp->bpdir);
	}
	return bp->bpdir;
}

/** **************************************************************************
************************************************************************** **/
uint32
TFatfs_AllocateCluster(TFatfs* fs)
{
	uint32 i;
	uint32 n;

	for (i = 2; i < fs->s_data_clusters + 2 ; i++) {
		n = FAT_GET_FAT12(fs, i);
		if (!n) {	/* found */
			FAT_SET_FAT12(fs, i, 0x0FF8);
			return i;
		}
	}
	errexit("No free space (couldn't allocate a cluster)");
}


/** **************************************************************************
************************************************************************** **/
/*
	0    ... string terminator
	1    ... illegal charactor
	o.w. ... legal charactor (and it means it's code)

	[RESTRICTION]
	We assume our charactor code is ASCII.
*/

#define DOSTRM		0u
#define DOSILG		1u
static uint8 char_check_table[128] = {
/*	+0	+1	+2	+3	+4	+5	+6	+7	*/
	DOSTRM,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	
	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	/*0x*/
	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	
	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	/*1x*/
	DOSILG,	'!',	DOSILG,	'#',	'$',	'%',	'&',	'\'',	
	'(',	')',	DOSILG,	DOSILG,	DOSILG,	'-',	'.',	DOSILG,	/*2x*/
	'0',	'1',	'2',	'3',	'4',	'5',	'6',	'7',	
	'8',	'9',	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	DOSILG,	/*3x*/
	'@',	'A',	'B',	'C',	'D',	'E',	'F',	'G',	
	'H',	'I',	'J',	'K',	'L',	'M',	'N',	'O',	/*4x*/
	'P',	'Q',	'R',	'S',	'T',	'U',	'V',	'W',	
	'X',	'Y',	'Z',	DOSILG,	DOSILG,	DOSILG,	'^',	'_',	/*5x*/
	DOSILG,	'A',	'B',	'C',	'D',	'E',	'F',	'G',	
	'H',	'I',	'J',	'K',	'L',	'M',	'N',	'O',	/*6x*/
	'P',	'Q',	'R',	'S',	'T',	'U',	'V',	'W',	
	'X',	'Y',	'Z',	'{',	DOSILG,	'}',	'~',	DOSILG,	/*7x*/
};

/* [FIXME]
	configure script need to check whether char-type is signed or unsigned !
*/
#if 0
#define CONV_DOS_FILE_CHAR(c)	(((c)<128) ? char_check_table[(c)] : 1)
#else
#define CONV_DOS_FILE_CHAR(c)	(((c)>=0) ? char_check_table[(c)] : 1)
#endif

/** **************************************************************************
************************************************************************** **/
int
TFatfs_NormalizeName(TFatDirEnt* ep, const char* name)
{
	int i;
	const char* p = name;
	char c;
	uint8 oc;

	memset(ep->name, ' ', 8);
	memset(ep->ext, ' ', 3);

	for (i = 0 ; i < 8 ; i++) {
		c = *p++;
		oc = CONV_DOS_FILE_CHAR(c);
		if (oc == DOSTRM) {
			return 1;
		} else if (oc == DOSILG) {
			return 0;
		} else if (oc == '.') {
			--p;
			break;
		}
		ep->name[i] = oc;
	}
	if (*p == '.') {
		p++;
	}
	for (i = 0 ; i < 3 ; i++) {
		c = *p++;
		oc = CONV_DOS_FILE_CHAR(c);
		if (oc == DOSTRM) {
			return 1;
		} else if (oc == DOSILG) {
			return 0;
		} else if (oc == '.') {
			--p;
			break;
		}
		ep->ext[i] = oc;
	}
	return (*p == 0) ? 1 : 0;
}

/** **************************************************************************
************************************************************************** **/
uint32
TFatfs_FindDir(TFatfs *fs, uint32 nod, const char * name)
{
	TFatDirEnt de;
	TBpFs bp;
	uint32 bk;
	uint32 n;
	uint32 i;
	TFatDirEnt* p;

	TBpFs_Init(&bp, fs, nod);

	if (!TFatfs_NormalizeName(&de, name)) {
		errexit("bad name '%s' (contains bad char.)", name);
	}

	if (nod == FAT_ROOT_INO) {
		for (i = 0 ; i < fs->s_dir_entries ; i++) {
			n = GET_INODE_NO(fs, fs->mRootDir + i);
			p = GET_INODE_PTR(fs, n);
			if (memcmp(p->name, de.name, 8) == 0 &&
			    memcmp(p->ext, de.ext, 3) == 0) {
				return n;
			}
		}
	} else {
		while((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
			TFatDirEnt* b;
			b = (TFatDirEnt*)TFatfs_GetBlockPtr(fs, bk);
			for (i = 0 ; i < fs->s_cluster_size / sizeof(TFatDirEnt) ; i++) {
				n = GET_INODE_NO(fs, b + i);
				p = GET_INODE_PTR(fs, n);
				if (memcmp(p->name, de.name, 8) == 0 &&
				    memcmp(p->ext, de.ext, 3) == 0) {
					return n;
				}
			}
		}
	}
	return 0;
}

/** **************************************************************************
************************************************************************** **/
uint32
TFatfs_FindInodeOf(TVfs *vfs, uint32 nod, const char * name)
{
	TFatfs* fs = (TFatfs*)vfs;
	char *p, *n, *n2 = strdup(name);
	n = n2;

	while(*n == '/') {
		nod = FAT_ROOT_INO;
		n++;
	}

	while(*n) {
		if((p = strchr(n, '/')))
			(*p) = 0;
		if (!(nod = TFatfs_FindDir(fs, nod, n)))
			break;
		if (p)
			n = p + 1;
		else
			break;
	}

	free(n2);
	return nod;
}

/** **************************************************************************
************************************************************************** **/
static TSrcInode* s_ref_fakeSi;

uint32
TFatfs_AllocateBlockForInode(TVfs* vfs, uint32 nod__unused, TSrcInode* si)
{
	s_ref_fakeSi = si;
	return 0;
}

/** **************************************************************************
	add blocks to an inode (file/dir/etc...)
************************************************************************** **/
uint32
TFatfs_ExtendBlock(TFatfs *fs, uint32 nod, uint8* b, int amount)
{
	int create = amount;
	TBpFs bp;
	TBpFs lbw;
	uint32 bk;
	uint32 bsize;

	bsize = fs->s_cluster_size;

	TBpFs_Init(&bp, fs, nod);

	lbw = bp;
	while ((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		lbw = bp;
	}
	bp = lbw;


	while (create) {
		int i;
		int hf = 1;

		if ( 1 /* !(fs->holef & OP_HOLES) */) {
			hf = 0;
		} else {
			/* check that block is filled with zero */
			uint32* d = (int32*)(b + bsize * (amount - create));
			for (i = 0; i < bsize / sizeof(uint32); i++) {
				if (d[i]) {
					hf = 0;
					break;
				}
			}
		}
		if ((bk = TBpFs_Next(&bp, &create, hf)) == WALK_END) {
			break;
		}
		if (!hf) {
			memcpy(TFatfs_GetBlockPtr(fs, bk),
				b + bsize * (amount - (create+1)),
				bsize);
		}
	}
	return bk; /* last allocated cluster no. */
}

/** **************************************************************************
************************************************************************** **/
uint32
TFatfs_AddEntryToDirectory(TVfs *vfs, uint32 dnod, uint32 nod__unused, const char* name)
{
	TFatfs* fs = (TFatfs*)vfs;
	TFatDirEnt* pde;
	TFatDirEnt de;
	TFatDirEnt* p;
	uint32 nod = 0;
	uint32 n;
	uint32 i;
	TBpFs bp;
	uint32 bk;


	if (strcmp(name, ".") == 0 || strcmp(name, "..") == 0) {
		return;
	}

	pde = GET_INODE_PTR(fs, dnod);

	if (dnod != FAT_ROOT_INO && !(pde->attr & ATTR_DIR)) {
		errexit("can't add '%s' to a non-directory", name);
	}

	if (TFatfs_FindDir(fs, dnod, name)) {
		errexit("entry '%s' already exist.", name);
	}

	if (!*name) {
		errexit("bad name '%s' (not meaningful)", name);
	}

	if (!TFatfs_NormalizeName(&de, name)) {
		errexit("bad name '%s' (contains bad char.)", name);
	}

	if (dnod == FAT_ROOT_INO) {
		for (i = 0 ; i < fs->s_dir_entries ; i++) {
			n = GET_INODE_NO(fs, fs->mRootDir + i);
			p = GET_INODE_PTR(fs, n);
			if (p->name[0] == 0 || p->name[0] == 0xe5) {
				break; 
			}
		}
		if (i == fs->s_dir_entries) {
			errexit("cannot add '%s' (no free root entries)", name);
		}
		nod = n;
	} else {
		TBpFs_Init(&bp, fs, dnod);
		while((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
			TFatDirEnt* b;
			b = (TFatDirEnt*)TFatfs_GetBlockPtr(fs, bk);
			for (i = 0 ; i < fs->s_cluster_size / sizeof(TFatDirEnt) ; i++) {
				n = GET_INODE_NO(fs, b + i);
				p = GET_INODE_PTR(fs, n);
				if (p->name[0] == 0 || p->name[0] == 0xe5) {
					nod = n;
					break;
				}
			}
			if (nod) {
				break;
			}
		}
		// we found no free entry in the directory, so we add a block
		{
			uint8* b;
			uint32 bk;

			b = XMALLOC(uint8, fs->s_cluster_size);
			memset(b, 0, fs->s_cluster_size);

			bk = TFatfs_ExtendBlock(fs, dnod, b, 1);
			GET_INODE_PTR(fs, dnod)->size += fs->s_cluster_size;

			p = (TFatDirEnt*)TFatfs_GetBlockPtr(fs, bk);
			nod = GET_INODE_NO(fs, p);

			free(b);
		}
	}
	memcpy(p->name, de.name, 8);
	memcpy(p->ext, de.ext, 3);

	p->attr = ATTR_ARCH;
	if (!(s_ref_fakeSi->mMode & VFS_IWUSR)) {
		p->attr |= ATTR_RO;
	}
	if (!(s_ref_fakeSi->mMode & VFS_IRUSR)) {
		p->attr |= ATTR_HIDDEN;
	}
	if (s_ref_fakeSi->mMode & VFS_ISUID) {
		p->attr |= ATTR_SYS;
	}
	if ((s_ref_fakeSi->mMode & VFS_IFMT) == VFS_IFDIR) {
		p->attr |= ATTR_DIR;
	}
	p->time = 0;
	p->date = 0;
	p->start = 0;
	p->size = 0;

	return nod;
}

/** **************************************************************************
************************************************************************** **/
void
TFatfs_AddRegulerFile(TVfs* vfs, uint32 nod2, TSrcInode* si)
{
	TFatfs* fs = (TFatfs*)vfs;
	TFatDirEnt* dp;

	uint8* blk;
	struct stat st;
	FILE *fh;
	uint32 rlen;

	dp = GET_INODE_PTR(fs, nod2);

	lstat(si->mSourcePath, &st);
	if ((st.st_mode & S_IFMT) != S_IFREG) {
		pexit(si->mSourcePath);
	}
	if (!(fh = fopen(si->mSourcePath, "r"))) {
		pexit(si->mSourcePath);
	}
	dp->size = st.st_size;
	rlen = ut_RoundUp(dp->size, fs->s_cluster_size);
	blk = XMALLOC(uint8, rlen);
	if (rlen > dp->size) {
		memset(blk + dp->size, 0, rlen - dp->size);
	}

	if (fh) {
		fread(blk, dp->size, 1, fh);
	} else {
		memset(blk, 0, dp->size);
	}
	fclose(fh);

	TFatfs_ExtendBlock(fs, nod2, blk, rlen / fs->s_cluster_size);
	free(blk);
}

/** **************************************************************************
************************************************************************** **/
TFatfs*
TFatfs_Setup(TFatfs* fs)
{
	fs->s_sector_size = fs->mFSS->s_sector_size[0]
			    + (fs->mFSS->s_sector_size[1] << 8);
	if (fs->s_sector_size != SECTORSIZE) {
		errexit("unsupported sector size");
	}
	fs->s_dir_entries = fs->mFSS->s_dir_entries[0]
			    + (fs->mFSS->s_dir_entries[1] << 8);
	fs->s_fats = fs->mFSS->s_fats;
	fs->s_sectors_per_fat = fs->mFSS->s_sectors_per_fat;
	fs->s_sectors_per_cluster = fs->mFSS->s_sectors_per_cluster;
	fs->s_reserved_sectors = fs->mFSS->s_reserved_sectors;
	fs->s_total_sectors = fs->mFSS->s_total_sectors[0]
			     + (fs->mFSS->s_total_sectors[1] << 8);
	fs->mFAT = (uint8*)(fs->mRawImage + SECTORSIZE * (
				fs->s_reserved_sectors));
	fs->mRootDir = (TFatDirEnt*)(fs->mRawImage + SECTORSIZE * (
				fs->s_reserved_sectors
				+ fs->s_fats * fs->s_sectors_per_fat));
	fs->s_dir_sectors = (fs->s_dir_entries * sizeof(TFatDirEnt) + (SECTORSIZE-1)) / SECTORSIZE;
	fs->mData = (uint8*)(fs->mRawImage + SECTORSIZE * (
				fs->s_reserved_sectors
				+ fs->s_fats * fs->s_sectors_per_fat
				+ fs->s_dir_sectors));
	fs->s_data_sectors = fs->s_total_sectors
				- fs->s_reserved_sectors
				- fs->s_fats * fs->s_sectors_per_fat
				- fs->s_dir_sectors;
	fs->s_data_clusters = fs->s_data_sectors / fs->s_sectors_per_cluster;
	fs->s_cluster_size = fs->s_sectors_per_cluster * fs->s_sector_size;

	return fs;
}

/** **************************************************************************
	Create new filesystem
************************************************************************** **/
TVfs*
TFatfs_CreateFs(TVfsOps* vfsops, TVfsParam* param)
{
#define STW__(v, n) do {(v)[0] = (n)&0xff; (v)[1] = ((n)>>8)&0xff; } while (0)

	TFatfsParam* p = (TFatfsParam*)param;
	TFatfs* fs;
	TFat_DiskGeom* geo;
	uint32 dsc;
	uint32 spf;
	uint32 spc;
	uint32 cls;
	uint32 i;

	/* Get GEOMETRY */
	if (p->nbsectors == -1) {
		errexit("filesystem size unspecified");
	}
	for (geo = g_fat_disk_geom_table ; geo->mSectors != 0 ; geo++) {
		if (geo->mSectors == p->nbsectors) {
			break;
		}
	}
	if (!geo->mSectors) {
		errexit("unknown filesystem size (C/H/S is not found.)");
	}
	/* ENTRIES in ROOT-DIR */
	if (p->nbentries == -1) {
		p->nbentries = geo->mRootDirEntries;
	}
	p->nbentries = ut_RoundUp(p->nbentries, SECTORSIZE/sizeof(TFatDirEnt));

	if (p->nbresrvd == -1) {
		p->nbresrvd = 1;
	}
	/* FATs */
	if (p->nbfats != 1 && p->nbfats != 2) {
		p->nbfats = 2;
	}
	dsc = p->nbsectors
		- p->nbresrvd
		- p->nbentries / (SECTORSIZE/sizeof(TFatDirEnt));

	/* Calc. sectors-per-cluster */
	for (spc = 1; spc < 256 ; spc *= 2) {
		cls = 0;
		for (i = 0x8000; i != 0 ; i /= 2) {
			if ((cls|i)*spc + p->nbfats * ut_RoundUp((cls|i)*3+6, 1024)/1024 <= dsc) {
				cls |= i;
			}
		}
		if (cls < 0xff0) {
			break;
		}
	}
	if (spc == 256) {
		errexit("Too large fileystem");
	}
	spf = ut_RoundUp(cls*3+6, 1024)/1024;

	if (!(fs = (TFatfs*)calloc(1, sizeof(TFatfs)))) {
		errexit("not enough memory for filesystem");
	}
	if (!(fs->mRawImage = (uint8*)calloc(p->nbsectors, SECTORSIZE))) {
		errexit("not enough memory for filesystem");
	}

	fs->mCommon.mFsops = vfsops;
	fs->mFSS = (TFatSuperSector*)(fs->mRawImage);

	/* fs->mFSS->s_jump_to_bootcode[3]; */
	memcpy(fs->mFSS->s_system_id, "MKIMGFAT", 8);
	STW__(fs->mFSS->s_sector_size, SECTORSIZE);
	fs->mFSS->s_sectors_per_cluster = spc;
	fs->mFSS->s_reserved_sectors = p->nbresrvd;
	fs->mFSS->s_fats = p->nbfats;
	STW__(fs->mFSS->s_dir_entries, p->nbentries);
	if (p->nbsectors < 0x10000) {
		STW__(fs->mFSS->s_total_sectors, p->nbsectors);
	} else {
		fs->mFSS->s_total_sectors_2 = p->nbsectors;
	}
	fs->mFSS->s_media = geo->mMedia;
	fs->mFSS->s_sectors_per_fat = spf;
	fs->mFSS->s_sectors_per_track = geo->mSect;
	fs->mFSS->s_heads = geo->mHead;
	/* fs->mFSS->s_hidden_sectors; */
	/* fs->mFSS->s_reserved1; */
	/* fs->mFSS->s_reserved2; */
	fs->mFSS->s_extbootsign = 0x29;
	/* fs->mFSS->s_volume_id[4]; */
	/* fs->mFSS->s_volume_label[11]; */
	memcpy(fs->mFSS->s_fs_type, "FAT12   ", 8);
	/* fs->mFSS->s_boot_code[448]; */
	/* fs->mFSS->s_magic; */

	TFatfs_Setup(fs);

	fs->mFAT[0] = geo->mMedia;
	fs->mFAT[1] = 0xff;
	fs->mFAT[2] = 0xff;

	return (TVfs*)fs;
}

/** **************************************************************************
	load filesystem image
************************************************************************** **/
TVfs*
TFatfs_LoadFromFile(TVfsOps* vfsops, TVfsParam* vfsprm__unused, FILE * fh)
{
	size_t fssize = 0;
	uint32 sre;
	TFatfs *fs;

	if ((fseek(fh, 0, SEEK_END) < 0) || ((fssize = ftell(fh)) < 0)) {
		pexit("input filesystem image");
	}
	rewind(fh);

	fssize = (fssize + SECTORSIZE - 1) / SECTORSIZE;
	if (fssize < 16) {
		errexit("too small filesystem");
	}

	if (!(fs = (TFatfs*)calloc(1, sizeof(TFatfs)))) {
		errexit("not enough memory for filesystem");
	}
	if (!(fs->mRawImage = (uint8*)calloc(fssize, SECTORSIZE))) {
		errexit("not enough memory for filesystem");
	}

	if (fread(fs->mRawImage, SECTORSIZE, fssize, fh) != fssize) {
		pexit("input filesystem image");
	}

	fs->mCommon.mFsops = vfsops;
	fs->mFSS = (TFatSuperSector*)(fs->mRawImage);

	return (TVfs*)TFatfs_Setup(fs);
}

/** **************************************************************************
************************************************************************** **/
uint32
TFatfs_GetRootInodeNo(TVfs* fs)
{
	return FAT_ROOT_INO;
}

/** **************************************************************************
************************************************************************** **/
void
TFatfs_SaveToFile(TVfs *vfs, FILE * fh)
{
	TFatfs* fs = (TFatfs*)vfs;
	int n = fs->s_total_sectors;
	int i;
	int sz = fs->s_sectors_per_fat * fs->s_sector_size;

	for (i = 0 ; i < fs->s_fats ; i++) {
		memcpy(fs->mFAT + sz * i, fs->mFAT, sz);
	}

	if (fwrite(fs->mRawImage, fs->s_sector_size, n, fh) < n) {
		pexit("output filesystem image");
	}
}

/** **************************************************************************
************************************************************************** **/
void
TFatfs_FillUnallocatedBlock(TVfs* vfs, char c)
{
	TFatfs* fs = (TFatfs*)vfs;
	int i;

#if 0 /* FIX ME ! : */
	for (i = 1; i < fs->sb->s_blocks_count; i++) {
		if (!TBitmap_IsSet(fs->bbm, i)) {
			memset(TExt2fs_GetBlockPtr(fs, i), c, BLOCKSIZE);
		}
	}
#endif
}

/** **************************************************************************
	printout fields in the dir-entry
************************************************************************** **/
void
TFatDirEnt_Print(TFatDirEnt* ent, FILE* fp)
{
	uint8 c;
	char attr[7] = "------";

	c = ent->name[0];

	if (ent->attr & ATTR_ARCH) attr[0] = 'A';
	if (ent->attr & ATTR_DIR) attr[1] = 'D';
	if (ent->attr & ATTR_VOLUME) attr[2] = 'V';
	if (ent->attr & ATTR_SYS) attr[3] = 'S';
	if (ent->attr & ATTR_HIDDEN) attr[4] = 'H';
	if (ent->attr & ATTR_RO) attr[5] = 'R';

	if (c == '\0') {
		fprintf(fp, "[        .   ] FREE\n");
	} else if (c == 0xe5) {
		fprintf(fp, "[        .   ] DELETED\n");
	} else {
		fprintf(fp, "[%8.8s.%3.3s] %s (%d) <%x> \n",
			ent->name, ent->ext, attr,
			ent->size,
			ent->start
		);
	}
}

/** **************************************************************************
	printout fat in the filesystem
************************************************************************** **/
void
TFatfs_PrintFat(TFatfs *fs, FILE* fp)
{
	uint32 i;
	uint32 n;
	char c;

	for (i = 0; i < fs->s_data_clusters + 2 ; i++) {
		n = FAT_GET_FAT12(fs, i);

		switch (n) {
		case 0x000: c = '.'; break;
		case 0xff8: c = 'E'; break;
		default: c = n == i+1 ? 'O' : 'X'; break;
		}
		fprintf(fp, "%c", c);

	}
}

/** **************************************************************************
	printout fields in the filesystem
************************************************************************** **/
void
TFatfs_Print(TVfs *vfs, FILE* fp)
{
	TFatfs* fs = (TFatfs*)vfs;
	int i;
	char fty[9];

	memcpy(fty, fs->mFSS->s_fs_type, 8);
	fty[8] = 0;

	fprintf(fp, "<<%s>>\n", fty);
	fprintf(fp, "%d sectors (%d reserved, %d * %d fat, %d rootdir, %d data)\n",
		fs->s_total_sectors,
		fs->s_reserved_sectors,
		fs->s_sectors_per_fat,
		fs->s_fats,
		fs->s_dir_sectors,
		fs->s_data_sectors
	);
	fprintf(fp, "%d entries in root directory\n",
		fs->s_dir_entries
	);
	fprintf(fp, "%d data clusters (%d sectors/clustor, sector = %d bytes)\n",
		fs->s_data_clusters,
		fs->s_sectors_per_cluster,
		fs->s_sector_size
	);
	
	for (i = 0 ; i < fs->s_dir_entries ; i++) {
		fprintf(fp, "inode <%d> ", GET_INODE_NO(fs, fs->mRootDir + i));
		TFatDirEnt_Print(fs->mRootDir + i, fp);
	}
	TFatfs_PrintFat(fs, fp);
}

/** **************************************************************************
************************************************************************** **/
TVfsOps st_fatfsops = {
	.mFilesystemName = "fatfs",
	.mfLoadFromFile = TFatfs_LoadFromFile,
	.mfCreateFs = TFatfs_CreateFs,
	.mfGetRootInodeNo = TFatfs_GetRootInodeNo,
	.mfFillUnallocatedBlock = TFatfs_FillUnallocatedBlock,
	.mfPrint = TFatfs_Print,
	.mfSaveToFile = TFatfs_SaveToFile,
	.mfFindInodeOf = TFatfs_FindInodeOf,
	.mfAllocateBlockForInode = TFatfs_AllocateBlockForInode,
	.mfAddEntryToDirectory = TFatfs_AddEntryToDirectory,
#if 0
	.mfAddSymlink = TFatfs_AddSymlink,
#endif
	.mfAddRegulerFile = TFatfs_AddRegulerFile,
};

void showhelp(void)
{
	/* mkimage.fatfs */
	fprintf(stderr, "Usage: %s [options]\n"
	/* --------1---------2---------3---------4---------5---------6---------7---------8 */
	"mkimgfatfs " VERSION " (c) 2002-2005 N.Masuichi <nor at users.sourceforge.jp>\n"
	"  derived from: genext2fs (C) 2000 Xavier Bestel <xavier.bestel at free.fr>\n"
	"\n"
	"Base image:\n"
	"  -s image	Use this image as a starting point\n"
	"               (default: Create NEW filesystem image.)\n"
	"Filesystem option:\n"
	"  -b sectors   size in sectors\n"
	"  -i entries   Number of entries in root directory\n"
	"  -r reserved  Number of reserved sectors\n"
	"  -F fats      Number of FATs [default:2]\n"
	"  -e value     Fill unallocated blocks with value\n"
#if 0
	"  -H           Make files with holes\n"
#endif
	"Specify entries to add:\n"
	"  -f file      Add nodes from this spec file [default:stdin]\n"
#if 0
	"  -S dirctory  A starting directory to search file(s) with relative path\n"
	"  -p file	Use this file instead of /etc/passwd\n"
	/* ex.)
	 * -p /path/foo/bar/passwd (The file is in your real fs.)
	 * -p foo/bar/passwd (The file is in your real fs.)
	 * -p /_fsimage_/path/passwd (The file is in your image fs.)
	 *	Keyword : /_fsimag_
	 */
	"  -g file	Use this file instead of /etc/group\n"
#endif
	"Output image:\n"
	"  -o file      put result image to this file\n"
	"Listing:\n"
	"  -l file      Print resulting filesystem structure\n"
	"Help:\n"
	"  -h           Show this help\n\n",
	g_ThisExecName);
}

int
main(int argc, char **argv)
{
	TFatfsParam param = {
		.mCommon = {
			.fsin = NULL,
			.fsspec = NULL,
			.fsout = NULL,
			.fsrep = NULL,
			.fspasswd = NULL,
			.verbose = 0,
			.holes = 0,
			.emptyval = 0,
			.nosquash = 0,
		},
		.nbsectors = -1,
		.nbentries = -1,
		.nbresrvd = -1,
	};

	char c;

	g_ThisExecName = argv[0];
	while ((c = getopt(argc, argv, "s:f:o:l:b:i:r:e:zvnhp:F:")) != EOF) {
		switch (c) {
/** NOT IMPELMENTED **/
#if 0
		case 'S':
			g_PrefixOfRelatvePath = optarg;
			break;
#endif
		case 's':
			param.mCommon.fsin = optarg;
			break;
		case 'f':
			param.mCommon.fsspec = optarg;
			break;
		case 'l':
			param.mCommon.fsrep = optarg;
			break;
		case 'o':
			param.mCommon.fsout = optarg;
			break;
		case 'b':
			param.nbsectors = atoi(optarg);
			break;
		case 'i':
			param.nbentries = atoi(optarg);
			break;
		case 'r':
			param.nbresrvd = atoi(optarg);
			break;
		case 'F':
			param.nbfats = atoi(optarg);
			break;
		case 'e':
			param.mCommon.emptyval = atoi(optarg);
			break;
		case 'z':
			param.mCommon.holes = 1;
			break;
		case 'n':
			param.mCommon.nosquash = 1;
			break;
		case 'v':
			param.mCommon.verbose = 1;
			break;
		case 'h':
		case '?':
			showhelp();
			exit(0);
		default:
			exit(1);
		}
	}
	if (optind < (argc - 1)) {
		errexit("too many arguments");
	}

	return TVfs_MakeImage(&st_fatfsops, (TVfsParam*)&param);
}

/* END-OF-FILE */
