/* vi: sw=8 ts=8 : */
/**
	mkimgminixfs - create an minixfs image like as mkisofs

	Copyright (c) 2003 Masuichi Noritoshi <nor@users.sourceforge.jp>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version
	2 of the License, or (at your option) any later version.

	This code is derived from:

	genext2fs.c

	ext2 filesystem generator for embedded systems
	Copyright (C) 2000 Xavier Bestel <xavier.bestel@free.fr>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; version
	2 of the License.
**/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#ifdef HAVE_STRING_H
#       include <string.h>
#else
#       ifdef HAVE_STRINGS_H
#               include <strings.h>
#       endif
#endif
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>

#include <util.h>
#include <srcinode.h>
#include <mkimgfs.h>
#include <mkimgminixfs.h>

typedef struct {
	TVfsParam mCommon;
	int nbblocks;
	int nbinodes;
	int nbresrvd;
	int nbnamelen;
} TMinixfsParam;

TVfs* TMinixfs_LoadFromFile(TVfsOps*, TVfsParam*, FILE * fh);
TVfs* TMinixfs_CreateFs(TVfsOps*, TVfsParam*);
void TMinixfs_FillUnallocatedBlock(TVfs* fs, char c);
void TMinixfs_Print(TVfs *fs, FILE* fp);
void TMinixfs_SaveToFile(TVfs *fs, FILE * fh);
uint32 TMinixfs_FindInodeOf(TVfs *fs, uint32 nod, const char * name);
uint32 TMinixfs_AddEntryToDirectory(TVfs *fs, uint32 dnod, uint32 nod, const char* name);
uint32 TMinixfs_AllocateBlockForInode(TVfs* fs, uint32 nod, TSrcInode* si);
uint32 TMinixfs_GetRootInodeNo(TVfs* fs);
void TMinixfs_AddSymlink(TVfs* vfs, uint32 nod2, TSrcInode* si);
void TMinixfs_AddRegulerFile(TVfs* vfs, uint32 nod2, TSrcInode* si);


/** **************************************************************************
TBitmap_IsSet

RETURN: 
	0    : item-bit is 0
	o.w. : item-bit is 1

************************************************************************** **/
inline uint32
TBitmap_IsSet(TBitmap* b, uint32 item)
{
	return b->bm[item / 8] & (1 << (item % 8));
}

/** **************************************************************************
TBitmap_FindFreeBit

************************************************************************** **/
uint32
TBitmap_FindFreeBit(const TBitmap* b)
{
	int i;
	uint8 m;

	for(i = 0; i < BLOCKSIZE; i++) {
		m = b->bm[i];
		if (m == 0xff) continue;
		if (!(m & 0x01)) return i * 8 + 0;
		if (!(m & 0x02)) return i * 8 + 1;
		if (!(m & 0x04)) return i * 8 + 2;
		if (!(m & 0x08)) return i * 8 + 3;
		if (!(m & 0x10)) return i * 8 + 4;
		if (!(m & 0x20)) return i * 8 + 5;
		if (!(m & 0x40)) return i * 8 + 6;
		return i * 8 + 7;
	}
	return 0xffffffff;
}

/** **************************************************************************
TBitmap_SetBit
************************************************************************** **/
void
TBitmap_SetBit(TBitmap* b, uint32 item)
{
	b->bm[item / 8] |= (1 << (item % 8));
}

/** **************************************************************************
TBitmap_ResetBit
************************************************************************** **/
#if 0
void
TBitmap_ResetBit(TBitmap* b, uint32 item)
{
	b->bm[item / 8] &= ~(1 << (item % 8));
}
#endif

/** **************************************************************************
TBitmap_FindFreeBitAndSetIt
************************************************************************** **/
uint32
TBitmap_FindFreeBitAndSetIt(TBitmap* b)
{
	uint32 item = TBitmap_FindFreeBit(b);

	if (item != 0xffffffff) {
		TBitmap_SetBit(b, item);
	}
	return item;
}

/** **************************************************************************
************************************************************************** **/
void
TBitmap_Print(TBitmap* b, uint32 max, FILE* fp)
{
	uint32 i;

	fprintf(fp, "----+----1----+----2----+----3----+----4----+----5\n");
	for (i = 0 ; i < max; i++) {
		fputc((TBitmap_IsSet(b, i) ? '*' : '.'), fp);
		if (!((i+1) % 50)) {
			fprintf(fp, "\n");
		}
	}
	if (i % 50) {
		fprintf(fp, "\n");
	}
}

/** **************************************************************************
TMinixfs_GetBlockPtr

	returns pointer to given block

************************************************************************** **/
inline uint8*
TMinixfs_GetBlockPtr(TMinixfs *fs, uint32 blk)
{
	return fs->rawimg + blk*BLOCKSIZE;
}

/** **************************************************************************
TMinixfs_GetInodePtr

	returns pointer to given inode

************************************************************************** **/
inline inode*
TMinixfs_GetInodePtr(TMinixfs *fs, uint32 nod)
{
	if (!nod) {
		fprintf(stderr, "nod == 0\n"); /* FIX ME ! : */
	}
	return &fs->itab[nod-1];
}

/** **************************************************************************
TMinixfs_AllocateBlock

return:
	0	hole.
	!0	block no.
************************************************************************** **/
uint32
TMinixfs_AllocateBlock(TMinixfs *fs, uint32 hole)
{
	uint32 bn;

	if (hole) {
		return 0;
	}
		
	bn = TBitmap_FindFreeBitAndSetIt(fs->bbm);
	if (bn == 0xffffffff) {
		errexit("No free space (couldn't allocate a block)");
	}
	bn += fs->sb->s_firstdatazone -1;

	return bn;
}

/** **************************************************************************
TMinixfs_AllocateBlockForIndirect

	allocate block for indirect block
************************************************************************** **/
inline uint32
TMinixfs_AllocateBlockForIndirect(TMinixfs *fs)
{
	return TMinixfs_AllocateBlock(fs, 0);
}

/** **************************************************************************
TMinixfs_AllocateBlockForInode
	allocate a inode
************************************************************************** **/
uint32
TMinixfs_AllocateBlockForInode(TVfs* vfs, uint32 nod__unused, TSrcInode* si)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	uint32 nod2;
	inode* ip;

	nod2 = TBitmap_FindFreeBitAndSetIt(fs->ibm);

	if (!nod2) {
		errexit("No free inode (couldn't allocate a inode)");
	}
#if 0
	if (!(fs->gd->bg_free_inodes_count--)) {
		errexit("Corrupted FS (group descr. free inode count == 0)");
	}
	if (!(fs->sb->s_free_inodes_count--)) {
		errexit("Corrupted FS (superblock free inode count == 0)");
	}
#endif

	ip = TMinixfs_GetInodePtr(fs, nod2);

	ip->i_mode = si->mMode;
	ip->i_uid = si->mUid;
	ip->i_gid = si->mGid;

	/* MINIXFS cannot have atime, ctime and mtime. */

	switch (si->mMode & VFS_IFMT) {
	case VFS_IFBLK:
	case VFS_IFCHR:
		/* FIX ME ! : Can I use i_zone ??? */
		((uint8*)ip->i_zone)[0] = si->mMinor;
		((uint8*)ip->i_zone)[1] = si->mMajor;
		break;
	default:
		/* nothing to do. */
		break;
	}
	
	return nod2;
}

/** **************************************************************************
	initalize a TBpFs (iterator for blocks list)
************************************************************************** **/
void
TBpFs_Init(TBpFs* bp, TMinixfs* fs, uint32 nod)
{
	bp->asnfs = fs;
	bp->asnod = nod;
	bp->asnode = TMinixfs_GetInodePtr(bp->asnfs, bp->asnod);
	bp->bnum = 0;
	bp->bpdir = MINIX_BP_INIT;
	bp->state = MINIX_BP_STATE_INI;
}

/** **************************************************************************
	finalize a TBpFs
************************************************************************** **/
#if 0
void
TBpFs_Fini(TBpFs* bp)
{
	bp->asnfs = NULL;
}
#endif

/** **************************************************************************
************************************************************************** **/
void
TBpFs_ExtendIndirectBlock(TBpFs* bp)
{
	uint32* b;
	inode* node = bp->asnode;
	TMinixfs* fs = bp->asnfs;

	switch (bp->state) {
	case MINIX_BP_STATE_IND:
		node->i_zone[MINIX_BP_IND] =
			TMinixfs_AllocateBlockForIndirect(fs);
		break;
	case MINIX_BP_STATE_DIND:
		if (!(node->i_zone[MINIX_BP_DIND])) {
			node->i_zone[MINIX_BP_DIND] =
				TMinixfs_AllocateBlockForIndirect(fs);
		}
		b = (uint32*)TMinixfs_GetBlockPtr(fs,
			node->i_zone[MINIX_BP_DIND]
		    );
		b[bp->bpind] = TMinixfs_AllocateBlockForIndirect(fs);
		break;
	case MINIX_BP_STATE_TIND:
		errexit("file too big ! blocks list for "
			"inode %d extends past double "
			"indirect blocks!", bp->asnod);
		break;
	}
}

/** **************************************************************************
************************************************************************** **/
uint16*
TBpFs_GetRefToCurrentPos(TBpFs* bp)
{
	uint16* b;
	uint16* ref;
	inode* node = bp->asnode;
	TMinixfs* fs = bp->asnfs;

	switch (bp->state) {
	case MINIX_BP_STATE_DIR:
	case MINIX_BP_STATE_DIR_LAST:
		ref = &(node->i_zone[bp->bpdir]);
		break;
	case MINIX_BP_STATE_IND:
	case MINIX_BP_STATE_IND_LAST:
		b = (uint16*)TMinixfs_GetBlockPtr(fs,node->i_zone[MINIX_BP_IND]);
		ref = &b[bp->bpind];
		break;
	case MINIX_BP_STATE_DIND:
	case MINIX_BP_STATE_DIND_LAST:
		b = (uint16*)TMinixfs_GetBlockPtr(fs,
			node->i_zone[MINIX_BP_DIND]
		    );
		b = (uint16*)TMinixfs_GetBlockPtr(fs, b[bp->bpind]);
		ref = &b[bp->bpdind];
		break;
	case MINIX_BP_STATE_TIND:
	case MINIX_BP_STATE_TIND_LAST:
		b = (uint16*)TMinixfs_GetBlockPtr(fs,
			node->i_zone[MINIX_BP_TIND]
		    );
		b = (uint16*)TMinixfs_GetBlockPtr(fs, b[bp->bpind]);
		b = (uint16*)TMinixfs_GetBlockPtr(fs, b[bp->bpdind]);
		ref = &b[bp->bptind];
		break;
	}
	return ref;
}

/** **************************************************************************
************************************************************************** **/
uint32
TBpFs_AllocateBlock(TBpFs* bp, uint32 hole)
{
	return TMinixfs_AllocateBlock(bp->asnfs, hole);
}

/** **************************************************************************
RETURN:
	next block of inode (WALK_END for end)
	if create>0, append a newly allocated block at the end
	if hole!=0, create a hole in the file
************************************************************************** **/
uint32
TBpFs_Next(TBpFs* bp, uint32 *create, uint32 hole)
{
	uint16 *bkref = NULL;
	uint32 *b;
	uint32 extend = 0;
	uint32 ipb = BLOCKSIZE/sizeof(uint32);

	if (bp->asnode->i_size <= bp->bnum * BLOCKSIZE) {
		if (create && (*create)--) {
			extend = 1;
		} else {
			return WALK_END;
		}
	}

	 /* first direct block */
	switch (bp->state) {
	case MINIX_BP_STATE_INI:
		bp->bpdir = 0;
		bp->state = MINIX_BP_STATE_DIR;
		break;

	case MINIX_BP_STATE_DIR:
		bp->bpdir++;
		if (bp->bpdir == MINIX_BP_IND-1) {
			bp->state = MINIX_BP_STATE_DIR_LAST;
		}
		break;

	case MINIX_BP_STATE_DIR_LAST:
		bp->bpdir = MINIX_BP_IND;
		bp->bpind = 0;
		bp->state = MINIX_BP_STATE_IND;
		if (extend) {
			TBpFs_ExtendIndirectBlock(bp);
		}
		break;

	case MINIX_BP_STATE_IND:
		bp->bpind++;
		if (bp->bpind == ipb-1) {
			bp->state = MINIX_BP_STATE_IND_LAST;
		}
		break;

	case MINIX_BP_STATE_IND_LAST:
		bp->bpdir = MINIX_BP_DIND;
		bp->bpind = 0;
		bp->bpdind = 0;
		bp->state = MINIX_BP_STATE_DIND;
		if (extend) {
			TBpFs_ExtendIndirectBlock(bp);
		}
		break;

	case MINIX_BP_STATE_DIND:
		bp->bpdind++;
		if (bp->bpdind == ipb-1) {
			bp->state = MINIX_BP_STATE_DIND_LAST;
		}
		break;

	case MINIX_BP_STATE_DIND_LAST:
		bp->bpdind = 0;
		bp->bpind++;
		if (bp->bpind == ipb-1) {
			bp->state = MINIX_BP_STATE_TIND;
		} else {
			bp->state = MINIX_BP_STATE_DIND;
		}
		if (extend) {
			TBpFs_ExtendIndirectBlock(bp);
		}
		break;
	default:
		errexit("file too big ! blocks list for "
			"inode %d extends past double indirect blocks!",
			bp->asnod
		);
		break;
	}

	bkref = TBpFs_GetRefToCurrentPos(bp);
	if (extend) {
		*bkref = TBpFs_AllocateBlock(bp, hole);
	}

	bp->bnum++;
	if (*bkref) {
		if (!TBitmap_IsSet(bp->asnfs->bbm, *bkref - bp->asnfs->sb->s_firstdatazone +1)) {
			errexit("[block %d of inode %d is unallocated !]",
				*bkref, bp->asnod
			);
		}
	}

	return *bkref;
}

/** **************************************************************************
	add blocks to an inode (file/dir/etc...)
************************************************************************** **/
void
TMinixfs_ExtendBlock(TMinixfs *fs, uint32 nod, uint8 b[BLOCKSIZE], int amount)
{
	int create = amount;
	TBpFs bp;
	TBpFs lbw;
	uint32 bk;

	TBpFs_Init(&bp, fs, nod);

	lbw = bp;
	while ((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		lbw = bp;
	}
	bp = lbw;

	while (create) {
		int i;
		int hf = 1;

		if (!(fs->mCommon.holes)) {
			hf = 0;
		} else {
			/* check that block is filled with zero */
			uint32* d = (int32*)(b + BLOCKSIZE * (amount - create));
			for (i = 0; i < BLOCKSIZE / sizeof(uint32); i++) {
				if (d[i]) {
					hf = 0;
					break;
				}
			}
		}
		if ((bk = TBpFs_Next(&bp, &create, hf)) == WALK_END) {
			break;
		}
		if (!hf) {
			memcpy(TMinixfs_GetBlockPtr(fs, bk),
				b + BLOCKSIZE * (amount - (create+1)),
				BLOCKSIZE);
		}
	}
}

/** **************************************************************************
************************************************************************** **/
uint32
TMinixfs_AddEntryToDirectory(TVfs *vfs, uint32 dnod, uint32 nod, const char* name)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	TBpFs bp;
	uint32 bk;
	directory* d;
	int nlen;
	uint8 *b;
	inode *node;
	inode *pnode;

	pnode = TMinixfs_GetInodePtr(fs, dnod);

	if (!S_ISDIR(pnode->i_mode)) {
		/* FIX ME ! : using S_ISDIR is not good idea ! */
		errexit("can't add '%s' to a non-directory", name);
	}
	if (!*name) {
		errexit("bad name '%s' (not meaningful)", name);
	}
	if (strchr(name, '/')) {
		errexit("bad name '%s' (contains a slash)", name);
	}

	nlen = strlen(name);
	if(nlen > fs->mDirNameLen) {
		errexit("bad name '%s' (too long)", name);
	}

	TBpFs_Init(&bp, fs, dnod);

	while ((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		b = TMinixfs_GetBlockPtr(fs, bk);

#define XNEXT_DENTRY(fs,d)		\
		((directory*)((int8*)(d) + 2 + (fs)->mDirNameLen))
#define XISVALID_DENTRY(base,pos,size)	\
		((uint8*)(pos) < ((uint8*)(base) + (size)))

		// for all dir entries in block
		for (d = (directory*)b ;
		     XISVALID_DENTRY(b, d, BLOCKSIZE) ;
		     d = XNEXT_DENTRY(fs, d)) {
			if (!d->d_inode) {
				/* Update link count */
				d->d_inode = nod;
				node = TMinixfs_GetInodePtr(fs, nod);
				node->i_links_count++;
				memset(d->d_name, '\0', fs->mDirNameLen);
				memcpy(d->d_name, name, nlen);
				return nod;
			}
		}
	}
	// we found no free entry in the directory, so we add a block
	{
		uint8 b[BLOCKSIZE];

		memset(b, 0, sizeof(b));
		d = (directory*)b;

		d->d_inode = nod;
		node = TMinixfs_GetInodePtr(fs, nod);
		node->i_links_count++;

		memset(d->d_name, '\0', fs->mDirNameLen);
		memcpy(d->d_name, name, nlen);

		TMinixfs_ExtendBlock(fs, dnod, b, 1);
		TMinixfs_GetInodePtr(fs, dnod)->i_size += BLOCKSIZE;
	}
	return nod;
}

/** **************************************************************************
************************************************************************** **/
void
TMinixfs_AddSymlink(TVfs* vfs, uint32 nod2, TSrcInode* si)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	inode* ip;
	uint32 nlen;
	uint8* blk;
	uint32 rlen;

	nlen = strlen(si->mSourcePath);
	ip = TMinixfs_GetInodePtr(fs, nod2);

	rlen = ut_RoundUp(nlen, BLOCKSIZE);
	blk = XMALLOC(uint8, rlen);
	if (rlen > nlen) {
		memset(blk + nlen, 0, rlen - nlen);
	}
	strncpy(blk, si->mSourcePath, nlen);
	TMinixfs_ExtendBlock(fs, nod2, blk, rlen / BLOCKSIZE);
	free(blk);
	ip->i_size = nlen;
}

/** **************************************************************************
************************************************************************** **/
void
TMinixfs_AddRegulerFile(TVfs* vfs, uint32 nod2, TSrcInode* si)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	inode* ip;
	uint8* blk;
	struct stat st;
	FILE *fh;
	uint32 size;
	uint32 rlen;

	ip = TMinixfs_GetInodePtr(fs, nod2);
	lstat(si->mSourcePath, &st);
	if ((st.st_mode & S_IFMT) != S_IFREG) {
		pexit(si->mSourcePath);
	}
	if (!(fh = fopen(si->mSourcePath, "r"))) {
		pexit(si->mSourcePath);
	}
	size = st.st_size;
	rlen = ut_RoundUp(size, BLOCKSIZE);
	blk = XMALLOC(uint8, rlen);
	if (rlen > size) {
		memset(blk + size, 0, rlen - size);
	}
	if (fh) {
		fread(blk, size, 1, fh);
	} else {
		memset(blk, 0, size);
	}
	TMinixfs_ExtendBlock(fs, nod2, blk, rlen / BLOCKSIZE);
	ip->i_size = st.st_size;
	free(blk);
	fclose(fh);
}

/** **************************************************************************
************************************************************************** **/
uint32
TMinixfs_FindDir(TMinixfs *fs, uint32 nod, const char * name)
{
	TBpFs bp;
	uint32 bk;
	int nlen;

	nlen = strlen(name);
	if (nlen > fs->mDirNameLen) { /* too long name */
		return 0;
	}

	TBpFs_Init(&bp, fs, nod);
	while((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		directory *d;
		uint8 *b;
		b = TMinixfs_GetBlockPtr(fs, bk);
		for (d = (directory*)b;
		     (int8*)d < (int8*)b + BLOCKSIZE;
		     d = (directory*)((int8*)d + sizeof(*d)
					       + fs->mDirNameLen)) {
			if (d->d_inode &&
			   !memcmp(d->d_name, name, nlen) &&
			   (nlen == fs->mDirNameLen || d->d_name[nlen] == 0)) {
				return d->d_inode;
			}
		}
	}
	return 0;
}

/** **************************************************************************
************************************************************************** **/
uint32
TMinixfs_FindInodeOf(TVfs *vfs, uint32 nod, const char * name)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	char *p, *n, *n2 = strdup(name);
	n = n2;

	while(*n == '/') {
		nod = MINIX_ROOT_INO;
		n++;
	}

	while(*n) {
		if((p = strchr(n, '/')))
			(*p) = 0;
		if (!(nod = TMinixfs_FindDir(fs, nod, n)))
			break;
		if (p)
			n = p + 1;
		else
			break;
	}

	free(n2);
	return nod;
}

/** **************************************************************************
************************************************************************** **/
uint32
TMinixfs_GetRootInodeNo(TVfs* fs)
{
	return MINIX_ROOT_INO;
}

/** **************************************************************************
	Create new filesystem
************************************************************************** **/
TVfs*
TMinixfs_CreateFs(TVfsOps* vfsops, TVfsParam* param)
{
	TMinixfsParam* p = (TMinixfsParam*)param;
	int i;
	TMinixfs *fs;
	directory *d;
	uint8 b[BLOCKSIZE];
	uint32 nod;
	uint32 offset;
	uint32 ibs;
	uint32 dbs;
	
	
	if (p->nbblocks == -1) {
		errexit("filesystem size unspecified");
	}
	if (p->nbinodes == -1) {
		p->nbinodes = p->nbblocks * BLOCKSIZE
				 / ut_RoundUp(BYTES_PER_INODE, BLOCKSIZE);
	}
	p->nbinodes = ut_RoundUp(p->nbinodes, BLOCKSIZE/sizeof(inode));

	if (p->nbblocks < 16) {
		errexit("too small filesystem");
	}
	if (p->nbblocks >MINIX_MAX_ZONES) {
		errexit("too big filesystem");
	}

	if (!(fs = (TMinixfs*)calloc(1, sizeof(TMinixfs)))) {
		errexit("not enough memory for filesystem");
	}
	if (!(fs->rawimg = (uint8*)calloc(p->nbblocks, BLOCKSIZE))) {
		errexit("not enough memory for filesystem");
	}

	fs->mCommon.mFsops = vfsops;
	fs->sb = (superblock*)(fs->rawimg + BLOCKSIZE * 1);
	fs->sb->s_zones_count = p->nbblocks;
	fs->sb->s_log_zone_size = BLOCKSIZE >> 11;
	fs->sb->s_state = MINIX_VALID_FS;
	switch (p->nbnamelen) {
	case 14:
		fs->sb->s_magic = MINIX_SUPER_MAGIC;
		fs->mDirNameLen = 14;
		break;
	case 30:
		fs->sb->s_magic = MINIX_SUPER_MAGIC2;
		fs->mDirNameLen = 30;
		break;
	default:
		errexit("not supported dirctory name length");
		break;
	}

#define XUT_CEILING(n,d)	(((n)+(d)-1)/(d))
#define XUT_FLOOR(n,d)		((n)/(d))

	fs->sb->s_imap_blocks = XUT_CEILING(p->nbinodes + 1, BLOCKSIZE*8); 
	fs->sb->s_inodes_count = p->nbinodes;
	ibs = XUT_CEILING(p->nbinodes * sizeof(inode), BLOCKSIZE);

	dbs = p->nbblocks - (2 + fs->sb->s_imap_blocks + ibs);
	fs->sb->s_zmap_blocks = XUT_CEILING(dbs - 1, BLOCKSIZE*8); 
	dbs -= fs->sb->s_zmap_blocks;

	offset = 2;
	fs->ibm = (TBitmap*)(fs->rawimg + BLOCKSIZE * offset);
	offset += fs->sb->s_zmap_blocks;
	fs->bbm = (TBitmap*)(fs->rawimg + BLOCKSIZE * offset);
	offset += fs->sb->s_imap_blocks;
	fs->itab = (inode*)(fs->rawimg + BLOCKSIZE * offset);
	offset += ibs;
	fs->sb->s_firstdatazone = offset;


	// mark non-filesystem blocks and inodes as TBitmap_IsSet
	TBitmap_SetBit(fs->bbm, 0);
	for (i = dbs + 1; i < ut_RoundUp(dbs + 1, BLOCKSIZE * 8) ; i++) {
		TBitmap_SetBit(fs->bbm, i);
	}

	/* mark ineffective bits in imap */
	TBitmap_SetBit(fs->ibm, 0);
	for (i = fs->sb->s_inodes_count + 1 ;
	     i < ut_RoundUp(fs->sb->s_inodes_count + 1, BLOCKSIZE * 8) ;
	     i++) {
		TBitmap_SetBit(fs->ibm, i);
	}

	// make root inode and directory
	TBitmap_SetBit(fs->ibm, MINIX_ROOT_INO);
	fs->itab[MINIX_ROOT_INO-1].i_mode = MINIX_IFDIR | MINIX_IRWXU | MINIX_IRWXG | MINIX_IRWXO;
	TMinixfs_AddEntryToDirectory((TVfs*)fs, MINIX_ROOT_INO, MINIX_ROOT_INO, ".");
	TMinixfs_AddEntryToDirectory((TVfs*)fs, MINIX_ROOT_INO, MINIX_ROOT_INO, "..");


	/* internal working flags */
	if (p->mCommon.holes) {
		fs->mCommon.holes = 1;
	}
	
	return (TVfs*)fs;
}

/** **************************************************************************
	load filesystem image
************************************************************************** **/
TVfs*
TMinixfs_LoadFromFile(TVfsOps* vfsops, TVfsParam* vfsprm__unused, FILE * fh)
{
	size_t fssize = 0;
	TMinixfs *fs;
	uint32 offset;

	if ((fseek(fh, 0, SEEK_END) < 0) || ((fssize = ftell(fh)) < 0)) {
		pexit("input filesystem image");
	}
	rewind(fh);

	fssize = (fssize + BLOCKSIZE - 1) / BLOCKSIZE;
	if(fssize < 16) // totally arbitrary
		errexit("too small filesystem");

	if (fssize > MINIX_MAX_ZONES) {
		errexit("too big filesystem");
	}

	if (!(fs = (TMinixfs*)calloc(1, sizeof(TMinixfs)))) {
		errexit("not enough memory for filesystem");
	}
	if (!(fs->rawimg = (uint8*)calloc(fssize, BLOCKSIZE))) {
		errexit("not enough memory for filesystem");
	}

	if(fread(fs->rawimg, BLOCKSIZE, fssize, fh) != fssize)
		pexit("input filesystem image");

	fs->mCommon.mFsops = vfsops;
	fs->sb = (superblock*)(fs->rawimg + BLOCKSIZE * 1);
	switch (fs->sb->s_magic) {
	case MINIX_SUPER_MAGIC:
		fs->mDirNameLen = 14;
		break;
	case MINIX_SUPER_MAGIC2:
		fs->mDirNameLen = 30;
		break;
	case MINIX2_SUPER_MAGIC:
	case MINIX2_SUPER_MAGIC2:
		errexit("not support : MINIX V2 filesystem");
		break;
	default:
		errexit("not a suitable minix filesystem");
		break;
	}

	offset = 2;
	fs->ibm = (TBitmap*)(fs->rawimg + BLOCKSIZE * offset);
	offset += fs->sb->s_zmap_blocks;
	fs->bbm = (TBitmap*)(fs->rawimg + BLOCKSIZE * offset);
	offset += fs->sb->s_imap_blocks;
	fs->itab = (inode*)(fs->rawimg + BLOCKSIZE * offset);

	return (TVfs*)fs;
}

/** **************************************************************************
	finalize filesystem image object
************************************************************************** **/
void
TMinixfs_Fini(TMinixfs *fs)
{
	free(fs->rawimg);
	free(fs);
}

/** **************************************************************************
	print list of data blocks
************************************************************************** **/
void
TMinixfs_PrintListDataBlockNo(TMinixfs *fs, uint32 nod, FILE* fp)
{
	int i = 0;
	TBpFs bp;
	uint32 bn;
	TBpFs_Init(&bp, fs, nod);

	fprintf(fp, "[");

	while ((bn = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		fprintf(fp, "%d ", bn), i++;
	}

	fprintf(fp, "] %d(%d)\n", i, i * BLOCKSIZE);
}

/** **************************************************************************
	print block/char device minor and major
************************************************************************** **/
void
TMinixfs_PrintInodeDev(TMinixfs *fs, uint32 nod, FILE* fp)
{
	int minor = ((uint8*)TMinixfs_GetInodePtr(fs, nod)->i_zone)[0];
	int major = ((uint8*)TMinixfs_GetInodePtr(fs, nod)->i_zone)[1];
	fprintf(fp, "%d %d\n", major, minor);
}

/** **************************************************************************
	print an inode as a directory
************************************************************************** **/
void
TMinixfs_PrintInodeDir(TMinixfs *fs, uint32 nod, FILE* fp)
{
	TBpFs bp;
	uint32 bn;
	int i;

	TBpFs_Init(&bp, fs, nod);
	fprintf(fp, "directory for inode %d:\n", nod);

	while ((bn = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		uint8* b = TMinixfs_GetBlockPtr(fs, bn);
		directory* d = (directory*)b;
		while ((uint8*)d < b + BLOCKSIZE) {
			if (d->d_inode) {
				fprintf(fp, "[\"");
				for(i = 0;
				    i < fs->mDirNameLen && d->d_name[i];
				    i++) {
					fputc(d->d_name[i], fp);
				}
				fprintf(fp, "\" %d]\n", d->d_inode);
			}
			d = (directory*)((uint8*)d + sizeof(*d)
						   + fs->mDirNameLen);
		}
	}
}

/** **************************************************************************
	saves blocks to FILE*
************************************************************************** **/
void
TMinixfs_WriteBlockImage(TMinixfs *fs, uint32 nod, FILE* f)
{
	TBpFs bp;
	uint32 bk;
	int32 fsize = TMinixfs_GetInodePtr(fs, nod)->i_size;

	TBpFs_Init(&bp, fs, nod);
	while ((bk = TBpFs_Next(&bp, 0, 0)) != WALK_END) {
		if (fsize <= 0) {
			errexit("wrong size while saving inode %d", nod);
		}
		if (fwrite(TMinixfs_GetBlockPtr(fs, bk), (fsize > BLOCKSIZE) ? BLOCKSIZE : fsize, 1, f) != 1) {
			errexit("error while saving inode %d", nod);
		}
		fsize -= BLOCKSIZE;
	}
}

/** **************************************************************************
	print a symbolic link
************************************************************************** **/
void
TMinixfs_PrintInodeLink(TMinixfs *fs, uint32 nod, FILE* fp)
{
	fprintf(fp, "(L:");
	TMinixfs_WriteBlockImage(fs, nod, fp);
	fprintf(fp, ")\n");
}

/** **************************************************************************
	make a ls-like printout of permissions
************************************************************************** **/
void
make_perms(uint32 mode, char perms[11])
{
	strcpy(perms, "----------");
	if (mode & MINIX_IRUSR) perms[1] = 'r';
	if (mode & MINIX_IWUSR) perms[2] = 'w';
	if (mode & MINIX_IXUSR) perms[3] = 'x';
	if (mode & MINIX_IRGRP) perms[4] = 'r';
	if (mode & MINIX_IWGRP) perms[5] = 'w';
	if (mode & MINIX_IXGRP) perms[6] = 'x';
	if (mode & MINIX_IROTH) perms[7] = 'r';
	if (mode & MINIX_IWOTH) perms[8] = 'w';
	if (mode & MINIX_IXOTH) perms[9] = 'x';
	if (mode & MINIX_ISUID) perms[3] = 's';
	if (mode & MINIX_ISGID) perms[6] = 's';
	if (mode & MINIX_ISVTX) perms[9] = 't';

	switch (mode & MINIX_IFMT) {
	case MINIX_IFSOCK:*perms = 's'; break;
	case MINIX_IFLNK: *perms = 'l'; break;
	case MINIX_IFREG: *perms = '-'; break;
	case MINIX_IFBLK: *perms = 'b'; break;
	case MINIX_IFDIR: *perms = 'd'; break;
	case MINIX_IFCHR: *perms = 'c'; break;
	case MINIX_IFIFO: *perms = 'p'; break;
	default:	 *perms = '?'; break;
	}
}

/** **************************************************************************
	print an inode
************************************************************************** **/
void
print_inode(TMinixfs *fs, uint32 nod, FILE* fp)
{
	char perms[11];
	inode* in;

	in = TMinixfs_GetInodePtr(fs, nod);

	if(!in->i_mode) {
		return;
	}

	make_perms(TMinixfs_GetInodePtr(fs, nod)->i_mode, perms);
		/* --inode-- type -link- -uid- -gid- */
	fprintf(fp, "%09d %s %6d %5d %5d %10d ",
		nod,
		perms,
		in->i_links_count,
		in->i_uid, in->i_gid,
		in->i_size
	);

	if (!TBitmap_IsSet(fs->ibm, nod)) {
		fprintf(fp, "(unallocated)\n");
		return;
	}

	switch (TMinixfs_GetInodePtr(fs, nod)->i_mode & MINIX_IFMT) {
	case MINIX_IFLNK:
		TMinixfs_PrintInodeLink(fs, nod, fp);
		break;
	case MINIX_IFBLK:
	case MINIX_IFCHR:
		TMinixfs_PrintInodeDev(fs, nod, fp);
		break;
	case MINIX_IFDIR:
		TMinixfs_PrintListDataBlockNo(fs, nod, fp);
		TMinixfs_PrintInodeDir(fs, nod, fp);
		break;
	case MINIX_IFSOCK:
	case MINIX_IFREG:
	case MINIX_IFIFO:
	default:
		TMinixfs_PrintListDataBlockNo(fs, nod, fp);
		break;
	}
}

/** **************************************************************************
	printout fields in the filesystem
************************************************************************** **/
void
TMinixfs_Print(TVfs *vfs, FILE* fp)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	int i;

	fprintf(fp, "%d zones, first data zone: %d\n",
			fs->sb->s_zones_count,
			fs->sb->s_firstdatazone
	);
	fprintf(fp, "%d inodes\n",
			fs->sb->s_inodes_count
	);
	fprintf(fp, "zone size = %d\n",
			fs->sb->s_log_zone_size ?
				(fs->sb->s_log_zone_size << 11) : 1024
	);
	fprintf(fp, "zone bitmap: zone %d, inode bitmap: zone %d\n",
			fs->sb->s_zmap_blocks,
			fs->sb->s_imap_blocks
	);
	fprintf(fp, "%d, %x, %x, %d\n",
			fs->sb->s_max_size,
			fs->sb->s_magic,
			fs->sb->s_state,
			fs->sb->s_zones
	);

	fprintf(fp, "block bitmap allocation:\n");
	TBitmap_Print(fs->bbm, fs->sb->s_zones_count - fs->sb->s_firstdatazone + 1, fp);
	fprintf(fp, "inode bitmap allocation:\n");
	TBitmap_Print(fs->ibm, fs->sb->s_inodes_count + 1, fp);
	for(i=1; i<=fs->sb->s_inodes_count; i++) {
		if(TBitmap_IsSet(fs->ibm, i)) {
			print_inode(fs, i, fp);
		}
	}
}

/** **************************************************************************
************************************************************************** **/
void
TMinixfs_SaveToFile(TVfs *vfs, FILE * fh)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	int n = fs->sb->s_zones_count;

	if (fwrite(fs->rawimg, BLOCKSIZE, n, fh) < n) {
		pexit("output filesystem image");
	}
}

/** **************************************************************************
************************************************************************** **/
void
TMinixfs_FillUnallocatedBlock(TVfs* vfs, char c)
{
	TMinixfs* fs = (TMinixfs*)vfs;
	int i;

	for (i = fs->sb->s_firstdatazone; i < fs->sb->s_zones_count; i++) {
		if (!TBitmap_IsSet(fs->bbm, i - fs->sb->s_firstdatazone +1)) {
			memset(TMinixfs_GetBlockPtr(fs, i), c, BLOCKSIZE);
		}
	}
}

/** **************************************************************************
************************************************************************** **/
TVfsOps st_minixfsops = {
	.mFilesystemName = "minixfs",
	.mfLoadFromFile = TMinixfs_LoadFromFile,
	.mfCreateFs = TMinixfs_CreateFs,
	.mfGetRootInodeNo = TMinixfs_GetRootInodeNo,
	.mfFillUnallocatedBlock = TMinixfs_FillUnallocatedBlock,
	.mfPrint = TMinixfs_Print,
	.mfSaveToFile = TMinixfs_SaveToFile,
	.mfFindInodeOf = TMinixfs_FindInodeOf,
	.mfAllocateBlockForInode = TMinixfs_AllocateBlockForInode,
	.mfAddEntryToDirectory = TMinixfs_AddEntryToDirectory,
	.mfAddSymlink = TMinixfs_AddSymlink,
	.mfAddRegulerFile = TMinixfs_AddRegulerFile,
};

/** **************************************************************************
************************************************************************** **/
void showhelp(void)
{
	/* mkimage.ext2fs */
	fprintf(stderr, "Usage: %s [options]\n"
	/* --------1---------2---------3---------4---------5---------6---------7---------8 */
	"mkimgminixfs " VERSION " (c) 2002-2005 N.Masuichi <nor at users.sourceforge.jp>\n"
	"  derived from: genext2fs (C) 2000 Xavier Bestel <xavier.bestel at free.fr>\n"
	"\n"
	"Base image:\n"
	"  -s image	Use this image as a starting point\n"
	"               (default: Create NEW filesystem image.)\n"
	"Filesystem option:\n"
	"  -b blocks    size in blocks\n"
	"  -i inodes    Number of inodes\n"
	"  -r reserved  Number of reserved blocks\n"
	"  -e value     Fill unallocated blocks with value\n"
	"  -H           Make files with holes\n"
	"Specify entries to add:\n"
	"  -f file      Add nodes from this spec file [default:stdin]\n"
#if 0
	"  -S dirctory  A starting directory to search file(s) with relative path\n"
	"  -p file	Use this file instead of /etc/passwd\n"
	/* ex.)
	 * -p /path/foo/bar/passwd (The file is in your real fs.)
	 * -p foo/bar/passwd (The file is in your real fs.)
	 * -p /_fsimage_/path/passwd (The file is in your image fs.)
	 *	Keyword : /_fsimag_
	 */
	"  -g file	Use this file instead of /etc/group\n"
#endif
	"Output image:\n"
	"  -o file      put result image to this file\n"
	"Listing:\n"
	"  -l file      Print resulting filesystem structure\n"
	"Help:\n"
	"  -h           Show this help\n"
	, g_ThisExecName);
}

/** **************************************************************************
************************************************************************** **/
int
main(int argc, char **argv)
{
	TMinixfsParam param = {
		.mCommon = {
			.fsin = NULL,
			.fsspec = NULL,
			.fsout = NULL,
			.fsrep = NULL,
			.fspasswd = NULL,
			.verbose = 0,
			.holes = 0,
			.emptyval = 0,
			.nosquash = 0,
		},
		.nbblocks = -1,
		.nbinodes = -1,
		.nbresrvd = -1,
		.nbnamelen = 30,
	};

	char c;

	g_ThisExecName = argv[0];
	while ((c = getopt(argc, argv, "s:f:o:l:b:i:r:e:zvnhp:")) != EOF) {
		switch (c) {
/** NOT IMPELMENTED **/
#if 0
		case 'S':
			g_PrefixOfRelatvePath = optarg;
			break;
#endif
		case 's':
			param.mCommon.fsin = optarg;
			break;
		case 'f':
			param.mCommon.fsspec = optarg;
			break;
		case 'l':
			param.mCommon.fsrep = optarg;
			break;
		case 'o':
			param.mCommon.fsout = optarg;
			break;
		case 'b':
			param.nbblocks = atoi(optarg);
			break;
		case 'i':
			param.nbinodes = atoi(optarg);
			break;
		case 'r':
			param.nbresrvd = atoi(optarg);
			break;
		case 'e':
			param.mCommon.emptyval = atoi(optarg);
			break;
		case 'z':
			param.mCommon.holes = 1;
			break;
		case 'n':
			param.mCommon.nosquash = 1;
			break;
		case 'v':
			param.mCommon.verbose = 1;
			break;
		case 'h':
		case '?':
			showhelp();
			exit(0);
		default:
			exit(1);
		}
	}
	if (optind < (argc - 1)) {
		errexit("too many arguments");
	}

	return TVfs_MakeImage(&st_minixfsops, (TVfsParam*)&param);
}

/* END-OF-FILE */
