/*
** itunesdb.cpp:
**   based on itunesdb.c from GtkPod, highly hacked up for this plug-in.
**   GtkPod was written by Jorg Schuler and others.
**  This file is covered under the GNU Lesser General Public License
**    http://www.gnu.org/licenses/lgpl.txt
** 
** 
**    Copyright (C) 2003  Jorg Schuler and others
**
**    This library is free software; you can redistribute it and/or
**    modify it under the terms of the GNU Lesser General Public
**    License as published by the Free Software Foundation; either
**    version 2.1 of the License, or (at your option) any later version.
**
**    This library is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**    Lesser General Public License for more details.
**
**    You should have received a copy of the GNU Lesser General Public
**    License along with this library; if not, write to the Free Software
**    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** 
** For more information on GtkPod, please see: http://gtkpod.sourceforge.net/
**
*/
// #define _WIN32_WINNT 0x400
#define _CRT_SECURE_NO_DEPRECATE
#define ML_IPOD

#ifdef ML_IPOD
#include "view_ipod.h"
#endif

#include <windows.h>
#include <stdio.h>
#include "itunesdb.h"
#include "resource.h"

// for our copying/transcoding/etc
#include "../../winamp/wa_ipc.h"
#include "wa_transcode.h"
#include "../ml.h"
#include "strings.h"


/* Some notes on how to use the functions in this file:


   *** Reading the iTunesDB ***

   BOOL itunesdb_parse (char *path); /+ path to mountpoint /+
   will read an iTunesDB and pass the data over to your program. Your
   programm is responsible to keep a representation of the data.

   For each song itunesdb_parse() will pass a filled out Song structure
   to "it_add_song()", which has to be provided. The minimal Song
   structure looks like this (feel free to have it_add_song() do with it
   as it pleases -- and yes, you are responsible to free the memory):

   typedef struct
   {
     gunichar2 *album_utf16;    /+ album (utf16)         /+
     gunichar2 *artist_utf16;   /+ artist (utf16)        /+
     gunichar2 *title_utf16;    /+ title (utf16)         /+
     gunichar2 *genre_utf16;    /+ genre (utf16)         /+
     gunichar2 *comment_utf16;  /+ comment (utf16)       /+
     gunichar2 *composer_utf16; /+ Composer (utf16)      /+
     gunichar2 *fdesc_utf16;    /+ ? (utf16)             /+
     gunichar2 *ipod_path_utf16;/+ name of file on iPod: uses ":" instead of "/" /+
     unsigned int ipod_id;           /+ unique ID of song     /+
     int  size;              /+ size of file in bytes /+
     int  songlen;           /+ Length of song in ms  /+
     int  cd_nr;             /+ CD number             /+
     int  cds;               /+ number of CDs         /+
     int  track_nr;          /+ track number          /+
     int  tracks;            /+ number of tracks      /+
     int  year;              /+ year                  /+
     int  bitrate;           /+ bitrate               /+
   } Song;

   By #defining ITUNESDB_PROVIDE_ANSI, itunesdb_parse() will also
   provide ansi versions of the above utf16 strings. You must then add
   members "char *album"... to the Song structure.

   For each new playlist in the iTunesDB, it_add_playlist() is
   called with a pointer to the following Playlist struct:

   typedef struct
   {
     gunichar2 *name_utf16;
     unsigned int type;         /+ 1: master play list (PL_TYPE_MPL) /+
   } Playlist;

   Again, by #defining ITUNESDB_PROVIDE_ANSI, a member "char *name"
   will be initialized with a ansi version of the playlist name.

   it_add_playlist() must return a pointer under which it wants the
   playlist to be referenced when it_add_song_to_playlist() is called.

   For each song in the playlist, it_add_songid_to_playlist() is called
   with the above mentioned pointer to the playlist and the songid to
   be added.

   BOOL it_add_song (Song *song);
   Playlist *it_add_playlist (Playlist *plitem);
   void it_add_songid_to_playlist (Playlist *plitem, unsigned int id);


   *** Writing the iTunesDB ***

   BOOL itunesdb_write (char *path), /+ path to mountpoint /+
   will write an updated version of the iTunesDB.

   It uses the following functions to retrieve the data necessary data
   from memory:

   guint it_get_nr_of_songs (void);
   Song *it_get_song_by_nr (unsigned int n);
   unsigned int it_get_nr_of_playlists (void);
   Playlist *it_get_playlist_by_nr (unsigned int n);
   unsigned int it_get_nr_of_songs_in_playlist (Playlist *plitem);
   Song *it_get_song_in_playlist_by_nr (Playlist *plitem, unsigned int n);

   The master playlist is expected to be "it_get_playlist_by_nr(0)". Only
   the utf16 strings in the Playlist and Song struct are being used.

   Jorg Schuler, 19.12.2002 */


/* We instruct itunesdb_parse to provide ansi versions of the strings */
#define ITUNESDB_PROVIDE_ANSI

#define ITUNESDB_DEBUG 0

#ifdef ITUNESDB_DEBUG
  static char _debug[512];
#endif

LPCRITICAL_SECTION lpCriticalSection;

/* list with the contents of the Play Count file for use when
 * importing the iTunesDB */
static C_ItemList *playcounts = NULL;

/* structure to hold the contents of one entry of the Play Count file */
typedef struct playcount {
    unsigned int count;
    unsigned int rating;
    time_t lastplayed;
} playcount;

static playcount *get_next_playcount (void);

extern char * conf_file;
/* call itunesdb_parse () to read the iTunesDB  */
/* call itunesdb_write () to write the iTunesDB */

/* Header definitions for parsing the iTunesDB */
char ipodmagic[] = {'m', 'h', 'b', 'd', 0x68, 0x00, 0x00, 0x00};
static unsigned int utf16_strlen(gunichar2 *utf16_string);
static unsigned int sort_type;
static unsigned int count_mhod;

/* concat_dir (): concats "dir" and "file" into full filename, taking
   into account that "dir" may or may not end with "/".
   You must free the return string after use. Defined in misc.c */


time_t mactime_to_wintime (const unsigned long mactime)
{
    if((int)mactime == -1) return -1;
    if((int)mactime == 0) return -1;
    if (mactime != 0) return (time_t)(mactime - 2082844800);
    else return (time_t)mactime;
}

// convert windows timestamp to Macintosh timestamp
unsigned long wintime_to_mactime (const time_t time)
{
    if((int)time == -1) return 0;
    if((int)time == 0) return 0;
    return (unsigned long)(time + 2082844800);
}


gunichar2 *g_ansi_to_utf16(const char  *str, long len, long  *items_read, long  *items_written, int error)
{
  if(!str) return NULL;
  gunichar2 *dest=(gunichar2*)calloc((strlen(str)+1),sizeof(gunichar2));
  MultiByteToWideChar(CP_ACP,0,str,-1,dest,strlen(str)+1);
  /*while(*str) *(d++)=(gunichar2)*(str++); old (not as) gay utf8 to utf16.
  *d=0;*/
  return dest;
}

char *g_utf16_to_ansi(const gunichar2 *str, long len, long *items_read, long *items_written, int error)
{
  if(!str) return NULL;
  char dest[2048]="";
  char * d=dest;
  WideCharToMultiByte(CP_ACP,0,str,-1,d,sizeof(dest)-1,NULL,NULL);
  dest[2047]=0;
  /*while(*p) *(d++)=(unsigned char)*(p++); old gay utf16 to utf8.
  *d=0;*/
  return _strdup(dest);
}


char *concat_dir(char *dir1, char *dir2)
{
  char blah[2048];
  int nosep=0;
  int c=dir1[strlen(dir1)-1];
  if(c=='/' || c=='\\') nosep=1;
  wsprintf(blah,"%s%s%s",dir1,nosep?"":"/",dir2);
  return _strdup(blah);
}

/* Compare the two data. TRUE if identical */
static BOOL cmp_n_bytes (char *data1, char *data2, int n)
{
  int i;

  for(i=0; i<n; ++i)
    {
      if (data1[i] != data2[i]) return FALSE;
    }
  return TRUE;
}


/* Seeks to position "seek", then reads "n" bytes. Returns -1 on error
   during seek, or the number of bytes actually read */
static int seek_get_n_bytes (FILE *file, char *data, long seek, int n)
{
  int i;
  int read;

  if (fseek (file, seek, SEEK_SET) != 0) return -1;

  for(i=0; i<n; ++i)
    {
      read = fgetc (file);
      if (read == EOF) return i;
      *data++ = (char)read;
    }
  return i;
}

static int get1int(FILE *file, long seek) {
	unsigned char data[1];
	int n;

	if (seek_get_n_bytes (file, (char *)data, seek, 1) != 1) return -1;
	n = (unsigned int)data[0];
	return n;
}

static int get2int(FILE *file, long seek)
{
  unsigned char data[2];
  int n;

  if (seek_get_n_bytes (file, (char *)data, seek, 2) != 2) return -1;
  n = ((unsigned int)data[1]) << 8;
  n += ((unsigned int)data[0]);
  return n;
}

/* Get the 4-byte-number stored at position "seek" in "file"
   (or -1 when an error occured) */
static int get4int(FILE *file, long seek)
{
  unsigned char data[4];
  int n;

  if (seek_get_n_bytes (file, (char *)data, seek, 4) != 4) return -1;
  n =  ((unsigned int)data[3]) << 24;
  n += ((unsigned int)data[2]) << 16;
  n += ((unsigned int)data[1]) << 8;
  n += ((unsigned int)data[0]);
  return n;
}




/* Fix UTF16 String for BIGENDIAN machines (like PPC) */
static gunichar2 *fixup_utf16(gunichar2 *utf16_string) {
#if 0
#if (G_BYTE_ORDER == G_BIG_ENDIAN)
int i;
 for(i=0; i<utf16_strlen(utf16_string); i++)
 {
    utf16_string[i] = ((utf16_string[i]<<8) & 0xff00) | 
                  ((utf16_string[i]>>8) & 0xff);
 
 }
#endif
#endif
return utf16_string;
}


/* return the length of the header *ml, the genre number *mty,
   and a string with the entry (in UTF16?). After use you must
   free the string with free (). Returns NULL in case of error. */
static gunichar2 *get_mhod (FILE *file, long seek, int *ml, int *mty)
{
  char data[4];
  gunichar2 *entry_utf16=NULL;
  int xl;

#if ITUNESDB_DEBUG
  sprintf(_debug, "get_mhod seek: %x\n", (int)seek);
  OutputDebugString(_debug);
#endif

  if (seek_get_n_bytes (file, data, seek, 4) != 4) 
    {
      *ml = -1;
      return NULL;
    }
  if (cmp_n_bytes (data, "mhod", 4) == FALSE )
    {
      *ml = -1;
      return NULL;
    }
  *ml = get4int (file, seek+8);       /* length         */
  *mty = get4int (file, seek+12);     /* mhod_id number */
  xl = get4int (file, seek+28);       /* entry length   */

#if ITUNESDB_DEBUG
  sprintf(_debug, "ml: %x mty: %x, xl: %x\n", *ml, *mty, xl);
  OutputDebugString(_debug);
#endif
/* this replaced with below (up to end of proc). Syncing with latest GTKpod.
  entry_utf16 = (unsigned short *)malloc (xl+2);
  if (seek_get_n_bytes (file, (char *)entry_utf16, seek+40, xl) != xl) {
    free (entry_utf16);
    *ml = -1;
    return NULL;
  }
  entry_utf16[xl/2] = 0; // add trailing 0

  return fixup_utf16(entry_utf16);
  */
  switch (*mty)
  {
  case MHOD_ID_PLAYLIST: /* do something with the "weird mhod" */
    break;
  case MHOD_ID_PODCAST_URL: // we HATE podcasts
  case MHOD_ID_PODCAST_ENCL:
    break;
  default:
    entry_utf16 = (gunichar2*)malloc (xl+2);
    if (seek_get_n_bytes (file, (char *)entry_utf16, seek+40, xl) != xl) {
	    free (entry_utf16);
	    entry_utf16 = NULL;
	    *ml = -1;
    }
    else
    {
	    entry_utf16[xl/2] = 0; /* add trailing 0 */
    }
    break;
  }
  return fixup_utf16(entry_utf16);

}

static int get_result(int n) 
{
	switch ( n ) {
      case CSTR_LESS_THAN : return -1;
      case CSTR_EQUAL     : return  0;
      case CSTR_GREATER_THAN : return  1;
	  default: return  0;
      }
}

static int STRCMP_NULLOK(const char *pa, const char *pb)
{
  if (!pa) pa="";
  if (!pb) pb="";
  return get_result(CompareString(LOCALE_SYSTEM_DEFAULT,0,pa,-1,pb,-1));
}

static long get_mhip(FILE *file, long seek, void *ipod, Playlist *plitem)
{
	char data[4];
	/* We read the mhip headers and skip everything else. If we
   find a mhyp header before n==songnum, something is wrong */
	if (seek_get_n_bytes (file, data, seek, 4) != 4) return -1;
  if (cmp_n_bytes (data, "mhyp", 4) == TRUE) return -1; /* Wrong!!! */
  if (cmp_n_bytes (data, "mhip", 4) == TRUE)
	 {
		 Playlistitem *ip = (Playlistitem *)calloc (sizeof (Playlistitem),1);
		 ip->podcastflag = get4int(file, seek+16);
		 ip->groupid = get4int(file, seek+20);
		 ip->trackid = get4int(file, seek+24);
		 ip->added_t = mactime_to_wintime(get4int(file, seek+28));
		 ip->podcastref = get4int(file, seek+32);
		 int mhipsize = get4int(file, seek+4);
	   if (seek_get_n_bytes (file, data, seek+mhipsize, 4) != 4) return -1;
	   if (cmp_n_bytes (data, "mhod", 4) == TRUE) 
		 {
			 ip->m_id = get4int(file ,seek+mhipsize+24);
		 }
		 #ifdef ML_IPOD
		 ((view_ipod*)ipod)->it_add_songid_to_playlist(plitem,ip->trackid);
		 ((view_ipod*)ipod)->it_add_plitem_to_playlist(plitem,ip);
		 #else
		 it_add_songid_to_playlist(plitem,ip->trackid);
		 #endif
 	 }
	int mhipSize = get4int (file, seek+4);
  int mhodSize = get4int (file, seek+8+mhipSize);
  seek += mhipSize + mhodSize;

	return seek;
}

/* get a PL, return pos where next PL should be, name and content */
static long get_pl(FILE *file, long seek, void * ipod) 
{
  gunichar2 *plname_utf16 = NULL, *plname_utf16_maybe;
#ifdef ITUNESDB_PROVIDE_ANSI
  char *plname_ansi;
#endif ITUNESDB_PROVIDE_ANSI
  unsigned int type, pltype, songnum, n;
  unsigned int nextseek;
  int zip;
  Playlist *plitem;

  char data[4];


#if ITUNESDB_DEBUG
  sprintf(_debug, "mhyp seek: %x\n", (int)seek);
  OutputDebugString(_debug);
#endif

  if (seek_get_n_bytes (file, data, seek, 4) != 4) return -1;
  if (cmp_n_bytes (data, "mhyp", 4) == FALSE)      return -1; /* not pl */
  pltype = get4int (file, seek+20) & 0xff;  /* Type of playlist (1= MPL) */
  songnum = get4int (file, seek+16); /* number of songs in playlist */
  nextseek = seek + get4int (file, seek+8); /* possible begin of next PL */
  zip = get4int(file, seek+4);
  if(zip==0) return -1; // error
  do {
    seek += zip;

    //if (seek_get_n_bytes (file, data, seek, 4) != 4) return -1; this was preventing reading of last&empty playlist
    plname_utf16_maybe = get_mhod(file, seek, &zip, (int*)&type); /* PL name */
    if (zip != -1) switch (type)
    {
    case MHOD_ID_PLAYLIST:
      break; /* here we could do something about the "weird mhod" */
    case MHOD_ID_TITLE:
      if (plname_utf16_maybe)
      {
        /* sometimes there seem to be two mhod TITLE headers */
        if (plname_utf16) free (plname_utf16);
        plname_utf16 = plname_utf16_maybe;
      }
      break;
    }
  } while (zip != -1); /* read all MHODs */
  if (!plname_utf16)
  {   /* we did not read a valid mhod TITLE header -> */
      /* we simply make up our own name */
    if (pltype == 1) plname_utf16 = g_ansi_to_utf16 (("iPod"),-1, NULL, NULL, NULL);
    else             plname_utf16 = g_ansi_to_utf16 (("Playlist"),-1, NULL, NULL, NULL);
  }
#ifdef ITUNESDB_PROVIDE_ANSI
  plname_ansi = g_utf16_to_ansi (plname_utf16, -1, NULL, NULL, NULL);
#endif 


#if ITUNESDB_DEBUG
  sprintf(_debug, "pln: %s(%d Tracks) \n", plname_ansi, (int)songnum);
  OutputDebugString(_debug);
#endif

  plitem = new Playlist;

#ifdef ITUNESDB_PROVIDE_ANSI
  plitem->name = plname_ansi;
#endif ITUNESDB_PROVIDE_ANSI
  plitem->name_utf16 = plname_utf16;
  plitem->type = pltype;

  /* create new playlist */
#ifdef ML_IPOD
  plitem = ((view_ipod*)ipod)->it_add_playlist(plitem);
#else
  plitem = it_add_playlist(plitem);
#endif

  n = 0;  /* number of songs read */
  while (n < songnum)
    {
			seek = get_mhip(file, seek, ipod, plitem);
			n++;
    }
  return nextseek;
}

static long get_mhit(FILE *file, long seek, void *ipod)
{
  Song *song;
  char data[4];
#ifdef ITUNESDB_PROVIDE_ANSI
  char *entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
  gunichar2 *entry_utf16;
  int type;
  int zip = 0;

#if ITUNESDB_DEBUG
  sprintf(_debug, "get_mhit seek: %x\n", (int)seek);
  OutputDebugString(_debug);
//  if(seek==0x1598b4)
//    __asm int 3;
#endif

  if (seek_get_n_bytes (file, data, seek, 4) != 4) return -1;
  if (cmp_n_bytes (data, "mhit", 4) == FALSE ) return -1; /* we are lost! */

  song = (Song *)calloc (sizeof (Song),1);

  song->ipod_id = get4int(file, seek+16);     /* iPod ID          */
  song->visible = get4int(file, seek+20);
  song->filetype = get4int(file, seek+24); /* file type */
  song->type1 = get1int(file, seek+28);       /* type1            */
  song->type2 = get1int(file, seek+29);       /* type2            */
  song->compilation = get1int(file, seek+30); /* compilation flag */
  song->rating = get1int(file, seek+31);      /* rating           */
  song->added = mactime_to_wintime(get4int(file,seek+32));/* added time       */
  song->size = get4int(file, seek+36);        /* file size        */
  song->songlen = get4int(file, seek+40);     /* time             */
  song->track_nr = get4int(file, seek+44);    /* track number     */
  song->tracks = get4int(file, seek+48);      /* nr of tracks     */
  song->year = get4int(file, seek+52);        /* year             */
  song->bitrate = get4int(file, seek+56);     /* bitrate          */
  song->samplerate = get4int(file, seek+60);  /* sample rate      */
  song->volume = get4int(file, seek+64);      /* volume           */
  song->start_t = get4int(file, seek+68);     /* start time       */
  song->stop_t = get4int(file, seek+72);      /* stop time        */
  song->soundcheck = get4int(file, seek+76);    /* sound check      */
  song->playcount = get4int(file, seek+80);   /* playcount        */
  song->playcount2 = get4int(file, seek+84);  /* playcount2       */
  song->lastplayed = mactime_to_wintime(get4int(file, seek+88)); /* last time played */
  song->cd_nr = get4int(file, seek+92);       /* CD nr            */
  song->cds = get4int(file, seek+96);         /* CD nr (total)    */
  song->userid = get4int(file, seek+96);      /* User ID          */
  song->lastupdate = mactime_to_wintime(get4int(file, seek+104)); /* last update      */
  song->bookmark_t = get4int(file, seek+108); /* bookmark time    */
  song->dbid_p = get4int(file, seek+112);       /* dbid             */
  song->dbid_s = get4int(file, seek+116); 
  song->checked = get1int(file, seek+120);
  song->app_rating = get1int(file, seek+121);
  song->BPM = get2int(file, seek+122);
  song->artwork_count = get2int(file, seek+124);
  song->unk9 = get2int(file, seek+126);
  song->artwork_size = get4int(file, seek+128);
  song->unk11 = get4int(file, seek+132);
  song->samplerate2 = get4int(file, seek+136);
  song->date_released = mactime_to_wintime(get4int(file, seek+140));
  song->unk141 = get2int(file, seek+144);
  song->unk142 = get2int(file, seek+146);
  song->unk15 =get4int(file, seek+148);
  song->unk16 =get4int(file, seek+152);

  if(get4int(file, seek+4) > 156) {        /* For dbversion >= 0x0c (iTunes 4.71 and later) 244 */
	  song->unk17 =get4int(file, seek+156);
	  song->unk18 =get4int(file, seek+160);
	  song->has_artwork = get1int(file, seek+164);
	  song->skip_shuff = get1int(file, seek+165);
	  song->resume_flag = get1int(file, seek+166);
	  song->podcast_flag = get1int(file, seek+167);
	  song->dbid2_p = get4int(file, seek+168);
	  song->dbid2_s = get4int(file, seek+172);
	  song->lyrics_flag = get1int(file, seek+176);
	  song->movie_flag = get1int(file, seek+177);
	  song->played_mark = get1int(file, seek+178);
	  song->unk37 = get1int(file, seek+179);
	  song->unk21 = get4int(file, seek+180);
	  song->unk22 = get4int(file, seek+184);
	  song->sample_count = get4int(file, seek+188);
	  song->unk24 = get4int(file, seek+192);
	  song->unk25 = get4int(file, seek+196);
	  song->unk26 = get4int(file, seek+200);
	  song->unk27 = get4int(file, seek+204);
	  song->mediatype = get4int(file, seek+208);
	  song->season_nr = get4int(file, seek+212);
	  song->episode_nr = get4int(file, seek+216);
	  song->unk31 = get4int(file, seek+220);
	  song->unk32 = get4int(file, seek+224);
	  song->unk33 = get4int(file, seek+228);
	  song->unk34 = get4int(file, seek+232);
	  song->unk35 = get4int(file, seek+236);
	  song->unk36 = get4int(file, seek+240);
  }
  seek += get4int (file, seek+4); //156 or 244;      /* 1st mhod starts here! */
  while(zip != -1)
    {
     seek += zip;
     entry_utf16 = get_mhod (file, seek, &zip, &type);
     if (entry_utf16 != NULL) {
#ifdef ITUNESDB_PROVIDE_ANSI
       entry_ansi = g_utf16_to_ansi (entry_utf16, -1, NULL, NULL, NULL);
#endif ITUNESDB_PROVIDE_ANSI
       switch (type)
   {
   case MHOD_ID_ALBUM:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->album = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->album_utf16 = entry_utf16;
     break;
   case MHOD_ID_ARTIST:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->artist = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->artist_utf16 = entry_utf16;
     break;
   case MHOD_ID_TITLE:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->title = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->title_utf16 = entry_utf16;
     break;
   case MHOD_ID_GENRE:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->genre = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->genre_utf16 = entry_utf16;
     break;
   case MHOD_ID_PATH:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->ipod_path = entry_ansi;
     song->pcfile = (char *)malloc(MAX_PATH);
#ifdef ML_IPOD
     ((view_ipod*)ipod)->getPcFilename(song->ipod_path,song->pcfile);
#endif ML_IPOD
#endif ITUNESDB_PROVIDE_ANSI
     song->ipod_path_utf16 = entry_utf16;
     break;
   case MHOD_ID_FDESC:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->fdesc = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->fdesc_utf16 = entry_utf16;
     break;
   case MHOD_ID_COMMENT:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->comment = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->comment_utf16 = entry_utf16;
     break;
   case MHOD_ID_COMPOSER:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->composer = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->composer_utf16 = entry_utf16;
     break;
   case MHOD_ID_CATEGORY:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->category = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->category_utf16 = entry_utf16;
     break;
   case MHOD_ID_GROUPING:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->grouping = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->grouping_utf16 = entry_utf16;
     break;
   case MHOD_ID_DESCRIPTION:
#ifdef ITUNESDB_PROVIDE_ANSI
     song->description = entry_ansi;
#endif ITUNESDB_PROVIDE_ANSI
     song->description_utf16 = entry_utf16;
     break;
   default: /* unknown entry -- discard */
#ifdef ITUNESDB_PROVIDE_ANSI
     free (entry_ansi);
#endif ITUNESDB_PROVIDE_ANSI
     free (entry_utf16);
     break;
       }
     }
  }
  playcount * playcount = get_next_playcount ();
  if (playcount)
  {
      if (playcount->rating)  song->rating = playcount->rating;
      song->playcount += playcount->count;
      if (playcount->lastplayed!=-1) {
        //if(song->lastplayed < playcount->lastplayed)
        time(&song->lastupdate); // = playcount->lastplayed;
        //song->lastupdate = playcount->lastplayed;
        song->lastplayed = playcount->lastplayed;
      }
      free (playcount);
  }

#ifdef ML_IPOD
  ((view_ipod*)ipod)->it_add_song (song);
#else
  it_add_song (song);
#endif
  
  return seek;   /* no more black magic */
}

int CurrentPlaycount;
/* get next playcount, that is the first entry of GList
 * playcounts. This entry is removed from the list. You must free the
 * return value after use */
static playcount *get_next_playcount (void)
{
    return (playcount*)playcounts->Get(CurrentPlaycount++);
}

/* delete all entries of GList *playcounts */
static void reset_playcounts (void)
{
    if(playcounts) delete playcounts;
    playcounts=new C_ItemList;
}

/* Read the Play Count file (formed by adding "Play Counts" to the
 * directory contained in @filename) and set up the GList *playcounts
 * */
static void init_playcounts (const char *filename)
{
  CurrentPlaycount=0;
  //char *dirname = g_path_get_dirname (filename);
  //char *plcname = g_build_filename (dirname, "Play Counts", NULL);
  char temp[MAX_PATH];
  //char *plcname=temp;
  strcpy(temp,filename);
  strcpy(strrchr(temp,'/')+1,"Play Counts");

  FILE *plycts = fopen (temp, "r");
  BOOL error = TRUE;

  reset_playcounts ();

  if (plycts) do
  {
    char data[4];
    unsigned int header_length, entry_length, entry_num, i=0;
    time_t tt = time (NULL);

    localtime (&tt);  /* set the ext. variable 'timezone' (see below) */
    if (seek_get_n_bytes (plycts, data, 0, 4) != 4)  break;
    if (cmp_n_bytes (data, "mhdp", 4) == FALSE)      break;
    header_length = get4int (plycts, 4);
      /* all the headers I know are 0x60 long -- if this one is longer
        we can simply ignore the additional information */
    if (header_length < 0x60)                        break;
    entry_length = get4int (plycts, 8);
      /* all the entries I know are 0x0c (firmware 1.3) or 0x10
       * (firmware 2.0) in length */
    if (entry_length < 0x0c)                         break;
      /* number of entries */
      entry_num = get4int (plycts, 12);
    for (i=0; i<entry_num; ++i)
    {
      playcount *pc = (playcount*)calloc (sizeof (playcount),1);
      long seek = header_length + i*entry_length;

      playcounts->Add(pc);
      /* check if entry exists by reading its last four bytes */
      if (seek_get_n_bytes (plycts, data,
                seek+entry_length-4, 4) != 4) break;
      pc->count = get4int (plycts, seek);
      pc->lastplayed = mactime_to_wintime(get4int (plycts, seek+4));
      /* NOTE:
	   *
	   * The iPod (firmware 1.3) doesn't seem to use the timezone
	   * information correctly -- no matter what you set iPod's
	   * timezone to it will always record in UTC -- we need to
	   * subtract the difference between current timezone and UTC
	   * to get a correct display. 'timezone' (initialized above)
	   * contains the difference in seconds.
           */
	  if (pc->lastplayed!=-1)
    {
      pc->lastplayed += _timezone;
      if(_daylight) pc->lastplayed += _dstbias;
    }

      /* rating only exists if the entry length is at least 0x10 */
      if (entry_length >= 0x10) pc->rating = get4int (plycts, seek+12);
      if(pc->rating>100 || pc->rating<0) pc->rating=0;
    }
    if (i == entry_num)  error = FALSE;
  } while (FALSE);
  if (plycts)  fclose (plycts);
  if (error)   reset_playcounts ();
  //unlink(temp);
}

BOOL parse_OTG_Playlist(char * filename, void* ipod)
{
  FILE * file = fopen(filename,"rb");
  if(!file) return false;
  char data[4];

  do { // dummy loop for easier error handling i.e. break on error.
  
    if(seek_get_n_bytes(file,data,0,4)!=4 || !cmp_n_bytes(data,"mhpo",4)) break;

    int songs_start = get4int(file,4);
    int pltype = get4int(file,8);
    int songs_count = get4int(file,12);
    if(songs_start<0 || pltype<0 || songs_count<0)  break;

    Playlist * pl = add_new_playlist(_strdup ("On The Go"),ipod);
    pl->type = pltype;
    int songnum=0;
#ifdef ML_IPOD
    Playlist * mpl=((view_ipod*)ipod)->it_get_playlist_by_nr(0);
#else
    Playlist * mpl=it_get_playlist_by_nr(0);
#endif
    for(int seek=songs_start; (seek-songs_start)/4<=songs_count; seek+=4)
    {
      songnum = get4int(file,seek);
      if(songnum<0) break;
      if(songnum<mpl->members.GetSize()) pl->members.Add(mpl->members.Get(songnum));
    }
    if(songnum<0) break;
    fclose(file);
    return true;
  } while (false);

  if(file) fclose(file);
  return false;
}

int read_3(FILE * f)
{
  char buf[3]={0,0,0,};
  fread(&buf[0],3,1,f);
  unsigned int ret = 0;
  ret += ((unsigned int)buf[0]);
  ret += ((unsigned int)buf[1]) << 8;
  ret += ((unsigned int)buf[2]) << 16;
  return ret;
}

BOOL parse_ShuffleStats(char * filename, void * ipod)
{
#ifdef ML_IPOD
	//get master playlist
  Playlist * p = ((view_ipod*)ipod)->it_get_playlist_by_nr(0);
#else
  Playlist * p = it_get_playlist_by_nr(0);
#endif

  FILE * f = fopen(filename,"rb");
  if(!f) return FALSE;
  
  int n = read_3(f);
  /*char buf2[100];
  sprintf(buf2,"num: %d",n);
  MessageBox(NULL,buf2,"df",0);*/
  read_3(f); // blank
  for(int i=0; i<n; i++) {
    int size = read_3(f);
    int bm = read_3(f); //bookmark time
    read_3(f); //unk 1
    read_3(f); //unk 2
    int pc = read_3(f); //playcount
    int sk = read_3(f); //skippedcount
    size-=18;
    while(size>0) { read_3(f); size-=3; }
    if(bm != 0xffffff && (pc > 0 || sk > 0)) { //changed!
      /*char buf[100];
      sprintf(buf,"track %d of %d pc: %d sk: %d",i,n,pc,sk);
      MessageBox(NULL,buf,"df",0);*/
      Song * s = ((view_ipod *)ipod)->it_get_song_in_playlist_by_nr(p,i);
      if(s) {
        s->playcount+=pc;
        time(&(s->lastplayed));
        time(&(s->lastupdate));
      }
    }
  }
  fclose(f);
  return TRUE;
}

void parse_SP_Info(char * filename, void * ipod)
{
  FILE * file = fopen(filename,"rt");
  char * buf = (char*)calloc(sizeof(char),2048);
  while(!!file)
  {
    if(!fgets(buf,sizeof(buf)-1,file)) break;
    while(STRCMP_NULLOK(buf,"#\n")!=0) if(!fgets(buf,sizeof(buf)-1,file)) break; //wait for entry start

    if(!fgets(buf,2047,file)) break;
    if(strlen(buf)>1) buf[strlen(buf)-1]=0;  //find playlist of matching name
    Playlist * pl;
#ifdef ML_IPOD
    for(unsigned int i=1; i<((view_ipod*)ipod)->it_get_nr_of_playlists(); i++)
    {
      pl=((view_ipod*)ipod)->it_get_playlist_by_nr(i);
#else
    for(unsigned int i=1; i<it_get_nr_of_playlists(); i++)
    {
      pl=it_get_playlist_by_nr(i);
#endif
      if(STRCMP_NULLOK(pl->name,buf)==0) break;
      pl=NULL;
    }
    if(pl==NULL) continue;
    if(!!pl->sp) continue;
    //found
    pl->sp=(SmartPlaylist*)calloc(sizeof(SmartPlaylist),1);
    //fill in data about this smart playlist
    if(!fgets(buf,2047,file)) break; //read query string
    if(strlen(buf)>1) buf[strlen(buf)-1]=0;
    if(STRCMP_NULLOK(buf,"(null)")!=0) pl->sp->query=_strdup(buf);
    else pl->sp->query=_strdup("");
  
    if(!fgets(buf,2047,file)) break; //read limit
    if(strlen(buf)>1) buf[strlen(buf)-1]=0;
    pl->sp->limit=atoi(buf);

    if(!fgets(buf,2047,file)) break; //read limit type
    if(strlen(buf)>1) buf[strlen(buf)-1]=0;
    pl->sp->limitby=atoi(buf);

    if(!fgets(buf,2047,file)) break; //read sort info
    if(strlen(buf)>1) buf[strlen(buf)-1]=0;
    pl->sp->sort=atoi(buf);
    //up to here is mlipod SP 1.0
  }
  if(file) fclose(file);
}

BOOL itunesdb_parse_nocc (char *path, void * ipod) {
  char *filename = NULL;
  BOOL result;
  filename = concat_dir (path, "iPod_Control/iTunes/iTunesDB");
  result = itunesdb_parse_file (filename, ipod);
  if(!result) return 0;
  if (filename)  free (filename);
  
  filename = concat_dir(path,"iPod_Control/iTunes/OTGPlaylistInfo");
  parse_OTG_Playlist(filename, ipod);
  if (filename)  free (filename);

  filename = concat_dir(path,"iPod_Control/iTunes/mlipodSmartPlaylists");
  parse_SP_Info(filename, ipod);
  if (filename)  free (filename);
  
  filename = concat_dir(path,"iPod_Control/iTunes/iTunesStats");
  parse_ShuffleStats(filename,ipod);
  if (filename)  free (filename);

  return result;
}

/* Parse the iTunesDB and store the songs 
   using it_addsong () defined in song.c. 
   Returns TRUE on success, FALSE on error.
   "path" should point to the mount point of the
   iPod, e.e. "/mnt/ipod" */
BOOL itunesdb_parse (char *path, void * ipod)
{
  EnterCriticalSection(lpCriticalSection);
  BOOL ret=itunesdb_parse_nocc(path,ipod);
  LeaveCriticalSection(lpCriticalSection);
  return ret;
}

/* Same as itunesdb_parse(), but let's specify the filename directly */
BOOL itunesdb_parse_file (char *filename, void * ipod)
{
  FILE *itunes = NULL;
  BOOL result = FALSE;
  char data[8];
  long seek=0, pl_mhsd=0;
  int zip,nr_tracks=0, nr_playlists=0;
  BOOL swapped_mhsd=false;
#if ITUNESDB_DEBUG
  sprintf(_debug, "Parsing %s\nenter: %4d\n", filename, it_get_nr_of_songs ());
  OutputDebugString(_debug);
#endif

  itunes = fopen (filename, "rb");
  do { /* dummy loop for easier error handling */
    if (itunes == NULL) break; // couldn't open
    if (seek_get_n_bytes (itunes, data, seek, 4) != 4) // was data,0,8)!=8
    {
      //MessageBox(NULL,"Error reading","failed",0);
      //gtkpod_warning (_("Error reading \"%s\".\n"), filename);
      break;
    }
    /* for(i=0; i<8; ++i)  printf("%02x ", data[i]); printf("\n");*/
    if (cmp_n_bytes (data, "mhbd", 4) == FALSE)  //if (cmp_n_bytes (data, ipodmagic, 8) == FALSE) 
    {  
      //gtkpod_warning (_("\"%s\" is not a iTunesDB.\n"), filename);
      //MessageBox(NULL,"Not an itbd","failed",0);
      break;
    }
    seek = get4int (itunes, 4); // 292; /* the magic number!! (the HARDCODED start of the first mhit) */

    /* get every file entry */

    do
    {
      if (seek_get_n_bytes (itunes, data, seek, 8) != 8)  break;
      if (cmp_n_bytes (data, "mhsd", 4) == TRUE)
      { /* mhsd header -> determine start of playlists */
        if (get4int (itunes, seek + 12) == 1)
        { /* OK, tracklist, save start of playlists */
          if (!swapped_mhsd) pl_mhsd = seek + get4int (itunes, seek+8);
        }
        else if (get4int (itunes, seek + 12) == 2)
        { /* bad: these are playlists... switch */
          if (swapped_mhsd)
          { /* already switched once -> forget it */
            break;
          }
          else
          {
            pl_mhsd = seek;
            seek += get4int (itunes, seek+8);
            swapped_mhsd = TRUE;
          }
        }
        else
        { /* neither playlist nor track MHSD --> skip it */
          seek += get4int (itunes, seek+8);
        }
      }
      if (cmp_n_bytes (data, "mhlt", 4) == TRUE)
      { /* mhlt header -> number of tracks */
        nr_tracks = get4int (itunes, seek+8);
        if (nr_tracks == 0)
        {   /* no tracks -- skip directly to next mhsd */
          result = TRUE;
          break;
        }
      }
      if (cmp_n_bytes (data, "mhit", 4) == TRUE)
      { /* mhit header -> start of tracks*/
        result = TRUE;
        break;
      }
      zip = get4int (itunes, seek+4);
      if (zip == 0)  break;
      seek += zip;
    } while (result == FALSE);
    if (result == FALSE)  break; /* some error occured */
    result = FALSE;
    /* now we should be at the first MHIT */
    /* Read Play Count file if available */
    init_playcounts (filename);
      /* get every file entry */
    if (nr_tracks)  while(seek != -1) {
      /* get_mhit returns where it's guessing the next MHIT,
       if it fails, it returns '-1' */
      seek = get_mhit (itunes, seek,ipod);
    }
    
    /* next: playlists */
    seek = pl_mhsd;
    do
    {
      if (seek_get_n_bytes (itunes, data, seek, 8) != 8)  break;
      if (cmp_n_bytes (data, "mhsd", 4) == TRUE)
      { /* mhsd header */
        if (get4int (itunes, seek + 12) != 2)
        {  /* this is not a playlist MHSD -> skip it */
          seek += get4int (itunes, seek+8);
        }
      }
      if (cmp_n_bytes (data, "mhlp", 4) == TRUE)
      { /* mhlp header -> number of playlists */
        nr_playlists = get4int (itunes, seek+8);
      }
      if (cmp_n_bytes (data, "mhyp", 4) == TRUE)
      { /* mhyp header -> start of playlists */
        result = TRUE;
        break;
      }
      zip = get4int (itunes, seek+4);
      if (zip == 0)  break;
      seek += zip;
    } while (result == FALSE);
    if (result == FALSE)  break; /* some error occured */
    result = FALSE;

#if ITUNESDB_DEBUG
    //fprintf(stderr, "iTunesDB part2 starts at: %x\n", (int)seek);
#endif
    
    while(seek != -1) {
      seek = get_pl(itunes, seek,ipod);
    }

    result=true;
  } while (false);
  if (itunes != NULL) fclose (itunes);
  reset_playcounts();
  return result;
}


/* up to here we had the routines for reading the iTunesDB                */
/* ---------------------------------------------------------------------- */
/* from here on we have the routines for writing the iTunesDB             */

/* Name of the device in utf16 */
//gunichar2 ipod_name[] = { 'g', 't', 'k', 'p', 'o', 'd', 0 };

void write_SP_Info(char * filename, void * ipod)
{
  FILE * file = fopen(filename,"wt");
#ifdef ML_IPOD
  for(unsigned int plnum=1; plnum<((view_ipod*)ipod)->it_get_nr_of_playlists(); plnum++)
  {
    Playlist * pl=((view_ipod*)ipod)->it_get_playlist_by_nr(plnum);
#else
  for(unsigned int plnum=1; plnum<it_get_nr_of_playlists(); plnum++)
  {
    Playlist * pl=it_get_playlist_by_nr(plnum);
#endif

    if(pl) if(pl->sp)
    {
      fprintf(file,"#\n"); /* this specifys the beginning of an entry.
                              We can have arbitary amounts of data below
                              this, but it must be in order. */
      fprintf(file,"%s\n",pl->name);
      fprintf(file,"%s\n",pl->sp->query);
      fprintf(file,"%d\n",pl->sp->limit);
      fprintf(file,"%d\n",pl->sp->limitby);
      fprintf(file,"%d\n",pl->sp->sort); // up to here is mlipod SP 1.0
      // fill in new entries after this
    }
  }
  fclose(file);
}


/* Get length of utf16 string in number of characters (words) */
static unsigned int utf16_strlen (gunichar2 *utf16)
{
  if(!utf16) return 0;
  unsigned int i=0;
  while (utf16[i] != 0) ++i;
  return i;
}


/* return dummy mac time stamp -- maybe we can improve later */
/* iPod doesn't seem to care...? */
static int mactime()
{
  return 0x8c3abf9b;
}


/* Write 4-byte-integer "n" in correct order to "data".
   "data" must be sufficiently long ... */
static void store4int (unsigned int n, unsigned char *data)
{
  data[3] = (n >> 24) & 0xff;
  data[2] = (n >> 16) & 0xff;
  data[1] = (n >>  8) & 0xff;
  data[0] =  n & 0xff;
}

static void store4intrev (unsigned int n, unsigned char *data)
{
  data[0] = (n >> 24) & 0xff;
  data[1] = (n >> 16) & 0xff;
  data[2] = (n >>  8) & 0xff;
  data[3] =  n & 0xff;
}


/* Write "data", "n" bytes long to current position in file.
   Returns TRUE on success, FALSE otherwise */
static BOOL put_data_cur (FILE *file, char *data, int n)
{
  if (fwrite (data, 1, n, file) != (unsigned int)n) return FALSE;
  return TRUE;
}

/* Write 4-byte integer "n" to "file".
   Returns TRUE on success, FALSE otherwise */
static BOOL put_1int_cur (FILE *file, unsigned int n) {
	char data[1];

	data[0] = n & 0xff;
	return put_data_cur (file, data, 1);
}

static BOOL put_2int_cur (FILE *file, unsigned int n) {
	char data[2];

	data[1] = (n >> 8) & 0xff;
	data[0] = n & 0xff;
	return put_data_cur (file, data, 2);
}

static BOOL put_4int_cur (FILE *file, unsigned int n)
{
  char data[4];

  store4int (n, (unsigned char *)data);
  return put_data_cur (file, data, 4);
}

static BOOL put_4int_currev (FILE *file, unsigned int n)
{
  char data[4];

  store4intrev (n, (unsigned char *)data);
  return put_data_cur (file, data, 4);
}


/* Write 4-byte integer "n" to "file" at position "seek".
   After writing, the file position indicator is set
   to the end of the file.
   Returns TRUE on success, FALSE otherwise */
static BOOL put_4int_seek (FILE *file, unsigned int n, int seek)
{
  BOOL result;

  if (fseek (file, seek, SEEK_SET) != 0) return FALSE;
  result = put_4int_cur (file, n);
  if (fseek (file, 0, SEEK_END) != 0) return FALSE;
  return result;
}


/* Write "n" times 4-byte-zero at current position
   Returns TRUE on success, FALSE otherwise */
static BOOL put_n0_cur (FILE*file, unsigned int n)
{
  unsigned int i;
  BOOL result = TRUE;

  for (i=0; i<n; ++i)  result &= put_4int_cur (file, 0);
  return result;
}



/* Write out the mhbd header. Size will be written later */
static void mk_mhbd (FILE *file)
{
  put_data_cur (file, "mhbd", 4);
  put_4int_cur (file, 104); /* header size */
  put_4int_cur (file, -1);  /* size of whole mhdb -- fill in later */
  put_4int_cur (file, 1);   /* ? */
  put_4int_cur (file, 0x0c);   /* appears to be a version number of the database type. 
							   0x09 = iTunes 4.2, 0x0a = iTunes 4.5,
							   0x0b = iTunes 4.7, 0x0c = iTunes 4.71/4.8, 
							   0x0d = iTunes 4.9, 0x0e = iTunes 5, 
							   0x0f = iTunes 6, 0x10 = iTunes 6.0.1(?), 
							   0x11 = iTunes 6.0.2. */
  put_4int_cur (file, 3);   /* the number of MHSD children */
  put_n0_cur (file, 2);     /* dummy space 8byte */
  put_4int_cur (file, 2);   /* unknown always seems to be 2 */
  put_n0_cur (file, 17);    /* dummy space 68byte */
}

/* Fill in the missing items of the mhsd header:
   total size and number of mhods */
static void fix_mhbd (FILE *file, long mhbd_seek, long cur)
{
  put_4int_seek (file, cur-mhbd_seek, mhbd_seek+8); /* size of whole mhit */
}


/* Write out the mhsd header. Size will be written later */
static void mk_mhsd (FILE *file, unsigned int type)
{
  put_data_cur (file, "mhsd", 4);
  put_4int_cur (file, 96);   /* Headersize */
  put_4int_cur (file, -1);   /* size of whole mhsd -- fill in later */
  put_4int_cur (file, type); /* type: 1 = song, 2 = playlist */
  put_n0_cur (file, 20);    /* dummy space */
}  


/* Fill in the missing items of the mhsd header:
   total size and number of mhods */
static void fix_mhsd (FILE *file, long mhsd_seek, long cur)
{
  put_4int_seek (file, cur-mhsd_seek, mhsd_seek+8); /* size of whole mhit */
}


/* Write out the mhlt header. */
static void mk_mhlt (FILE *file, unsigned int song_num)
{
  put_data_cur (file, "mhlt", 4);
  put_4int_cur (file, 92);         /* Headersize */
  put_4int_cur (file, song_num);   /* songs in this itunesdb */
  put_n0_cur (file, 20);           /* dummy space */
}  


/* Write out the mhit header. Size will be written later */
static void mk_mhit (FILE *file, Song *song)
{

	if(_stricmp(song->ipod_path+strlen(song->ipod_path)-4,".mp4")==0) song->mediatype = 2;
	else if(_stricmp(song->ipod_path+strlen(song->ipod_path)-4,".m4v")==0) song->mediatype = 2;
	else if(_stricmp(song->ipod_path+strlen(song->ipod_path)-4,".mov")==0) song->mediatype = 2;
	else song->mediatype = 1;
  put_data_cur (file, "mhit", 4);
  put_4int_cur (file, 244);  /* header size was 244 */
  put_4int_cur (file, -1);   /* size of whole mhit -- fill in later */
  put_4int_cur (file, -1);   /* nr of mhods in this mhit -- later   */
  put_4int_cur (file, song->ipod_id); /* song index number          */
  put_4int_cur (file, song->visible);
  put_4int_cur (file, song->filetype); /* type               */
  put_1int_cur (file, song->type1);
  put_1int_cur (file, song->type2);
  put_1int_cur (file, song->compilation);
  put_1int_cur (file, song->rating);  /* rating        */
  put_4int_cur (file, wintime_to_mactime(song->added));/* timestamp */
  put_4int_cur (file, song->size);    /* filesize                   */
  put_4int_cur (file, song->songlen); /* length of song in ms       */
  put_4int_cur (file, song->track_nr);/* track number               */
  put_4int_cur (file, song->tracks);  /* number of tracks           */
  put_4int_cur (file, song->year);    /* the year                   */
  put_4int_cur (file, song->bitrate); /* bitrate              56byte */
  put_4int_cur (file, song->samplerate);
  put_4int_cur (file, song->volume);
  put_4int_cur (file, song->start_t);
  put_4int_cur (file, song->stop_t);
  put_4int_cur (file, song->soundcheck);
  put_4int_cur (file, song->playcount);  /*playcount            80byte */
  put_4int_cur (file, song->playcount2);
  put_4int_cur (file, wintime_to_mactime(song->lastplayed));/* last play time 88byte */
  put_4int_cur (file, song->cd_nr);		/* 92byte */
  put_4int_cur (file, song->cds);		/* 96byte */
  put_4int_cur (file, song->userid);		/* 100byte */
  put_4int_cur (file, wintime_to_mactime(song->lastupdate));		/* 104byte */
  put_4int_cur (file, song->bookmark_t);		/* 108byte */
  put_4int_cur (file, song->dbid_p);		/* 112byte */
  put_4int_cur (file, song->dbid_s);
  put_1int_cur (file, song->checked);		/* 120byte */
  put_1int_cur (file, song->app_rating);
  put_2int_cur (file, song->BPM);
  put_2int_cur (file, song->artwork_count);
  put_2int_cur (file, song->unk9);		/* 126byte */
  put_4int_cur (file, song->artwork_size);		/* 128byte */
  put_4int_cur (file, song->unk11);		/* 132byte */
  put_4int_cur (file, song->samplerate2);		/* 136byte */
  put_4int_cur (file, (int)song->date_released);		/* 140byte */
  put_2int_cur (file, song->unk141);
  put_2int_cur (file, song->unk142);
  put_4int_cur (file, song->unk15);
  put_4int_cur (file, song->unk16);
  put_4int_cur (file, song->unk17);
  put_4int_cur (file, song->unk18);		/* 160byte */
  put_1int_cur (file, song->has_artwork);
  put_1int_cur (file, song->skip_shuff);
  put_1int_cur (file, song->resume_flag);
  put_1int_cur (file, song->podcast_flag);
  put_4int_cur (file, song->dbid2_p);		/* 168byte */
  put_4int_cur (file, song->dbid2_s);
  put_1int_cur (file, song->lyrics_flag);
  put_1int_cur (file, song->movie_flag);
  put_1int_cur (file, song->played_mark);
  put_1int_cur (file, song->unk37);
  put_4int_cur (file, song->unk21);
  put_4int_cur (file, song->unk22);
  put_4int_cur (file, song->sample_count);		/* 188byte */
  put_4int_cur (file, song->unk24);
  put_4int_cur (file, song->unk25);
  put_4int_cur (file, song->unk26);
  put_4int_cur (file, song->unk27);
  put_4int_cur (file, song->mediatype);		/* 208byte */
  put_4int_cur (file, song->season_nr);
  put_4int_cur (file, song->episode_nr);
  put_4int_cur (file, song->unk31);
  put_4int_cur (file, song->unk32);
  put_4int_cur (file, song->unk33);
  put_4int_cur (file, song->unk34);
  put_4int_cur (file, song->unk35);
  put_4int_cur (file, song->unk36);		/* 244byte */
}  


/* Fill in the missing items of the mhit header:
   total size and number of mhods */
static void fix_mhit (FILE *file, long mhit_seek, long cur, int mhod_num)
{
  put_4int_seek (file, cur-mhit_seek, mhit_seek+8); /* size of whole mhit */
  put_4int_seek (file, mhod_num, mhit_seek+12);     /* nr of mhods        */
}


/* Write out one mhod header.
     type: see enum of MHMOD_IDs;
     string: utf16 string to pack
     fqid: will be used for playlists -- use 1 for songs */
static void mk_mhod (FILE *file, unsigned int type,
         gunichar2 *string, unsigned int fqid)
{
  unsigned int mod;
  unsigned int len;

  if (fqid == 1) mod = 40;   /* normal mhod */
  else           mod = 44;   /* playlist entry */

  len = utf16_strlen (string);         /* length of string in _words_     */

  put_data_cur (file, "mhod", 4);      /* header                          */
  put_4int_cur (file, 24);             /* size of header                  */
  put_4int_cur (file, 2*len+mod);      /* size of header + body           */
  put_4int_cur (file, type);           /* type of the entry               */
  put_n0_cur (file, 2);                /* dummy space                     */
  put_4int_cur (file, fqid);           /* refers to this ID if a PL item,
            otherwise always 1              */
  put_4int_cur (file, 2*len);          /* size of string                  */
  if (type < 100)
    {                                     /* no PL mhod */
      put_n0_cur (file, 2);               /* trash      */
      /* FIXME: this assumes "string" is writable. 
   However, this might not be the case,
   e.g. ipod_name might be in read-only mem. */
      string = fixup_utf16(string); 
      put_data_cur (file, (char *)string, 2*len); /* the string */
      string = fixup_utf16(string);
    }
  else
    {                                     
      put_n0_cur (file, 3);     /* PL mhods are different ... */
    }
}


extern int STRCMP_NULLOK2(const wchar_t *pa, const wchar_t *pb);

static int sortFunc(const void *elem1, const void *elem2)
{
  Song *a=(Song *)*(void **)elem1;
  Song *b=(Song *)*(void **)elem2;

  int use_by=sort_type;

#define RETIFNZ(v) if ((v)<0) return -1; if ((v)>0) return 1;

  // this might be too slow, but it'd be nice
  int x;
  for (x = 0; x < 5; x ++)
  {
    if (use_by == LPI_TITLE) // title 
    {
      int v=STRCMP_NULLOK2(a->title_utf16,b->title_utf16);
      RETIFNZ(v)
    }
    else if (use_by == LPI_ARTIST) // artist -> album -> title
    {
       int v=STRCMP_NULLOK2(a->artist_utf16,b->artist_utf16);
      RETIFNZ(v)
      use_by=LPI_ALBUM;
    }
    else if (use_by == LPI_ALBUM) // album -> title
    {
      int v=STRCMP_NULLOK2(a->album_utf16,b->album_utf16);
      RETIFNZ(v)
      use_by=LPI_TITLE;
    }
    else if (use_by == LPI_GENRE) // genre -> artist -> album -> title
    {
      int v=STRCMP_NULLOK2(a->genre_utf16,b->genre_utf16);
      RETIFNZ(v)
      use_by=LPI_ARTIST;
    }
    else if (use_by == LPI_COMPOSER) // composer -> title
    {
			int v=STRCMP_NULLOK2(a->composer_utf16,b->composer_utf16);
      RETIFNZ(v)
      use_by=LPI_TITLE;
    }
    else break; // no sort order?
  } 

  return 0;
}

static void mk_plindex (FILE *file,int song_num, unsigned int index_type,void *ipod)
{
	long plindex_seek = ftell(file);
	int i;
	Playlist *pl;
	C_ItemList *m_songs_sorted;
	m_songs_sorted = new C_ItemList;
	pl = ((view_ipod *)ipod)->it_get_playlist_by_nr(0);
  for(unsigned int i=0; i<((view_ipod *)ipod)->it_get_nr_of_songs_in_playlist(pl); i++)
  {
    Song *s=(Song *)((view_ipod *)ipod)->it_get_song_in_playlist_by_nr(pl,i);
		s->id = i;
		m_songs_sorted->Add((void *)s);
  }
	put_data_cur (file, "mhod", 4);			/* header									*/
	put_4int_cur (file, 24);						/* header size						*/
	put_4int_cur (file, -1);						/* size -> later          */
	put_4int_cur (file, 52);						/* mhod type							*/
	put_4int_cur (file, 0);							/* always zero						*/
	put_4int_cur (file, 0);							/* always zero						*/
  put_4int_cur (file, index_type);    /* sort on this type      */
	put_4int_cur (file, song_num);      /* number of songs				*/
	put_n0_cur (file, 10);							/* 40byte padding					*/
	sort_type = index_type;
	qsort(m_songs_sorted->GetAll(), song_num, sizeof(void*), sortFunc);
	for ( i=0; i<song_num; i++)
	{
		Song *s = (Song *)m_songs_sorted->Get(i);
		put_4int_cur (file, s->id);
	}
	delete m_songs_sorted;
	long cur = ftell(file);
	put_4int_seek (file, cur-plindex_seek, plindex_seek+8);
	count_mhod += 1;
	
}

/* Write out the mhlp header. Size will be written later */
static void mk_mhlp (FILE *file, unsigned int lists)
{
  put_data_cur (file, "mhlp", 4);      /* header                   */
  put_4int_cur (file, 92);             /* size of header           */
  put_4int_cur (file, lists);          /* playlists on iPod (including main!) */
  put_n0_cur (file, 20);               /* dummy space              */
}


/* Fix the mhlp header */
static void fix_mhlp (FILE *file, long mhlp_seek, int playlist_num)
{
  put_4int_seek (file, playlist_num, mhlp_seek+8); /* nr of playlists    */
}


typedef struct TokenisedQuery {
  //these following are for the SPL_PREFS section
  unsigned char limitype;
  unsigned char checklimits;
  unsigned char limitsort;
  unsigned char limitvalue;
  unsigned char limitsort_opposite;
  int ruleoperator; //(and==0 or==1)
  int size;
  C_ItemList rules;
} TokenisedQuery;

TokenisedQuery * tokeniseQuery(Playlist * pl, void * ipod);

static void write_SP_mhods(FILE * file, TokenisedQuery * tq);

/* Write out the mhyp header. Size will be written later */
static void mk_mhyp (FILE *file, gunichar2 *listname, unsigned int type, unsigned int song_num, Playlist * pl,long mhyp_seek,void * ipod)
{
  count_mhod = 1; //default case is just the name
  TokenisedQuery * tq = tokeniseQuery(pl,ipod);
  if(!!tq) count_mhod+=2; //two more mhods for smart playlist data
  put_data_cur (file, "mhyp", 4);      /* header                   */
  put_4int_cur (file, 108);            /* header size              */
  put_4int_cur (file, -1);             /* size -> later            */
  put_4int_cur (file, -1);						/* mhod children count -> later */
  put_4int_cur (file, song_num);       /* number of songs in plist -> later */
  put_4int_cur (file, type);           /* 1 = main, 0 = visible    */
  put_4int_cur (file, 0);              /* ?                        */
  put_4int_cur (file, 0);              /* ?                        */
  put_4int_cur (file, 0);              /* ?                        */
  put_n0_cur (file, 18);               /* dummy space              */
	/* put_data_cur (file, "mhod", 4);
  put_4int_cur (file, 24);
	put_4int_cur (file, 648);
	put_4int_cur (file, 100);
	put_n0_cur (file, 6);
	put_2int_cur (file, 130);
	put_2int_cur (file, 1);
	put_4int_cur (file, 3);
	put_4int_cur (file, 9);
	put_4int_cur (file, 3);
  put_2int_cur (file, 1);
	put_2int_cur (file, 0x2D);
  put_n0_cur (file, 3);
	put_2int_cur (file, 2);
	put_2int_cur (file, 0xC8);
  put_n0_cur (file, 3);
	put_2int_cur (file, 0xD);
	put_2int_cur (file, 0x3B);
  put_n0_cur (file, 3);
	put_2int_cur (file, 4);
	put_2int_cur (file, 0x7D);
  put_n0_cur (file, 3);
	put_2int_cur (file, 3);
	put_2int_cur (file, 0x7D);
  put_n0_cur (file, 3);
	put_2int_cur (file, 8);
	put_2int_cur (file, 0x50);
  put_n0_cur (file, 3);
	put_2int_cur (file, 0x17);
	put_2int_cur (file, 0x4F);
	put_4int_cur (file, 1);
	put_n0_cur (file, 2);
	put_2int_cur (file, 0x14);
	put_2int_cur (file, 0x50);
	put_4int_cur (file, 1);
  put_n0_cur (file, 2);
	put_2int_cur (file, 0x15);
	put_2int_cur (file, 0x7D);
	put_4int_cur (file, 1);
  put_n0_cur (file, 2);
	put_n0_cur (file, 92);
	put_4int_cur (file, 0x8C);
	put_n0_cur (file, 19); */
  mk_mhod (file, MHOD_ID_TITLE, listname, 1); //playlist name
	/* Library Playlist Index (type 52) */
	if (type == 1) {
		mk_plindex (file,song_num,LPI_TITLE,ipod);
		mk_plindex (file,song_num,LPI_ARTIST,ipod);
		mk_plindex (file,song_num,LPI_ALBUM,ipod);
		mk_plindex (file,song_num,LPI_GENRE,ipod);
		mk_plindex (file,song_num,LPI_COMPOSER,ipod);
	}
  if(!!tq) write_SP_mhods(file,tq);           //smart playlist data, if avaliable.
}


/* Fix the mhyp header */
static void fix_mhyp (FILE *file, long mhyp_seek, long cur)
{
  put_4int_seek (file, cur-mhyp_seek, mhyp_seek+8);    /* size */
	put_4int_seek (file, count_mhod, mhyp_seek+12);			/* mhod children count */
}


/* Header for new PL item */
static void mk_mhip (FILE *file, Playlistitem *ip)
{
  put_data_cur (file, "mhip", 4);
  put_4int_cur (file, 76);
  put_4int_cur (file, 120);
  put_4int_cur (file, 1);
  put_4int_cur (file, ip->podcastflag);
  put_4int_cur (file, ip->groupid);  
  put_4int_cur (file, ip->trackid);  /* song id in playlist */
  put_4int_cur (file, wintime_to_mactime(ip->added_t));
  put_4int_cur (file, ip->podcastref);
  put_n0_cur (file, 10); 
}


static void write_mhsd_one(FILE *file, void * ipod)
{
    Song *song;
    unsigned int i, song_num, mhod_num;
    long mhsd_seek, mhit_seek, mhlt_seek; 
#ifdef ML_IPOD
    song_num = ((view_ipod*)ipod)->it_get_nr_of_songs();
#else
    song_num = it_get_nr_of_songs();
#endif

    mhsd_seek = ftell (file);  /* get position of mhsd header */
    mk_mhsd (file, 1);         /* write header: type 1: song  */
    mhlt_seek = ftell (file);  /* get position of mhlt header */
    mk_mhlt (file, song_num);  /* write header with nr. of songs */
    for (i=0; i<song_num; ++i)  /* Write each song */
    {
#ifdef ML_IPOD
  if((song = ((view_ipod*)ipod)->it_get_song_by_nr (i)) == 0)
#else
  if((song = it_get_song_by_nr (i)) == 0)
#endif
  {
      //g_warning ("Invalid song Index!\n");
      break;
  }
  mhit_seek = ftell (file);
  mk_mhit (file, song);
  mhod_num = 0;
  if (utf16_strlen (song->title_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_TITLE, song->title_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->ipod_path_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_PATH, song->ipod_path_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->album_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_ALBUM, song->album_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->artist_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_ARTIST, song->artist_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->genre_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_GENRE, song->genre_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->fdesc_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_FDESC, song->fdesc_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->comment_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_COMMENT, song->comment_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->composer_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_COMPOSER, song->composer_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->category_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_CATEGORY, song->category_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->grouping_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_GROUPING, song->grouping_utf16, 1);
      ++mhod_num;
  }
  if (utf16_strlen (song->description_utf16) != 0)
  {
      mk_mhod (file, MHOD_ID_DESCRIPTION, song->description_utf16, 1);
      ++mhod_num;
  }
        /* Fill in the missing items of the mhit header */
  fix_mhit (file, mhit_seek, ftell (file), mhod_num);
    }
    fix_mhsd (file, mhsd_seek, ftell (file));
}

typedef struct QueryRule {
  unsigned char field;
  int action;
  int fromvalue;
  time_t fromdate;
  int fromunits;
  int tovalue;
  time_t todate;
  int tounits;
  unsigned int length; //68 for non-string or string size in bytes
  gunichar2 * string;
} QueryRule;

#define SKIP_WHITESPACE(x) { while (*x == ' ') x++; }
time_t IntegerField_ApplyConversion(const char *format);
TokenisedQuery * tokeniseQuery(Playlist * pl,void * ipod)
{
  if(pl==NULL) return NULL;
  SmartPlaylist * s = pl->sp;
  if(s==NULL) return NULL;
  s->live=false;

  if(!GetPrivateProfileInt("ml_ipod","livesmartplaylists",0,((view_ipod*)ipod)->ini_file)) return NULL;
 
  TokenisedQuery * out = new TokenisedQuery;
  out->size=136; // size is 136 + rulecount*56 + sum(rule->length)
  out->checklimits= s->limitby==0?0:1;
  out->limitvalue=s->limit;

  switch(s->limitby) {
  case 0: out->limitype=0;                 break;
  case 1: out->limitype=LIMITTYPE_MINUTES; break;
  case 2: out->limitype=LIMITTYPE_SONGS;   break;
  }
  
  out->limitsort_opposite=0;
  switch(s->sort) {
  case ID_IPODSORT_ARTIST: out->limitsort=LIMITSORT_ARTIST;     break;
  case ID_IPODSORT_ALBUM:  out->limitsort=LIMITSORT_ALBUM;      break;
  case ID_IPODSORT_TRACK:  out->limitsort=LIMITSORT_SONG_NAME;  break;
  case ID_IPODSORT_RANDOM: out->limitsort=LIMITSORT_RANDOM;     break;
  default: { delete out; return NULL; }
  }
  // now to determine the rule operator and rules
  if(s->query==NULL) return NULL;
  char * c = s->query;
  bool fail=false;
  out->ruleoperator=69; //unset
  //while(*c != 0)
  while(true)
  {
    QueryRule * rule = new QueryRule;
    ZeroMemory(rule,sizeof(QueryRule));
    char field[15];
    char * f= &field[0];
    SKIP_WHITESPACE(c);
    while(*c!=' ' && *c!='<' && *c!='>' && *c!='=' && *c!='!' && *c!=0) *(f++) = *(c++);
    *f = 0; //now we should have the first field name
    if     (!STRCMP_NULLOK(field,"artist"))    rule->field = SPLFIELD_ARTIST;
    else if(!STRCMP_NULLOK(field,"album"))     rule->field = SPLFIELD_ALBUM;
    else if(!STRCMP_NULLOK(field,"title"))     rule->field = SPLFIELD_SONG_NAME;
    else if(!STRCMP_NULLOK(field,"trackno"))   rule->field = SPLFIELD_TRACKNUMBER;
    else if(!STRCMP_NULLOK(field,"genre"))     rule->field = SPLFIELD_GENRE;
    else if(!STRCMP_NULLOK(field,"year"))      rule->field = SPLFIELD_YEAR;
    else if(!STRCMP_NULLOK(field,"comment"))   rule->field = SPLFIELD_COMMENT;
    else if(!STRCMP_NULLOK(field,"playcount")) rule->field = SPLFIELD_PLAYCOUNT;
    else if(!STRCMP_NULLOK(field,"lastplay"))  rule->field = SPLFIELD_LAST_PLAYED;
    else if(!STRCMP_NULLOK(field,"rating"))    rule->field = SPLFIELD_RATING;
    else break;
    
    bool nocomp=false;
    SKIP_WHITESPACE(c);
    if(*c == 0) break;
    if(*(c+1) == 0) break;

    if(*c == '=') rule->action=SPLACTION_IS_INT; //rule is =
    else if(*c == '!' && *(c+1) == '=') { c++; rule->action=SPLACTION_IS_NOT_INT; /*rule is !=*/}
    else if(*c == '>') {
      if(*(c+1) == '=') { c++; /*rule is >=*/ rule->action=SPLACTION_IS_GREATER_THAN_OR_EQUAL_TO;}
      else rule->action=SPLACTION_IS_GREATER_THAN; //rule is >
    }
    else if(*c == '<') {
      if(*(c+1) == '=') { c++; /*rule is <=*/ rule->action=SPLACTION_IS_LESS_THAN_OR_EQUAL_TO; }
      else rule->action=SPLACTION_IS_LESS_THAN; //rule is <
    }
    else {
      char op[255]="";
      char * f = &op[0];
      while(*c!=' ' && *c!=0) *(f++) = *(c++);
      *f=0;
      
      if (*c==0) return NULL;
      if (_stricmp(op,"HAS")==0) rule->action=SPLACTION_CONTAINS;
      else if(_stricmp(op,"NOTHAS")==0) rule->action=SPLACTION_DOES_NOT_CONTAIN;
      else if(_stricmp(op,"LIKE")==0)   rule->action=SPLACTION_CONTAINS; // hmm. Not quite right.
      else if(_stricmp(op,"BEGINS")==0) rule->action=SPLACTION_STARTS_WITH;
      else if(_stricmp(op,"ENDS")==0)   rule->action=SPLACTION_ENDS_WITH;
      else if(_stricmp(op,"ISEMPTY")==0) { rule->action=SPLACTION_IS_INT; nocomp=true; }
      else if(_stricmp(op,"ISNOTEMPTY")==0) { rule->action=SPLACTION_IS_NOT_INT; nocomp=true; }
      
    }

    bool isstring=false;
    switch(rule->field) {
    case SPLFIELD_ARTIST:
    case SPLFIELD_ALBUM:
    case SPLFIELD_SONG_NAME:
    case SPLFIELD_GENRE:
    case SPLFIELD_YEAR:
    case SPLFIELD_COMMENT:
      rule->field |= 0x01000000; //turn INT comparators into STRING ones.
      isstring=true;
      break;
    }
    char value[512]="";
    char end=' ';
    bool reldate=false;
    if(nocomp) {
      rule->string=g_ansi_to_utf16("", -1, NULL, NULL, NULL);
      rule->length=strlen("")*2;
      goto finishrule;
    }
    c++;
    SKIP_WHITESPACE(c);
    if(*c == 0) break;
    f=&value[0];
    if(*c =='\"') { end='\"'; c++; }
    if(*c =='[') { end=']'; c++; reldate=true; }
    while(*c!=end && *c!=0) *(f++) = *(c++);
    c++;
    *f=0;
    *(f+1)=0;
    //now we have value!
    rule->string=NULL;
    if(isstring) {
      rule->string=g_ansi_to_utf16(value, -1, NULL, NULL, NULL);
      rule->length=strlen(value)*2;
    } else if(reldate) {
      if(rule->action == SPLACTION_IS_LESS_THAN || rule->action == SPLACTION_IS_LESS_THAN_OR_EQUAL_TO) {
        rule->action = SPLACTION_IS_NOT_IN_THE_LAST;
      } else if (rule->action == SPLACTION_IS_GREATER_THAN || rule->action == SPLACTION_IS_GREATER_THAN_OR_EQUAL_TO) {
        rule->action = SPLACTION_IS_IN_THE_LAST;
      } else return NULL; // no need for live smart playlist.
      
      time_t period = time(NULL) - IntegerField_ApplyConversion(value);
      
      if((period % 86400) < 2 && period >= SPLACTION_LAST_DAYS_VALUE)
      {
        rule->tounits = rule->fromunits = SPLACTION_LAST_DAYS_VALUE;
        rule->fromdate = rule->todate = - (period / SPLACTION_LAST_DAYS_VALUE);
      } else { // itunes unfriendly but hopefully iPod friendly
        rule->tounits = rule->fromunits = 1;
        rule->fromdate = rule->todate = -period;
      }
      rule->fromvalue = rule->tovalue = 0x2dae2dae;
      rule->todate = 0;
      rule->tounits = 1;
      rule->length=68;
    } else{
      rule->tovalue = atoi(value);
      if(rule->field==SPLFIELD_RATING) rule->tovalue*=20;
      rule->fromvalue = rule->tovalue;
      rule->length=68;
    }
finishrule:
    out->size=out->size + rule->length + 56;
    out->rules.Add(rule);

    SKIP_WHITESPACE(c);
    if(*c == 0) break;

//    c++;
    //determine next operator (should be && or ||)
    f=&value[0];
    while(*c!=' ' && *c!=0) *(f++) = *(c++);
    *f=0;
    if(_stricmp(value,"and")==0 || _stricmp(value,"&")==0 || _stricmp(value,"&&")==0)
    { // and
      if(out->ruleoperator == SPLMATCH_OR) return NULL;
      out->ruleoperator = SPLMATCH_AND;
    }
    else if(_stricmp(value,"or")==0 || _stricmp(value,"|")==0 || _stricmp(value,"||")==0)
    { // or
      if(out->ruleoperator == SPLMATCH_AND) return NULL;
      out->ruleoperator = SPLMATCH_OR;
    }
    value[0]=0;
  }
  
  if(out->ruleoperator==69) out->ruleoperator=SPLMATCH_AND;
  s->live=true;
  return out;
  /*
  out->ruleoperator=SPLMATCH_AND;
  QueryRule * qr = new QueryRule;
  ZeroMemory(qr,sizeof(QueryRule));
  out->rules.Add(qr);
  out->size+=56;
  qr->action=SPLACTION_IS_GREATER_THAN;
  qr->field=SPLFIELD_RATING;
  qr->fromvalue=60;
  qr->length=68;
  out->size+=68;
  qr->string=NULL;

  qr = new QueryRule;
  ZeroMemory(qr,sizeof(QueryRule));
  out->rules.Add(qr);
  out->size+=56;
  qr->action=SPLACTION_CONTAINS;
  char * ar="Beastie";
  qr->field=SPLFIELD_ARTIST;
  qr->length=strlen(ar)*2;
  out->size+=qr->length;
  qr->string = g_ansi_to_utf16(ar, -1, NULL, NULL, NULL);
  return out;
  */
}

static void write_SP_mhods(FILE * file, TokenisedQuery * tq)
{
  if(!tq) return;
  put_data_cur (file, "mhod", 4);      // header                          /
  put_4int_cur (file, 24);             // size of header                  /
  put_4int_cur (file, 24+72);          // size of header + body           /
  put_4int_cur (file, 50);             // type of the entry               /
  put_n0_cur (file, 2);                // dummy space                     /

  char data[8];
  data[0]=1; //liveupdate
  data[1]=1; //checkrules
  data[2]=tq->checklimits; //checklimits
  data[3]=tq->limitype; //limittype
  data[4]=tq->limitsort; //limitsort
  data[5]=0; 
  data[6]=0;
  data[7]=0;
  put_data_cur(file,&data[0],8);
  put_4int_cur(file,tq->limitvalue); // limitvalue); 
  data[0]=0; //match checked only
  data[1]=tq->limitsort_opposite; //limitsort_opposite (1 or 0)
  data[2]=0;
  data[3]=0;
  put_data_cur(file,&data[0],4);
  put_n0_cur(file, 14);

  put_data_cur (file, "mhod", 4);      // header                          /
  put_4int_cur (file, 24);             // size of header                  /
  put_4int_cur (file, 24 + tq->size);  // size of header + body           /
  put_4int_cur (file, 51);             // type of the entry               /
  put_n0_cur (file, 2);                // dummy space                     /

  put_data_cur(file,"SLst",4);
  put_n0_cur(file, 1);
  put_4int_currev (file, tq->rules.GetSize()); //rulecount
  put_4int_currev (file, tq->ruleoperator); //rules operator (and==0 or==1)
  put_n0_cur(file, 30);
  int ruleCount=tq->rules.GetSize();
  for(int i=0; i<ruleCount; i++)
  {
    QueryRule * rule = (QueryRule *)tq->rules.Get(i);
    put_4int_currev(file,rule->field); //field
    put_4int_currev(file,rule->action); // action (use a SPLACTION_*)
    put_n0_cur(file, 11);
    put_4int_currev(file,rule->length); //length
    if(!!rule->string) //has string
    { //write out a utf-16 string with 2-byte pairs in reverse order
      unsigned char *c = (unsigned char*)rule->string;
      for(unsigned int j=0; j<rule->length; j+=2)
      {
        data[0] = *(c + j + 1);
        data[1] = *(c + j);
        put_data_cur(file,&data[0],2);
      }
      free(rule->string);
    }
    else
    {
      if(rule->fromvalue == 0x2dae2dae) put_4int_currev(file,rule->fromvalue);
      else put_n0_cur (file, 1);
      put_4int_currev(file,rule->fromvalue); //fromvalue

      if(rule->fromvalue == 0x2dae2dae) put_4int_currev(file,0xffffffff);
      else put_n0_cur (file, 1);
      put_4int_currev(file,(int)rule->fromdate);  //fromdate

      put_n0_cur (file, 1);
      put_4int_currev(file,rule->fromunits); //fromunits

      if(rule->tovalue == 0x2dae2dae) put_4int_currev(file,rule->tovalue);
      else put_n0_cur (file, 1);
      put_4int_currev(file,rule->tovalue);   //tovalue

      /*if(rule->tovalue == 0x2dae2dae) put_4int_currev(file,0xffffffff);
      else*/ put_n0_cur (file, 1);
      put_4int_currev(file,(int)rule->todate);    //todate

      put_n0_cur (file, 1);
      put_4int_currev(file,rule->tounits);   //tounits

      put_n0_cur (file, 5); //unks
    }
    delete rule;
  }
  delete tq;
}

static void write_playlist(FILE *file, Playlist *pl, void * ipod)
{
  Song *s;
  Playlistitem *ip;
  unsigned int i, n;
  long mhyp_seek;
  gunichar2 empty = 0;
  mhyp_seek = ftell(file);
#ifdef ML_IPOD
  n = ((view_ipod*)ipod)->it_get_nr_of_songs_in_playlist (pl);
#else
  n = it_get_nr_of_songs_in_playlist (pl);
#endif
#if ITUNESDB_DEBUG
  sprintf(_debug, "Playlist: %s (%d tracks)\n", pl->name, n);
  OutputDebugString(_debug);
#endif    
  mk_mhyp(file, pl->name_utf16, pl->type, n,pl,mhyp_seek,ipod);
  
  for (i=0; i<n; ++i)
  {
#ifdef ML_IPOD
		if((s = ((view_ipod*)ipod)->it_get_song_in_playlist_by_nr (pl, i))) {
			ip = ((view_ipod*)ipod)->it_get_plitem_in_playlist_by_nr (pl ,i);
#else
		if((s = it_get_song_in_playlist_by_nr (pl, i))) {
#endif
			mk_mhip(file, ip);
			mk_mhod(file, MHOD_ID_PLAYLIST, &empty, ip->trackid); 
    }
  }
  fix_mhyp (file, mhyp_seek, ftell(file));
}



/* Expects the master playlist to be (it_get_playlist_by_nr (0)) */
static void
write_mhsd_two(FILE *file, void * ipod)
{
    unsigned int playlists, i;
    long mhsd_seek, mhlp_seek;
    int otg=0;
    
    mhsd_seek = ftell (file);  /* get position of mhsd header */
    mk_mhsd (file, 2);         /* write header: type 2: playlists  */
    mhlp_seek = ftell (file);
#ifdef ML_IPOD
    playlists = ((view_ipod*)ipod)->it_get_nr_of_playlists();
    for(i = 0; i < playlists; i++) if(((view_ipod*)ipod)->it_get_playlist_by_nr(i)->type==PL_TYPE_ONTHEGO) otg++;
#else
    playlists = it_get_nr_of_playlists();
    for(i = 0; i < playlists; i++) if(it_get_playlist_by_nr(i)->type==PL_TYPE_ONTHEGO) otg++;
#endif
    mk_mhlp (file, playlists-otg);
    for(i = 0; i < playlists; i++)
    { 
#ifdef ML_IPOD
      if(((view_ipod*)ipod)->it_get_playlist_by_nr(i)->type!=PL_TYPE_ONTHEGO) write_playlist(file, ((view_ipod*)ipod)->it_get_playlist_by_nr(i),ipod);
#else
      if(it_get_playlist_by_nr(i)->type!=PL_TYPE_ONTHEGO) write_playlist(file, it_get_playlist_by_nr(i),ipod);
#endif
    }
    fix_mhlp (file, mhlp_seek, playlists-otg);
    fix_mhsd (file, mhsd_seek, ftell (file));
}


/* Do the actual writing to the iTunesDB */
BOOL 
write_it (FILE *file,void * ipod)
{
    long mhbd_seek;

    mhbd_seek = 0;             
    mk_mhbd (file);            
    write_mhsd_one(file,ipod);    /* write songs mhsd */
    write_mhsd_two(file,ipod);    /* write playlists mhsd */
		write_mhsd_two(file,ipod);    /* write playlists mhsd */
    fix_mhbd (file, mhbd_seek, ftell (file));
    return TRUE;
}

static void write_3(FILE * f, int x)
{
  char * y = (char*)&x;
  fwrite((y+2),1,1,f);
  fwrite((y+1),1,1,f);
  fwrite(y,1,1,f);
}
bool deb=false;
BOOL write_ShuffleDb(char *file,void * ipod)
{
  FILE * f = fopen(file,"wb");
  // write header....
#ifdef ML_IPOD
  write_3(f,((view_ipod*)ipod)->it_get_nr_of_songs());  // num songs
#else
  write_3(f,it_get_nr_of_songs());  // num songs
#endif
  write_3(f,0x010600);              // unk 1
  write_3(f,0x12);                  // header size
  write_3(f,0x0);                   // unk 2
  write_3(f,0x0);                   // unk 3
  write_3(f,0x0);                   // unk 4
#ifdef ML_IPOD
  int l = ((view_ipod*)ipod)->it_get_nr_of_songs();
	// get master playlist
  Playlist * p = ((view_ipod*)ipod)->it_get_playlist_by_nr(0);
#else
  int l = it_get_nr_of_songs();
  Playlist * p = it_get_playlist_by_nr(0);
#endif
  
  for(int i=0; i<l; i++)
  {
#ifdef ML_IPOD
    Song * s = ((view_ipod*)ipod)->it_get_song_in_playlist_by_nr (p,i);
#else
    Song * s = it_get_song_in_playlist_by_nr (p,i);
#endif
    if(!s) continue;
    // write record....
    write_3(f,0x22e);                // entry size
    write_3(f,0x5aa501);             // unk 1
    write_3(f,0x0);                  // start time
    write_3(f,0x0);                  // unk 2
    write_3(f,0x0);                  // unk 3
    write_3(f,0x0);                  // stop time
    write_3(f,0x0);                  // unk 4
    write_3(f,0x0);                  // unk 5
    write_3(f,0x64);                 // volume
    // file type (0x01 = MP3, 0x02 = AAC, 0x04 = WAV)
    if(_stricmp(s->ipod_path+strlen(s->ipod_path)-4,".wav")==0) write_3(f,0x4);
    else if(_stricmp(s->ipod_path+strlen(s->ipod_path)-4,".mp3")==0) write_3(f,0x1);
    else if(_stricmp(s->ipod_path+strlen(s->ipod_path)-4,".mp2")==0) write_3(f,0x1);
    else write_3(f,0x2);
    write_3(f,0x200);                // unk 6
    char * filename = (char *)calloc(522,1);
    int cl = strlen(s->ipod_path)*2;
    if(cl > 520) goto error;
    memcpy(filename,s->ipod_path_utf16,cl);
    // if(i==0) MessageBox(NULL,s->ipod_path,"dsf",0);
    for(int j=0; j<522; j++) if(*(filename+j) == ':' && *(filename+j+1) == 0) *(filename+j) = '/';
    put_data_cur(f,filename,522);    // filename
    free(filename);
    write_3(f,0x10000);              // unk 7
  }
  
  fclose(f);

  return TRUE;
error:
  if(f) fclose(f);
  return FALSE;
}


HANDLE writeHandle=NULL;

void itunesdb_init_cc() {
  lpCriticalSection = new CRITICAL_SECTION;
  InitializeCriticalSection(lpCriticalSection);
}

void itunesdb_del_cc() {
  EnterCriticalSection(lpCriticalSection);
  if(writeHandle) WaitForSingleObject(writeHandle,INFINITE);
  LeaveCriticalSection(lpCriticalSection);
  DeleteCriticalSection(lpCriticalSection);
}

/* Write out an iTunesDB.
   Note: only the _utf16 entries in the Song-struct are used
   Returns TRUE on success, FALSE on error.
   "path" should point to the mount point of the
   iPod, e.e. "/mnt/ipod" 
   This has no concurrency controls. */
BOOL itunesdb_write_nocc (char *path,void * ipod)
{
  
  if(((view_ipod*)ipod)->delayDBWrite) return true;
  char *filename = NULL;
  BOOL result = FALSE;
  filename = concat_dir(path,"iPod_Control/iTunes/mlipodSmartPlaylists");
  write_SP_Info(filename,ipod);
  if (filename != NULL) free (filename);  
  filename = concat_dir(path,"iPod_Control/iTunes/iTunesSD");
  write_ShuffleDb(filename,ipod);
  if (filename != NULL) free (filename);  
  filename = concat_dir (path, "iPod_Control/iTunes/iTunesDB.new");
  _unlink(filename);
  result = itunesdb_write_to_file (filename,ipod);
  if(result) {
    if (filename != NULL) free (filename);
    filename = concat_dir (path, "iPod_Control/iTunes/Play Counts");
    _unlink(filename);
    if (filename != NULL) free (filename);
    filename = concat_dir (path, "iPod_Control/iTunes/iTunesStats");
    _unlink(filename);
    if (filename != NULL) free (filename);
    char * filenamenew = concat_dir (path, "iPod_Control/iTunes/iTunesDB.new");
    char * filenameold = concat_dir (path, "iPod_Control/iTunes/iTunesDB.old");
    filename = concat_dir (path, "iPod_Control/iTunes/iTunesDB");
    _unlink(filenameold);
    rename(filename, filenameold);
    rename(filenamenew, filename);
    free(filename);
    free(filenameold);
    free(filenamenew);
  }
  return result;
}

/* this is like the above, but with concurrency controls. You need to
  call itunesdb_init_cc() at some point (preferably
  at the beginning of the program) if you want to use this.
*/
BOOL itunesdb_write (char *path, void * ipod, HANDLE wh)
{
  EnterCriticalSection(lpCriticalSection);
  writeHandle=wh;
  BOOL ret = itunesdb_write_nocc(path,ipod);
  writeHandle=NULL;
  LeaveCriticalSection(lpCriticalSection);
  return ret;
}

typedef struct WriteDBParams {
  void * ipod;
  char * path;
} WriteDBParams;

DWORD WINAPI ThreadFunc_WriteDB(LPVOID lpParam) {
  WriteDBParams * p = (WriteDBParams*)lpParam;
  itunesdb_write (p->path,p->ipod);
  delete p;
  return NULL;
}

/* itunesdb_write (), but in a new thread. */
void itunesdb_writeA (char *path,void * ipod) {
  DWORD dwThreadId;
  WriteDBParams * p = new WriteDBParams;
  p->ipod=ipod;
  p->path=path;
  HANDLE writeHandle = CreateThread(NULL, 0, ThreadFunc_WriteDB, (LPVOID)p, 0, &dwThreadId);
}

/* Same as itnuesdb_write (), but you specify the filename directly */
BOOL itunesdb_write_to_file (char *filename,void * ipod)
{
  FILE *file = NULL;
  BOOL result = FALSE;

#if ITUNESDB_DEBUG
  sprintf(_debug, "Writing to %s\n", filename);
  OutputDebugString(_debug);
#endif

  if((file = fopen (filename, "w+b")))
    {
      write_it (file,ipod);
      fclose(file);
      result = TRUE;
    }
  else
    {
      //gtkpod_warning (_("Could not open iTunesDB \"%s\" for writing.\n"),
          //filename);
    }
  return result;
}

static void type_to_string(unsigned int t, char *out)
{
  out[0]=(t)&0xff;
  out[1]=(t>>8)&0xff;
  out[2]=(t>>16)&0xff;
  out[3]=(t>>24)&0xff;
  out[4]=0;
  int x=3;
  while (out[x]==' ' && x > 0) out[x--]=0;
}

extern winampMediaLibraryPlugin plugin;


typedef struct songCopyInst {
  char ipod_file[2048], ipod_fullfile[2048];
  FILE *file_in, *file_out;
  int finished;
  Song *song;

  int is_transcode; // 2 == wa transcode, 1 == lame, 3 == ml_transcoder
  HANDLE hTranscodeProcess;
  convertFileStruct m_fcs;
#ifdef ML_IPOD
  view_ipod * ipod;
#else
  void * ipod;
#endif
  char * pcfile;
} songCopyInst;

/*
extern char g_transcode_cmd[1024];
extern int g_transcode2;
extern HWND g_sendfiles_hwnd;
extern int g_hideconsole;
extern char g_transcode_cmd2[1024];
extern int g_hideconsole2;
extern int g_transcode3;
extern int m_blocks_xferred;
extern int m_pooper_pos;
*/
static int dir_num = -1;
extern bool os2k;

//char * g_pcfile;
int last_m_pooper_pos=0;

BOOL TrancodeCallback(int pc,int bw,void * lpParam) {
  songCopyInst * p = (songCopyInst *)lpParam;
  last_m_pooper_pos = p->ipod->m_pooper_pos;
  p->ipod->m_pooper_pos = bw / 65536;
  p->ipod->m_blocks_xferred += (p->ipod->m_pooper_pos - last_m_pooper_pos);
  return p->finished;
}

static DWORD WINAPI ThreadFunc_Transcode(LPVOID lpParam) {
  songCopyInst * p = (songCopyInst *)lpParam;
  
  transcodeInst t={NULL,p->pcfile,p->ipod_fullfile,p->ipod->ini_file,&TrancodeCallback,p};

  UINT ipc_transcode=SendMessage(plugin.hwndWinampParent,WM_WA_IPC,(WPARAM)&"IpcTranscode",IPC_REGISTER_WINAMP_IPCMESSAGE);
  tTranscodeProc transcodeProc = (tTranscodeProc)SendMessage(plugin.hwndWinampParent,WM_WA_IPC,NULL,ipc_transcode);
  int ret = transcodeProc(&t);
  //if(ret==0)
  {
    p->finished=TRUE;
  }
  return 0;
}

typedef DWORD (WINAPI *LPPROGRESS_ROUTINE)(
    LARGE_INTEGER TotalFileSize,
    LARGE_INTEGER TotalBytesTransferred,
    LARGE_INTEGER StreamSize,
    LARGE_INTEGER StreamBytesTransferred,
    DWORD dwStreamNumber,
    DWORD dwCallbackReason,
    HANDLE hSourceFile,
    HANDLE hDestinationFile,
    LPVOID lpData OPTIONAL
    );

#define PROGRESS_CONTINUE   0
#define PROGRESS_CANCEL     1
#define PROGRESS_STOP       2
#define PROGRESS_QUIET      3

DWORD CALLBACK CopyToIpodProgressRoutine(
  LARGE_INTEGER TotalFileSize,
  LARGE_INTEGER TotalBytesTransferred,
  LARGE_INTEGER StreamSize,
  LARGE_INTEGER StreamBytesTransferred,
  DWORD dwStreamNumber,
  DWORD dwCallbackReason,
  HANDLE hSourceFile,
  HANDLE hDestinationFile,
  LPVOID lpData
)
{
  songCopyInst *p=(songCopyInst *)lpData;
  last_m_pooper_pos =p->ipod->m_pooper_pos;
  p->ipod->m_pooper_pos = (unsigned int)(TotalBytesTransferred.QuadPart / 65536);
  p->ipod->m_blocks_xferred += (p->ipod->m_pooper_pos - last_m_pooper_pos);
  if(p->finished) { /*free(p);*/ return PROGRESS_CANCEL; } // PROGRESS_CANCEL
  return PROGRESS_CONTINUE; //0; //PROGRESS_CONTINUE
}

extern void deleteSongPtr(Song *song);

typedef BOOL (WINAPI*COPYFILEEXTYPE)(LPCTSTR lpExistingFileName,
                              LPCTSTR lpNewFileName,
                              LPPROGRESS_ROUTINE lpProgressRoutine,
                              LPVOID lpData,
                              LPBOOL pbCancel,
                              DWORD dwCopyFlags);
DWORD Win98Copy(songCopyInst * p);
static DWORD WINAPI ThreadFunc_CopyFile(LPVOID lpParam) 
{
  songCopyInst *p=(songCopyInst *)lpParam;
  if(os2k) //GetPrivateProfileInt("ml_ipod", "wincopy", os2k, conf_file))
  { //windows routines
    //MessageBox(NULL,"poop","poop",0);
    HMODULE hmod = LoadLibrary("kernel32.dll");
    COPYFILEEXTYPE CopyFileEx;
    CopyFileEx=(COPYFILEEXTYPE)GetProcAddress(hmod, "CopyFileExA");
    if(!CopyFileEx) 
    { // win9x!
      //MessageBox(plugin.hwndWinampParent,"The use of the windows copy routines is not\npossible under windows 9x/ME. Please hit \"Abort\" and change to the\ninternal copy routines via the preferences.","Error",0);
      //return 0; 
      return Win98Copy(p);
    }
    else {
      BOOL ret = CopyFileEx(p->ipod->g_pcfile,p->ipod_fullfile,CopyToIpodProgressRoutine,lpParam,NULL,NULL);
      if(p->finished || !ret)
      {
        //MessageBox(NULL,"failed","sdf",0);
        DeleteFile(p->ipod_fullfile);
        //cleanupAfterFile(p->ipod_fullfile);
        if(p->song) deleteSongPtr(p->song);
        //p->song=0;
        free(p);
      }
      else p->finished=TRUE;
      
    }
    FreeLibraryAndExitThread(hmod,0x0);
    return 0;
  }
  return Win98Copy(p);
}

DWORD Win98Copy(songCopyInst * p) {
  //char buf[65536];
  char buf[65536*15];
  //char * buf = (char *)malloc(65536*10);
  unsigned int l;
  do {
    p->ipod->m_blocks_xferred+=15;
    p->ipod->m_pooper_pos+=15;
    //if(l<1) break;
    l=fread(buf,1,sizeof(buf),p->file_in);    
  } while(l>0 && fwrite(buf,1,l,p->file_out) == l && !p->finished);
  //free(buf);
  
  if(p->finished) { //cancelled!
    //MessageBox(NULL,"cancel","dsf",0);  
    _unlink(p->ipod_fullfile);
    free(p);
  } else  p->finished=TRUE;
  fclose(p->file_out);
  fclose(p->file_in);
  return 0;
}

BOOL mustTranscode(char * song, char * inifile) {
  if(!GetPrivateProfileInt("ml_ipod", "newtranscode", 1,inifile)) return FALSE;
  char * ext = strrchr(song,'.');
  if(!ext) return TRUE;
  if(_stricmp(ext,".mp3")==0) return FALSE;
  if(_stricmp(ext,".m4a")==0) return FALSE;
  if(_stricmp(ext,".m4b")==0) return FALSE;
  if(_stricmp(ext,".m4p")==0) return FALSE;
  return TRUE;
}

void *start_song_copy(char *path, Song *song, char *pcfile, void * ip)
{
  if (song == NULL) return NULL;
  if (dir_num == -1) dir_num = (int) (19.0*rand()/(RAND_MAX));

  songCopyInst *p=(songCopyInst *)calloc(1,sizeof(songCopyInst));
#ifdef ML_IPOD
  p->ipod=(view_ipod*)ip;
#else
  p->ipod=ip;
#endif
  p->song=song;
  p->pcfile=_strdup(pcfile);
  if (!GetPrivateProfileInt("ml_ipod","noipodfs",0,p->ipod->ini_file))
  {
    wsprintf(p->ipod_fullfile,"%c:\\iPod_Control\\Music\\F%02d",*path,dir_num);
    wsprintf(p->ipod_file,"/iPod_Control/Music/F%02d/ml_ipod%05d",dir_num, song->ipod_id);
  }
  else
  {
    char buf[4096];
    GetPrivateProfileString("ml_ipod","noipodfs_fmt","<Artist> - <Album>\\## - <Title>",buf,sizeof(buf),p->ipod->ini_file);

    Song nsong = *song;
    nsong.album = _strdup(nsong.album);
    nsong.artist = _strdup(nsong.artist);
    nsong.title = _strdup(nsong.title);
    if(!!STRCMP_NULLOK(nsong.title, ""))
    {
      removebadchars(nsong.title);
    }else{
      free(nsong.title);
      nsong.title = _strdup("Unknown Title");
    }

    if(!!STRCMP_NULLOK(nsong.artist, ""))
    {      
      removebadchars(nsong.artist);
    }else{
      free(nsong.artist);
      nsong.artist = _strdup("Unknown Artist");
    }

    if(!!STRCMP_NULLOK(nsong.album, ""))
    {
      removebadchars(nsong.album);
    }else{
      free(nsong.album);
      nsong.album = _strdup("Unknown Album");
    }

    if(!!STRCMP_NULLOK(nsong.genre, ""))
    {
      removebadchars(nsong.genre);
    }

    FixReplacementVars(buf,sizeof(buf),&nsong,true);

    free(nsong.artist);      
    free(nsong.album);
    free(nsong.title);


    wsprintf(p->ipod_fullfile,"%c:\\%s",*path,buf);


    // remove last dir portion
    char *ptr=p->ipod_fullfile+3;
    while (*ptr) ptr++;
    while (ptr >= p->ipod_fullfile+3 && *ptr != '/' && *ptr != '\\') ptr--;
//    if (ptr >= p->ipod_fullfile+3) 
    *ptr=0;

    ptr=buf;
    while (*ptr) { if (*ptr == '\\') *ptr='/'; ptr++; }
    wsprintf(p->ipod_file,"/%s",buf);
  }

  if (strlen(p->ipod_fullfile) > 3)  // if it has a dir portion
  {
    char * foo = _strdup(p->ipod_fullfile);
    //*strrchr(foo,'\\') = 0;
    RecursiveCreateDirectory(foo);
    free(foo);
  }

  int c=path[strlen(path)-1];
  wsprintf(p->ipod_fullfile,"%s%s%s",path,c=='/' || c=='\\'?"":"/",p->ipod_file+1);
  if(mustTranscode(p->pcfile,p->ipod->ini_file)) {
    char ext[5];
    ext[0]='.';
    GetExt(&ext[1],p->ipod->ini_file);
    strcat(p->ipod_file,ext);
    
    strcat(p->ipod_fullfile,ext);
    p->is_transcode=3;
    DWORD dwThreadId;
    HANDLE hThread = CreateThread(NULL, 0, ThreadFunc_Transcode, (LPVOID)p, 0, &dwThreadId);
    //if(hThread) CloseHandle(hThread);
    p->hTranscodeProcess = hThread;
  }
  else if(p->ipod->g_transcode_cmd2[0] && strlen(pcfile) > 4 && !_stricmp(pcfile+strlen(pcfile)-4,".m4p"))
  {
    strcat(p->ipod_file,".m4a");
    strcat(p->ipod_fullfile,".m4a");
    //wsprintf(p->ipod_file,"/iPod_Control/Music/F%02d/%s",dir_num, strrchr(pcfile,'\\')+1);
    strcpy(p->ipod_file+strlen(p->ipod_file)-1,"a");
    int c=path[strlen(path)-1];
    wsprintf(p->ipod_fullfile,"%s%s%s",path,c=='/' || c=='\\'?"":"/",p->ipod_file+1);
    
    char runcmd[8192];
    char hymn_bat[MAX_PATH]="";
    GetModuleFileName(plugin.hDllInstance,hymn_bat,sizeof(hymn_bat)-1);
    strcpy(strrchr(hymn_bat,'\\'),"\\ml_ipod_hymn.bat");
    strcpy(pcfile+strlen(pcfile)-1,"");
    wsprintf(runcmd,"\"%s\" \"%s\" \"%sp\" \"%sa\" \"%s\"",hymn_bat,p->ipod->g_transcode_cmd2,pcfile,pcfile,p->ipod_fullfile);
    strcat(pcfile,"p");
    char *tmpp=runcmd;
    while (*tmpp)
    {
      if (*tmpp == '/') *tmpp='\\';
      tmpp++;
    }
    STARTUPINFO si={sizeof(si),0};
    PROCESS_INFORMATION pi={0};
    
    if (!CreateProcess(NULL,runcmd,NULL,NULL,FALSE,p->ipod->g_hideconsole2?CREATE_NO_WINDOW:0,NULL,NULL,&si,&pi) || !pi.hProcess)
    {
      free(p);
      return FALSE;
    }
    
    p->is_transcode=1;
    p->hTranscodeProcess = pi.hProcess;
    CloseHandle(pi.hThread);
  }
  else
  {
    strcat(p->ipod_file,strrchr (pcfile,'.'));
    strcat(p->ipod_fullfile,strrchr (pcfile,'.'));

    if(!os2k) //if(!GetPrivateProfileInt("ml_ipod", "wincopy", os2k, conf_file))
    { // internal copy routines
      p->file_in = fopen(pcfile,"rb");
      if (!p->file_in)
      {
        free(p);
        return FALSE;
      }
      p->file_out = fopen(p->ipod_fullfile,"wb");
      if (!p->file_out)
      {
        fclose(p->file_in);
        free(p);
        return FALSE;
      }
    }
    else
    { //windows copy routines
      p->ipod->g_pcfile = _strdup(pcfile);
    }
    DWORD dwThreadId;
    HANDLE hThread = CreateThread(NULL, 0, ThreadFunc_CopyFile, (LPVOID)p, 0, &dwThreadId);
    if(hThread) CloseHandle(hThread);
    
  }

  p->finished=0;

  return (songCopyInst*)p;
}

Song *close_song_copy(void *inst)
{
  bool freep=true;
  songCopyInst *p=(songCopyInst*)inst;
  if (p)
  {
    if (p->is_transcode)
    {
      if (p->is_transcode==2)
      {
        SendMessage(plugin.hwndWinampParent,WM_WA_IPC,(WPARAM)&p->m_fcs,IPC_CONVERTFILE_END);
      }
      else if(p->is_transcode==3)
      {
        //must fix filesize and bitrate!
        p->song->size = fileSize(p->song->pcfile);
        p->song->bitrate = p->song->size / (1 + p->song->songlen);
        goto foo;
      }
      else
      {
        WaitForSingleObject(p->hTranscodeProcess,INFINITE);
        CloseHandle(p->hTranscodeProcess);
      }
    }
    else
    /*{
      if (p->file_in) fclose(p->file_in);
      if (p->file_out) fclose(p->file_out);
    }*/

    if (p->finished)
    {
foo:
      int i, len;
      len = strlen(p->ipod_file);
      for (i=0; i<len; ++i)     /* replace '/' by ':' */
        if (p->ipod_file[i] == '/')  p->ipod_file[i] = ':';
#ifdef ITUNESDB_PROVIDE_ANSI
      p->song->ipod_path = _strdup (p->ipod_file);
#endif ITUNESDB_PROVIDE_ANSI
      p->song->ipod_path_utf16 = g_ansi_to_utf16 (p->ipod_file, -1, NULL, NULL, NULL);
      ++dir_num;
      if (dir_num == 20) dir_num = 0;
    }
    else 
    {
      //if(os2k) {
        freep=false;
        p->finished=TRUE;
      //} else {
        //unlink(p->ipod_fullfile); // remove our partial upload
        //p->ipod->cleanupAfterFile(p->ipod_fullfile);
        //if(p->song) deleteSongPtr(p->song);
        //p->song=0;
      //}
    }
    Song *s=p->song;
    if(freep) {
      p->file_in=p->file_out=0;
      free(p);
    }
    return s;
  }
  return NULL;
}

void copy_on_complete(void *inst)
{
  songCopyInst *p=(songCopyInst*)inst;
  p->finished=1;
}

int run_song_copy(void *inst) // returns -1 on error > 0, if needs more calls or 0 if done
{
  songCopyInst *p=(songCopyInst*)inst;
  if(p) { if (p->finished) return 0; }

  if (p)
  {
    if (p->is_transcode==2)
    {
      Sleep(100);
      return 1;
    }
    else if(p->is_transcode==1)
    {
      if (WaitForSingleObject(p->hTranscodeProcess,10)==WAIT_OBJECT_0) 
      {
        DWORD d;
        if (GetExitCodeProcess(p->hTranscodeProcess,&d) && d) // if return value != 0
          return -1;
        p->finished=1;
        return 0;
      }
      return 1;
    }
    else
    {
      if(p->finished) return 0;
      else return 1;
      /*char buf[65536];
      unsigned int l=fread(buf,1,sizeof(buf),p->file_in);
      if (fwrite(buf,1,l,p->file_out) == l) return !(p->finished = (l != sizeof(buf)));
      */
    }
  }
  return -1;
}

Playlist *add_new_playlist(char *plname, void * ipod) {
  Playlist *plitem;
  plitem = new Playlist;
  plitem->type = PL_TYPE_NORM;
  plitem->name = plname;
  plitem->name_utf16 = g_ansi_to_utf16 (plname, -1, NULL, NULL, NULL);
  plitem->sp=NULL;
#ifdef ML_IPOD
  return ((view_ipod*)ipod)->it_add_playlist(plitem);
#else
  return it_add_playlist(plitem);
#endif
}

#if 0


/* Does this really belong here? -- Maybe, because it
   requires knowledge of the iPod's filestructure */
/* Copy one song to the ipod. The PC-Filename is
   "pcfile" and is taken literally.
   "path" is assumed to be the mountpoint of the iPod.
   For storage, the directories "f00 ... f19" will be
   cycled through. The filename is constructed from
   "song->ipod_id": "gtkpod_id" and written to
   "song->ipod_path_ansi" and "song->ipod_path_utf16" */
BOOL copy_song_to_ipod (char *path, Song *song, char *pcfile)
{
  static int dir_num = -1;
  char *ipod_file, *ipod_fullfile;
  BOOL success;

  if (dir_num == -1) dir_num = (int) (19.0*rand()/(RAND_MAX));
  if (song == NULL)
    {
      //g_warning ("Programming error: copy_song_to_ipod () called NULL-song\n");
      return FALSE;
    }

  /* The iPod seems to need the .mp3 ending to play the song.
     Of course the following line should be changed once gtkpod
     also supports other formats. CHANGED! now carries extentions forwards regardless */
//  ipod_file = g_strdup_printf ("/iPod_Control/Music/F%02d/gtkpod%05d.mp3",
//             dir_num, song->ipod_id);
  {
    char temp[2048];
    wsprintf(p->ipod_file,"/iPod_Control/Music/F%02d/ml_ipod%05d%03s",dir_num, song->ipod_id, strrchr (pcfile,'.'));
    ipod_file=_strdup(temp);
  }
#if ITUNESDB_DEBUG
  sprintf(_debug, "ipod_file: %s\n", ipod_file);
  OutputDebugString(_debug);
#endif

  ipod_fullfile = concat_dir (path, ipod_file+1);
  success = cp (pcfile, ipod_fullfile);
  if (success)
    { /* need to store ipod_filename */
      int i, len;
      len = strlen (ipod_file);
      for (i=0; i<len; ++i)     /* replace '/' by ':' */
  if (ipod_file[i] == '/')  ipod_file[i] = ':';
#ifdef ITUNESDB_PROVIDE_ANSI
      song->ipod_path = _strdup (ipod_file);
#endif ITUNESDB_PROVIDE_ANSI
      song->ipod_path_utf16 = g_ansi_to_utf16 (ipod_file,
                 -1, NULL, NULL, NULL);
      ++dir_num;
      if (dir_num == 20) dir_num = 0;
    }
  free (ipod_file);
  free (ipod_fullfile);
  return success;
}


/* Copy file "from_file" to "to_file".
   Returns TRUE on success, FALSE otherwise */
BOOL cp (char *from_file, char *to_file)
{
  char data[ITUNESDB_COPYBLK];
  long bread, bwrite;
  BOOL success = TRUE;
  FILE *file_in = NULL;
  FILE *file_out = NULL;

  do { /* dummy loop for easier error handling */
    file_in = fopen (from_file, "rb");
    if (file_in == NULL)
      {
  //gtkpod_warning (_("Could not open file \"%s\" for reading.\n"), from_file);
  success = FALSE;
  break;
      }
    file_out = fopen (to_file, "wb");
    if (file_out == NULL)
      {
  //gtkpod_warning (_("Could not open file \"%s\" for writing.\n"), to_file);
  success = FALSE;
  break;
      }
    do {
      bread = fread (data, 1, ITUNESDB_COPYBLK, file_in);
      if (bread == 0)
  {
    if (feof (file_in) == 0)
      { /* error -- not end of file! */
        //gtkpod_warning (_("Error reading file \"%s\"."), from_file);
        success = FALSE;
      }
  }
      else
  {
    bwrite = fwrite (data, 1, bread, file_out);
    if (bwrite != bread)
      {
        //gtkpod_warning (_("Error writing PC file \"%s\"."),to_file);
        success = FALSE;
      }
  } 
    } while (success && (bread != 0));
  } while (FALSE);
  if (file_in)  fclose (file_in);
  if (file_out)
    {
      fclose (file_out);
      if (!success) { /* error occured -> delete to_file */
  remove (to_file);
      }
    }
  return success;
} 

#endif




typedef struct tokenstruct {
  char *token;
  int tid;
} tokenstruct;

enum token {
  TOKEN_AGO = 128,
  TOKEN_NOW,
  TOKEN_YESTERDAY,
  TOKEN_TOMORROW,
  TOKEN_TODAY,
  TOKEN_OF,
  TOKEN_THE,
  TOKEN_DATE,
  TOKEN_FROM,
  TOKEN_BEFORE,
  TOKEN_AFTER,
  TOKEN_THIS,
  TOKEN_SUNDAY,
  TOKEN_MONDAY,
  TOKEN_TUESDAY,
  TOKEN_WEDNESDAY,
  TOKEN_THURSDAY,
  TOKEN_FRIDAY,
  TOKEN_SATURDAY,

  TOKEN_MIDNIGHT,
  TOKEN_NOON,

  TOKEN_AM,
  TOKEN_PM,

  TOKEN_JANUARY,
  TOKEN_FEBRUARY,
  TOKEN_MARCH,
  TOKEN_APRIL,
  TOKEN_MAY,
  TOKEN_JUNE,
  TOKEN_JULY,
  TOKEN_AUGUST,
  TOKEN_SEPTEMBER,
  TOKEN_OCTOBER,
  TOKEN_NOVEMBER,
  TOKEN_DECEMBER,

  TOKEN_TIME,
  TOKEN_SECOND,
  TOKEN_MINUTE,
  TOKEN_HOUR,
  TOKEN_DAY,
  TOKEN_WEEK,
  TOKEN_MONTH,
  TOKEN_YEAR,
  TOKEN_AT,
  TOKEN_IDENTIFIER,
  TOKEN_EOQ,
};

//tokens
tokenstruct Int_Tokens[] = { // Feel free to add more...
  {"ago", TOKEN_AGO},
  {"now", TOKEN_NOW},
  {"am", TOKEN_AM},
  {"pm", TOKEN_PM},
  {"this", TOKEN_THIS},
  {"date", TOKEN_DATE},
  {"time", TOKEN_TIME},
  {"of", TOKEN_OF},
  {"at", TOKEN_AT},
  {"the", TOKEN_THE},
  {"yesterday", TOKEN_YESTERDAY},
  {"tomorrow", TOKEN_TOMORROW},
  {"today", TOKEN_TODAY},
  {"from", TOKEN_FROM},
  {"before", TOKEN_BEFORE},
  {"after", TOKEN_AFTER},
  {"past", TOKEN_AFTER},
  {"monday", TOKEN_MONDAY},
  {"mon", TOKEN_MONDAY},
  {"tuesday", TOKEN_TUESDAY},
  {"tue", TOKEN_TUESDAY},
  {"wednesday", TOKEN_WEDNESDAY},
  {"wed", TOKEN_WEDNESDAY},
  {"thursday", TOKEN_THURSDAY},
  {"thu", TOKEN_THURSDAY},
  {"friday", TOKEN_FRIDAY},
  {"fri", TOKEN_FRIDAY},
  {"saturday", TOKEN_SATURDAY},
  {"sat", TOKEN_SATURDAY},
  {"sunday", TOKEN_SUNDAY},
  {"sun", TOKEN_SUNDAY},
  {"midnight", TOKEN_MIDNIGHT},
  {"noon", TOKEN_NOON},
  {"second", TOKEN_SECOND},
  {"seconds", TOKEN_SECOND},
  {"sec", TOKEN_SECOND},
  {"s", TOKEN_SECOND},
  {"minute", TOKEN_MINUTE},
  {"minutes", TOKEN_MINUTE},
  {"min", TOKEN_MINUTE},
  {"mn", TOKEN_MINUTE},
  {"m", TOKEN_MINUTE},
  {"hour", TOKEN_HOUR},
  {"hours", TOKEN_HOUR},
  {"h", TOKEN_HOUR},
  {"day", TOKEN_DAY},
  {"days", TOKEN_DAY},
  {"d", TOKEN_DAY},
  {"week", TOKEN_WEEK},
  {"weeks", TOKEN_WEEK},
  {"w", TOKEN_WEEK},
  {"month", TOKEN_MONTH},
  {"months", TOKEN_MONTH},
  {"year", TOKEN_YEAR},
  {"years", TOKEN_YEAR},
  {"y", TOKEN_YEAR},
  {"january", TOKEN_JANUARY},
  {"jan", TOKEN_JANUARY},
  {"february", TOKEN_FEBRUARY},
  {"feb", TOKEN_FEBRUARY},
  {"march", TOKEN_MARCH},
  {"mar", TOKEN_MARCH},
  {"april", TOKEN_APRIL},
  {"apr", TOKEN_APRIL},
  {"may", TOKEN_MAY},
  {"june", TOKEN_JUNE},
  {"jun", TOKEN_JUNE},
  {"july", TOKEN_JULY},
  {"jul", TOKEN_JULY},
  {"august", TOKEN_AUGUST},
  {"aug", TOKEN_AUGUST},
  {"september", TOKEN_SEPTEMBER},
  {"sep", TOKEN_SEPTEMBER},
  {"october", TOKEN_OCTOBER},
  {"oct", TOKEN_OCTOBER},
  {"november", TOKEN_NOVEMBER},
  {"nov", TOKEN_NOVEMBER},
  {"december", TOKEN_DECEMBER},
  {"dec", TOKEN_DECEMBER},
};
//---------------------------------------------------------------------------
int IntegerField_LookupToken(char *t) {
  for (int i=0;i<sizeof(Int_Tokens)/sizeof(tokenstruct);i++) {
    if (!_stricmp(Int_Tokens[i].token, t))
      return Int_Tokens[i].tid;
  }
  return TOKEN_IDENTIFIER;
}

int myatoi(char *p, int len) {
  char *w = (char *)malloc(len+1);
  strncpy(w, p, len);
  w[len] = 0;
  int a = atoi(w);
  free(w);
  return a;
}

int isallnum(char *p) {
  while (p && *p) {
    if (*p < '0' || *p > '9') return 0;
    p++;
  }
  return 1;
}

int Scanner_Query_GetNextToken(char *p, int *size, char ** token) {
  char * c=p;
  char tok[256];
  char * f = &tok[0];
  tok[0]=0;
  while (*c == ' ') c++;
  if(*c == 0) return TOKEN_EOQ;
  while(*c != 0 && *c != ' ') *(f++) = *(c++);
  *f=0;
  *size = strlen(tok) + 1;
  if(*size == 1) return TOKEN_EOQ;
  *token = _strdup(tok);
  return IntegerField_LookupToken(tok);
}
//---------------------------------------------------------------------------
time_t IntegerField_ApplyConversion(const char *format) {
  //return 0;
  int size;

  time_t value = 0;
  char *token = NULL;
  int ago = 0;
  int from = 0;
  int kthis = 0;
  int what = TOKEN_MINUTE;

  int getwhat = 1;
  time_t lastnumber = value;

  time_t now;
  time(&now);

  struct tm *o = localtime(&now);
  struct tm origin = *o;
  struct tm origin_flags = {0,0,0,0,0,0,0,0,0};

  struct tm onow = *o;

  char *p = (char *)format;
  int t = -1;
  int lastt = -1;

  origin.tm_isdst = -1;

  while (1) {
    int save_lastt = lastt;
    lastt = t;

    t = Scanner_Query_GetNextToken(p, &size, &token);

    if (t == TOKEN_EOQ) break;

    switch (t) {
      case TOKEN_THIS:
        kthis = 1;
        break;
      case TOKEN_AGO:
      case TOKEN_BEFORE: // before defaults to before now (= ago)
        ago = 1;
        break;
      case TOKEN_AFTER: // if after, ago is discarded, coz 5 mn ago after x has no meaning, so we get it as 5 mn after x
        ago = 0;
        // no break
      case TOKEN_FROM:
        from = 1;
        break;
      case TOKEN_DATE: {
        if (!kthis) break;
        kthis = 0;
        origin.tm_year = onow.tm_year;
        origin_flags.tm_year = 1;
        origin.tm_mon = onow.tm_mon;
        origin_flags.tm_mon = 1;
        origin.tm_mday = onow.tm_mday - onow.tm_wday;
        origin_flags.tm_mday = 1;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
                       } 
      case TOKEN_TIME: {
        if (!kthis) break;
        kthis = 0;
        origin.tm_hour = onow.tm_hour;
        origin_flags.tm_hour = 1;
        origin.tm_min = onow.tm_min;
        origin_flags.tm_min = 1;
        origin.tm_sec = onow.tm_sec;
        origin_flags.tm_sec = 1;
        break;
                       } 
      case TOKEN_SECOND: 
      case TOKEN_MINUTE: 
      case TOKEN_HOUR: 
      case TOKEN_DAY: 
      case TOKEN_WEEK: 
      case TOKEN_MONTH: 
      case TOKEN_YEAR: 
        if (kthis) {
          kthis = 0;
          switch (t) {
      case TOKEN_SECOND: 
        origin.tm_sec = onow.tm_sec;
        origin_flags.tm_sec = 1;
        break;
      case TOKEN_MINUTE: 
        origin.tm_min = onow.tm_min;
        origin_flags.tm_min = 1;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
      case TOKEN_HOUR: 
        origin.tm_hour = onow.tm_hour;
        origin_flags.tm_hour = 1;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
      case TOKEN_DAY: 
        origin.tm_mday = onow.tm_mday;
        origin_flags.tm_mday = 1;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
      case TOKEN_WEEK: 
        origin.tm_mday = onow.tm_mday - onow.tm_wday;
        origin_flags.tm_mday = 1;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
      case TOKEN_MONTH: 
        origin.tm_mon = onow.tm_mon;
        origin_flags.tm_mon = 1;
        if (!origin_flags.tm_mday)
          origin.tm_mday = 1;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
      case TOKEN_YEAR: 
        origin.tm_year = onow.tm_year;
        origin_flags.tm_year = 1;
        if (!origin_flags.tm_mon)
          origin.tm_mon = 0;
        if (!origin_flags.tm_mday)
          origin.tm_mday = 1;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
          }
          break;
        } 
        if (lastnumber > 0) {
          value = lastnumber;
          lastnumber = 0;
        }
        what = t;
        break;
      case TOKEN_SUNDAY:
      case TOKEN_MONDAY:
      case TOKEN_TUESDAY:
      case TOKEN_WEDNESDAY:
      case TOKEN_THURSDAY:
      case TOKEN_FRIDAY:
      case TOKEN_SATURDAY: {
        kthis = 0;
        int dow = t-TOKEN_MONDAY;
        if (dow > onow.tm_mday)
          origin.tm_mday = 7 - (dow - onow.tm_mday);
        else
          origin.tm_mday = dow;
        origin_flags.tm_mday = 1;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
                           }
                           break;
      case TOKEN_MIDNIGHT:
        kthis = 0;
        origin.tm_hour = 0;
        origin_flags.tm_hour = 1;
        if (!origin_flags.tm_min) {
          origin.tm_min = 0;
          origin_flags.tm_min = 1;
        }
        if (!origin_flags.tm_sec) {
          origin.tm_sec = 0;
          origin_flags.tm_sec = 1;
        }
        break;
      case TOKEN_NOON:
        kthis = 0;
        origin.tm_hour = 12;
        origin_flags.tm_hour = 1;
        if (!origin_flags.tm_min) {
          origin.tm_min = 0;
          origin_flags.tm_min = 1;
        }
        if (!origin_flags.tm_sec) {
          origin.tm_sec = 0;
          origin_flags.tm_sec = 1;
        }
        break;
      case TOKEN_AM:
        kthis = 0;
        if (lastnumber > 0) {
          origin.tm_hour = lastnumber;
          if (!origin_flags.tm_min) {
            origin.tm_min = 0;
            origin_flags.tm_min = 1;
          }
          if (!origin_flags.tm_sec) {
            origin.tm_sec = 0;
            origin_flags.tm_sec = 1;
          }
          lastnumber = 0;
        } else {
          if (origin.tm_hour > 12) origin.tm_hour -= 12;
        }
        origin_flags.tm_hour = 1;
        break;
      case TOKEN_PM: 
        kthis = 0;
        if (lastnumber > 0) {
          origin.tm_hour = lastnumber > 12 ? lastnumber : lastnumber + 12;
          if (!origin_flags.tm_min) {
            origin.tm_min = 0;
            origin_flags.tm_min = 1;
          }
          if (!origin_flags.tm_sec) {
            origin.tm_sec = 0;
            origin_flags.tm_sec = 1;
          }
          lastnumber = 0;
        } else {
          if (origin.tm_hour <= 12) origin.tm_hour += 12;
        }
        origin_flags.tm_hour = 1;
        break;
      case TOKEN_NOW:
        kthis = 0;
        if (!origin_flags.tm_year) {
          origin.tm_year = onow.tm_year;
        }
        origin_flags.tm_year = 1;
        if (!origin_flags.tm_mon) {
          origin.tm_mon = onow.tm_mon;
        }
        origin_flags.tm_mon = 1;
        if (!origin_flags.tm_mday) {
          origin.tm_mday = onow.tm_mday;
        }
        origin_flags.tm_mday = 1;
        if (!origin_flags.tm_hour) {
          origin.tm_hour = onow.tm_hour;
        }
        origin_flags.tm_hour = 1;
        if (!origin_flags.tm_min) {
          origin.tm_min = onow.tm_min;
        }
        origin_flags.tm_min = 1;
        if (!origin_flags.tm_sec) {
          origin.tm_sec = onow.tm_sec;
        }
        break;
      case TOKEN_YESTERDAY:
        kthis = 0;
        origin.tm_mday = onow.tm_mday - 1;
        origin_flags.tm_mday = 1;
        break;
      case TOKEN_TODAY:
        origin.tm_mday = onow.tm_mday;
        origin_flags.tm_mday = 1;
        break;
      case TOKEN_TOMORROW:
        kthis = 0;
        origin.tm_mday = onow.tm_mday + 1;
        origin_flags.tm_mday = 1;
        break;
      case TOKEN_JANUARY:
      case TOKEN_FEBRUARY:
      case TOKEN_MARCH:
      case TOKEN_APRIL:
      case TOKEN_MAY:
      case TOKEN_JUNE:
      case TOKEN_JULY:
      case TOKEN_AUGUST:
      case TOKEN_SEPTEMBER:
      case TOKEN_OCTOBER:
      case TOKEN_NOVEMBER:
      case TOKEN_DECEMBER:
        kthis = 0;
        if (lastnumber > 0) {
          origin.tm_mday = lastnumber;
          origin_flags.tm_mday = 1;
          lastnumber = 0;
        } 
        origin.tm_mon = t-TOKEN_JANUARY;
        if (!origin_flags.tm_mday)
          origin.tm_mday = 1;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        origin_flags.tm_mon = 1;
        break;
      case TOKEN_IDENTIFIER: {
        kthis = 0;
        int i = atoi(token);
        if (i > 1970 && i < 2038 && isallnum(token)) { // max time_t range
          origin.tm_year = i-1900;
          if (!origin_flags.tm_mday)
            origin.tm_mday = 1;
          if (!origin_flags.tm_mon)
            origin.tm_mon = 0;
          if (!origin_flags.tm_hour)
            origin.tm_hour = 0;
          if (!origin_flags.tm_min)
            origin.tm_min = 0;
          if (!origin_flags.tm_sec)
            origin.tm_sec = 0;
          break;
        }
        char *z = token+strlen(token)-2;
        if (!_stricmp(z, "st") || !_stricmp(z, "nd") || !_stricmp(z, "rd") || !_stricmp(z, "th")) {
          char *w = (char *)malloc(z-token+1);
          strncpy(w, token, z-token);
          w[z-token] = 0;
          int j = atoi(w);
          free(w);
          if (j >= 1 && j <= 31) {
            origin.tm_mday = j;
            origin_flags.tm_mday = 1;
            break;
          }
        }
        z = strchr(token, ':');
        if (z) {
          char *zz = strchr(z+1, ':');
          int a, b, c=0;
          a = myatoi(token, z-token);
          if (zz && *(zz+1) == 0) zz = NULL;
          if (zz && !isallnum(zz+1)) zz = NULL;
          if (zz) { b = myatoi(z+1, zz-(z+1)); c = atoi(zz+1); }
          else b = atoi(z+1);
          origin.tm_hour = a;
          origin.tm_min = b;
          if (zz && !origin_flags.tm_sec) {
            origin.tm_sec = c;
          } else if (!origin_flags.tm_sec) {
            origin.tm_sec = 0;
          }
          origin_flags.tm_sec = 1;
          origin_flags.tm_hour = 1;
          origin_flags.tm_min = 1;
          break;
        }
        z = strchr(token, '/');
        if (z) {
          char *zz = strchr(z+1, '/');
          time_t a, b, c=onow.tm_year;
          a = myatoi(token, z-token);
          if (zz && !isallnum(zz+1)) zz = NULL;
          if (zz && *(zz+1) == 0) zz = NULL;
          if (zz) { b = myatoi(z+1, zz-(z+1)); c = atoi(zz+1); }
          else b = atoi(z+1);
          if (b > 1969 && b < 2038) {
            // mm/yyyy
            origin.tm_year = b-1900;
            origin_flags.tm_year = 1;
            origin.tm_mon = a-1;
            origin_flags.tm_mon = 1;
            if (!origin_flags.tm_mday)
              origin.tm_mday = 1;
            if (!origin_flags.tm_hour)
              origin.tm_hour = 0;
            if (!origin_flags.tm_min)
              origin.tm_min = 0;
            if (!origin_flags.tm_sec)
              origin.tm_sec = 0;
          } else {
            // mm/dd(/yy[yy])
            if (c < 70) c += 100;
            if (c > 138) c -= 1900;
            origin.tm_year = c;
            origin.tm_mon = a-1;
            origin.tm_mday = b == 0 ? 1 : b;
            origin_flags.tm_year = 1;
            origin_flags.tm_mon = 1;
            origin_flags.tm_mday = 1;
            if (!origin_flags.tm_hour)
              origin.tm_hour = 0;
            if (!origin_flags.tm_min)
              origin.tm_min = 0;
            if (!origin_flags.tm_sec)
              origin.tm_sec = 0;
          }
          origin_flags.tm_year = 1;
          origin_flags.tm_mon = 1;
          origin_flags.tm_mday = 1;
          break;
        }
        if (isallnum(token)) {
          lastnumber = i;
          switch (lastt) {
      case TOKEN_JANUARY:
      case TOKEN_FEBRUARY:
      case TOKEN_MARCH:
      case TOKEN_APRIL:
      case TOKEN_MAY:
      case TOKEN_JUNE:
      case TOKEN_JULY:
      case TOKEN_AUGUST:
      case TOKEN_SEPTEMBER:
      case TOKEN_OCTOBER:
      case TOKEN_NOVEMBER:
      case TOKEN_DECEMBER:
        origin.tm_mday = lastnumber;
        origin_flags.tm_mday = 1;
        lastnumber = 0;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
      case TOKEN_AT: {
        origin.tm_hour = lastnumber;
        origin.tm_min = 0;
        origin.tm_sec = 0;
        origin_flags.tm_hour = 1;
        origin_flags.tm_min = 1;
        origin_flags.tm_sec = 1;
        lastnumber = 0;
        break;
                     }
          }
        }
        break;
                             }
      default:
        lastt = save_lastt;
        break;
    }
    p += size;
  }

  if (lastnumber) {
    switch (lastt) {
      case TOKEN_JANUARY:
      case TOKEN_FEBRUARY:
      case TOKEN_MARCH:
      case TOKEN_APRIL:
      case TOKEN_MAY:
      case TOKEN_JUNE:
      case TOKEN_JULY:
      case TOKEN_AUGUST:
      case TOKEN_SEPTEMBER:
      case TOKEN_OCTOBER:
      case TOKEN_NOVEMBER:
      case TOKEN_DECEMBER:
        origin.tm_mday = lastnumber;
        lastnumber = 0;
        if (!origin_flags.tm_hour)
          origin.tm_hour = 0;
        if (!origin_flags.tm_min)
          origin.tm_min = 0;
        if (!origin_flags.tm_sec)
          origin.tm_sec = 0;
        break;
    }
  }

  if (ago) { // if ago (or before), from is optional since if it wasn't specified we use now
    switch (what) {
      case TOKEN_SECOND:
        origin.tm_sec -= value;
        break;
      case TOKEN_MINUTE:
        origin.tm_min -= value;
        break;
      case TOKEN_HOUR:
        origin.tm_hour -= value;
        break;
      case TOKEN_DAY:
        origin.tm_mday -= value;
        break;
      case TOKEN_WEEK:
        origin.tm_mday -= value*7;
        break;
      case TOKEN_MONTH:
        origin.tm_mon -= value;
        break;
      case TOKEN_YEAR:
        origin.tm_year -= value;
        break;
    }
    time_t o = mktime(&origin);
    struct tm *g = localtime(&o);
    value=o;
    if (token) free(token);
    return value;
  } else if (from) { // from (or after) was specified, but not ago, 5 mn from x is x + 5 mn
    time_t v = value;
    switch (what) {
      case TOKEN_SECOND:
        origin.tm_sec += value;
        break;
      case TOKEN_MINUTE:
        origin.tm_min += value;
        break;
      case TOKEN_HOUR:
        origin.tm_hour += value;
        break;
      case TOKEN_DAY:
        origin.tm_mday += value;
        break;
      case TOKEN_WEEK:
        origin.tm_mday += value*7;
        break;
      case TOKEN_MONTH:
        origin.tm_mon += value;
        break;
      case TOKEN_YEAR:
        origin.tm_year += value;
        break;
    }
    time_t o = mktime(&origin);
    struct tm *g = localtime(&o);
    value=o;
    if (token) free(token);
    return value;
  } else { // none of ago/from/before/after were specified, just make a date/time with what we got and ignore our old value
    time_t o = mktime(&origin);
    struct tm *g = localtime(&o);
    value=o;
    if (token) free(token);
    return value;
  }

  if (token) free(token);
  return value;
}
