/*
    Melody maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "add.h"
#include "abort.h"
#include "charlist.h"
#include "command.h"
#include "edit.h"
#include "export.h"
#include "file.h"
#include "general.h"
#include "mm_dialog.h"
#include "mm_list.h"
#include "reload.h"
#include "root.h"
#include "misc/fileio.h"
#include "misc/misc.h"
#include "misc/misc_conf.h"
#ifdef G_OS_WIN32
# include <gdk/gdkwin32.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:メニュー関数群                                                           *
*                                                                             *
******************************************************************************/
void
command_new (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  file_open_edit (NULL);
}


static void
command_open_ok (GtkWidget *widget,
                 GtkWidget *dialog)
{
  gchar **file;

  file = g_object_get_data (G_OBJECT (dialog), "user_data");
  *file
    = g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog)));
  gtk_widget_destroy (dialog);
}


void
command_open (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *file = NULL;
  GtkWidget *dialog;

  dialog = gtk_file_selection_new (_("Open"));
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), dialog);
  g_signal_connect_swapped
                        (G_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", &file);
  if (open_path)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), open_path);
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
  if (file)
    {
      g_free (open_path);
      open_path = fileio_get_full_path (file);
      *(gchar *)g_basename (open_path) = '\0';
      file_open_edit (file);
      g_free (file);
    }
}


void
command_close (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  gint page;

  if ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook))) >= 0)
    {
      MmaidWindow *mmaid;

      mmaid = g_object_get_data (G_OBJECT (gtk_notebook_get_nth_page
                                (GTK_NOTEBOOK (notebook), page)), "user_data");
      if (prompt_close (mmaid))
        gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), page);
    }
}


void
command_save (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *root = NULL, *tmp;
  gint i = G_MAXINT;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  tmp = g_path_get_dirname (mmaid->file);
  root = g_strconcat (tmp, G_DIR_SEPARATOR_S, NULL);
  g_free (tmp);
  if (!mmaid->create)
    {
      gchar *path;
      gint leng;

      path = g_filename_to_utf8 (root, -1, NULL, NULL, NULL);
      leng = g_strlen (path);
      for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0; i--)
        {
          const ID3Tag *id3tag;

          id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
          if (g_strfilencmp (id3tag->file, path, leng))
            break;
        }
      g_free (path);
    }
  if (i >= 0)
    {
      command_saveas (callback_data, callback_action, widget);
    }
  else
    {
      gchar *text, *utf8str;

      utf8str = g_filename_to_utf8 (mmaid->file, -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("%s\nThis file is existed.\n\nOver write?"),
                                                                    utf8str);
      g_free (utf8str);
      if ((!fileio_isfile (mmaid->file)
            || misc_message_box ("Melody maid", text, 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
                    && file_save_db (mmaid->file, mmaid, root, mmaid->charset))
        mmaid_change_edit (mmaid, FALSE);
    }
  g_free (root);
}


typedef struct _CommandSaveOk CommandSaveOk;
struct _CommandSaveOk
{
  gchar *file, *root, *charset;
  GtkWidget *dialog, *combo0, *combo1;
};


static void
command_save_ok (GtkWidget     *widget,
                 CommandSaveOk *cmd_sv)
{
  cmd_sv->file    = g_strdup (gtk_file_selection_get_filename
                                        (GTK_FILE_SELECTION (cmd_sv->dialog)));
  cmd_sv->root    = g_strdup (gtk_entry_get_text
                            (GTK_ENTRY (GTK_COMBO (cmd_sv->combo0)->entry)));
  cmd_sv->charset = g_strdup (gtk_entry_get_text
                            (GTK_ENTRY (GTK_COMBO (cmd_sv->combo1)->entry)));
  gtk_widget_destroy (cmd_sv->dialog);
}


/* ja:コンボボックスが変更された */
static void
command_save_changed_combo0 (GtkWidget     *widget,
                             CommandSaveOk *cmd_sv)
{
  gchar *file, *name;

  name = g_path_get_basename (gtk_file_selection_get_filename
                                        (GTK_FILE_SELECTION (cmd_sv->dialog)));
  file = g_strconcat (gtk_entry_get_text (GTK_ENTRY (widget)), name, NULL);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (cmd_sv->dialog), file);
  g_free (file);
  g_free (name);
}


/* ja:コンボボックスが変更された */
static void
command_save_changed_combo1 (GtkWidget     *widget,
                             CommandSaveOk *cmd_sv)
{
  gtk_widget_set_sensitive (GTK_FILE_SELECTION (cmd_sv->dialog)->ok_button,
                charlist_is_valid (gtk_entry_get_text (GTK_ENTRY (widget))));
}


void
command_saveas (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *path, *root = NULL, **ary;
  gint i;
  GList *glist0 = NULL, *glist1 = NULL;
  GtkWidget *label0, *label1, *hbox, *vbox;
  CommandSaveOk cmd_sv;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  g_memset (&cmd_sv, 0 ,sizeof (CommandSaveOk));
  /* ja:共通のルートを求める */
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0; i--)
    {
      const ID3Tag *id3tag;

      id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
      if (root)
        {
          gint j;

          for (j = 0; root[j] != '\0'; j++)
            if (root[j] != id3tag->file[j])
              break;
          root[j] = '\0';
        }
      else
        {
          root = g_strdup (id3tag->file);
        }
    }
  path = g_strrchr (root, G_DIR_SEPARATOR);
  if (!path)
    {
      misc_message_box ("Melody maid", _("Root Directory Error"), 0,
                                                            _("OK"), NULL);
      return;
    }
  *path = '\0';
#ifdef G_OS_WIN32
  if (g_strlen (root) > 2 && root[0] == G_DIR_SEPARATOR && root[1] == G_DIR_SEPARATOR)
    {
      path = g_strdup (G_DIR_SEPARATOR_S G_DIR_SEPARATOR_S);
      ary = g_strsplit (root + 2, G_DIR_SEPARATOR_S, G_MAXINT);
    }
  else
    {
      ary = g_strsplit (root, G_DIR_SEPARATOR_S, G_MAXINT);
    }
#else /* not G_OS_WIN32 */
  ary = g_strsplit (root, G_DIR_SEPARATOR_S, G_MAXINT);
  path = NULL;
#endif /* not G_OS_WIN32 */
  g_free (root);
  root = NULL;
  for (i = 0; ary[i]; i++)
    {
      path = path ? g_strconcat (path, ary[i], G_DIR_SEPARATOR_S, NULL)
                  : g_strconcat (      ary[i], G_DIR_SEPARATOR_S, NULL);
      glist0 = g_list_append (glist0, path);
    }
  g_strfreev (ary);
  if (save_path)
    {
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
      if (!g_strfilencmp (utf8str, path, g_strlen (utf8str)))
        root = g_strdup (save_path);
      g_free (utf8str);
    }
  if (!root)
    root = g_filename_from_utf8 (path, -1, NULL, NULL, NULL);
  ary = g_strsplit (charset_list, ",", G_MAXINT);
  for (i = 0; ary[i]; i++)
    glist1 = g_list_insert_sorted (glist1, ary[i], (GCompareFunc)strcmp);
  /* ja:コンボボックス */
  cmd_sv.combo0 = gtk_combo_new ();
  cmd_sv.combo1 = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (cmd_sv.combo0), glist0);
  gtk_combo_set_popdown_strings (GTK_COMBO (cmd_sv.combo1), glist1);
  gtk_combo_set_value_in_list (GTK_COMBO (cmd_sv.combo0), FALSE, FALSE);
  gtk_combo_set_value_in_list (GTK_COMBO (cmd_sv.combo1), FALSE, FALSE);
  gtk_combo_set_case_sensitive (GTK_COMBO (cmd_sv.combo0), TRUE);
  gtk_combo_set_case_sensitive (GTK_COMBO (cmd_sv.combo1), TRUE);
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (cmd_sv.combo0)->entry),
                                                                        FALSE);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (cmd_sv.combo0)->entry), root);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (cmd_sv.combo1)->entry),
                                                            mmaid->charset);
  for (i = g_list_length (glist0) - 1; i >= 0; i--)
    g_free (g_list_nth_data (glist0, i));
  g_strfreev (ary);
  g_list_free (glist0);
  g_list_free (glist1);
  /* ja:ラベル */
  label0 = gtk_label_new_with_mnemonic (_("_Root"));
  label1 = gtk_label_new_with_mnemonic (_("Character _Encoding"));
  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), label0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), cmd_sv.combo0, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), label1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), cmd_sv.combo1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show_all (vbox);
  /* ja:ダイアログ */
  cmd_sv.dialog = gtk_file_selection_new (_("Save As"));
  g_signal_connect (G_OBJECT (cmd_sv.dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->ok_button),
                            "clicked", G_CALLBACK (command_save_ok), &cmd_sv);
  g_signal_connect_swapped
                (G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->cancel_button),
                                    "clicked", G_CALLBACK (gtk_widget_destroy),
                                                GTK_OBJECT (cmd_sv.dialog));
  g_signal_connect_swapped (G_OBJECT (GTK_COMBO (cmd_sv.combo0)->entry),
                    "activate", G_CALLBACK (gtk_button_clicked),
                    G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->ok_button));
  g_signal_connect_swapped (G_OBJECT (GTK_COMBO (cmd_sv.combo1)->entry),
                    "activate", G_CALLBACK (gtk_button_clicked),
                    G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->ok_button));
  g_signal_connect (G_OBJECT (GTK_COMBO (cmd_sv.combo0)->entry), "changed",
                            G_CALLBACK (command_save_changed_combo0), &cmd_sv);
  g_signal_connect (G_OBJECT (GTK_COMBO (cmd_sv.combo1)->entry), "changed",
                            G_CALLBACK (command_save_changed_combo1), &cmd_sv);
  gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (cmd_sv.dialog)->main_vbox),
                                                        vbox, FALSE, FALSE, 0);
  if (root)
    {
      gchar *tmp;

      tmp = g_strconcat (root, "irivnavi.idb", NULL);
      gtk_file_selection_set_filename
                                    (GTK_FILE_SELECTION (cmd_sv.dialog), tmp);
      g_free (tmp);
      g_free (root);
      root = NULL;
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (cmd_sv.dialog));
  gtk_widget_show (cmd_sv.dialog);
  gtk_grab_add (cmd_sv.dialog);
  gtk_main ();
  if (cmd_sv.file)
    {
      gchar *text, *label, *utf8str;

      utf8str = g_filename_to_utf8 (cmd_sv.file, -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("%s\nThis file is existed.\n\nOver write?"),
                                                                    utf8str);
      g_free (utf8str);
      if ((!fileio_isfile (cmd_sv.file)
                                || misc_message_box ("Melody maid", text, 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
            && file_save_db (cmd_sv.file, mmaid, cmd_sv.root, cmd_sv.charset))
        {
          g_free (save_path);
          save_path = fileio_get_full_path (cmd_sv.file);
          *(gchar *)g_basename (save_path)='\0';
          /* ja:ファイル名を調べる */
          file_delete_edit (mmaid->file);
          mmaid->file[0]='\0';
          label = file_add_edit (cmd_sv.file, &mmaid->same);
          gtk_label_set_text (GTK_LABEL (mmaid->label), label);
          g_free (label);
          /* ja:ウインドウメニュー */
          gtk_label_set_text (GTK_LABEL (gtk_bin_get_child
                                (GTK_BIN (mmaid->menu_item))), cmd_sv.file);
          /* ja:キャラクターセット */
          g_free (mmaid->charset);
          mmaid->charset = g_strdup (cmd_sv.charset);
          /* ja:諸設定 */
          mmaid->create = FALSE;
          mmaid_change_edit (mmaid, FALSE);
          g_free (mmaid->file);
          mmaid->file = g_strdup (cmd_sv.file);
          file_set_history (mmaid->file);
        }
    }
  g_free (cmd_sv.file);
  g_free (cmd_sv.root);
  g_free (cmd_sv.charset);
}


void
command_reload (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *text;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  text = mmaid->edit ? g_strdup_printf
                        (_("File %s was edited.\nRestore to existed file?"),
                                                        mmaid->file) : NULL;
  if (reload_dialog (mmaid, text))
    {
      /* ja:削除 */
      mm_list_delete_all (MM_LIST (mmaid->list));
      /* ja:再読み込み */
      file_open_db (mmaid);
      mmaid_change_edit (mmaid, FALSE);
      charlist_renewal_all (mmaid->charset);
      set_menu_bar (mmaid);
    }
}


static void
command_dialog_abort (GtkWidget *widget,
                      GtkWidget *dialog)
{
  userbreak = FALSE;
}


void
command_export (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gint i, count;
  GtkWidget *dialog;
  ExportInfo *expinfo;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  expinfo = export_dialog (mmaid->charset);
  if (!expinfo)
    return;
  def_hold = expinfo->hold;
  def_gnum = expinfo->gnum;
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Export"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 0; i < count && userbreak; i++)
    {
      while (gtk_events_pending ())
        gtk_main_iteration ();
      if (mm_list_is_selected (MM_LIST (mmaid->list), i))
        {
          gchar *file, *str;
          gint j, k, n, gnum;
          ID3Tag *id3tag;

          id3tag = copy_id3tag (mm_list_get_nth (MM_LIST (mmaid->list), i));
          file = g_filename_from_utf8 (id3tag->file, -1, NULL, NULL, NULL);
          switch (file_is_type (file))
            {
              case FILE_KIND_ERROR:
                /* ja:ファイルが存在しないとき */
                str = _("File Open Error");
                break;
              case FILE_KIND_TAG:
                if (expinfo->hold)
                  {
                    /* ja:既存のタグを保持し、かつ既にタグがあるとき */
                    str = _("Have Tag");
                    break;
                  }
              default:
                gnum = -1;
                if (id3tag->genre)
                  for (j = 0; j < genres; j++)
                    for (k = 0; k < 256; k++)
                      if (genre[j][k]
                                    && !g_strcmp (genre[j][k], id3tag->genre))
                        {
                          gnum = k;
                          goto loop;
                        }
                loop:
                if (gnum == -1)
                  {
                    gnum = expinfo->gnum;
                    g_free (id3tag->genre);
                    id3tag->genre = NULL;
                    for (j = 0; j < genres; j++)
                      if (genre[j][gnum])
                        {
                          id3tag->genre = g_strdup (genre[j][gnum]);
                          break;
                        }
                  }
                if (!id3tag->charset)
                  id3tag->charset = g_strdup (expinfo->charset);
                switch (file_save_id3tag (file, id3tag, gnum))
                  {
                    case FILE_SAVE_SUCCESS:
                      str = _("Success");
                      break;
                    case FILE_SAVE_CHARSET_TITLE:
                      str = _("Invalid Character Encoding of Title");
                      break;
                    case FILE_SAVE_CHARSET_ARTIST:
                      str = _("Invalid Character Encoding of Artist");
                      break;
                    case FILE_SAVE_CHARSET_ALBUM:
                      str = _("Invalid Character Encoding of Album");
                      break;
                    case FILE_SAVE_CHARSET_YEAR:
                      str = _("Invalid Character Encoding of Year");
                      break;
                    case FILE_SAVE_CHARSET_COMMENT:
                      str = _("Invalid Character Encoding of Comment");
                      break;
                    case FILE_SAVE_LENGTH_TITLE:
                      str = _("Invalid Length of Title");
                      break;
                    case FILE_SAVE_LENGTH_ARTIST:
                      str = _("Invalid Length of Artist");
                      break;
                    case FILE_SAVE_LENGTH_ALBUM:
                      str = _("Invalid Length of Album");
                      break;
                    case FILE_SAVE_LENGTH_YEAR:
                      str = _("Invalid Length of Year");
                      break;
                    case FILE_SAVE_LENGTH_COMMENT:
                      str = _("Invalid Length of Comment");
                      break;
                    default:
                      str = _("Error");
                  }
            }
          g_free (file);
          n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list), id3tag);
          mm_list_set_status (MM_LIST (MM_DIALOG (dialog)->list), n, str);
        }
    }
  g_free (expinfo->charset);
  g_free (expinfo);
  gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (MM_DIALOG (dialog)->ok_button);
  gtk_main ();
}


void
command_exit (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GdkEvent event;

  event.any.type = GDK_DELETE;
  event.any.window = window->window;
  event.any.send_event = FALSE;
  gdk_event_put (&event);
}


void
command_up (gpointer   callback_data,
            guint      callback_action,
            GtkWidget *widget)
{
  gint i, count;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  mm_list_sort_off (MM_LIST (mmaid->list));
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 1; i < count; i++)
    if (!mm_list_is_selected (MM_LIST (mmaid->list), i - 1)
                            && mm_list_is_selected (MM_LIST (mmaid->list), i))
      mm_list_swap (MM_LIST (mmaid->list), i - 1, i);
  mmaid_change_edit (mmaid, TRUE);
}


void
command_down (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gint i;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  mm_list_sort_off (MM_LIST (mmaid->list));
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 2; i >= 0 ; i--)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i)
                        && !mm_list_is_selected (MM_LIST (mmaid->list), i + 1))
      mm_list_swap (MM_LIST (mmaid->list), i, i + 1);
  mmaid_change_edit (mmaid, TRUE);
}


void
command_root (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  root_dialog (mmaid);
}


static void
command_dialog_ok (GtkWidget *widget,
                    GtkWidget *dialog)
{
  gboolean edited = FALSE;
  gint i, count;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT (dialog), "user_data");
  count = mm_list_length (MM_LIST (MM_DIALOG (dialog)->list));
  for (i = 0; i < count; i++)
    {
      gint n;
      const ID3Tag *id3tag;

      id3tag = mm_list_get_nth (MM_LIST (MM_DIALOG (dialog)->list), i);
      n = mm_list_search (MM_LIST (mmaid->list),
                                            MM_LIST_FILE, id3tag->file, -1);
      if (n >= 0)
        {
          if (mm_list_get_active_refresh
                                    (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_change (MM_LIST (mmaid->list), n,
                                    MM_LIST_ID3TAG, copy_id3tag (id3tag), -1);
              edited = TRUE;
            }
          else if (mm_list_get_active_delete
                                    (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_delete (MM_LIST (mmaid->list), n);
              edited = TRUE;
            }
        }
      else
        {
          if (mm_list_get_active_add (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_append (MM_LIST (mmaid->list), copy_id3tag (id3tag));
              edited = TRUE;
            }
        }
    }
  if (edited)
    mmaid_change_edit (mmaid, TRUE);
  gtk_widget_destroy (dialog);
}


void
command_refresh (gpointer   callback_data,
                 guint      callback_action,
                 GtkWidget *widget)
{
  gboolean selected = FALSE;
  gint i, count;
  GtkWidget *dialog;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  /* ja:選択の数を数える */
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 0; i < count; i++)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i))
      {
        selected = TRUE;
        break;
      }
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Refresh"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                        "clicked", G_CALLBACK (command_dialog_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->cancel_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  for (i = 0; i < count && userbreak; i++)
    {
      while (gtk_events_pending ())
        gtk_main_iteration ();
      if (!selected || mm_list_is_selected (MM_LIST (mmaid->list), i))
        {
          gchar *file;
          gint n;
          const ID3Tag *id3tag0;
          ID3Tag *id3tag1;

          id3tag0 = mm_list_get_nth (MM_LIST (mmaid->list), i);
          file = g_filename_from_utf8 (id3tag0->file, -1, NULL, NULL, NULL);
          switch (file_is_type (file))
            {
              case FILE_KIND_ERROR:
              case FILE_KIND_DB:
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                        copy_id3tag (id3tag0));
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                mm_list_set_active_delete (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    n, TRUE);
                break;
              case FILE_KIND_UNKNOWN:
                id3tag1 = file_open_id3tag (file, id3tag0->charset
                                            ? id3tag0->charset : charset_list);
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                if (!cmp_id3tag (id3tag0, id3tag1))
                  mm_list_show_refresh (MM_LIST (MM_DIALOG (dialog)->list), n);
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                break;
              case FILE_KIND_TAG:
                id3tag1 = file_open_id3tag (file, id3tag0->charset
                                            ? id3tag0->charset : charset_list);
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                if (!cmp_id3tag (id3tag0, id3tag1))
                  {
                    mm_list_set_sensitive_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n, FALSE);
                    mm_list_show_refresh (MM_LIST (MM_DIALOG (dialog)->list),
                                                                            n);
                    mm_list_set_active_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n, TRUE);
                  }
            }
          g_free (file);
        }
    }
  if (userbreak)
    {
      gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
      gtk_widget_show (MM_DIALOG (dialog)->ok_button);
      gtk_widget_show (MM_DIALOG (dialog)->cancel_button);
      gtk_main ();
    }
  else
    {
      gtk_widget_destroy (dialog);
    }
}


void
command_add (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  add_dialog (mmaid);
}


void
command_edit (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  edit_dialog (mmaid);
}

static void
command_search_recursive (const gchar *path,
                          GtkWidget   *dialog)
{
  GDir *gdir;

  gdir = g_dir_open (path, 0, NULL);
  if (gdir)
    {
      const gchar *name;

      while (userbreak && (name = g_dir_read_name (gdir)))
        {
          gchar *file;

          while (gtk_events_pending ())
            gtk_main_iteration ();
#ifndef G_OS_WIN32
          if (name[0] == '.')
            continue;
#endif /* not G_OS_WIN32 */
          file = g_strconcat (path, G_DIR_SEPARATOR_S, name, NULL);
          if (g_file_test (file, G_FILE_TEST_IS_DIR))
            {
              command_search_recursive (file, dialog);
            }
          else
            {
              gchar *utf8file;
              gint n0, n1;
              const ID3Tag *id3tag0;
              ID3Tag *id3tag1;
              MmaidWindow *mmaid;

              mmaid = g_object_get_data (G_OBJECT (dialog), "user_data");
              utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
              n0 = mm_list_search (MM_LIST (mmaid->list),
                                                MM_LIST_FILE, utf8file, -1);
              g_free (utf8file);
              id3tag0 = n0 >= 0 ? mm_list_get_nth (MM_LIST (mmaid->list), n0)
                                : NULL;
              switch (file_is_type (file))
                {
                  case FILE_KIND_ERROR:
                  case FILE_KIND_DB:
                    if (n0 >= 0)
                      {
                        n1 = mm_list_append
                                (MM_LIST (MM_DIALOG (dialog)->list),
                                                        copy_id3tag (id3tag0));
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_set_active_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                    break;
                  case FILE_KIND_UNKNOWN:
                    id3tag1 = file_open_id3tag (file, charset_list);
                    n1 = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                    if (n0 >= 0)
                      {
                        if (!cmp_id3tag (id3tag0, id3tag1))
                          mm_list_show_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                      }
                    else
                      {
                        mm_list_show_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        if (g_strlen (file) > 4 && !g_strcasecmp (".mp3",
                                                file + g_strlen (file) - 4))
                          mm_list_set_active_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                    break;
                  case FILE_KIND_TAG:
                    id3tag1 = file_open_id3tag (file, charset_list);
                    n1 = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                    if (n0 >= 0)
                      {
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        if (!cmp_id3tag (id3tag0, id3tag1))
                          {
                            mm_list_set_sensitive_delete
                                (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    n1, FALSE);
                            mm_list_show_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                            mm_list_set_active_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                          }
                      }
                    else
                      {
                        mm_list_show_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_set_active_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                }
            }
          g_free (file);
        }
      g_dir_close (gdir);
    }
}


void
command_search (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *path, *tmp;
  GtkWidget *dialog;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  dialog = gtk_file_selection_new (_("Browse"));
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), dialog);
  g_signal_connect_swapped
                        (G_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", &path);
  path = g_path_is_absolute (mmaid->file) ? g_path_get_dirname (mmaid->file)
       : open_path ? g_strdup (open_path)
       : save_path ? g_strdup (save_path)
       : NULL;
  if (path)
    {
      if (path[0] != '\0')
        {
          gchar *tmp;

          if (path[g_strlen (path) - 1] != G_DIR_SEPARATOR)
            {
              tmp = g_strconcat (path, G_DIR_SEPARATOR_S, NULL);
              g_free (path);
              path = tmp;
            }
          gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), path);
        }
      g_free (path);
      path = NULL;
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_hide
            (gtk_widget_get_parent (GTK_FILE_SELECTION (dialog)->file_list));
  gtk_widget_hide (GTK_FILE_SELECTION (dialog)->selection_entry);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();

  if (!path)
    return;
  tmp = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
  g_free (path);
  if (!tmp || tmp[0] =='\0')
    {
      g_free (tmp);
      return;
    }
  if (tmp[g_strlen (tmp) - 1] == G_DIR_SEPARATOR)
    tmp[g_strlen (tmp) - 1] = '\0';
  path = g_filename_from_utf8 (tmp, -1, NULL, NULL, NULL);
  g_free (tmp);
  if (!path)
    return;
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Search"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                        "clicked", G_CALLBACK (command_dialog_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->cancel_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  command_search_recursive (path, dialog);
  g_free (path);
  gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (MM_DIALOG (dialog)->ok_button);
  gtk_widget_show (MM_DIALOG (dialog)->cancel_button);
  gtk_main ();
}


void
command_cut (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  command_copy   (callback_data, callback_action, widget);
  command_delete (callback_data, callback_action, widget);
}


void
command_copy (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
#ifndef G_OS_WIN32
  if (gtk_selection_owner_set (window, GDK_SELECTION_CLIPBOARD,
                                                            GDK_CURRENT_TIME))
    {
#endif /* not G_OS_WIN32 */
      gint i, count;
      MmaidWindow *mmaid;

      mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
      g_free (clipboard_id3tag);
      clipboard_id3tag = NULL;
      count = mm_list_length (MM_LIST (mmaid->list));
      for (i = 0; i < count; i++)
        if (mm_list_is_selected (MM_LIST (mmaid->list), i))
          {
            gchar *str;
            const ID3Tag *id3tag;

            id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
            str = g_strdup_printf ("%s\t%s\t%s\t%s\t%s\t%s\t%d\t%s\t%s",
                                    id3tag->file    ? id3tag->file     : "",
                                    id3tag->title   ? id3tag->title    : "",
                                    id3tag->artist  ? id3tag->artist   : "",
                                    id3tag->album   ? id3tag->album    : "",
                                    id3tag->year    ? id3tag->year     : "",
                                    id3tag->comment ? id3tag->comment  : "",
                                    id3tag->track,
                                    id3tag->genre   ? id3tag->genre    : "",
                                    id3tag->charset ? id3tag->charset  : "");
            if (clipboard_id3tag)
              {
                gchar *tmp;

                tmp = g_strconcat (clipboard_id3tag, "\n", str, NULL);
                g_free (clipboard_id3tag);
                g_free (str);
                clipboard_id3tag = tmp;
              }
            else
              {
                clipboard_id3tag = str;
              }
          }
#ifdef G_OS_WIN32
      if (clipboard_id3tag)
        {
          HGLOBAL hGlobal;
          LPSTR lpID3Tag;

          hGlobal = GlobalAlloc (GMEM_MOVEABLE,
                        (g_strlen (clipboard_id3tag) + 1) * sizeof (gchar));
          if (hGlobal)
            {
              lpID3Tag = GlobalLock (hGlobal);
              if (lpID3Tag)
                {
                  g_strcpy (lpID3Tag, clipboard_id3tag);
                  GlobalUnlock (hGlobal);
                  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
                    {
                      if (!EmptyClipboard ()
                                    || !SetClipboardData (uFormat, hGlobal))
                        GlobalFree (hGlobal);
                      CloseClipboard ();
                    }
                  else
                    {
                      GlobalFree (hGlobal);
                    }
                }
              else
                {
                  GlobalFree (hGlobal);
                }
            }
        }
#else /* not G_OS_WIN32 */
    }
#endif /* not G_OS_WIN32 */
}


void
command_paste (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  /* ja:Windowsのセレクションの実装があやしいので暫定的処理 */
#ifdef G_OS_WIN32
  gchar *id3tag = NULL;
  HGLOBAL hGlobal;
  LPSTR lpID3Tag;

  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
    {
      if ((hGlobal = GetClipboardData (uFormat)) != NULL)
        {
          if ((lpID3Tag = GlobalLock (hGlobal)) != NULL)
            id3tag = g_strdup (lpID3Tag);
          GlobalUnlock (hGlobal);
        }
      CloseClipboard ();
      if (id3tag)
        {
          GtkSelectionData data;

          data.selection = GDK_SELECTION_CLIPBOARD;
          data.target = atom_id3tag;
          data.type = atom_id3tag;
          data.format = 8;
          data.data = id3tag;
          data.length = g_strlen (id3tag);
          signal_selection_received (window, &data, GDK_CURRENT_TIME, NULL);
          g_free (id3tag);
        }
    }
#else /* not G_OS_WIN32 */
  gtk_selection_convert (window, GDK_SELECTION_CLIPBOARD, atom_targets,
                                                            GDK_CURRENT_TIME);
#endif /* not G_OS_WIN32 */
}


void
command_delete (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gint i;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0 ; i--)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i))
      mm_list_delete (MM_LIST (mmaid->list), i);
  mmaid_change_edit (mmaid, TRUE);
}


void
command_all (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  gint i;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT
                (gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)))),
                                                                "user_data");
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0 ; i--)
    if (!mm_list_is_selected (MM_LIST (mmaid->list), i))
      break;
  if (i >= 0)
    mm_list_select_all (MM_LIST (mmaid->list));
  else
    mm_list_unselect_all (MM_LIST (mmaid->list));
}


static void
command_conf_ok (GtkWidget *widget,
                 GtkWidget *dialog)
{
  gint i;
  GList *glist;
  GtkWidget *menu_shell;

  history = misc_conf_get_history (MISC_CONF (dialog));
  newfile = misc_conf_get_newfile (MISC_CONF (dialog));
  second = misc_conf_get_second (MISC_CONF (dialog));
  misc_conf_get_size (MISC_CONF (dialog), &def_width, &def_height);
  n_pos = misc_conf_get_tab (MISC_CONF (dialog));
  gtk_widget_destroy (dialog);
  /* ja:履歴の整理 */
  menu_shell = gtk_item_factory_get_widget (ifactory_menu, "<main>/File");
  glist = gtk_container_children (GTK_CONTAINER (menu_shell));
  for (i = g_list_length (glist) - 3;
                            i > history- (history == 0) + MENUFILE - 2; i--)
    gtk_container_remove (GTK_CONTAINER (menu_shell),
                                                g_list_nth_data (glist, i));
  g_list_free (glist);
  /* ja:ノートの位置 */
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), n_pos);
}


void
command_conf (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GtkWidget *dialog;

  dialog = misc_conf_new ();
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MISC_CONF (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_conf_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MISC_CONF (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  misc_conf_set_history (MISC_CONF(dialog), history);
  misc_conf_set_newfile (MISC_CONF(dialog), newfile);
  misc_conf_set_second (MISC_CONF(dialog), second);
  misc_conf_set_size (MISC_CONF(dialog), def_width, def_height);
  misc_conf_set_tab (MISC_CONF(dialog), n_pos);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
}


void
command_charset (gpointer   callback_data,
                 guint      callback_action,
                 GtkWidget *widget)
{
  gchar *result;

  result = charlist_dialog (charset_list);
  if (result)
    {
      g_free (charset_list);
      charset_list = result;
      charlist_renewal_all (charset_list);
    }
}
