/*
    charlist
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "charuty.h"
#include "misc_charlist.h"
#include "misc_charset.h"
#include <gdk/gdkkeysyms.h>


static void misc_charlist_class_init (MiscCharlistClass *klass);
static void misc_charlist_init       (MiscCharlist      *charlist);
static void misc_charlist_close      (MiscCharlist      *charlist);


static GtkWindowClass *parent_class = NULL;


enum
{
  CLOSE_SIGNAL,
  LAST_SIGNAL
};


static gint misc_charlist_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
misc_charlist_get_type (void)
{
  static GType misc_charlist_type = 0;

  if (!misc_charlist_type)
    {
      static const GTypeInfo charlist_info =
      {
        sizeof (MiscCharlistClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)misc_charlist_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (MiscCharlist),
        0,              /* n_preallocs */
        (GInstanceInitFunc)misc_charlist_init,
      };

      misc_charlist_type = g_type_register_static (GTK_TYPE_WINDOW,
                                            "MiscCharlist", &charlist_info, 0);
    }

  return misc_charlist_type;
}


static void
misc_charlist_class_init (MiscCharlistClass *klass)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkBindingSet *binding_set;

  gobject_class = (GObjectClass*) klass;
  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;

  parent_class = g_type_class_peek_parent (klass);

  klass->close = misc_charlist_close;

  misc_charlist_signals[CLOSE_SIGNAL]
                = g_signal_new ("close",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                                G_STRUCT_OFFSET (MiscCharlistClass, close),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);

  binding_set = gtk_binding_set_by_class (klass);
  gtk_binding_entry_add_signal (binding_set, GDK_Escape, 0, "close", 0);
}


/* ja:リストボックスの選択 */
static void
misc_charlist_changed (GtkTreeSelection *select,
                       MiscCharlist     *charlist)
{
  gint count, pos = -1;
  GtkTreeIter iter;

  for (count = 0; gtk_tree_model_iter_nth_child
            (GTK_TREE_MODEL (charlist->store), &iter, NULL, count); count++)
    if (gtk_tree_selection_iter_is_selected (select, &iter))
      pos = count;
  if (pos >= 0)
    {
      gtk_widget_set_sensitive (charlist->remove_button, count > 1);
      gtk_widget_set_sensitive (charlist->up_button, pos > 0);
      gtk_widget_set_sensitive (charlist->down_button, pos < count - 1);
    }
  else
    {
      gtk_widget_set_sensitive (charlist->remove_button, FALSE);
      gtk_widget_set_sensitive (charlist->up_button, FALSE);
      gtk_widget_set_sensitive (charlist->down_button, FALSE);
    }
}


/* ja:新規 */
static void
misc_charlist_clicked_new (GtkWidget    *widget,
                           MiscCharlist *charlist)
{
  gchar *charset;
  gint i;
  GtkTreeIter iter;

  /* ja:挿入 */
  charset = misc_charset_get_charset (MISC_CHARSET (charlist->charset));
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (charlist->select, &iter))
      {
        i++;
        break;
      }
  gtk_list_store_insert (charlist->store, &iter, i);
  gtk_list_store_set (charlist->store, &iter, 0, g_strdup (charset), -1);
  gtk_tree_selection_select_iter (charlist->select, &iter);
  /* ja:コンボ */
  misc_charset_remove_candidate (MISC_CHARSET (charlist->charset), charset);
  g_free (charset);
  misc_charset_set_charset (MISC_CHARSET (charlist->charset), NULL);
  /* ja:OKボタン */
  gtk_widget_set_sensitive (charlist->ok_button, TRUE);
}


/* ja:削除 */
static void
misc_charlist_clicked_remove (GtkWidget    *widget,
                              MiscCharlist *charlist)
{
  gint i;
  GtkTreeIter iter;

  /* ja:削除 */
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (charlist->select, &iter))
      {
        gchar *charset;

        gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                    &iter, 0, &charset, -1);
        gtk_list_store_remove (charlist->store, &iter);
        /* ja:コンボ */
        misc_charset_add_candidate (MISC_CHARSET (charlist->charset), charset);
        misc_charset_set_charset (MISC_CHARSET (charlist->charset), charset);
        g_free (charset);
      }
  /* ja:OKボタン */
  gtk_widget_set_sensitive (charlist->ok_button, TRUE);
}


/* ja:上へ */
static void
misc_charlist_clicked_up (GtkWidget    *widget,
                          MiscCharlist *charlist)
{
  gint i;
  GtkTreeIter iter0;

  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter0, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (charlist->select, &iter0))
      {
        gchar *charset0, *charset1;
        GtkTreeIter iter1;

        gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                    &iter0, 0, &charset0, -1);
        gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter1, NULL, i - 1);
        gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                    &iter1, 0, &charset1, -1);
        gtk_list_store_set (charlist->store, &iter0, 0, charset1, -1);
        gtk_list_store_set (charlist->store, &iter1, 0, charset0, -1);
        gtk_tree_selection_select_iter (charlist->select, &iter1);
        g_free (charset0);
        g_free (charset1);
        break;
      }
  /* ja:OKボタン */
  gtk_widget_set_sensitive (charlist->ok_button, TRUE);
}


/* ja:下へ */
static void
misc_charlist_clicked_down (GtkWidget    *widget,
                            MiscCharlist *charlist)
{
  gint i;
  GtkTreeIter iter0;

  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter0, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (charlist->select, &iter0))
      {
        gchar *charset0, *charset1;
        GtkTreeIter iter1;

        gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                    &iter0, 0, &charset0, -1);
        gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter1, NULL, i + 1);
        gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                    &iter1, 0, &charset1, -1);
        gtk_list_store_set (charlist->store, &iter0, 0, charset1, -1);
        gtk_list_store_set (charlist->store, &iter1, 0, charset0, -1);
        gtk_tree_selection_select_iter (charlist->select, &iter1);
        g_free (charset0);
        g_free (charset1);
        break;
      }
  /* ja:OKボタン */
  gtk_widget_set_sensitive (charlist->ok_button, TRUE);
}


/* ja:エントリーでリターンが押された */
static void
misc_charlist_activate (GtkWidget    *widget,
                        MiscCharlist *charlist)
{
  if (GTK_WIDGET_IS_SENSITIVE (charlist->ok_button))
    misc_charlist_clicked_new (widget, charlist);
}


/* ja:エントリーが変更された */
static void
misc_charlist_edited (GtkWidget    *widget,
                      MiscCharlist *charlist)
{
  const gchar *text;
  gint i;
  GtkTreeIter iter;

  text = misc_charset_get_charset (MISC_CHARSET (widget));
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter, NULL, i); i++)
    {
      gchar *charset;

      gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                    &iter, 0, &charset, -1);
      if (!g_strcasecmp (charset, text))
        {
          gtk_widget_set_sensitive (charlist->new_button, FALSE);
          g_free (charset);
          return;
        }
      g_free (charset);
    }
  gtk_widget_set_sensitive (charlist->new_button, charuty_is_valid (text));
}


static void
misc_charlist_close (MiscCharlist *charset)
{
  if (GTK_WIDGET_VISIBLE (charset->cancel_button)
                            && GTK_WIDGET_SENSITIVE (charset->cancel_button))
    g_signal_emit_by_name (G_OBJECT (charset->cancel_button), "clicked");
}


/* ja:閉じるボタンが押された */
static gboolean
misc_charlist_delete (GtkWidget *widget,
                      GdkEvent  *event,
                      gpointer   user_data)
{
  if (GTK_WIDGET_VISIBLE (MISC_CHARLIST (widget)->cancel_button)
            && GTK_WIDGET_SENSITIVE (MISC_CHARLIST (widget)->cancel_button))
    g_signal_emit_by_name
                (G_OBJECT (MISC_CHARLIST (widget)->cancel_button), "clicked");
  return TRUE;
}


static void
misc_charlist_init (MiscCharlist *charlist)
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkWidget *scroll, *hbox, *vbox;

  /* ja:メインウインドウ */
  gtk_window_set_title (GTK_WINDOW (charlist), _("Character Encoding"));
  gtk_window_set_default_size (GTK_WINDOW (charlist), 320, 240);
  g_signal_connect (G_OBJECT (charlist), "delete-event",
                                    G_CALLBACK (misc_charlist_delete), NULL);
  /* ja:リストボックス */
  charlist->store = gtk_list_store_new (1, G_TYPE_STRING);
  charlist->tview = gtk_tree_view_new_with_model
                                            (GTK_TREE_MODEL (charlist->store));
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (charlist->tview), FALSE);
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes
            (_("Registered Character Encoding"), renderer, "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (charlist->tview), column);
  charlist->select = gtk_tree_view_get_selection
                                            (GTK_TREE_VIEW (charlist->tview));
  g_signal_connect (G_OBJECT (charlist->select),
                    "changed", G_CALLBACK (misc_charlist_changed), charlist);
  /* ja:スクロールウインドウ */
  scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scroll), charlist->tview);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  /* ja:コンボボックス */
  charlist->charset = misc_charset_new ();
  g_signal_connect (G_OBJECT (charlist->charset), "activate",
                                G_CALLBACK (misc_charlist_activate), charlist);
  g_signal_connect (G_OBJECT (charlist->charset), "edited",
                                G_CALLBACK (misc_charlist_edited), charlist);
  /* ja:ボタン */
  charlist->new_button = gtk_button_new_with_mnemonic (_("_New"));
  charlist->remove_button = gtk_button_new_with_mnemonic (_("De_lete"));
  charlist->up_button = gtk_button_new_with_mnemonic (_("_Up"));
  charlist->down_button = gtk_button_new_with_mnemonic (_("_Down"));
  charlist->ok_button = gtk_button_new_with_label (_("OK"));
  charlist->cancel_button = gtk_button_new_with_label (_("Cancel"));
  g_signal_connect (G_OBJECT (charlist->new_button), "clicked",
                        G_CALLBACK (misc_charlist_clicked_new), charlist);
  g_signal_connect (G_OBJECT (charlist->remove_button), "clicked",
                        G_CALLBACK (misc_charlist_clicked_remove), charlist);
  g_signal_connect (G_OBJECT (charlist->up_button), "clicked",
                        G_CALLBACK (misc_charlist_clicked_up), charlist);
  g_signal_connect (G_OBJECT (charlist->down_button), "clicked",
                        G_CALLBACK (misc_charlist_clicked_down), charlist);
  /* ja:ボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), scroll, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), charlist->charset, FALSE, FALSE, 0);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox), charlist->new_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox),
                                    charlist->remove_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), charlist->up_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), charlist->down_button, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), charlist->cancel_button, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), charlist->ok_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (charlist), vbox);

  /* ja:表示 */
  gtk_window_set_policy (GTK_WINDOW (charlist), FALSE, TRUE, FALSE);
  gtk_widget_show_all (vbox);
  gtk_widget_set_sensitive (charlist->new_button, FALSE);
  gtk_widget_set_sensitive (charlist->remove_button, FALSE);
  gtk_widget_set_sensitive (charlist->up_button, FALSE);
  gtk_widget_set_sensitive (charlist->down_button, FALSE);
  gtk_widget_set_sensitive (charlist->ok_button, FALSE);
  gtk_widget_grab_focus (charlist->cancel_button);
}


/******************************************************************************
*                                                                             *
* ja:文字符号化方式リスト関数群                                               *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget*
misc_charlist_new (void)
{
  return GTK_WIDGET (g_object_new (MISC_TYPE_CHARLIST, NULL));
}


/*  ja:文字符号化方式のリストを取得する
    charlist,ウィジェット
         RET,文字符号化方式のリスト                                         */
gchar *
misc_charlist_get_charlist (MiscCharlist *charlist)
{
  gchar *list = NULL;
  gint i;
  GtkTreeIter iter;

  if (!charlist)
    return NULL;
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charlist->store),
                                                        &iter, NULL, i); i++)
    {
      gchar *charset, *tmp;

      gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                    &iter, 0, &charset, -1);
      tmp = list ? g_strjoin (",", list, charset, NULL) : g_strdup (charset);
      g_free (charset);
      g_free (list);
      list = tmp;
    }
  return list;
}


/*  ja:文字符号化方式のリストを設定する
    charlist,ウィジェット
         RET,TRUE:正しい文字符号化方式,FALSE:不正な文字符号化方式           */
gboolean
misc_charlist_set_charlist (MiscCharlist *charlist,
                            const gchar  *list)
{
  gboolean result = TRUE;
  gchar **charset;
  gint i;

  if (!charlist || !list)
    return FALSE;
  gtk_list_store_clear (charlist->store);
  charset = g_strsplit (list, ",", G_MAXINT);
  for (i = 0; charset[i]; i++)
    if (charuty_is_valid (charset[i]))
      {
        GtkTreeIter iter;

        gtk_list_store_append (charlist->store, &iter);
        gtk_list_store_set (charlist->store, &iter, 0, charset[i], -1);
      }
    else
      {
        result = FALSE;
      }
  g_strfreev (charset);
  return result;
}


/*  ja:文字符号化方式を取得する
    charlist,ウィジェット
         RET,文字符号化方式                                                 */
/*gchar *
misc_charlist_get_charset (MiscCharlist *charlist)
{
  return charlist ? misc_charset_get_charset (MISC_CHARSET (charlist->charset))
                  : NULL;
}*/


/*  ja:文字符号化方式を設定する
    charlist,ウィジェット
         RET,TRUE:正しい文字符号化方式,FALSE:不正な文字符号化方式           */
gboolean
misc_charlist_set_charset (MiscCharlist *charlist,
                           const gchar  *charset)
{
  if (!charlist)
    return FALSE;
  misc_charset_set_charset (MISC_CHARSET (charlist->charset), charset);
  return TRUE;
}


/*  ja:文字符号化方式の候補を取得する
    charlist,ウィジェット
         RET,文字符号化方式の候補                                           */
/*gchar *
misc_charlist_get_candidate (MiscCharlist *charlist)
{
  return charlist
        ? misc_charset_get_candidate (MISC_CHARSET (charlist->charset)) : NULL;
}*/


/*  ja:文字符号化方式の候補を追加する
     charlist,ウィジェット
    candidate,文字符号化方式のリスト
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
misc_charlist_add_candidate (MiscCharlist *charlist,
                             const gchar  *candidate)
{
  gboolean result = TRUE;
  gchar **charset;
  gint i;

  if (!charlist || !candidate)
    return FALSE;
  charset = g_strsplit (candidate, ",", G_MAXINT);
  for (i = 0; charset[i]; i++)
    {
      gboolean exist = FALSE;
      gint j;
      GtkTreeIter iter;

      for (j = 0; gtk_tree_model_iter_nth_child
                    (GTK_TREE_MODEL (charlist->store), &iter, NULL, j); j++)
        {
          gchar *str;

          gtk_tree_model_get (GTK_TREE_MODEL (charlist->store),
                                                        &iter, 0, &str, -1);
          if (g_strcasecmp (charset[i], str) == 0)
            exist = TRUE;
          g_free (str);
        }
      if (!exist && !misc_charset_add_candidate
                                (MISC_CHARSET (charlist->charset), charset[i]))
        result = FALSE;
    }
  g_strfreev (charset);
  return result;
}


/*  ja:文字符号化方式の候補を削除する
     charlist,ウィジェット
    candidate,文字符号化方式のリスト,NULL:すべて削除
          RET,TRUE:正常終了,FALSE:エラー                                    */
/*gboolean
misc_charlist_remove_candidate (MiscCharlist *charlist,
                                const gchar  *candidate)
{
  return charlist ? misc_charset_remove_candidate
                        (MISC_CHARSET (charlist->charset), candidate) : FALSE;
}*/
