/*
    history
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "fileio.h"
#include "misc_history.h"


static void misc_history_class_init (MiscHistoryClass *klass);
static void misc_history_init       (MiscHistory      *history);
static void misc_history_dispose    (GObject          *object);


static GObjectClass *parent_class = NULL;


enum
{
  ACTIVATE_SIGNAL,
  LAST_SIGNAL
};


struct _MiscHistoryItem
{
  gchar *file;
  GtkWidget *menu_item;
};


static gint misc_history_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
GType
misc_history_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (MiscHistoryClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)misc_history_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (MiscHistory),
        0,              /* n_preallocs */
        (GInstanceInitFunc)misc_history_init,
      };

      type = g_type_register_static (G_TYPE_OBJECT, "MiscHistory", &info, 0);
    }

  return type;
}


static void
misc_history_class_init (MiscHistoryClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass *) klass;

  object_class->dispose = misc_history_dispose;

  misc_history_signals[ACTIVATE_SIGNAL]
                = g_signal_new ("activate",
                                G_TYPE_FROM_CLASS (object_class),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (MiscHistoryClass, activate),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__POINTER,
                                G_TYPE_NONE, 1,
                                G_TYPE_POINTER);
}


/*  ja:新規作成
    RET,オブジェクト                                                        */
static void
misc_history_init (MiscHistory *history)
{
  history->num             = MISC_HISTORY_NUM_DEF;
  history->position        = 0;
  history->handler_destroy = 0;
  history->menu_shell      = NULL;
  history->menu_item_head  = NULL;
  history->menu_item_foot  = NULL;
  history->glist           = NULL;
}


static void
misc_history_remove (MiscHistory *history)
{

  if (history && history->glist)
    {
      MiscHistoryItem *history_item;

      history_item = (g_list_last (history->glist))->data;
      if (history_item)
        {
          if (history->menu_shell && history_item->menu_item)
            gtk_container_remove (GTK_CONTAINER (history->menu_shell),
                                                    history_item->menu_item);
          history->glist = g_list_remove (history->glist, history_item);
          g_free (history_item->file);
          g_free (history_item);
        }
      if (history->menu_shell && !history->glist)
        {
          if (history->menu_item_head)
            {
              gtk_container_remove (GTK_CONTAINER (history->menu_shell),
                                                    history->menu_item_head);
              history->menu_item_head = NULL;
            }
          if (history->menu_item_foot)
            {
              gtk_container_remove (GTK_CONTAINER (history->menu_shell),
                                                    history->menu_item_foot);
              history->menu_item_foot = NULL;
            }
          if (GTK_IS_MENU (history->menu_shell))
            {
              GList *glist;

              glist = gtk_container_get_children
                                        (GTK_CONTAINER (history->menu_shell));
              if (glist)
                {
                  g_free (glist);
                }
              else
                {
                  GtkWidget *menu_item;

                  menu_item = gtk_menu_get_attach_widget
                                            (GTK_MENU (history->menu_shell));
                  if (menu_item)
                    gtk_widget_set_sensitive (menu_item, FALSE);
                }
            }
        }
    }
}


static void
misc_history_dispose (GObject *object)
{
  MiscHistory *history = MISC_HISTORY (object);

  while (history->glist)
    misc_history_remove (history);
  if (history->menu_shell)
    {
      if (history->handler_destroy)
        g_signal_handler_disconnect (G_OBJECT (history->menu_shell),
                                                    history->handler_destroy);
      history->menu_shell = NULL;
    }
  if (G_OBJECT_CLASS (parent_class)->dispose)
    G_OBJECT_CLASS (parent_class)->dispose (object);
}


static void
misc_history_activate (GtkWidget   *widget,
                       MiscHistory *history)
{
  MiscHistoryItem *history_item;

  history_item = g_object_get_data (G_OBJECT (widget), "user_data");
  g_signal_emit (history,
                 misc_history_signals[ACTIVATE_SIGNAL],
                 0,
                 history_item->file);
}


static void
misc_history_destroy_menu_item (GtkWidget *widget,
                                gpointer   user_data)
{
  MiscHistoryItem *history_item;

  history_item = g_object_get_data (G_OBJECT (widget), "user_data");
  history_item->menu_item = NULL;
}


static void
misc_history_destroy_menu_shell (GtkWidget   *widget,
                                 MiscHistory *history)
{
  if (history && history->menu_shell == widget)
    {
      history->menu_item_head = NULL;
      history->menu_item_foot = NULL;
      history->menu_shell = NULL;
    }
}


/******************************************************************************
*                                                                             *
* ja:ウインドウメニュー関数群                                                 *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,オブジェクト                                                        */
GObject*
misc_history_new (void)
{
  return g_object_new (MISC_TYPE_HISTORY, NULL);
}


/*  ja:メニューにファイルを追加する
    history,オブジェクト
       file,ファイル                                                        */
void
misc_history_add_file (MiscHistory *history,
                       const gchar *file)
{
  if (history && file && history->num > 0)
    {
      gint i;
      MiscHistoryItem *history_item;

      history_item = g_malloc0 (sizeof (MiscHistoryItem));
      history_item->file = fileio_get_full_path (file);
      if (history->menu_shell)
        {
          gchar *label;
          gint length;
          GList *glist;
          GtkWidget *menu_item;

          glist = gtk_container_get_children
                                        (GTK_CONTAINER (history->menu_shell));
          length = g_list_length (glist);
          label = g_filename_to_utf8 (history_item->file, -1,
                                                            NULL, NULL, NULL);
          history_item->menu_item = gtk_menu_item_new_with_label (label);
          g_free (label);
          g_signal_connect (G_OBJECT (history_item->menu_item), "activate",
                            G_CALLBACK (misc_history_activate), history);
          g_signal_connect (G_OBJECT (history_item->menu_item), "destroy",
                            G_CALLBACK (misc_history_destroy_menu_item), NULL);
          g_object_set_data (G_OBJECT (history_item->menu_item), "user_data",
                                                                history_item);
          gtk_widget_show (history_item->menu_item);
          if (history->glist)
            {
              menu_item = ((MiscHistoryItem *)
                            (g_list_first (history->glist))->data)->menu_item;
              for (i = 0; i < length; i++)
                if (g_list_nth_data (glist, i) == menu_item)
                  gtk_menu_shell_insert (GTK_MENU_SHELL (history->menu_shell),
                                                history_item->menu_item, i);
            }
          else
            {
              gint position;

              position = 0 <= history->position
                        && history->position < length ? history->position : -1;
              if (length > 0)
                {
                  menu_item = g_list_nth_data (glist, position);
                  if (position >= 0 && !GTK_IS_SEPARATOR_MENU_ITEM (menu_item)
                                    && gtk_bin_get_child (GTK_BIN (menu_item)))
                    {
                      history->menu_item_foot = gtk_separator_menu_item_new ();
                      gtk_menu_shell_insert
                                        (GTK_MENU_SHELL (history->menu_shell),
                                            history->menu_item_foot, position);
                    }
                  menu_item = g_list_nth_data (glist,
                                    (position < 0 ? length : position) - 1);
                  if (position != 0 && !GTK_IS_SEPARATOR_MENU_ITEM (menu_item)
                                    && gtk_bin_get_child (GTK_BIN (menu_item)))
                    {
                      history->menu_item_head = gtk_separator_menu_item_new ();
                      gtk_menu_shell_insert
                                        (GTK_MENU_SHELL (history->menu_shell),
                                            history->menu_item_head, position);
                      if (position > 0)
                        position++;
                    }
                }
              if (position < 0)
                gtk_menu_shell_append (GTK_MENU_SHELL (history->menu_shell),
                                                    history_item->menu_item);
              else if (position == 0)
                gtk_menu_shell_prepend (GTK_MENU_SHELL (history->menu_shell),
                                                    history_item->menu_item);
              else
                gtk_menu_shell_insert (GTK_MENU_SHELL (history->menu_shell),
                                            history_item->menu_item, position);
              if (GTK_IS_MENU (history->menu_shell))
                {
                  GtkWidget *menu_item_attach;

                  menu_item_attach = gtk_menu_get_attach_widget
                                            (GTK_MENU (history->menu_shell));
                  if (menu_item_attach)
                    gtk_widget_set_sensitive (menu_item_attach, TRUE);
                }
            }
          g_list_free (glist);
        }
      history->glist = g_list_prepend (history->glist, history_item);
      for (i = g_list_length (history->glist) - 1; i > 0; i--)
        {
          MiscHistoryItem *history_item_nth;

          history_item_nth = g_list_nth_data (history->glist, i);
          if (g_strfilecmp (history_item->file, history_item_nth->file) == 0)
            {
              if (history->menu_shell && history_item_nth->menu_item)
                gtk_container_remove (GTK_CONTAINER (history->menu_shell),
                                                history_item_nth->menu_item);
              history->glist = g_list_remove (history->glist,
                                                            history_item_nth);
              g_free (history_item_nth->file);
              g_free (history_item_nth);
            }
        }
      while (g_list_length (history->glist) > history->num)
        misc_history_remove (history);
    }
}


/*  ja:メニューのファイルを取得する
    history,オブジェクト
        RET,ファイルのリスト                                                */
GList *
misc_history_get_file (MiscHistory *history)
{
  GList *glist = NULL;

  if (history)
    {
      gint i;

      for (i = g_list_length (history->glist) - 1; i >= 0; i--)
        {
          MiscHistoryItem *history_item;

          history_item = g_list_nth_data (history->glist, i);
          glist = g_list_prepend (glist, history_item->file);
        }
    }
  return glist;
}


/*  ja:メニューを設定する
       history,オブジェクト
    menu_shell,メニュー
      position,位置(0...,-1:末尾)                                           */
void
misc_history_set_menu (MiscHistory *history,
                       GtkWidget   *menu_shell,
                       gint         position)
{
  if (history)
    {
      gint i;

      if (history->menu_shell)
        {
          if (history->handler_destroy)
            {
              g_signal_handler_disconnect (G_OBJECT (history->menu_shell),
                                                    history->handler_destroy);
              history->handler_destroy = 0;
            }
          if (history->glist)
            for (i = g_list_length (history->glist) - 1; i >= 0; i--)
              {
                MiscHistoryItem *history_item;

                history_item = g_list_nth_data (history->glist, i);
                if (history_item->menu_item)
                  gtk_container_remove (GTK_CONTAINER (history->menu_shell),
                                                    history_item->menu_item);
              }
        }
      history->menu_shell = menu_shell;
      history->position = position;
      if (history->menu_shell)
        {
          GList *glist;

          history->handler_destroy
                        = g_signal_connect (G_OBJECT (menu_shell), "destroy",
                        G_CALLBACK (misc_history_destroy_menu_shell), history);
          glist = history->glist;
          history->glist = NULL;
          for (i = g_list_length (glist) - 1; i >= 0; i--)
            {
              MiscHistoryItem *history_item;

              history_item = g_list_nth_data (glist, i);
              misc_history_add_file (history, history_item->file);
              g_free (history_item->file);
              g_free (history_item);
            }
          g_list_free (glist);
          if (!history->glist && GTK_IS_MENU (history->menu_shell))
            {
              glist = gtk_container_get_children (GTK_CONTAINER (menu_shell));
              if (g_list_length (glist) <= 0)
                {
                  GtkWidget *menu_item;

                  menu_item = gtk_menu_get_attach_widget
                                                    (GTK_MENU (menu_shell));
                  if (menu_item)
                    gtk_widget_set_sensitive (menu_item, FALSE);
                }
              g_list_free (glist);
            }
        }
    }
}


/*  ja:メニューの数を取得する
    history,オブジェクト
        RET,メニューの数                                                    */
gint
misc_history_get_num (MiscHistory *history)
{
  return history ? history->num : 0;
}


/*  ja:メニューの数を設定する
    history,オブジェクト
        num,メニューの数
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
misc_history_set_num (MiscHistory *history,
                      const gint   num)
{
  if (history)
    {
      history->num = num > MISC_HISTORY_NUM_MAX ? MISC_HISTORY_NUM_MAX
                    : num < MISC_HISTORY_NUM_MIN ? MISC_HISTORY_NUM_MIN : num;
      while (g_list_length (history->glist) > history->num)
        misc_history_remove (history);
    }
  return history
                && MISC_HISTORY_NUM_MIN <= num  && num <= MISC_HISTORY_NUM_MAX;
}
