/*
    orzcharlist
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "orzcharlist.h"
#include "orzcharset.h"
#include "misc/charuty.h"
#include <gdk/gdkkeysyms.h>


static void orz_charlist_class_init (OrzCharlistClass *klass);
static void orz_charlist_init       (OrzCharlist      *charlist);
static void orz_charlist_escape     (OrzCharlist      *charlist);


static GtkDialogClass *parent_class = NULL;


enum
{
  ESCAPE_SIGNAL,
  LAST_SIGNAL
};


enum
{
  TARGET_GTK_TREE_MODEL_ROW
};


static gint orz_charlist_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
orz_charlist_get_type (void)
{
  static GType orz_charlist_type = 0;

  if (!orz_charlist_type)
    {
      const static GTypeInfo charlist_info =
      {
        sizeof (OrzCharlistClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)orz_charlist_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (OrzCharlist),
        0,              /* n_preallocs */
        (GInstanceInitFunc)orz_charlist_init,
      };

      orz_charlist_type = g_type_register_static (GTK_TYPE_DIALOG,
                                            "OrzCharlist", &charlist_info, 0);
    }

  return orz_charlist_type;
}


static void
orz_charlist_class_init (OrzCharlistClass *klass)
{
  GtkBindingSet *binding_set;

  parent_class = g_type_class_peek_parent (klass);

  klass->escape = orz_charlist_escape;

  orz_charlist_signals[ESCAPE_SIGNAL]
                = g_signal_new ("escape",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                                G_STRUCT_OFFSET (OrzCharlistClass, escape),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);

  binding_set = gtk_binding_set_by_class (klass);
  gtk_binding_entry_add_signal (binding_set, GDK_Escape, 0, "escape", 0);
}


/* ja:リストボックスの選択 */
static void
orz_charlist_changed (GtkTreeSelection *select,
                      OrzCharlist      *charlist)
{
  gint i, pos = -1;
  GtkTreeIter iter;
  GtkTreeModel *model;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (charlist->tview));
  for (i = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (select, &iter))
      {
        pos = i;
        break;
      }
  if (pos >= 0)
    {
      gint count;

      count = gtk_tree_model_iter_n_children (model, NULL);
      gtk_widget_set_sensitive (charlist->remove_button, count > 1);
      gtk_widget_set_sensitive (charlist->up_button, pos > 0);
      gtk_widget_set_sensitive (charlist->down_button, pos < count - 1);
    }
  else
    {
      gtk_widget_set_sensitive (charlist->remove_button, FALSE);
      gtk_widget_set_sensitive (charlist->up_button, FALSE);
      gtk_widget_set_sensitive (charlist->down_button, FALSE);
    }
}


/* ja:追加 */
static void
orz_charlist_clicked_add (GtkWidget   *widget,
                          OrzCharlist *charlist)
{
  gchar *charset;
  gint i;
  GtkListStore *store;
  GtkTreeIter iter;
  GtkTreeSelection *select;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (charlist->tview)));
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (charlist->tview));
  /* ja:挿入 */
  charset = orz_charset_get_charset (ORZ_CHARSET (charlist->charset));
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (select, &iter))
      {
        i++;
        break;
      }
  gtk_list_store_insert (store, &iter, i);
  gtk_list_store_set (store, &iter, 0, g_strdup (charset), -1);
  gtk_tree_selection_select_iter (select, &iter);
  /* ja:コンボ */
  orz_charset_remove_candidate (ORZ_CHARSET (charlist->charset), charset);
  g_free (charset);
  orz_charset_set_charset (ORZ_CHARSET (charlist->charset), NULL);
  /* ja:OKボタン */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (charlist),
                                                        GTK_RESPONSE_OK, TRUE);
}


/* ja:削除 */
static void
orz_charlist_clicked_remove (GtkWidget   *widget,
                             OrzCharlist *charlist)
{
  gint i;
  GtkListStore *store;
  GtkTreeIter iter;
  GtkTreeSelection *select;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (charlist->tview)));
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (charlist->tview));
  /* ja:削除 */
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (select, &iter))
      {
        gchar *charset;

        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 0, &charset, -1);
        gtk_list_store_remove (store, &iter);
        /* ja:コンボ */
        orz_charset_add_candidate (ORZ_CHARSET (charlist->charset), charset);
        orz_charset_set_charset (ORZ_CHARSET (charlist->charset), charset);
        g_free (charset);
      }
  /* ja:OKボタン */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (charlist),
                                                        GTK_RESPONSE_OK, TRUE);
}


/* ja:上へ */
static void
orz_charlist_clicked_up (GtkWidget   *widget,
                         OrzCharlist *charlist)
{
  gint i;
  GtkListStore *store;
  GtkTreeIter iter0;
  GtkTreeSelection *select;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (charlist->tview)));
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (charlist->tview));
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter0, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (select, &iter0))
      {
        gchar *charset0, *charset1;
        GtkTreeIter iter1;

        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter0, 0, &charset0, -1);
        gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter1, NULL, i - 1);
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter1, 0, &charset1, -1);
        gtk_list_store_set (store, &iter0, 0, charset1, -1);
        gtk_list_store_set (store, &iter1, 0, charset0, -1);
        gtk_tree_selection_select_iter (select, &iter1);
        g_free (charset0);
        g_free (charset1);
        break;
      }
  /* ja:OKボタン */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (charlist),
                                                        GTK_RESPONSE_OK, TRUE);
}


/* ja:下へ */
static void
orz_charlist_clicked_down (GtkWidget   *widget,
                           OrzCharlist *charlist)
{
  gint i;
  GtkListStore *store;
  GtkTreeIter iter0;
  GtkTreeSelection *select;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (charlist->tview)));
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (charlist->tview));
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter0, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (select, &iter0))
      {
        gchar *charset0, *charset1;
        GtkTreeIter iter1;

        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter0, 0, &charset0, -1);
        gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter1, NULL, i + 1);
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter1, 0, &charset1, -1);
        gtk_list_store_set (store, &iter0, 0, charset1, -1);
        gtk_list_store_set (store, &iter1, 0, charset0, -1);
        gtk_tree_selection_select_iter (select, &iter1);
        g_free (charset0);
        g_free (charset1);
        break;
      }
  /* ja:OKボタン */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (charlist),
                                                        GTK_RESPONSE_OK, TRUE);
}


/* ja:エントリーでリターンが押された */
static void
orz_charlist_activate (GtkWidget   *widget,
                       OrzCharlist *charlist)
{
  if (GTK_WIDGET_IS_SENSITIVE (charlist->ok_button))
    orz_charlist_clicked_add (widget, charlist);
}


/* ja:エントリーが変更された */
static void
orz_charlist_edited (GtkWidget   *widget,
                     OrzCharlist *charlist)
{
  const gchar *text;
  gint i;
  GtkTreeIter iter;
  GtkTreeModel *model;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (charlist->tview));
  text = orz_charset_get_charset (ORZ_CHARSET (widget));
  for (i = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
    {
      gchar *charset;

      gtk_tree_model_get (model, &iter, 0, &charset, -1);
      if (g_ascii_strcasecmp (charset, text) == 0)
        {
          gtk_widget_set_sensitive (charlist->add_button, FALSE);
          g_free (charset);
          return;
        }
      g_free (charset);
    }
  gtk_widget_set_sensitive (charlist->add_button, charuty_is_valid (text));
}


/* ja:ドロップされたとき */
static void
orz_charlist_dialog_drag_end (GtkWidget      *widget,
                              GdkDragContext *drag_context,
                              OrzCharlist    *charlist)
{
  /* ja:OKボタン */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (charlist),
                                                        GTK_RESPONSE_OK, TRUE);
}


static void
orz_charlist_escape (OrzCharlist *charlist)
{
  if (GTK_WIDGET_VISIBLE (charlist->cancel_button)
                            && GTK_WIDGET_SENSITIVE (charlist->cancel_button))
    g_signal_emit_by_name (G_OBJECT (charlist->cancel_button), "clicked");
}


/* ja:閉じるボタンが押された */
static gboolean
orz_charlist_delete (GtkWidget *widget,
                     GdkEvent  *event,
                     gpointer   user_data)
{
  if (GTK_WIDGET_VISIBLE (ORZ_CHARLIST (widget)->cancel_button)
                && GTK_WIDGET_SENSITIVE (ORZ_CHARLIST (widget)->cancel_button))
    g_signal_emit_by_name
                (G_OBJECT (ORZ_CHARLIST (widget)->cancel_button), "clicked");
  return TRUE;
}


static void
orz_charlist_init (OrzCharlist *charlist)
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkWidget *scroll, *hbox, *vbox;
  GtkTargetEntry dnd_entries[1] = {{"GTK_TREE_MODEL_ROW",
    GTK_TARGET_SAME_APP | GTK_TARGET_SAME_WIDGET, TARGET_GTK_TREE_MODEL_ROW}};

  /* ja:メインウインドウ */
  gtk_window_set_title (GTK_WINDOW (charlist), _("Character Encoding"));
  gtk_window_set_default_size (GTK_WINDOW (charlist), 320, 240);
  g_signal_connect (G_OBJECT (charlist), "delete-event",
                                    G_CALLBACK (orz_charlist_delete), NULL);
  /* ja:ボタン */
#ifdef G_OS_WIN32
  charlist->ok_button = gtk_dialog_add_button (GTK_DIALOG (charlist),
                                        GTK_STOCK_OK, GTK_RESPONSE_OK);
  charlist->cancel_button = gtk_dialog_add_button (GTK_DIALOG (charlist),
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
#else /* not G_OS_WIN32 */
  charlist->cancel_button = gtk_dialog_add_button (GTK_DIALOG (charlist),
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
  charlist->ok_button = gtk_dialog_add_button (GTK_DIALOG (charlist),
                                        GTK_STOCK_OK, GTK_RESPONSE_OK);
#endif /* not G_OS_WIN32 */
  /* ja:リストボックス */
  charlist->tview = gtk_tree_view_new_with_model
                    (GTK_TREE_MODEL (gtk_list_store_new (1, G_TYPE_STRING)));
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (charlist->tview), FALSE);
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes
            (_("Registered Character Encoding"), renderer, "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (charlist->tview), column);
  g_signal_connect (G_OBJECT (gtk_tree_view_get_selection
                                            (GTK_TREE_VIEW (charlist->tview))),
                    "changed", G_CALLBACK (orz_charlist_changed), charlist);
  /* en:Drag & Drop */
  gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW (charlist->tview),
                                            dnd_entries, 1, GDK_ACTION_MOVE);
  gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (charlist->tview),
                            GDK_BUTTON1_MASK, dnd_entries, 1, GDK_ACTION_MOVE);
  g_signal_connect (G_OBJECT (charlist->tview), "drag-end",
                        G_CALLBACK (orz_charlist_dialog_drag_end), charlist);
  /* ja:スクロールウインドウ */
  scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scroll), charlist->tview);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  /* ja:コンボボックス */
  charlist->charset = orz_charset_new ();
  g_signal_connect (G_OBJECT (charlist->charset), "activate",
                                G_CALLBACK (orz_charlist_activate), charlist);
  g_signal_connect (G_OBJECT (charlist->charset), "edited",
                                G_CALLBACK (orz_charlist_edited), charlist);
  /* ja:ボタン */
  charlist->add_button = gtk_button_new_from_stock (GTK_STOCK_ADD);
  charlist->remove_button = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
  charlist->up_button = gtk_button_new_from_stock (GTK_STOCK_GO_UP);
  charlist->down_button = gtk_button_new_from_stock (GTK_STOCK_GO_DOWN);
  g_signal_connect (G_OBJECT (charlist->add_button), "clicked",
                        G_CALLBACK (orz_charlist_clicked_add), charlist);
  g_signal_connect (G_OBJECT (charlist->remove_button), "clicked",
                        G_CALLBACK (orz_charlist_clicked_remove), charlist);
  g_signal_connect (G_OBJECT (charlist->up_button), "clicked",
                        G_CALLBACK (orz_charlist_clicked_up), charlist);
  g_signal_connect (G_OBJECT (charlist->down_button), "clicked",
                        G_CALLBACK (orz_charlist_clicked_down), charlist);
  /* ja:ボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), scroll, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), charlist->charset, FALSE, FALSE, 0);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox), charlist->add_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox),
                                    charlist->remove_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), charlist->up_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), charlist->down_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (charlist)->vbox), vbox);

  /* ja:表示 */
  gtk_widget_set_sensitive (charlist->add_button, FALSE);
  gtk_widget_set_sensitive (charlist->remove_button, FALSE);
  gtk_widget_set_sensitive (charlist->up_button, FALSE);
  gtk_widget_set_sensitive (charlist->down_button, FALSE);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (charlist),
                                                    GTK_RESPONSE_OK, FALSE);
  gtk_widget_show_all (vbox);
}


/******************************************************************************
*                                                                             *
* ja:文字符号化方式リスト関数群                                               *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
orz_charlist_new (void)
{
  return GTK_WIDGET (g_object_new (ORZ_TYPE_CHARLIST, NULL));
}


/*  ja:文字符号化方式のリストを取得する
    charlist,ウィジェット
         RET,文字符号化方式のリスト                                         */
gchar *
orz_charlist_get_charlist (OrzCharlist *charlist)
{
  gchar *list = NULL;
  gint i;
  GtkTreeIter iter;
  GtkTreeModel *model;

  if (!charlist)
    return NULL;
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (charlist->tview));
  for (i = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
    {
      gchar *charset, *tmp;

      gtk_tree_model_get (model, &iter, 0, &charset, -1);
      tmp = list ? g_strconcat (list, ",", charset, NULL) : g_strdup (charset);
      g_free (charset);
      g_free (list);
      list = tmp;
    }
  return list;
}


/*  ja:文字符号化方式のリストを設定する
    charlist,ウィジェット
         RET,TRUE:正しい文字符号化方式,FALSE:不正な文字符号化方式           */
gboolean
orz_charlist_set_charlist (OrzCharlist *charlist,
                           const gchar *list)
{
  gboolean result = TRUE;
  gchar **charset;
  gint i;
  GtkListStore *store;

  if (!charlist || !list)
    return FALSE;
  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (charlist->tview)));
  gtk_list_store_clear (store);
  charset = g_strsplit (list, ",", 0);
  for (i = 0; charset[i]; i++)
    if (charuty_is_valid (charset[i]))
      {
        GtkTreeIter iter;

        gtk_list_store_append (store, &iter);
        gtk_list_store_set (store, &iter, 0, charset[i], -1);
      }
    else
      {
        result = FALSE;
      }
  g_strfreev (charset);
  return result;
}


/*  ja:文字符号化方式を取得する
    charlist,ウィジェット
         RET,文字符号化方式                                                 */
gchar *
orz_charlist_get_charset (OrzCharlist *charlist)
{
  return charlist ? orz_charset_get_charset (ORZ_CHARSET (charlist->charset))
                  : NULL;
}


/*  ja:文字符号化方式を設定する
    charlist,ウィジェット
         RET,TRUE:正しい文字符号化方式,FALSE:不正な文字符号化方式           */
gboolean
orz_charlist_set_charset (OrzCharlist *charlist,
                          const gchar *charset)
{
  if (!charlist)
    return FALSE;
  orz_charset_set_charset (ORZ_CHARSET (charlist->charset), charset);
  return TRUE;
}


/*  ja:文字符号化方式の候補を取得する
    charlist,ウィジェット
         RET,文字符号化方式の候補                                           */
gchar *
orz_charlist_get_candidate (OrzCharlist *charlist)
{
  return charlist
        ? orz_charset_get_candidate (ORZ_CHARSET (charlist->charset)) : NULL;
}


/*  ja:文字符号化方式の候補を追加する
     charlist,ウィジェット
    candidate,文字符号化方式のリスト
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
orz_charlist_add_candidate (OrzCharlist *charlist,
                            const gchar *candidate)
{
  gboolean result = TRUE;
  gchar **charset;
  gint i;
  GtkTreeModel *model;

  if (!charlist || !candidate)
    return FALSE;
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (charlist->tview));
  charset = g_strsplit (candidate, ",", 0);
  for (i = 0; charset[i]; i++)
    {
      gboolean exist = FALSE;
      gint j;
      GtkTreeIter iter;

      for (j = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, j); j++)
        {
          gchar *str;

          gtk_tree_model_get (model, &iter, 0, &str, -1);
          if (g_ascii_strcasecmp (charset[i], str) == 0)
            exist = TRUE;
          g_free (str);
        }
      if (!exist && !orz_charset_add_candidate
                                (ORZ_CHARSET (charlist->charset), charset[i]))
        result = FALSE;
    }
  g_strfreev (charset);
  return result;
}


/*  ja:文字符号化方式の候補を削除する
     charlist,ウィジェット
    candidate,文字符号化方式のリスト,NULL:すべて削除
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
orz_charlist_remove_candidate (OrzCharlist *charlist,
                               const gchar *candidate)
{
  return charlist ? orz_charset_remove_candidate
                        (ORZ_CHARSET (charlist->charset), candidate) : FALSE;
}
