/*
    gcommon
    copyright (c) 1998-2012 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif /* HAVE_FCNTL_H */
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif /* HAVE_SYS_STAT_H */
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef USE_GTK_EMULATE
# ifdef HAVE_DIRENT_H
#  include <dirent.h>
# endif /* HAVE_DIRENT_H */
# ifdef HAVE_SYS_PARAM_H
#  include <sys/param.h>
# endif /* HAVE_SYS_PARAM_H */
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# endif /* HAVE_SYS_TIME_H */
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# endif /* HAVE_SYS_TIME_H */
# ifdef HAVE_UNISTD_H
#  include <unistd.h>
# endif /* HAVE_UNISTD_H */
#endif /* USE_GTK_EMULATE */
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
# include <shlwapi.h>
# include <shlobj.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
* Byte Order Macros                                                           *
******************************************************************************/
guint16
gcommon_swap_16 (const guint16 val)
{
  return (val >> 8) | (val << 8);
}


guint32
gcommon_swap_32 (const guint32 val)
{
  return ((val & 0x000000ff) << 24) | ((val & 0x0000ff00) << 8)
                    | ((val & 0x00ff0000) >> 8) | ((val & 0xff000000) >> 24);
}


guint64
gcommon_swap_64 (const guint64 val)
{
  guint8 *p, temp;
  guint64 v;

  v = val;
  p = (guint8 *)&v;
  temp = p[0]; p[0] = p[7]; p[7] = temp;
  temp = p[1]; p[1] = p[6]; p[6] = temp;
  temp = p[2]; p[2] = p[5]; p[5] = temp;
  temp = p[3]; p[3] = p[4]; p[4] = temp;
  return v;
}


/******************************************************************************
* Memory Allocation                                                           *
******************************************************************************/
#ifdef USE_GTK_EMULATE
# if ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
static gulonglong g_memory_max_size = 0;
static gulonglong g_memory_used_size = 0;
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */


gpointer
g_malloc (gsize n_bytes)
{
  gpointer mem = NULL;

  if (n_bytes > 0)
    {
      mem = g_try_malloc (n_bytes);
      if (!mem)
        {
          g_fprintf (stderr,
                    "Failed to allocate %"G_GSIZE_FORMAT" bytes\n", n_bytes);
          abort ();
        }
    }
  return mem;
}


gpointer
g_malloc0 (gsize n_bytes)
{
  gpointer mem;

  mem = g_malloc (n_bytes);
  g_memset (mem, 0, n_bytes);
  return mem;
}


gpointer
g_realloc (gpointer mem,
           gsize    n_bytes)
{
  if (n_bytes > 0)
    {
      mem = g_try_realloc (mem, n_bytes);
      if (!mem)
        {
          g_fprintf (stderr,
                    "Failed to allocate %"G_GSIZE_FORMAT" bytes\n", n_bytes);
          abort ();
        }
    }
  else
    {
      g_free (mem);
      mem = NULL;
    }
  return mem;
}


gpointer
g_try_malloc (gsize n_bytes)
{
  gpointer mem = NULL;

  if (n_bytes > 0)
    {
# if defined (G_OS_WIN32)
      MEMORYSTATUSEX ms;

      ms.dwLength = sizeof (MEMORYSTATUSEX);
      GlobalMemoryStatusEx (&ms);
      if (ms.ullAvailPhys >= n_bytes)
        mem = malloc (n_bytes);
# elif defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
      if (g_memory_max_size == 0)
        g_memory_max_size = (gulonglong)sysconf (_SC_PAGESIZE)
                                        * (gulonglong)sysconf (_SC_PHYS_PAGES);
      if (g_memory_max_size >= g_memory_used_size + (gulonglong)n_bytes)
        mem = malloc (n_bytes + sizeof (gsize));
      if (mem)
        {
          g_memory_used_size += n_bytes;
          *(gsize *)mem = n_bytes;
          mem = (gsize *)mem + 1;
        }
# else /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
      mem = malloc (n_bytes);
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
    }
  return mem;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,8,0)
gpointer
g_try_malloc0 (gsize n_bytes)
{
  gpointer mem;

  mem = g_try_malloc (n_bytes);
  g_memset (mem, 0, n_bytes);
  return mem;
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */


#ifdef USE_GTK_EMULATE
gpointer
g_try_realloc (gpointer mem,
               gsize    n_bytes)
{
  gpointer ret = NULL;
# ifdef G_OS_WIN32
  MEMORYSTATUSEX ms;
# endif /* G_OS_WIN32 */

  if (!mem)
    return g_try_malloc (n_bytes);
  if (n_bytes <= 0)
    {
      g_free (mem);
      return NULL;
    }
# if defined (G_OS_WIN32)
  ms.dwLength = sizeof (MEMORYSTATUSEX);
  GlobalMemoryStatusEx (&ms);
  if (ms.ullAvailPhys >= n_bytes)
    ret = realloc (mem, n_bytes);
# elif defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
  if (g_memory_max_size >= g_memory_used_size + (gulonglong)n_bytes)
    ret = realloc ((gsize *)mem - 1, n_bytes + sizeof (gsize));
  if (ret)
    {
      g_memory_used_size = g_memory_used_size - *(gsize *)ret + n_bytes;
      *(gsize *)ret = n_bytes;
      ret = (gsize *)ret + 1;
    }
# else /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
  ret = realloc (mem, n_bytes);
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
  return ret;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,24,0)
gpointer
g_malloc_n (gsize n_blocks,
            gsize n_block_bytes)
{
  return g_malloc (n_blocks * n_block_bytes);
}


gpointer
g_malloc0_n (gsize n_blocks,
             gsize n_block_bytes)
{
  return g_malloc0 (n_blocks * n_block_bytes);
}


gpointer
g_realloc_n (gpointer mem,
             gsize    n_blocks,
             gsize    n_block_bytes)
{
  return g_realloc (mem, n_blocks * n_block_bytes);
}


gpointer
g_try_malloc_n (gsize n_blocks,
                gsize n_block_bytes)
{
  return g_try_malloc (n_blocks * n_block_bytes);
}


gpointer
g_try_malloc0_n (gsize n_blocks,
                 gsize n_block_bytes)
{
  return g_try_malloc0 (n_blocks * n_block_bytes);
}


gpointer
g_try_realloc_n (gpointer mem,
                 gsize    n_blocks,
                 gsize    n_block_bytes)
{
  return g_try_realloc (mem, n_blocks * n_block_bytes);
}
#endif /* not GLIB_CHECK_VERSION(2,24,0) */


#ifdef USE_GTK_EMULATE
void
g_free (gpointer mem)
{
  if (mem)
    {
# if ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
      mem = (gsize *)mem - 1;
      g_memory_used_size -= *(gsize *)mem;
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
      free (mem);
    }
}


gpointer
g_memmove (gpointer      dest,
           gconstpointer src,
           gsize         len)
{
  return dest && src ? memmove (dest, src, len) : NULL;
}


gpointer
g_memdup (gconstpointer mem,
          guint         byte_size)
{
  gpointer ret = NULL;

  if (mem)
    {
      ret = g_malloc (byte_size);
      g_memmove (ret, mem, byte_size);
    }
  return ret;
}
#endif /* USE_GTK_EMULATE */


/******************************************************************************
* String Utility Functions                                                    *
******************************************************************************/
#ifdef USE_GTK_EMULATE
gchar*
g_strdup (const gchar *str)
{
  gchar *s = NULL;

  if (str)
    {
      s = g_malloc ((g_strlen (str) + 1) * sizeof (gchar));
      g_strcpy (s, str);
    }
  return s;
}


gchar *
g_strndup (const gchar *str,
           gsize        n)
{
  gchar *s = NULL;

  if (str)
    {
      s = g_malloc ((n + 1) * sizeof (gchar));
      g_strncpy (s, str, n);
      s[n] = '\0';
    }
  return s;
}


gchar **
g_strdupv (gchar **str_array)
{
  gchar **str = NULL;

  if (str_array)
    {
      gint i;

      for (i = 0; str_array[i]; i++);
      str = g_malloc ((i + 1) * sizeof (gchar *));
      for (i = 0; str_array[i]; i++)
        str[i] = g_strdup (str_array[i]);
      str[i] = NULL;
    }
  return str;
}


gchar *
g_strnfill (gsize length,
            gchar fill_char)
{
  gchar *str;

  str = g_malloc ((length + 1) * sizeof (gchar));
  g_memset (str, fill_char, length * sizeof (gchar));
  str[length] = '\0';
  return str;
}


gchar *
g_stpcpy (gchar       *dest,
          const gchar *src)
{
  gchar *d;
  const gchar *s;

  if (!dest || !src)
    return NULL;
  d = dest;
  s = src;
  do
    *d++ = *s;
  while (*s++ != '\0');
  return d - 1;
}


gchar *
g_strstr_len (const gchar *haystack,
              gssize       haystack_len,
              const gchar *needle)
{
  gsize needle_len;
  const gchar *p;

  if (!haystack || !needle)
    return NULL;
  if (haystack_len < 0)
    haystack_len = g_strlen (haystack);
  needle_len = g_strlen (needle);
  if (needle_len < 1)
    return (gchar *)haystack;
  if (haystack_len < needle_len)
    return NULL;
  for (p = haystack; *p != '\0'; p++)
    if (g_strncmp (p, needle, needle_len) == 0)
      return (gchar *)p;
  return NULL;
}


gchar *
g_strrstr (const gchar *haystack,
           const gchar *needle)
{
  return g_strrstr_len (haystack, -1, needle);
}


gchar *
g_strrstr_len (const gchar *haystack,
               gssize       haystack_len,
               const gchar *needle)
{
  gsize needle_len;
  const gchar *p;

  if (!haystack || !needle)
    return NULL;
  needle_len = g_strlen (needle);
  if (needle_len < 1)
    return (gchar *)haystack;
  if (haystack_len < 0)
    haystack_len = g_strlen (haystack);
  if (haystack_len < needle_len)
    return NULL;
  for (p = haystack + haystack_len; p >= haystack; p--)
    if (g_strncmp (p, needle, needle_len) == 0)
      return (gchar *)p;
  return NULL;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,2,0)
gboolean
g_str_has_prefix (const gchar *str,
                  const gchar *prefix)
{
  return str && prefix && g_strncmp (str, prefix, g_strlen (prefix)) == 0;
}


gboolean
g_str_has_suffix (const gchar *str,
                  const gchar *suffix)
{
  if (str && suffix)
    {
      gsize len_str, len_suffix;

      len_str = g_strlen (str);
      len_suffix = g_strlen (suffix);
      if (len_str > len_suffix)
        return g_strcmp (str + len_str - len_suffix, suffix) == 0;
    }
  return FALSE;
}
#endif /* GLIB_CHECK_VERSION(2,2,0) */


#if ! GLIB_CHECK_VERSION(2,16,0)
int
g_strcmp0 (const char *str1,
           const char *str2)
{
  return g_strcmp (str1, str2);
}
#endif /* GLIB_CHECK_VERSION(2,16,0) */


#ifdef USE_GTK_EMULATE
gsize
g_strlcpy (gchar       *dest,
           const gchar *src,
           gsize        dest_size)
{
  gint i;

  if (!dest || !src)
    return 0;
  for (i = 0; i < dest_size - 1 && src[i] != '\0'; i++)
    dest[i] = src[i];
  while (i < dest_size)
    dest[i++] = '\0';
  return g_strlen (src);
}


gsize g_strlcat (gchar       *dest,
                 const gchar *src,
                 gsize        dest_size)
{
  gsize len;
  gint i, j;

  if (!dest || !src)
    return 0;
  len = g_strlen (dest);
  for (i = len, j = 0; i < dest_size - 1 && src[j] != '\0'; i++, j++)
    dest[i] = src[j];
  while (i < dest_size)
    dest[i++] = '\0';
  return len + g_strlen (src);
}
#endif /* USE_GTK_EMULATE */


#ifdef USE_GTK_EMULATE
gchar *
g_strdup_printf (const gchar *format,
                 ...)
{
  gchar *str;
  va_list args;

  va_start (args, format);
  str = g_strdup_vprintf (format, args);
  va_end (args);
  return str;
}


gchar *
g_strdup_vprintf (const char *format,
                  va_list     args)
{
  gchar *str;

  return g_vasprintf (&str, format, args) >= 0 ? str : NULL;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,4,0)
gint
g_vasprintf (gchar       **string,
             gchar const  *format,
             va_list       args)
{
  if (string)
    {
# ifdef HAVE_VASPRINTF
      gchar *str;

      if (vasprintf (&str, format, args) >= 0)
        {
          *string = g_strdup (str);
          free (str);
        }
      else
        {
          *string = NULL;
        }
# else /* not HAVE_VASPRINTF */
      gchar *str = NULL;
      gsize len, length = 0;

      do
        {
          length += 1024;
          if (length > 16416)
            {
              free (str);
              str = NULL;
              break;
            }
          str = g_realloc (str, length * sizeof (gchar));
#  ifdef _MSC_VER
          len = _vsnprintf (str, length, format, args);
#  else /* not _MSC_VER */
          len = vsnprintf (str, length, format, args);
#  endif /* not _MSC_VER */
        }
      while (len < 0 || length <= len);
      *string = str;
# endif /* not HAVE_VASPRINTF */
    }
  return string && *string ? g_strlen (*string) : -1;
}
#endif /* ! GLIB_CHECK_VERSION(2,4,0) */


#ifdef USE_GTK_EMULATE
gsize
g_printf_string_upper_bound (const gchar *format,
                             va_list      args)
{
  gsize len;
  gchar *str;

  str = g_strdup_vprintf (format, args);
  if (!str)
    return 0;
  len = g_strlen (str) + 1;
  g_free (str);
  return len;
}


gint
g_ascii_digit_value (gchar c)
{
  return g_ascii_isdigit (c) ? c - '0' : -1;
}


gint
g_ascii_xdigit_value (gchar c)
{
  if ('A' <= c && c <= 'F')
    return c - 'A' + 10;
  if ('a' <= c && c <= 'f')
    return c - 'a' + 10;
  return g_ascii_digit_value (c);
}


gint
g_ascii_strcasecmp (const gchar *s1,
                    const gchar *s2)
{
  if (!s1 || !s2)
    return 0;
  while (*s1 && *s2)
    {
      gint c1, c2;

      c1 = (gint)(guchar)g_ascii_tolower (*s1);
      c2 = (gint)(guchar)g_ascii_tolower (*s2);
      if (c1 != c2)
        return c1 - c2;
      s1++;
      s2++;
    }
  return (((gint)(guchar)*s1) - ((gint)(guchar)*s2));
}


gint
g_ascii_strncasecmp (const gchar *s1,
                     const gchar *s2,
                     gsize        n)
{
  if (!s1 || !s2)
    return 0;
  while (n > 0 && *s1 && *s2)
    {
      gint c1, c2;

      n--;
      c1 = (gint)(guchar)g_ascii_tolower (*s1);
      c2 = (gint)(guchar)g_ascii_tolower (*s2);
      if (c1 != c2)
        return c1 - c2;
      s1++;
      s2++;
    }
  return n > 0 ? (((gint)(guchar)*s1) - ((gint)(guchar)*s2)) : 0;
}


gchar *
g_ascii_strup (const gchar *str,
               gssize       len)
{
  gchar *ret = NULL;

  if (str)
    {
      gint i;

      if (len < 0)
        len = g_strlen (str);
      ret = g_malloc ((len + 1) * sizeof (gchar));
      g_memmove (ret, str, len);
      ret[len] = '\0';
      for (i = 0; i < len; i++)
        ret[i] = g_ascii_toupper (ret[i]);
    }
  return ret;
}


gchar *
g_ascii_strdown (const gchar *str,
                 gssize       len)
{
  gchar *ret = NULL;

  if (str)
    {
      gint i;

      if (len < 0)
        len = g_strlen (str);
      ret = g_malloc ((len + 1) * sizeof (gchar));
      g_memmove (ret, str, len);
      ret[len] = '\0';
      for (i = 0; i < len; i++)
        ret[i] = g_ascii_tolower (ret[i]);
    }
  return ret;
}


gchar *
g_strreverse (gchar *string)
{
  if (string)
    {
      gchar *p, *q;

      p = string;
      q = string + g_strlen (string) - 1;
      while (p < q)
        {
          gchar c;

          c = *p;
          *p = *q;
          *q = c;
          p++;
          q--;
        }
    }
  return string;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,12,0)
gint64
g_ascii_strtoll (const gchar  *nptr,
                 gchar       **endptr,
                 guint         base)
{
  gint64 ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = g_strtol (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}
#endif /* not GLIB_CHECK_VERSION(2,12,0) */


#if ! GLIB_CHECK_VERSION(2,2,0)
guint64
g_ascii_strtoull (const gchar  *nptr,
                  gchar       **endptr,
                  guint         base)
{
  guint64 ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = g_strtoul (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}
#endif /* not GLIB_CHECK_VERSION(2,2,0) */


#ifdef USE_GTK_EMULATE
gchar *
g_strchug (gchar *string)
{
  gchar *p;

  if (string)
    {
      for (p = string; *p != '\0' && g_ascii_isspace (*p); p++);
      g_memmove (string, p, (g_strlen (p) + 1) * sizeof (gchar));
    }
  return string;
}


gchar *
g_strchomp (gchar *string)
{
  if (string)
    {
      gsize len;

      len = g_strlen (string);
      while (len-- > 0)
        if (g_ascii_isspace (string[len]))
          string[len] = '\0';
        else
          break;
    }
  return string;
}


gchar *
g_strescape (const gchar *source,
             const gchar *exceptions)
{
  const guchar *p;
  gchar *escape;
  gchar *q;
  guchar excmap[256];
  gint i;

  if (!source)
    return NULL;
  for (i = 0; i < 256; i++)
    excmap[i] = g_ascii_isprint (i) ? 1 : 0;
  if (exceptions)
    {
      guchar *e;

      e = (guchar *)exceptions;
      while (*e != '\0')
        {
          excmap[*e] = 1;
          e++;
        }
    }
  q = escape = g_malloc ((g_strlen (source) * 4 + 1) * sizeof (gchar));
  for (p = (guchar *)source; *p != '\0'; p++)
    if (excmap[*p] == 1)
      *q++ = *p;
    else
      switch (*p)
        {
          case '\b': *q++ = '\\'; *q++ = 'b';  break;
          case '\f': *q++ = '\\'; *q++ = 'f';  break;
          case '\n': *q++ = '\\'; *q++ = 'n';  break;
          case '\r': *q++ = '\\'; *q++ = 'r';  break;
          case '\t': *q++ = '\\'; *q++ = 't';  break;
          case '\\': *q++ = '\\'; *q++ = '\\'; break;
          case '"':  *q++ = '\\'; *q++ = '"';  break;
          default:
            *q++ = '\\';
            *q++ = '0' + (((*p) >> 6) & 7);
            *q++ = '0' + (((*p) >> 3) & 7);
            *q++ = '0' + ((*p) & 7);
        }
  *q = '\0';
  return escape;
}


gchar *
g_strcompress (const gchar *source)
{
  gchar *p, *q, *str;

  if (!source)
    return NULL;
  p = (gchar *)source;
  q = str = g_malloc ((g_strlen (source) + 1) * sizeof (gchar));
  while (*p != '\0')
    if (*p == '\\')
      switch (*(++p))
        {
          case '\0': break;
          case 'b' : p++; *q++ = '\b'; break;
          case 'f' : p++; *q++ = '\f'; break;
          case 'n' : p++; *q++ = '\n'; break;
          case 'r' : p++; *q++ = '\r'; break;
          case 't' : p++; *q++ = '\t'; break;
          case '0': case '1': case '2': case '3':
          case '4': case '5': case '6': case '7':
            {
              gint i;

              *q = 0;
              for (i = 0; i < 3 && '0' <= p[i] && p[i] <= '7'; i++)
                *q = *q * 8 + p[i] - '0';
              p += i;
              q++;
            }
            break;
          default:
            *q++ = *p++;
        }
    else
      *q++ = *p++;
  *q = '\0';
  return str;
}


gchar **
g_strsplit (const gchar *string,
            const gchar *delimiter,
            gint         max_tokens)
{
  GList *glist = NULL, *gl;
  gchar **ret, *s;
  guint n = 0;
  const gchar *remainder;

  if (!string || !delimiter || delimiter[0] == '\0')
    return NULL;
  if (max_tokens < 1)
    max_tokens = G_MAXINT;
  remainder = string;
  s = g_strstr (remainder, delimiter);
  if (s)
    {
      gsize delimiter_len;

      delimiter_len = g_strlen (delimiter);
      while (--max_tokens && s)
        {
          gsize len;

          len = s - remainder;
          glist = g_list_append (glist, g_strndup (remainder, len));
          n++;
          remainder = s + delimiter_len;
          s = g_strstr (remainder, delimiter);
        }
    }
  if (*string)
    {
      n++;
      glist = g_list_append (glist, g_strdup (remainder));
    }

  ret = g_malloc ((n + 1) * sizeof (gchar *));
  ret[n] = NULL;
  n = 0;
  for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
    ret[n++] = gl->data;
  g_list_free (glist);

  return ret;
}


void
g_strfreev (gchar **str_array)
{
  if (str_array)
    {
      gint i;

      for (i = 0; str_array[i]; i++)
        g_free (str_array[i]);
      g_free (str_array);
    }
}


gchar *
g_strconcat (const gchar *str,
             ...)
{
  gchar *concat = NULL;

  if (str)
    {
      gsize len;
      gchar *s;
      va_list arg;

      len = strlen (str) + 1;
      va_start (arg, str);
      s = va_arg (arg, gchar *);
      while (s)
        {
          len += g_strlen (s);
          s = va_arg (arg, gchar *);
        }
      va_end (arg);
      concat = g_malloc (len * sizeof (gchar));
      g_strcpy (concat, str);
      va_start (arg, str);
      s = va_arg (arg, gchar *);
      while (s)
        {
          g_strcat (concat, s);
          s = va_arg (arg, gchar*);
        }
      va_end (arg);
    }
  return concat;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,6,0)
guint
g_strv_length (gchar **str_array)
{
  guint i = 0;

  if (str_array)
    while (str_array[i])
      i++;
  return i;
}
#endif /* not GLIB_CHECK_VERSION(2,6,0) */


/******************************************************************************
* Character Set Conversion                                                    *
******************************************************************************/
#ifdef USE_GTK_EMULATE
gchar *
g_locale_to_utf8 (const gchar  *opsysstring,
                  gssize        len,
                  gsize        *bytes_read,
                  gsize        *bytes_written,
                  GError      **error)
{
  gsize length;

  length = len < 0 ? g_strlen (opsysstring) + 1 : len;
  if (bytes_read)
    *bytes_read = length;
  if (bytes_written)
    *bytes_written = length;
  return g_memdup (opsysstring, length);
}


gchar *
g_locale_from_utf8 (const gchar  *utf8string,
                    gssize        len,
                    gsize        *bytes_read,
                    gsize        *bytes_written,
                    GError      **error)
{
  gsize length;

  length = len < 0 ? g_strlen (utf8string) + 1 : len;
  if (bytes_read)
    *bytes_read = length;
  if (bytes_written)
    *bytes_written = length;
  return g_memdup (utf8string, length);
}


gchar *
g_filename_to_utf8 (const gchar  *opsysstring,
                    gssize        len,
                    gsize        *bytes_read,
                    gsize        *bytes_written,
                    GError      **error)
{
  gsize length;

  length = len < 0 ? g_strlen (opsysstring) + 1 : len;
  if (bytes_read)
    *bytes_read = length;
  if (bytes_written)
    *bytes_written = length;
  return g_memdup (opsysstring, length);
}


gchar *
g_filename_from_utf8 (const gchar  *utf8string,
                      gssize        len,
                      gsize        *bytes_read,
                      gsize        *bytes_written,
                      GError      **error)
{
  gsize length;

  length = len < 0 ? g_strlen (utf8string) + 1 : len;
  if (bytes_read)
    *bytes_read = length;
  if (bytes_written)
    *bytes_written = length;
  return g_memdup (utf8string, length);
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,6,0)
gchar *
g_filename_display_name (const gchar *filename)
{
  return g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
}


gchar *
g_filename_display_basename (const gchar *filename)
{
  gchar *basename, *display;

  basename = g_path_get_basename (filename);
  display = g_filename_to_utf8 (basename, -1, NULL, NULL, NULL);
  g_free (basename);
  return display;
}
#endif /* not GLIB_CHECK_VERSION(2,6,0) */


/******************************************************************************
* Unicode Manipulation                                                        *
******************************************************************************/
#ifdef USE_GTK_EMULATE
gchar *
g_utf8_next_char (const gchar *p)
{
  const static gchar skip[256] = {
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                            2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                            2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                            3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
                            4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 6, 6, 1, 1};
  return p ? (gchar *)p + skip[*(guchar *)p] : NULL;
}

gunichar
g_utf8_get_char_validated (const gchar *p,
                           gssize       max_len)
{
  return max_len != 0 && p ? (guchar)*p : (gunichar)-2;
}


gchar *
g_utf8_strup (const gchar *str,
              gssize       len)
{
  return g_ascii_strup (str, len);
}


gchar *
g_utf8_strdown (const gchar *str,
                gssize       len)
{
  return g_ascii_strdown (str, len);
}


gunichar2 *
g_utf8_to_utf16 (const gchar  *str,
                 glong         len,
                 glong        *items_read,
                 glong        *items_written,
                 GError      **error)
{
  gunichar2 *utf16str = NULL;

  if (str)
    {
      gint i;

      if (len < 0)
        len = g_strlen (str);
      utf16str = g_malloc ((len + 1) * sizeof (gunichar2));
      for (i = 0; i < len; i++)
        utf16str[i] = (guchar)str[i];
      utf16str[i] = 0;
      if (items_read)
        *items_read = len;
      if (items_written)
        *items_written = len;
    }
  return utf16str;
}


gchar *
g_utf16_to_utf8 (const gunichar2  *str,
                 glong             len,
                 glong            *items_read,
                 glong            *items_written,
                 GError          **error)
{
  gchar *utf8str = NULL;

  if (str)
    {
      gint i;

      if (len < 0)
        for (len = 0; str[len] != 0; len++);
      utf8str = g_malloc ((len + 1) * sizeof (gchar));
      for (i = 0; i < len; i++)
        utf8str[i] = str[i];
      utf8str[i] = '\0';
      if (items_read)
        *items_read = len;
      if (items_written)
        *items_written = len;
    }
  return utf8str;
}


gint
g_unichar_to_utf8 (gunichar  c,
                   gchar    *outbuf)
{
  if (outbuf)
    {
      *outbuf = c;
      return 1;
    }
  return 0;
}
#endif /* USE_GTK_EMULATE */


/******************************************************************************
* Data Checksums                                                              *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,16,0)
# define SHA1_DATASIZE   64
# define SHA1_DIGEST_LEN 20


struct _GChecksum
{
  gchar *digest_str;
  guint32 buf[5];
  guint32 bits[2];
  guint32 data[16];
  guchar digest[SHA1_DIGEST_LEN];
};


gssize
g_checksum_type_get_length (GChecksumType checksum_type)
{
  return checksum_type == G_CHECKSUM_SHA1 ? SHA1_DIGEST_LEN : -1;
}


GChecksum *
g_checksum_new (GChecksumType checksum_type)
{
  GChecksum *checksum = NULL;

  if (checksum_type == G_CHECKSUM_SHA1)
    {
      checksum = g_malloc0 (sizeof (GChecksum));
      g_checksum_reset (checksum);
    }
  return checksum;
}


GChecksum *g_checksum_copy (const GChecksum *checksum)
{
  GChecksum *copy = NULL;

  if (checksum)
    {
      copy = g_malloc (sizeof (GChecksum));
      *copy = *checksum;
      copy->digest_str = g_strdup (checksum->digest_str);
    }
  return copy;
}


void
g_checksum_free (GChecksum *checksum)
{
  if (checksum)
    {
      g_free (checksum->digest_str);
      g_free (checksum);
    }
}


void
g_checksum_reset (GChecksum *checksum)
{
  if (checksum)
    {
      g_free (checksum->digest_str);
      checksum->digest_str = NULL;
      checksum->buf[0] = 0x67452301;
      checksum->buf[1] = 0xefcdab89;
      checksum->buf[2] = 0x98badcfe;
      checksum->buf[3] = 0x10325476;
      checksum->buf[4] = 0xc3d2e1f0;
      checksum->bits[0] = checksum->bits[1] = 0;
    }
}


static void
sha1_byte_reverse (guint32    *buf,
                   const gint  len)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
  gint i;

  for (i = len / sizeof (guint32); i > 0; i--)
    {
      *buf = GUINT32_SWAP_LE_BE (*buf);
      buf++;
    }
#endif /* G_BYTE_ORDER == G_LITTLE_ENDIAN */
}


# define f1(x,y,z) ((z)^((x)&((y)^(z))))
# define f2(x,y,z) ((x)^(y)^(z))
# define f3(x,y,z) (((x)&(y))|((z)&((x)|(y))))
# define f4(x,y,z) ((x)^(y)^(z))
# define k1 0x5a827999
# define k2 0x6ed9eba1
# define k3 0x8f1bbcdc
# define k4 0xca62c1d6
# define rotl(n,x) (((x)<<(n))|((x)>>(32-(n))))
# define expand(w,i) ((w)[(i)&15]=rotl(1,((w)[(i)&15]^(w)[((i)-14)&15]^(w)[((i)-8)&15]^(w)[((i)-3)&15])))
# define subround(a,b,c,d,e,f,k,data) ((e)+=rotl(5,(a))+f((b),(c),(d))+(k)+(data),(b)=rotl(30,(b)))


static void
sha1_transform (guint32 buf[5],
                guint32 in[16])
{
  guint32 a, b, c, d, e;

  a = buf[0];
  b = buf[1];
  c = buf[2];
  d = buf[3];
  e = buf[4];

  subround (a, b, c, d, e, f1, k1, in[0]);
  subround (e, a, b, c, d, f1, k1, in[1]);
  subround (d, e, a, b, c, f1, k1, in[2]);
  subround (c, d, e, a, b, f1, k1, in[3]);
  subround (b, c, d, e, a, f1, k1, in[4]);
  subround (a, b, c, d, e, f1, k1, in[5]);
  subround (e, a, b, c, d, f1, k1, in[6]);
  subround (d, e, a, b, c, f1, k1, in[7]);
  subround (c, d, e, a, b, f1, k1, in[8]);
  subround (b, c, d, e, a, f1, k1, in[9]);
  subround (a, b, c, d, e, f1, k1, in[10]);
  subround (e, a, b, c, d, f1, k1, in[11]);
  subround (d, e, a, b, c, f1, k1, in[12]);
  subround (c, d, e, a, b, f1, k1, in[13]);
  subround (b, c, d, e, a, f1, k1, in[14]);
  subround (a, b, c, d, e, f1, k1, in[15]);
  subround (e, a, b, c, d, f1, k1, expand (in, 16));
  subround (d, e, a, b, c, f1, k1, expand (in, 17));
  subround (c, d, e, a, b, f1, k1, expand (in, 18));
  subround (b, c, d, e, a, f1, k1, expand (in, 19));

  subround (a, b, c, d, e, f2, k2, expand (in, 20));
  subround (e, a, b, c, d, f2, k2, expand (in, 21));
  subround (d, e, a, b, c, f2, k2, expand (in, 22));
  subround (c, d, e, a, b, f2, k2, expand (in, 23));
  subround (b, c, d, e, a, f2, k2, expand (in, 24));
  subround (a, b, c, d, e, f2, k2, expand (in, 25));
  subround (e, a, b, c, d, f2, k2, expand (in, 26));
  subround (d, e, a, b, c, f2, k2, expand (in, 27));
  subround (c, d, e, a, b, f2, k2, expand (in, 28));
  subround (b, c, d, e, a, f2, k2, expand (in, 29));
  subround (a, b, c, d, e, f2, k2, expand (in, 30));
  subround (e, a, b, c, d, f2, k2, expand (in, 31));
  subround (d, e, a, b, c, f2, k2, expand (in, 32));
  subround (c, d, e, a, b, f2, k2, expand (in, 33));
  subround (b, c, d, e, a, f2, k2, expand (in, 34));
  subround (a, b, c, d, e, f2, k2, expand (in, 35));
  subround (e, a, b, c, d, f2, k2, expand (in, 36));
  subround (d, e, a, b, c, f2, k2, expand (in, 37));
  subround (c, d, e, a, b, f2, k2, expand (in, 38));
  subround (b, c, d, e, a, f2, k2, expand (in, 39));

  subround (a, b, c, d, e, f3, k3, expand (in, 40));
  subround (e, a, b, c, d, f3, k3, expand (in, 41));
  subround (d, e, a, b, c, f3, k3, expand (in, 42));
  subround (c, d, e, a, b, f3, k3, expand (in, 43));
  subround (b, c, d, e, a, f3, k3, expand (in, 44));
  subround (a, b, c, d, e, f3, k3, expand (in, 45));
  subround (e, a, b, c, d, f3, k3, expand (in, 46));
  subround (d, e, a, b, c, f3, k3, expand (in, 47));
  subround (c, d, e, a, b, f3, k3, expand (in, 48));
  subround (b, c, d, e, a, f3, k3, expand (in, 49));
  subround (a, b, c, d, e, f3, k3, expand (in, 50));
  subround (e, a, b, c, d, f3, k3, expand (in, 51));
  subround (d, e, a, b, c, f3, k3, expand (in, 52));
  subround (c, d, e, a, b, f3, k3, expand (in, 53));
  subround (b, c, d, e, a, f3, k3, expand (in, 54));
  subround (a, b, c, d, e, f3, k3, expand (in, 55));
  subround (e, a, b, c, d, f3, k3, expand (in, 56));
  subround (d, e, a, b, c, f3, k3, expand (in, 57));
  subround (c, d, e, a, b, f3, k3, expand (in, 58));
  subround (b, c, d, e, a, f3, k3, expand (in, 59));

  subround (a, b, c, d, e, f4, k4, expand (in, 60));
  subround (e, a, b, c, d, f4, k4, expand (in, 61));
  subround (d, e, a, b, c, f4, k4, expand (in, 62));
  subround (c, d, e, a, b, f4, k4, expand (in, 63));
  subround (b, c, d, e, a, f4, k4, expand (in, 64));
  subround (a, b, c, d, e, f4, k4, expand (in, 65));
  subround (e, a, b, c, d, f4, k4, expand (in, 66));
  subround (d, e, a, b, c, f4, k4, expand (in, 67));
  subround (c, d, e, a, b, f4, k4, expand (in, 68));
  subround (b, c, d, e, a, f4, k4, expand (in, 69));
  subround (a, b, c, d, e, f4, k4, expand (in, 70));
  subround (e, a, b, c, d, f4, k4, expand (in, 71));
  subround (d, e, a, b, c, f4, k4, expand (in, 72));
  subround (c, d, e, a, b, f4, k4, expand (in, 73));
  subround (b, c, d, e, a, f4, k4, expand (in, 74));
  subround (a, b, c, d, e, f4, k4, expand (in, 75));
  subround (e, a, b, c, d, f4, k4, expand (in, 76));
  subround (d, e, a, b, c, f4, k4, expand (in, 77));
  subround (c, d, e, a, b, f4, k4, expand (in, 78));
  subround (b, c, d, e, a, f4, k4, expand (in, 79));

  buf[0] += a;
  buf[1] += b;
  buf[2] += c;
  buf[3] += d;
  buf[4] += e;
}


void
g_checksum_update (GChecksum    *checksum,
                   const guchar *data,
                   gssize        length)
{
  if (checksum && data && length != 0 && !checksum->digest_str)
    {
      guint32 tmp;
      guint count;

      if (length < 0)
        length = g_strlen ((const gchar *)data);

      tmp = checksum->bits[0];
      checksum->bits[0] = tmp + ((guint32)length << 3);
      if (checksum->bits[0] < tmp)
        checksum->bits[1]++;
      checksum->bits[1] += length >> 29;

      count = (tmp >> 3) & 0x3f;
      if (count)
        {
          guchar *p;

          p = (guchar *)checksum->data + count;
          count = SHA1_DATASIZE - count;
          if (length < count)
            {
              g_memmove (p, data, length);
              return;
            }

          g_memmove (p, data, count);

          sha1_byte_reverse (checksum->data, SHA1_DATASIZE);
          sha1_transform (checksum->buf, checksum->data);

          data += count;
          length -= count;
        }

      while (length >= SHA1_DATASIZE)
        {
          g_memmove (checksum->data, data, SHA1_DATASIZE);

          sha1_byte_reverse (checksum->data, SHA1_DATASIZE);
          sha1_transform (checksum->buf, checksum->data);

          data += SHA1_DATASIZE;
          length -= SHA1_DATASIZE;
        }

      g_memmove (checksum->data, data, length);
    }
}


const gchar *
g_checksum_get_string (GChecksum *checksum)
{
  gint i, count;
  guchar *p;

  if (!checksum)
    return NULL;
  if (checksum->digest_str)
    return checksum->digest_str;

  count = (checksum->bits[0] >> 3) & 0x3f;
  p = (guchar *)checksum->data + count;
  *p++ = 0x80;
  count = SHA1_DATASIZE - 1 - count;
  if (count < 8)
    {
      g_memset (p, 0, count);
      sha1_byte_reverse (checksum->data, SHA1_DATASIZE);
      sha1_transform (checksum->buf, checksum->data);
      g_memset (checksum->data, 0, SHA1_DATASIZE - 8);
    }
  else
    {
      g_memset (p, 0, count - 8);
    }
  checksum->data[14] = checksum->bits[1];
  checksum->data[15] = checksum->bits[0];
  sha1_byte_reverse (checksum->data, SHA1_DATASIZE - 8);
  sha1_transform (checksum->buf, checksum->data);
  sha1_byte_reverse (checksum->buf, SHA1_DIGEST_LEN);
  g_memmove (checksum->digest, checksum->buf, SHA1_DIGEST_LEN);
  g_memset (checksum->buf, 0, sizeof (checksum->buf));
  g_memset (checksum->data, 0, sizeof (checksum->data));

  checksum->digest_str = g_malloc ((SHA1_DIGEST_LEN * 2 + 1) * sizeof (gchar));
  for (i = 0; i < SHA1_DIGEST_LEN; i++)
    {
      const static gchar hex[] = "0123456789abcdef";
      guint8 byte;

      byte = checksum->digest[i];
      checksum->digest_str[i * 2] = hex[byte >> 4];
      checksum->digest_str[i * 2 + 1] = hex[byte & 0xf];
    }
  checksum->digest_str[SHA1_DIGEST_LEN * 2] = 0;

  return checksum->digest_str;
}


void
g_checksum_get_digest (GChecksum *checksum,
                       guint8    *buffer,
                       gsize     *digest_len)
{
  if (checksum && buffer && digest_len
                && *digest_len >= g_checksum_type_get_length (G_CHECKSUM_SHA1)
                && g_checksum_get_string (checksum))
    {
      g_memmove (buffer, checksum->digest, SHA1_DIGEST_LEN);
      *digest_len = SHA1_DIGEST_LEN;
    }
}


gchar *
g_compute_checksum_for_data (GChecksumType  checksum_type,
                             const guchar  *data,
                             gsize          length)
{
  gchar *str = NULL;

  if (checksum_type == G_CHECKSUM_SHA1 && (data || length == 0))
    {
      GChecksum *checksum;

      checksum = g_checksum_new (checksum_type);
      g_checksum_update (checksum, data, length);
      str = g_strdup (g_checksum_get_string (checksum));
      g_checksum_free (checksum);
    }
  return str;
}


gchar *
g_compute_checksum_for_string (GChecksumType  checksum_type,
                               const gchar   *str,
                               gssize         length)
{
  if (!str && length != 0)
    return NULL;
  if (length < 0)
    length = g_strlen (str);
  return g_compute_checksum_for_data (checksum_type,
                                                (const guchar *)str, length);
}
#endif /* not GLIB_CHECK_VERSION(2,16,0) */


/******************************************************************************
* Date and Time Functions                                                     *
******************************************************************************/
#ifdef USE_GTK_EMULATE
void
g_get_current_time (GTimeVal *result)
{
  if (result)
    {
# ifdef G_OS_WIN32
      FILETIME ft;
      guint64 usec;

      GetSystemTimeAsFileTime (&ft);
      usec = (ft.dwLowDateTime | ((guint64)ft.dwHighDateTime << 32)) / 10
                                    - G_GINT64_CONSTANT (11644473600000000);
      result->tv_sec = usec / G_USEC_PER_SEC;
      result->tv_usec = usec % G_USEC_PER_SEC;
# else /* not G_OS_WIN32 */
#  ifdef HAVE_CLOCK_GETTIME
      struct timespec tp;
#  endif /* HAVE_CLOCK_GETTIME */
#  ifdef HAVE_GETTIMEOFDAY
      struct timeval tv;
#  endif /* HAVE_GETTIMEOFDAY */

#  ifdef HAVE_CLOCK_GETTIME
      if (clock_gettime (CLOCK_REALTIME, &tp) == 0)
        {
          result->tv_sec = tp.tv_sec;
          result->tv_usec = tp.tv_usec / 1000;
          return;
        }
#  endif /* HAVE_CLOCK_GETTIME */
#  ifdef HAVE_GETTIMEOFDAY
      if (gettimeofday (&tv, NULL) == 0)
        {
          result->tv_sec = tv.tv_sec;
          result->tv_usec = tv.tv_usec;
          return;
        }
#  endif /* HAVE_GETTIMEOFDAY */
      result->tv_sec = time (NULL);
      result->tv_usec = 0;
# endif /* not G_OS_WIN32 */
    }
}


void
g_usleep (gulong microseconds)
{
# ifdef G_OS_WIN32
  Sleep (microseconds / 1000);
# else /* not G_OS_WIN32 */
#  ifdef HAVE_NANOSLEEP
  struct timespec req, rem;

  req.tv_sec = microseconds / G_USEC_PER_SEC;
  req.tv_nsec = (microseconds % G_USEC_PER_SEC) * 1000;
  while (nanosleep (&req, &rem) == -1 && errno == EINTR)
    req = rem;
#  else /* not HAVE_NANOSLEEP */
#   ifdef HAVE_SLEEP
  unsigned int seconds;

  seconds = microseconds / G_USEC_PER_SEC;
  while (seconds > 0)
    seconds = sleep (seconds);
#   endif /* HAVE_SLEEP */
#   ifdef HAVE_USLEEP
#    ifdef HAVE_SLEEP
  usleep (microseconds % G_USEC_PER_SEC);
#    else /* not HAVE_SLEEP */
  usleep (microseconds);
#    endif /* not HAVE_SLEEP */
#   endif /* HAVE_USLEEP */
#  endif /* not HAVE_NANOSLEEP */
# endif /* not G_OS_WIN32 */
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,28,0)
gint64
g_get_monotonic_time (void)
{
# ifdef G_OS_WIN32
  gint64 monotonic = -1;
  HMODULE hLib;

  hLib = LoadLibrary (_T("kernel32.dll"));
  if (hLib)
    {
      ULONGLONG (WINAPI *pGetTickCount64)(VOID);

      pGetTickCount64 = (gpointer)GetProcAddress (hLib, "GetTickCount64");
      if (pGetTickCount64)
        monotonic = pGetTickCount64 ();
      FreeLibrary (hLib);
    }
  return (gint64)(monotonic >= 0 ? monotonic : GetTickCount ()) * 1000;
# else /* not G_OS_WIN32 */
#  if defined (HAVE_CLOCK_GETTIME) && defined (CLOCK_MONOTONIC)
  struct timespec tp;

  if (clock_gettime (CLOCK_MONOTONIC, &tp) == 0)
    return (gint64)tp.tv_sec * 1000000 + tp.tv_usec / 1000;
#  endif /* defined (HAVE_CLOCK_GETTIME) && defined (CLOCK_MONOTONIC) */
  return g_get_real_time ();
# endif /* not G_OS_WIN32 */
}


gint64
g_get_real_time (void)
{
  GTimeVal tv;

  g_get_current_time (&tv);
  return (gint64)tv.tv_sec * G_USEC_PER_SEC + tv.tv_usec;
}
#endif /* not GLIB_CHECK_VERSION(2,28,0) */


/******************************************************************************
* GDateTime                                                                   *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,26,0)
struct _GDateTime
{
  time_t t;
  struct tm tm;
  gint ref;
};


void
g_date_time_unref (GDateTime *datetime)
{
  if (datetime && --datetime->ref <= 0)
    g_free (datetime);
}


GDateTime *
g_date_time_ref (GDateTime *datetime)
{
  if (datetime)
    datetime->ref++;
  return datetime;
}


GDateTime *
g_date_time_new_now_utc (void)
{
  return g_date_time_new_from_unix_utc (g_get_real_time () / G_USEC_PER_SEC);
}


static void
g_date_time_gmtime (GDateTime *datetime)
{
  if (datetime)
    {
# ifdef HAVE_GMTIME_R
      gmtime_r (&datetime->t, &datetime->tm);
# else /* not HAVE_GMTIME_R */
      struct tm *tm;

      tm = gmtime (&datetime->t);
      if (tm)
        datetime->tm = *tm;
# endif /* not HAVE_GMTIME_R */
    }
}


static void
g_date_time_mktime (GDateTime *datetime)
{
  if (datetime)
    {
# if defined (G_OS_WIN32)
      datetime->tm.tm_isdst = 0;
      datetime->t = _mkgmtime (&datetime->tm);
# elif defined (HAVE_TIMEGM)
      datetime->tm.tm_isdst = 0;
      datetime->t = timegm (&datetime->tm);
# else /* not HAVE_TIMEGM */
      const gchar *tz;

      tz = g_getenv ("TZ");
      g_setenv ("TZ", "", TRUE);
#  ifdef HAVE_TZSET
      tzset ();
#  endif /* HAVE_TZSET */
      datetime->tm.tm_isdst = 0;
      datetime->t = mktime (&datetime->tm);
      if (tz)
        g_setenv ("TZ", tz, TRUE);
      else
        g_unsetenv ("TZ");
#  ifdef HAVE_TZSET
      tzset ();
#  endif /* HAVE_TZSET */
# endif /* not HAVE_TIMEGM */
    }
}


GDateTime *
g_date_time_new_from_unix_utc (gint64 t)
{
  GDateTime *datetime;

  datetime = g_malloc (sizeof (GDateTime));
  datetime->ref = 1;
  datetime->t = t;
  g_date_time_gmtime (datetime);
  return datetime;
}


GDateTime *
g_date_time_new_from_timeval_utc (const GTimeVal *tv)
{
  return tv ? g_date_time_new_from_unix_utc (tv->tv_sec) : NULL;
}


GDateTime *
g_date_time_new_utc (gint    year,
                     gint    month,
                     gint    day,
                     gint    hour,
                     gint    minute,
                     gdouble seconds)
{
  GDateTime *datetime;

  datetime = g_malloc (sizeof (GDateTime));
  datetime->ref = 1;
  datetime->tm.tm_year = year - 1900;
  datetime->tm.tm_mon = month - 1;
  datetime->tm.tm_mday = day;
  datetime->tm.tm_hour = hour;
  datetime->tm.tm_min = minute;
  datetime->tm.tm_sec = seconds;
  g_date_time_mktime (datetime);
  return datetime;
}


GDateTime *
g_date_time_add (GDateTime *datetime,
                 GTimeSpan  timespan)
{
  GDateTime *dt = NULL;

  if (datetime)
    {
      dt = g_malloc (sizeof (GDateTime));
      dt->ref = 1;
      dt->t = datetime->t + timespan / G_USEC_PER_SEC;
      g_date_time_gmtime (dt);
    }
  return dt;
}


GDateTime *
g_date_time_add_years (GDateTime *datetime,
                       gint       years)
{
  return g_date_time_add_full (datetime, years, 0, 0, 0, 0, 0);
}


GDateTime *
g_date_time_add_months (GDateTime *datetime,
                        gint       months)
{
  return g_date_time_add_full (datetime, 0, months, 0, 0, 0, 0);
}


GDateTime *
g_date_time_add_weeks (GDateTime *datetime,
                       gint       weeks)
{
  return g_date_time_add_days (datetime, weeks * 7);
}


GDateTime *
g_date_time_add_days (GDateTime *datetime,
                      gint       days)
{
  return g_date_time_add_full (datetime, 0, 0, days, 0, 0, 0);
}


GDateTime *
g_date_time_add_hours (GDateTime *datetime,
                       gint       hours)
{
  return g_date_time_add (datetime, (gint64)hours * 60 * 60 * G_USEC_PER_SEC);
}


GDateTime *
g_date_time_add_minutes (GDateTime *datetime,
                         gint       minutes)
{
  return g_date_time_add (datetime, (gint64)minutes * 60 * G_USEC_PER_SEC);
}


GDateTime *
g_date_time_add_seconds (GDateTime *datetime,
                         gdouble    seconds)
{
  return g_date_time_add (datetime, seconds * G_USEC_PER_SEC);
}


GDateTime *
g_date_time_add_full (GDateTime *datetime,
                      gint       years,
                      gint       months,
                      gint       days,
                      gint       hours,
                      gint       minutes,
                      gdouble    seconds)
{
  GDateTime *dt = NULL;

  if (datetime)
    {
      gint leap;
      const static guint16 days_in_months[2][12] = {
                            {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
                            {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}};

      dt = g_malloc (sizeof (GDateTime));
      dt->ref = 1;
      dt->tm.tm_year = datetime->tm.tm_year + (years * 12 + months) / 12;
      dt->tm.tm_mon = datetime->tm.tm_mon + (years * 12 + months) % 12;
      if (dt->tm.tm_mon < 0)
        {
          dt->tm.tm_mon += 12;
          dt->tm.tm_year--;
        }
      else if (dt->tm.tm_mon > 12)
        {
          dt->tm.tm_mon -= 12;
          dt->tm.tm_year++;
        }
      leap = dt->tm.tm_year % 400 == 0
            || (dt->tm.tm_year % 100 != 0 && dt->tm.tm_year % 4 == 0) ? 1 : 0;
      dt->tm.tm_mday = MIN (datetime->tm.tm_mday,
                                        days_in_months[leap][dt->tm.tm_mon]);
      dt->tm.tm_hour = datetime->tm.tm_hour;
      dt->tm.tm_min = datetime->tm.tm_min;
      dt->tm.tm_sec = datetime->tm.tm_sec;
      g_date_time_mktime (dt);
      dt->t += (((days * 24) + hours) * 60 + minutes) * 60 + seconds;
      g_date_time_gmtime (dt);
    }
  return dt;
}


gint
g_date_time_compare (gconstpointer dt1,
                     gconstpointer dt2)
{
  GTimeSpan difference;

  difference = g_date_time_difference ((GDateTime *)dt1, (GDateTime *)dt2);
  return difference < 0 ? -1 : difference > 0 ? 1 : 0;
}


GTimeSpan
g_date_time_difference (GDateTime *end,
                        GDateTime *begin)
{
  return end && begin ? (GTimeSpan)(end->t - begin->t) * G_USEC_PER_SEC : 0;
}


guint
g_date_time_hash (gconstpointer datetime)
{
  return datetime ? ((GDateTime *)datetime)->t : 0;
}


gboolean
g_date_time_equal (gconstpointer dt1,
                   gconstpointer dt2)
{
  return g_date_time_compare (dt1, dt2) == 0;
}


void
g_date_time_get_ymd (GDateTime *datetime,
                     gint      *year,
                     gint      *month,
                     gint      *day)
{
  if (datetime)
    {
      if (year)
        *year = datetime->tm.tm_year + 1900;
      if (month)
        *month = datetime->tm.tm_mon + 1;
      if (day)
        *day = datetime->tm.tm_mday;
    }
}


gint
g_date_time_get_year (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_year + 1900 : 0;
}


gint
g_date_time_get_month (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_mon + 1 : 0;
}


gint
g_date_time_get_day_of_month (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_mday : 0;
}


gint
g_date_time_get_hour (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_hour : 0;
}


gint
g_date_time_get_minute (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_min : 0;
}


gint
g_date_time_get_second (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_sec : 0;
}

gint
g_date_time_get_microsecond (GDateTime *datetime)
{
  return 0;
}


gdouble
g_date_time_get_seconds (GDateTime *datetime)
{
  return g_date_time_get_second (datetime);
}


gint64
g_date_time_to_unix (GDateTime *datetime)
{
  return datetime ? datetime->t : 0;
}


gboolean
g_date_time_to_timeval (GDateTime *datetime,
                        GTimeVal  *tv)
{
  if (datetime && tv)
    {
      tv->tv_sec = datetime->t;
      tv->tv_usec = 0;
      return TRUE;
    }
  return FALSE;
}
#endif /* not GLIB_CHECK_VERSION(2,26,0) */


/******************************************************************************
* Miscellaneous Utility Functions                                             *
******************************************************************************/
#ifdef USE_GTK_EMULATE
static gchar *g_prgname = NULL;


gchar *
g_get_prgname (void)
{
  return g_prgname;
}


void g_set_prgname (const gchar *prgname)
{
  g_free (g_prgname);
  g_prgname = g_strdup (prgname);
}


# ifdef G_OS_WIN32
static GHashTable *ghash_env = NULL;


gboolean
g_strcase_equal (gconstpointer v1,
                 gconstpointer v2)
{
  return g_ascii_strcasecmp ((const gchar *)v1, (const gchar *)v2) == 0;
}


guint
g_strcase_hash (gconstpointer v)
{
  const gchar *p;
  guint32 h = 5381;

  if (v)
    for (p = v; *p != '\0'; p++)
      h = (h << 5) + h + g_ascii_tolower (*p);
  return h;
}
# endif /* G_OS_WIN32 */


const gchar *
g_getenv (const gchar *variable)
{
# ifdef G_OS_WIN32
  gchar *env = NULL;

  if (!ghash_env)
    ghash_env = g_hash_table_new_full (g_strcase_hash, g_strcase_equal,
                                                            g_free, g_free);
  if (variable && !(env = g_hash_table_lookup (ghash_env, variable)))
    {
      DWORD dwSize, dwResult;
#  ifdef UNICODE
      LPWSTR lpszName, lpszBuffer;
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (variable, -1, NULL, NULL, NULL);
      lpszName = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      dwSize = GetEnvironmentVariableW (lpszName, NULL, 0);
      lpszBuffer = g_malloc (dwSize * sizeof (WCHAR));
      dwResult = GetEnvironmentVariableW (lpszName, lpszBuffer, dwSize);
      g_free (lpszName);
      if (dwResult == dwSize - 1)
        {
          utf8str = g_utf16_to_utf8 (lpszBuffer, -1, NULL, NULL, NULL);
          env = g_locale_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      g_free (lpszBuffer);
#  else /* not UNICODE */
      dwSize = GetEnvironmentVariableA (variable, NULL, 0);
      env = g_malloc (dwSize * sizeof (CHAR));
      dwResult = GetEnvironmentVariableA (variable, env, dwSize);
      if (dwResult != dwSize - 1)
        {
          g_free (env);
          env = NULL;
        }
#  endif /* not UNICODE */
      if (env)
        g_hash_table_insert (ghash_env, g_strdup (variable), env);
    }
  return env;
# else /* not G_OS_WIN32 */
  return variable ? getenv (variable) : NULL;
# endif /* not G_OS_WIN32 */
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,4,0)
gboolean
g_setenv (const gchar *variable,
          const gchar *value,
          gboolean     overwrite)
{
# ifdef G_OS_WIN32
  gboolean ret = FALSE;

  if (variable)
    {
      if (overwrite || !g_getenv (variable))
        {
#  ifdef UNICODE
          LPWSTR lpszName, lpszValue;
          gchar *utf8str;

          utf8str = g_locale_to_utf8 (variable, -1, NULL, NULL, NULL);
          lpszName = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          utf8str = g_locale_to_utf8 (value, -1, NULL, NULL, NULL);
          lpszValue = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          ret = SetEnvironmentVariableW (lpszName, lpzaValue);
          g_free (lpszName);
          g_free (lpzaValue);
#  else /* not UNICODE */
          ret = SetEnvironmentVariableA (variable, value);
#  endif /* not UNICODE */
          if (ghash_env)
            g_hash_table_remove (ghash_env, variable);
        }
      else
        {
          ret = TRUE;
        }
    }
  return ret;
# else /* not G_OS_WIN32 */
#  ifdef HAVE_SETENV
  return variable ? setenv (variable, value, overwrite) : FALSE;
#  else /* not HAVE_SETENV */
  gboolean ret = FALSE;

  if (variable)
    {
      if (overwrite || !g_getenv (variable))
        {
          gchar *string;

          string = g_strconcat (variable, "=", value, NULL);
          ret = putenv (string);
          g_free (string);
        }
      else
        {
          ret = TRUE;
        }
    }
  return ret;
#  endif /* not HAVE_SETENV */
# endif /* not G_OS_WIN32 */
}


void
g_unsetenv (const gchar *variable)
{
  if (variable)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      LPWSTR lpszName;
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (variable, -1, NULL, NULL, NULL);
      lpszName = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      SetEnvironmentVariableW (lpszName, NULL);
      g_free (lpszName);
#  else /* not UNICODE */
      SetEnvironmentVariableA (variable, NULL);
#  endif /* not UNICODE */
      if (ghash_env)
        g_hash_table_remove (ghash_env, variable);
# else /* not G_OS_WIN32 */
#  ifdef HAVE_UNSETENV
      unsetenv (variable);
#  else /* not HAVE_UNSETENV */
      gsize len;
      gchar **p, **q;

      len = g_strlen (variable);
      for (p = q = environ; *p; p++)
        if (g_strncmp (*p, variable, len) != 0 || (*p)[len] != '=')
          {
            *q = *p;
            q++;
          }
      *q = NULL;
#  endif /* not HAVE_UNSETENV */
# endif /* not G_OS_WIN32 */
    }
}
#endif /* not GLIB_CHECK_VERSION(2,4,0) */


#ifdef USE_GTK_EMULATE
const gchar *
g_get_user_name (void)
{
  static const gchar *name = NULL;

  if (!name)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
  DWORD dwSize = 0;
  LPWSTR lpszName;

  GetUserNameW (NULL, &dwSize);
  lpszName = g_malloc (dwSize * sizeof (WCHAR));
  if (GetUserNameW (lpszName, &dwSize))
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (lpszName, -1, NULL, NULL, NULL);
      name = g_locale_from_utf8 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
    }
  g_free (lpszName);
#  else /* not UNICODE */
  DWORD dwSize = 0;

  GetUserNameA (NULL, &dwSize);
  name = g_malloc (dwSize * sizeof (CHAR));
  if (!GetUserNameA ((LPSTR)name, &dwSize))
    {
      g_free ((gpointer)name);
      name = NULL;
    }
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      name = g_getenv ("LOGNAME");
# endif /* not G_OS_WIN32 */
    }
  return name;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,6,0)
# ifdef G_OS_WIN32
static gchar *
g_get_special_folder (INT nFolder)
{
  gchar *path = NULL;
  LPITEMIDLIST pidl = NULL;

  if (SHGetSpecialFolderLocation (NULL, nFolder, &pidl) == S_OK)
    {
      TCHAR szPath[MAX_PATH + 1];

      if (SHGetPathFromIDList (pidl, szPath))
        {
          gchar *utf8str;

#  ifdef UNICODE
          utf8str = g_utf16_to_utf8 (szPath, -1, NULL, NULL, NULL);
#  else /* not UNICODE */
          utf8str = g_locale_to_utf8 (szPath, -1, NULL, NULL, NULL);
#  endif /* not UNICODE */
          if (utf8str)
            {
              path = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
            }
        }
      CoTaskMemFree (pidl);
    }
  return path;
}
# endif /* G_OS_WIN32 */


const gchar *
g_get_user_cache_dir (void)
{
  static const gchar *cache = NULL;

  if (!cache)
    {
# ifdef G_OS_WIN32
      cache = g_get_special_folder (CSIDL_INTERNET_CACHE);
# else /* not G_OS_WIN32 */
      cache = g_getenv ("XDG_CACHE_HOME");
# endif /* not G_OS_WIN32 */
      if (!cache)
        {
          cache = g_build_filename (g_get_home_dir (), ".cache", NULL);
          if (!cache)
            cache = g_build_filename (g_get_tmp_dir (),
                                      g_get_user_name (), ".cache", NULL);
        }
    }
  return cache;
}


const gchar *
g_get_user_data_dir (void)
{
  static const gchar *data = NULL;

  if (!data)
    {
# ifdef G_OS_WIN32
      data = g_get_special_folder (CSIDL_LOCAL_APPDATA);
# else /* not G_OS_WIN32 */
      data = g_getenv ("XDG_DATA_HOME");
# endif /* not G_OS_WIN32 */
      if (!data)
        {
          data = g_build_filename (g_get_home_dir (), ".local", "share", NULL);
          if (!data)
            data = g_build_filename (g_get_tmp_dir (),
                                g_get_user_name (), ".local", "share", NULL);
        }
    }
  return data;
}


const gchar *
g_get_user_config_dir (void)
{
  static const gchar *config = NULL;

  if (!config)
    {
# ifdef G_OS_WIN32
      config = g_get_special_folder (CSIDL_LOCAL_APPDATA);
# else /* not G_OS_WIN32 */
      config = g_getenv ("XDG_CONFIG_HOME");
# endif /* not G_OS_WIN32 */
      if (!config)
        {
          config = g_build_filename (g_get_home_dir (), ".config", NULL);
          if (!config)
            config = g_build_filename (g_get_tmp_dir (),
                                       g_get_user_name (), ".config", NULL);
        }
    }
  return config;
}
#endif /* not GLIB_CHECK_VERSION(2,6,0) */


#ifdef USE_GTK_EMULATE
const gchar *
g_get_home_dir (void)
{
  static const gchar *home = NULL;

  if (!home)
    {
# ifdef G_OS_WIN32
      home = g_get_special_folder (CSIDL_PROFILE);
      if (!home)
        home = g_getenv ("USERPROFILE");
# else /* not G_OS_WIN32 */
      home = g_getenv ("HOME");
# endif /* not G_OS_WIN32 */
    }
  return home;
}


const gchar *
g_get_tmp_dir (void)
{
  static const gchar *tmp = NULL;

  if (!tmp)
    {
      tmp = g_getenv ("TMPDIR");
      if (!tmp)
        {
          tmp = g_getenv ("TMP");
          if (!tmp)
            tmp = g_getenv ("TEMP");
        }
    }
  return tmp;
}


gchar *
g_get_current_dir (void)
{
  gchar *dir = NULL;
# ifdef G_OS_WIN32
#  ifdef UNICODE
  DWORD dwLength;
  LPWSTR lpszDir;

  dwLength = GetCurrentDirectoryW (0, NULL);
  lpszDir = g_malloc (dwLength * sizeof (WCHAR));
  if (GetCurrentDirectoryW (dwLength, lpszDir) == dwLength - 1)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (lpszDir, -1, NULL, NULL, NULL);
      dir = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
    }
  g_free (lpszDir);
#  else /* not UNICODE */
  DWORD dwLength;

  dwLength = GetCurrentDirectoryA (0, NULL);
  dir = g_malloc (dwLength * sizeof (CHAR));
  if (GetCurrentDirectoryA (dwLength, dir) != dwLength - 1)
    {
      g_free (dir);
      dir = NULL;
    }
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
  gchar *buf;
#  ifdef HAVE_GETCWD
  buf = getcwd (NULL, 0);
  dir = g_strdup (buf);
  free (buf);
#  else /* not HAVE_GETCWD */
  buf = g_malloc (MAXPATHLEN * sizeof (gchar));
  dir = getwd (buf);
  if (!dir)
    g_free (buf);
#  endif /* not HAVE_GETCWD */
# endif /* not G_OS_WIN32 */
  if (!dir)
    dir = g_strdup (G_DIR_SEPARATOR_S);
  return dir;
}


gboolean
g_path_is_absolute (const gchar *file)
{
  if (file)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      gboolean ret = FALSE;
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          LPWSTR lpszPath;

          lpszPath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          if (lpszPath)
            {
              ret = ! PathIsRelativeW (lpszPath);
              g_free (lpszPath);
            }
          g_free (utf8str);
        }
      return ret;
#  else /* not UNICODE */
      return ! PathIsRelativeA (file);
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      return G_IS_DIR_SEPARATOR (file[0]);
# endif /* not G_OS_WIN32 */
    }
  return FALSE;
}


gchar *
g_path_get_basename (const gchar *file)
{
  gchar *name = NULL;

  if (file)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          LPWSTR lpszFile;

          lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          if (lpszFile)
            {
              LPWSTR lpszName;

              lpszName = PathFindFileNameW (lpszFile);
              if (lpszName)
                {
                  utf8str = g_utf16_to_utf8 (lpszName, -1, NULL, NULL, NULL);
                  if (utf8str)
                    {
                      name = g_filename_from_utf8 (utf8str, -1,
                                                            NULL, NULL, NULL);
                      g_free (utf8str);
                    }
                }
              g_free (lpszFile);
            }
        }
#  else /* not UNICODE */
      LPSTR lpszName;

      lpszName = PathFindFileNameA (file);
      if (lpszName)
        name = g_strdup (lpszName);
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      gchar *base;

      base = g_strrchr (file, G_DIR_SEPARATOR);
      if (base)
        name = g_strdup (base + 1);
# endif /* not G_OS_WIN32 */
      if (!name)
        name = g_strdup (*file != '\0' ? file : ".");
    }
  return name;
}


gchar *
g_path_get_dirname (const gchar *file)
{
  gchar *path = NULL;

  if (file)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          LPWSTR lpszFile;

          lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          if (lpszFile)
            {
              LPWSTR lpszName;

              lpszName = PathFindFileNameW (lpszFile);
              if (lpszName > lpszFile + 1)
                {
                  utf8str = g_utf16_to_utf8 (lpszFile, lpszName - lpszFile - 1,
                                                            NULL, NULL, NULL);
                  if (utf8str)
                    {
                      path = g_filename_from_utf8 (utf8str, -1,
                                                            NULL, NULL, NULL);
                      g_free (utf8str);
                    }
                }
            }
        }
      g_free (lpszFile);
#  else /* not UNICODE */
      LPSTR lpszName;

      lpszName = PathFindFileNameA (file);
      if (lpszName > file + 1)
        path = g_strndup (file, lpszName - file - 1);
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      gchar *base;

      base = g_strrchr (file, G_DIR_SEPARATOR);
      if (base)
        {
          while (base > file && G_IS_DIR_SEPARATOR (*base))
            base--;
          path = g_strndup (file, base - file + 1);
        }
# endif /* not G_OS_WIN32 */
    }
  return path ? path : g_strdup (".");
}


gchar *
g_build_filename (const gchar *first_element,
                  ...)
{
  gchar *str = NULL;

  if (first_element)
    {
      gchar **a, *s;
      gint i = 0;
      va_list args;

      va_start (args, first_element);
      while (va_arg (args, gchar *))
        i++;
      va_end (args);
      a = g_malloc ((i + 2) * sizeof (gchar *));
      a[0] = g_strdup (first_element);
      i = 1;
      va_start (args, first_element);
      while ((s = va_arg (args, gchar *)))
        a[i++] = g_strdup (s);
      va_end (args);
      a[i] = NULL;
      str = g_build_pathv (G_DIR_SEPARATOR_S, a);
      g_strfreev (a);
    }
  return str;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,8,0)
gchar *
g_build_filenamev (gchar **args)
{
  return g_build_pathv (G_DIR_SEPARATOR_S, args);
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */


#ifdef USE_GTK_EMULATE
gchar *
g_build_path (const gchar *separator,
              const gchar *first_element,
              ...)
{
  gchar *str = NULL;

  if (first_element)
    {
      gchar **a, *s;
      gint i = 0;
      va_list args;

      va_start (args, first_element);
      while (va_arg (args, gchar *))
        i++;
      va_end (args);
      a = g_malloc ((i + 2) * sizeof (gchar *));
      a[0] = g_strdup (first_element);
      i = 1;
      va_start (args, first_element);
      while ((s = va_arg (args, gchar *)))
        a[i++] = g_strdup (s);
      va_end (args);
      a[i] = NULL;
      str = g_build_pathv (separator, a);
      g_strfreev (a);
    }
  return str;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,8,0)
gchar *
g_build_pathv (const gchar  *separator,
               gchar       **args)
{
  gsize len, leng = 0;
  gchar **str, *path;
  gint i, j = 0;
  guint length;

  length = g_strv_length (args);
  str = g_malloc ((length + 1) * sizeof (gchar *));
  len = g_strlen (separator);
  for (i = 0; args[i]; i++)
    {
      gchar *p, *q;

      p = args[i];
      while (g_strncmp (p, separator, len) == 0)
        p += len;
      q = args[i] + g_strlen (args[i]) - len;
      while (p < q && g_strncmp (q, separator, len) == 0)
        q -= len;
      q += len;
      if (p < q)
        {
          str[j] = g_strndup (p, q - p);
          leng += g_strlen (str[j]);
          j++;
        }
    }
  str[j] = NULL;
  leng += g_strv_length (args) * len + 1;
  path = g_malloc (leng * sizeof (gchar));
  path[0] = '\0';
  for (i = 0; str[i]; i++)
    {
# ifdef G_OS_WIN32
      if (!(i == 0 && g_ascii_isalpha ((str[0])[0]) && (str[0])[1] == ':'))
# endif /* G_OS_WIN32 */
      g_strcat (path, separator);
      g_strcat (path, str[i]);
    }
  g_strfreev (str);
  return path;
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */


/******************************************************************************
* File Utilities                                                              *
******************************************************************************/
#ifdef USE_GTK_EMULATE
gboolean
g_file_test (const gchar *file,
             GFileTest    test)
{
# ifdef G_OS_WIN32
#  ifndef INVALID_FILE_ATTRIBUTES
#   define INVALID_FILE_ATTRIBUTES -1
#  endif /* not INVALID_FILE_ATTRIBUTES */
#  ifndef FILE_ATTRIBUTE_DEVICE
#   define FILE_ATTRIBUTE_DEVICE 64
#  endif /* not FILE_ATTRIBUTE_DEVICE */
  gboolean ret = FALSE;
  DWORD dwAttributes;
#  ifdef UNICODE
  gchar *utf8str;
  LPWSTR lpszFile;

  utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  dwAttributes = GetFileAttributesW (lpszFile);
  if (dwAttributes != INVALID_FILE_ATTRIBUTES)
    {
      if ((test & G_FILE_TEST_EXISTS)
        || ((test & G_FILE_TEST_IS_REGULAR) && (dwAttributes
                    & (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_DEVICE)) == 0)
        || ((test & G_FILE_TEST_IS_DIR)
                            && (dwAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0))
        ret = TRUE;
      if (test & G_FILE_TEST_IS_EXECUTABLE)
        {
          INT nLength;

          nLength = lstrlenW (lpszFile);
          if (nLength > 4
                        && lstrcmpiW (lpszFile + nLength - 4, L".exe") == 0)
            ret = TRUE;
        }
    }
  g_free (lpszFile);
#  else /* not UNICODE */
  dwAttributes = GetFileAttributesA (file);
  if (dwAttributes != INVALID_FILE_ATTRIBUTES)
    {
      if ((test & G_FILE_TEST_EXISTS)
        || ((test & G_FILE_TEST_IS_REGULAR) && (dwAttributes
                    & (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_DEVICE)) == 0)
        || ((test & G_FILE_TEST_IS_DIR)
                            && (dwAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0))
        ret = TRUE;
      if (test & G_FILE_TEST_IS_EXECUTABLE)
        {
          gsize len;

          len = g_strlen (file);
          if (len > 4 && g_ascii_strcasecmp (file + len - 4, ".exe") == 0)
            ret = TRUE;
        }
    }
#  endif /* not UNICODE */
  return ret;
# else /* not G_OS_WIN32 */
  if ((test & G_FILE_TEST_EXISTS) && (g_access (file, F_OK) == 0))
    return TRUE;
  if ((test & G_FILE_TEST_IS_EXECUTABLE) && (g_access (file, X_OK) == 0))
    {
      if (getuid () != 0)
        return TRUE;
    }
  else
    {
      test &= ~G_FILE_TEST_IS_EXECUTABLE;
    }
  if (test & G_FILE_TEST_IS_SYMLINK)
    {
      struct stat s;

      if ((g_lstat (file, &s) == 0) && S_ISLNK (s.st_mode))
        return TRUE;
    }
  if (test & (G_FILE_TEST_IS_REGULAR | G_FILE_TEST_IS_DIR
                                                | G_FILE_TEST_IS_EXECUTABLE))
    {
      GStatBuf s;

      if (g_stat (file, &s) == 0
              && (((test & G_FILE_TEST_IS_REGULAR) && S_ISREG (s.st_mode))
              || ((test & G_FILE_TEST_IS_DIR) && S_ISDIR (s.st_mode))
              || ((test & G_FILE_TEST_IS_EXECUTABLE) && ((s.st_mode & S_IXOTH)
                        || (s.st_mode & S_IXUSR) || (s.st_mode & S_IXGRP)))))
        return TRUE;
    }
  return FALSE;
# endif /* not G_OS_WIN32 */
}
#endif /* USE_GTK_EMULATE */
#if ! GLIB_CHECK_VERSION(2,6,0) && defined (G_OS_WIN32)
gint
g_mkdir (const gchar *filename,
         gint         mode)
{
# ifdef UNICODE
  gchar *utf8str;
  gint ret;
  LPWSTR lpszFile;

  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  ret = CreateDirectoryW (lpszFile, NULL);
  g_free (lpszFile);
  return ret ? 0 : -1;
# else /* UNICODE */
  return CreateDirectoryA (filename, NULL) ? 0 : -1;
# endif /* UNICODE */
}
#endif /* ! GLIB_CHECK_VERSION(2,6,0) && defined (G_OS_WIN32) */


#if ! GLIB_CHECK_VERSION(2,8,0)
gint
g_chmod (const gchar *filename,
         gint         mode)
{
#ifdef G_OS_WIN32
  gint ret;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      g_free (path);
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tchmod (lpszFile, mode);
  g_free (lpszFile);
  return ret;
#else /* not G_OS_WIN32 */
  return chmod (filename, mode);
#endif /* not G_OS_WIN32 */
}


gint
g_access (const gchar *filename,
          gint         mode)
{
#ifdef G_OS_WIN32
  gint ret;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _taccess (lpszFile, mode);
  g_free (lpszFile);
  return ret;
#else /* not G_OS_WIN32 */
  return access (filename, mode);
#endif /* not G_OS_WIN32 */
}


gint g_creat (const gchar *filename,
              gint         mode)
{
#ifdef G_OS_WIN32
  gint ret;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tcreat (lpszFile, mode);
  g_free (lpszFile);
  return ret;
#else /* not G_OS_WIN32 */
  return creat (filename, mode);
#endif /* not G_OS_WIN32 */
}


gint
g_chdir (const gchar *path)
{
#ifdef G_OS_WIN32
  gint ret;
  LPTSTR lpszPath;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszPath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszPath = g_win32_locale_filename_from_utf8 (path);
# endif /* not UNICODE */
  if (!lpszPath)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tchdir (lpszPath);
  g_free (lpszPath);
  return ret;
#else /* not G_OS_WIN32 */
  return chdir (path);
#endif /* not G_OS_WIN32 */
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */


#if ! GLIB_CHECK_VERSION(2,18,0)
gint
g_utime (const gchar    *filename,
         struct utimbuf *utb)
{
#ifdef G_OS_WIN32
  gint ret;
  HANDLE hFile;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tutime (lpszFile, (struct _utimbuf *)utb);
  g_free (lpszFile);
  return ret;
#else /* not G_OS_WIN32 */
# ifdef HAVE_UTIME
  return utime (filename, utb);
# else /* not HAVE_UTIME */
  return -1;
# endif /* not HAVE_UTIME */
#endif /* not G_OS_WIN32 */
}
#endif /* not GLIB_CHECK_VERSION(2,18,0) */


/******************************************************************************
* URI Utilities                                                               *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,16,0)
gchar *
g_uri_parse_scheme (const gchar *uri)
{
  gchar *scheme = NULL;

  if (uri && g_ascii_isalpha (*uri))
    {
      const gchar *p;

      for (p = uri + 1; g_ascii_isalnum (*p)
                                || *p == '+' || *p == '-' || *p == '.'; p++);
      if (*p == ':')
        scheme = g_strndup (uri, (p - uri) * sizeof (gchar));
    }
  return scheme;
}


gchar *
g_uri_escape_string (const gchar *unescaped,
                     const gchar *reserved_chars_allowed,
                     gboolean     allow_utf8)
{
  gchar *ret = NULL;

  if (unescaped)
    {
      const gchar *p;
      gchar *q, *allowed;

      allowed = g_strconcat (
        "-.0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~",
                                                reserved_chars_allowed, NULL);
      p = unescaped;
      q = ret = g_malloc ((g_strlen (unescaped) * 3 + 1) * sizeof (gchar));
      while (*p != '\0')
        {
          gunichar c;
          gchar *r;

          if (allow_utf8
                && (c = g_utf8_get_char_validated (p, -1)) != (gunichar)-1
                && c != (gunichar)-2
                && (r = g_utf8_next_char (p)) - p > 1)
            {
              while (p < r)
                *q++ = *p++;
            }
          else if (g_strchr (allowed, *p))
            {
              *q++ = *p++;
            }
          else
            {
              const static gchar hex[16] = {'0', '1', '2', '3',
                                            '4', '5', '6', '7',
                                            '8', '9', 'A', 'B',
                                            'C', 'D', 'E', 'F'};

              *q++ = '%';
              *q++ = hex[((guchar)*p) >> 4];
              *q++ = hex[((guchar)*p++) & 15];
            }
        }
      g_free (allowed);
    }
  return ret;
}


gchar *
g_uri_unescape_string (const gchar *escaped_string,
                       const gchar *illegal_characters)
{
  return g_uri_unescape_segment (escaped_string, NULL, illegal_characters);
}


gchar *
g_uri_unescape_segment (const gchar *escaped_string,
                        const gchar *escaped_string_end,
                        const gchar *illegal_characters)
{
  gchar *ret = NULL;

  if (escaped_string)
    {
      const gchar *end, *p;
      gchar *q;

      p = escaped_string;
      end = escaped_string_end ? escaped_string_end : p + g_strlen (p);
      q = ret = g_malloc ((end - escaped_string + 1) * sizeof (gchar));
      while (p < end)
        {
          gchar c;

          c = *p++;
          if (c == '%')
            {
              gint c1, c2;

              c = (c1 = g_ascii_xdigit_value (*p++)) >= 0
               && (c2 = g_ascii_xdigit_value (*p++)) >= 0
                                                    ? (c1 << 4) | c2 : '\0';
              if (c == '\0' || g_strchr (illegal_characters, c))
                {
                  g_free (ret);
                  ret = NULL;
                  break;
                }
            }
          *q++ = c;
        }
    }
  return ret;
}
#endif /* not GLIB_CHECK_VERSION(2,16,0) */


/******************************************************************************
* Directory Utilities                                                         *
******************************************************************************/
#ifdef USE_GTK_EMULATE
# if defined (G_OS_WIN32) && ! defined (HAVE_DIRENT_H)
#  ifdef UNICODE
#   define tdirent wdirent
#   define TDIR WDIR
#   define topendir wopendir
#   define treaddir wreaddir
#   define tclosedir wclosedir
typedef struct wdirent
{
  long           d_ino;
  unsigned short d_reclen;
  unsigned short d_namlen;
  wchar_t        d_name[MAX_PATH];
} WDIRENT;
typedef struct wdir
{
  int            nStat;
  LPWSTR         lpszPath;
  HANDLE         hFind;
  struct wdirent dd_dir;
} WDIR;
#  else /* not UNICODE */
#   define tdirent dirent
#   define TDIR DIR
#   define topendir opendir
#   define treaddir readdir
#   define tclosedir closedir
typedef struct dirent
{
  long           d_ino;
  unsigned short d_reclen;
  unsigned short d_namlen;
  char           d_name[MAX_PATH];
} DIRENT;
typedef struct dir
{
  int           nStat;
  LPSTR         lpszPath;
  HANDLE        hFind;
  struct dirent dd_dir;
} DIR;
#  endif /* not UNICODE */


TDIR *
topendir (const TCHAR *name)
{
  int i;
  DWORD dwResult;
  HANDLE hFind;
  LPTSTR lpszFile, lpszName;
  WIN32_FIND_DATA fnData;
  TDIR *nDir;

  if (!name)
    {
      errno = EFAULT;
      return NULL;
    }
  if (name[0] == '\0')
    {
      errno = ENOTDIR;
      return NULL;
    }
  lpszName = HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY,
                                        (lstrlen (name) + 1) * sizeof (TCHAR));
  lstrcpy (lpszName, name);
#  ifdef UNICODE
  for (i = 0; lpszName[i] != '\0'; i++)
    if (lpszName[i] == '/')
      lpszName[i] = '\\';
#  else /* not UNICODE */
  i = 0;
  while (lpszName[i] != '\0')
    if (IsDBCSLeadByteEx (CP_ACP, lpszName[i]))
      {
        i += 2;
      }
    else
      {
        if (lpszName[i] == '/')
          lpszName[i] = '\\';
        i++;
      }
#  endif /* not UNICODE */
  dwResult = GetFullPathName (lpszName, 0, NULL, &lpszFile);
  if (dwResult <= 0)
    {
      HeapFree (GetProcessHeap (), 0, lpszName);
      errno = ENOTDIR;
      return NULL;
    }
  nDir = HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY, sizeof (TDIR));
  nDir->lpszPath = HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY,
                                            (dwResult + 2) * sizeof (TCHAR));
  dwResult = GetFullPathName (lpszName, dwResult, nDir->lpszPath, &lpszFile);
  HeapFree (GetProcessHeap (), 0, lpszName);
  if (dwResult <= 0)
    {
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = ENOTDIR;
      return NULL;
    }
  if (nDir->lpszPath[dwResult - 1] != '\\')
    lstrcat (nDir->lpszPath, _T("\\"));
  lstrcat (nDir->lpszPath, _T("*"));
  hFind = FindFirstFile (nDir->lpszPath, &fnData);
  if (hFind == INVALID_HANDLE_VALUE)
    {
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = ENOTDIR;
      return NULL;
    }
  if (!FindClose (hFind))
    {
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = EFAULT;
      return NULL;
    }
  nDir->hFind = INVALID_HANDLE_VALUE;
  errno = 0;
  return nDir;
}


struct tdirent *
treaddir (TDIR *nDir)
{
  WIN32_FIND_DATA fnData;

  if (!nDir)
    {
      errno = EFAULT;
      return NULL;
    }
  if (nDir->nStat == 0)
    {
      nDir->hFind = FindFirstFile (nDir->lpszPath, &fnData);
      nDir->nStat = nDir->hFind == INVALID_HANDLE_VALUE ? -1 : 1;
    }
  else if (nDir->nStat > 0)
    {
      if (FindNextFile (nDir->hFind, &fnData))
        {
          nDir->nStat++;
        }
      else
        {
          FindClose (nDir->hFind);
          nDir->hFind = INVALID_HANDLE_VALUE;
          nDir->nStat = -1;
        }
    }
  errno = 0;
  if (nDir->nStat <= 0)
    return NULL;
  nDir->dd_dir.d_namlen = lstrlen (fnData.cFileName);
  lstrcpy (nDir->dd_dir.d_name, fnData.cFileName);
  return &nDir->dd_dir;
}


int
tclosedir (TDIR *nDir)
{
  int ret;

  if (nDir)
    {
      ret = nDir->hFind == INVALID_HANDLE_VALUE || FindClose (nDir->hFind)
                                                                    ? 0 : -1;
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = 0;
    }
  else
    {
      ret = -1;
      errno = EFAULT;
    }
  return ret;
}
# endif /* defined (G_OS_WIN32) && ! defined (HAVE_DIRENT_H) */


struct _GDir
{
# if defined (G_OS_WIN32) && defined (UNICODE)
  WDIR *dirp;
  gchar *name;
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
  DIR *dirp;
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
};


GDir *
g_dir_open (const gchar  *path,
            guint         flags,
            GError      **error)
{
  if (path)
    {
      GDir *dir;
# if defined (G_OS_WIN32) && defined (UNICODE)
      gchar *utf8str;
      wchar_t *wpath;

      dir = g_malloc (sizeof (GDir));
      utf8str = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
      wpath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      dir->dirp = wopendir (wpath);
      g_free (wpath);
      dir->name = NULL;
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
      dir = g_malloc (sizeof (GDir));
      dir->dirp = opendir (path);
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
      if (dir->dirp)
        return dir;
      g_free (dir);
    }
  return NULL;
}


const gchar *
g_dir_read_name (GDir *dir)
{
# if defined (G_OS_WIN32) && defined (UNICODE)
  struct wdirent *entry = NULL;

  g_free (dir->name);
  dir->name = NULL;
  if (dir)
    {
      entry = wreaddir (dir->dirp);
      while (entry && (wcscmp (entry->d_name, L".") == 0
                                        || wcscmp (entry->d_name, L"..") == 0))
        entry = wreaddir (dir->dirp);
    }
  if (entry)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (entry->d_name, -1, NULL, NULL, NULL);
      dir->name = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
    }
  return dir->name;
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
  struct dirent *entry = NULL;

  if (dir)
    {
      entry = readdir (dir->dirp);
      while (entry && (g_strcmp (entry->d_name, ".") == 0
                                    || g_strcmp (entry->d_name, "..") == 0))
        entry = readdir (dir->dirp);
    }
  return entry ? entry->d_name : NULL;
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
}


void
g_dir_close (GDir *dir)
{
  if (dir)
    {
# if defined (G_OS_WIN32) && defined (UNICODE)
      wclosedir (dir->dirp);
      g_free (dir->name);
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
      closedir (dir->dirp);
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
      g_free (dir);
    }
}
#endif /* USE_GTK_EMULATE */


/******************************************************************************
* Windows Compatibility Functions                                             *
******************************************************************************/
#if defined (G_OS_WIN32) && ! GLIB_CHECK_VERSION(2,8,0)
gchar *
g_win32_locale_filename_from_utf8 (const gchar *utf8filename)
{
# if GLIB_CHECK_VERSION(2,6,0)
  return g_locale_from_utf8 (utf8filename, -1, NULL, NULL, NULL);
# else /* not GLIB_CHECK_VERSION(2,6,0) */
  return g_strdup (utf8filename);
# endif /* not GLIB_CHECK_VERSION(2,6,0) */
}
#endif /* defined (G_OS_WIN32) && ! GLIB_CHECK_VERSION(2,8,0) */


/******************************************************************************
* Doubly-Linked Lists                                                         *
******************************************************************************/
#ifdef USE_GTK_EMULATE
GList *
g_list_append (GList    *glist,
               gpointer  data)
{
  GList *gl;

  gl = g_malloc (sizeof (GList));
  gl->data = data;
  gl->next = NULL;
  if (glist)
    {
      glist = g_list_last (glist);
      glist->next = gl;
      gl->prev = glist;
    }
  else
    {
      gl->prev = NULL;
    }
  return gl;
}


GList *
g_list_prepend (GList    *glist,
                gpointer  data)
{
  GList *gl;

  gl = g_malloc (sizeof (GList));
  gl->data = data;
  gl->prev = NULL;
  if (glist)
    {
      glist = g_list_first (glist);
      glist->prev = gl;
      gl->next = glist;
    }
  else
    {
      gl->next = NULL;
    }
  return gl;
}


GList *
g_list_insert_sorted (GList        *glist,
                      gpointer      data,
                      GCompareFunc  compare)
{
  if (compare)
    {
      GList *gl;

      for (gl = g_list_last (glist); gl; gl = g_list_previous (gl))
        if (compare (gl->data, data) <= 0)
          break;
      if (gl)
        {
          glist = gl;
          gl = g_malloc (sizeof (GList));
          gl->data = data;
          gl->prev = glist;
          gl->next = glist->next;
          if (glist->next)
            glist->next->prev = gl;
          glist->next = gl;
        }
      else
        {
          glist = g_list_prepend (glist, data);
        }
    }
  return glist;
}


GList *
g_list_remove (GList         *glist,
               gconstpointer  data)
{
  GList *gl;

  for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
    if (gl->data == data)
      return g_list_delete_link (glist, gl);
  return glist;
}


GList *
g_list_remove_link (GList *glist,
                    GList *glink)
{
  GList *gl = NULL;

  if (glink)
    {
      if (glink->prev)
        glink->prev->next = glink->next;
      if (glink->next)
        glink->next->prev = glink->prev;
      gl = glink->prev ? glink->prev : glink->next;
      glink->prev = glink->next = NULL;
    }
  return glist == glink ? gl : glist;
}


GList *
g_list_delete_link (GList *glist,
                    GList *glink)
{
  GList *gl;

  gl = g_list_remove_link (glist, glink);
  g_free (glink);
  return gl;
}


void
g_list_free (GList *glist)
{
  while (glist)
    glist = g_list_delete_link (glist, glist);
}


guint
g_list_length (GList *glist)
{
  guint n = 0;

  for (glist = g_list_first (glist); glist; glist = g_list_next (glist))
    n++;
  return n;
}


GList *
g_list_concat (GList *glist1,
               GList *glist2)
{
  if (glist1 && glist2)
    {
      glist1 = g_list_last (glist1);
      glist2 = g_list_first (glist2);
      glist1->next = glist2;
      glist2->prev = glist1;
    }
  return glist1 ? glist1 : glist2;
}


GList *
g_list_first (GList *glist)
{
  if (glist)
    while (glist->prev)
      glist = glist->prev;
  return glist;
}


GList *
g_list_last (GList *glist)
{
  if (glist)
    while (glist->next)
      glist = glist->next;
  return glist;
}


gpointer
g_list_nth_data (GList *glist,
                 guint  n)
{
  guint i;
  GList *gl;

  for (i = 0, gl = g_list_first (glist);
                                    i < n && gl; i++, gl = g_list_next (gl));
  return gl ? gl->data : NULL;
}


GList *
g_list_find (GList         *glist,
             gconstpointer  data)
{
  GList *gl;

  for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
    if (gl->data == data)
      return gl;
  return NULL;
}


GList *
g_list_find_custom (GList         *glist,
                    gconstpointer  data,
                    GCompareFunc   compare)
{
  if (compare)
    {
      while (glist)
        {
          if (compare (glist->data, data) == 0)
            return glist;
          glist = g_list_next (glist);
        }
    }
  return NULL;
}
#endif /* USE_GTK_EMULATE */


/******************************************************************************
* Hash Tables                                                                 *
******************************************************************************/
#ifdef USE_GTK_EMULATE
# define HASH_TABLE_SHIFT 13
# define HASH_TABLE_SIZE (1 << HASH_TABLE_SHIFT)
# define HASH_TABLE_MASK (HASH_TABLE_SIZE - 1)


struct _GHashTable
{
  GHashFunc hash_func;
  GEqualFunc key_equal_func;
  GDestroyNotify key_destroy_func;
  GDestroyNotify value_destroy_func;
  GList *glist[HASH_TABLE_SIZE];
};
typedef struct _GHashNode
{
  gpointer key;
  gpointer value;
} GHashNode;


GHashTable *
g_hash_table_new (GHashFunc  hash_func,
                  GEqualFunc key_equal_func)
{
  return g_hash_table_new_full (hash_func, key_equal_func, NULL, NULL);
}


GHashTable *
g_hash_table_new_full (GHashFunc      hash_func,
                       GEqualFunc     key_equal_func,
                       GDestroyNotify key_destroy_func,
                       GDestroyNotify value_destroy_func)
{
  GHashTable *hash_table;

  hash_table = g_malloc0 (sizeof (GHashTable));
  hash_table->hash_func = hash_func ? hash_func : g_direct_hash;
  hash_table->key_equal_func = key_equal_func ? key_equal_func
                                              : g_direct_equal;
  hash_table->key_destroy_func = key_destroy_func;
  hash_table->value_destroy_func = value_destroy_func;
  return hash_table;
}


static GHashNode *
g_hash_table_lookup_node (GHashTable    *hash_table,
                          gconstpointer  key)
{
  if (hash_table)
    {
      guint hash;
      GList *glist;

      hash = hash_table->hash_func (key) & HASH_TABLE_MASK;
      for (glist = g_list_first (hash_table->glist[hash]);
                                            glist; glist = g_list_next (glist))
        {
          GHashNode *node;

          node = glist->data;
          if (hash_table->key_equal_func (node->key, key))
            {
              hash_table->glist[hash]
                        = g_list_delete_link (hash_table->glist[hash], glist);
              hash_table->glist[hash]
                        = g_list_prepend (hash_table->glist[hash], node);
              return node;
            }
        }
    }
  return NULL;
}


void
g_hash_table_insert (GHashTable *hash_table,
                     gpointer    key,
                     gpointer    value)
{
  if (hash_table)
    {
      GHashNode *node;

      node = g_hash_table_lookup_node (hash_table, key);
      if (node)
        {
          if (hash_table->key_destroy_func)
            hash_table->key_destroy_func (key);
          if (hash_table->value_destroy_func)
            hash_table->value_destroy_func (node->value);
          node->value = value;
        }
      else
        {
          guint hash;

          hash = hash_table->hash_func (key) & HASH_TABLE_MASK;
          node = g_malloc (sizeof (GHashNode));
          node->key = key;
          node->value = value;
          hash_table->glist[hash] = g_list_append (hash_table->glist[hash],
                                                                        node);
        }
    }
}


guint
g_hash_table_size (GHashTable *hash_table)
{
  guint size = 0;

  if (hash_table)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        size += g_list_length (hash_table->glist[i]);
    }
  return size;
}


gpointer
g_hash_table_lookup (GHashTable    *hash_table,
                     gconstpointer  key)
{
  GHashNode *node;

  node = g_hash_table_lookup_node (hash_table, key);
  return node ? node->value : NULL;
}


void
g_hash_table_foreach (GHashTable *hash_table,
                      GHFunc      func,
                      gpointer    user_data)
{
  if (hash_table && func)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        {
          GList *glist;

          for (glist = g_list_first (hash_table->glist[i]);
                                            glist; glist = g_list_next (glist))
            {
              GHashNode *node;

              node = glist->data;
              func (node->key, node->value, user_data);
            }
        }
    }
}


gboolean
g_hash_table_remove (GHashTable    *hash_table,
                     gconstpointer  key)
{
  if (hash_table)
    {
      guint hash;
      GList *glist;

      hash = hash_table->hash_func (key) & HASH_TABLE_MASK;
      for (glist = g_list_first (hash_table->glist[hash]);
                                            glist; glist = g_list_next (glist))
        {
          GHashNode *node;

          node = glist->data;
          if (hash_table->key_equal_func (node->key, key))
            {
              hash_table->glist[hash]
                        = g_list_delete_link (hash_table->glist[hash], glist);
              if (hash_table->key_destroy_func)
                hash_table->key_destroy_func (node->key);
              if (hash_table->value_destroy_func)
                hash_table->value_destroy_func (node->value);
              g_free (node);
              return TRUE;
            }
        }
    }
  return FALSE;
}


guint
g_hash_table_foreach_remove (GHashTable *hash_table,
                             GHRFunc     func,
                             gpointer    user_data)
{
  guint removed = 0;

  if (hash_table && func)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        {
          GList *glist;

          glist = g_list_first (hash_table->glist[i]);
          while (glist)
            {
              GList *gl;
              GHashNode *node;

              gl = g_list_next (glist);
              node = glist->data;
              if (func (node->key, node->value, user_data))
                {
                  hash_table->glist[i]
                            = g_list_delete_link (hash_table->glist[i], glist);
                  if (hash_table->key_destroy_func)
                    hash_table->key_destroy_func (node->key);
                  if (hash_table->value_destroy_func)
                    hash_table->value_destroy_func (node->value);
                  g_free (node);
                  removed++;
                }
              glist = gl;
            }
        }
    }
  return removed;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,14,0)
static void
g_hash_table_get_keys_func (gpointer key,
                            gpointer value,
                            gpointer user_data)
{
  GList **glist;

  glist = user_data;
  *glist = g_list_append (*glist, key);
}


GList *
g_hash_table_get_keys (GHashTable *hash_table)
{
  GList *glist = NULL;

  if (hash_table)
    g_hash_table_foreach (hash_table, g_hash_table_get_keys_func, &glist);
  return glist;
}


static void
g_hash_table_get_values_func (gpointer key,
                              gpointer value,
                              gpointer user_data)
{
  GList **glist;

  glist = user_data;
  *glist = g_list_append (*glist, value);
}


GList *
g_hash_table_get_values (GHashTable *hash_table)
{
  GList *glist = NULL;

  if (hash_table)
    g_hash_table_foreach (hash_table, g_hash_table_get_values_func, &glist);
  return glist;
}
#endif /* not GLIB_CHECK_VERSION(2,14,0) */


#ifdef USE_GTK_EMULATE
void
g_hash_table_destroy (GHashTable *hash_table)
{
  if (hash_table)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        while (hash_table->glist[i])
          {
            GHashNode *node;

            node = hash_table->glist[i]->data;
            if (hash_table->key_destroy_func)
              hash_table->key_destroy_func (node->key);
            if (hash_table->value_destroy_func)
              hash_table->value_destroy_func (node->value);
            g_free (node);
            hash_table->glist[i] = g_list_delete_link (hash_table->glist[i],
                                                       hash_table->glist[i]);
          }
      g_free (hash_table);
    }
}


gboolean
g_direct_equal (gconstpointer v1,
                gconstpointer v2)
{
  return v1 == v2;
}


guint
g_direct_hash (gconstpointer v)
{
  return GPOINTER_TO_UINT (v);
}


gboolean
g_int_equal (gconstpointer v1,
             gconstpointer v2)
{
  return *((const gint *)v1) == *((const gint *)v2);
}


guint
g_int_hash (gconstpointer v)
{
  return *(const gint *)v;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,22,0)
gboolean
g_int64_equal (gconstpointer v1,
               gconstpointer v2)
{
  return *((const gint64 *)v1) == *((const gint64 *)v2);
}


guint
g_int64_hash (gconstpointer v)
{
  return (guint)*(const gint64 *)v;
}


gboolean
g_double_equal (gconstpointer v1,
                gconstpointer v2)
{
  return *((const gdouble *)v1) == *((const gdouble *)v2);
}


guint
g_double_hash (gconstpointer v)
{
  return (guint)*(const gdouble *)v;
}
#endif /* not GLIB_CHECK_VERSION(2,22,0) */


#ifdef USE_GTK_EMULATE
gboolean
g_str_equal (gconstpointer v1,
             gconstpointer v2)
{
  return g_strcmp ((const gchar *)v1, (const gchar *)v2) == 0;
}


guint
g_str_hash (gconstpointer v)
{
  const gchar *p;
  guint32 h = 5381;

  if (v)
    for (p = v; *p != '\0'; p++)
      h = (h << 5) + h + *p;
  return h;
}
#endif /* USE_GTK_EMULATE */


/******************************************************************************
* Main loop and Events                                                        *
******************************************************************************/
#ifdef USE_GTK_EMULATE
gchar *
gtk_set_locale (void)
{
  return setlocale (LC_ALL, NULL);
}

void
gtk_init (int    *argc,
          char ***argv)
{
  g_set_prgname (argv && *argv ? g_path_get_basename ((*argv)[0]) : NULL);
}
#endif /* USE_GTK_EMULATE */


/******************************************************************************
* Selections                                                                  *
******************************************************************************/
#ifndef USE_GTK_EMULATE
# if ! GTK_CHECK_VERSION(2,16,0)
GdkAtom
gtk_selection_data_get_selection (GtkSelectionData *selection_data)
{
  return selection_data ? selection_data->selection : 0;
}
# endif /* not GTK_CHECK_VERSION(2,16,0) */


# if ! GTK_CHECK_VERSION(2,14,0)
const guchar *
gtk_selection_data_get_data (GtkSelectionData *selection_data)
{
  return selection_data ? selection_data->data : NULL;
}


gint
gtk_selection_data_get_length (GtkSelectionData *selection_data)
{
  return selection_data ? selection_data->length : -1;
}


GdkAtom
gtk_selection_data_get_data_type (GtkSelectionData *selection_data)
{
  return selection_data ? selection_data->type : 0;
}


GdkDisplay *
gtk_selection_data_get_display (GtkSelectionData *selection_data)
{
  return selection_data ? selection_data->display : NULL;
}


gint
gtk_selection_data_get_format (GtkSelectionData *selection_data)
{
  return selection_data ? selection_data->format : 0;
}


GdkAtom
gtk_selection_data_get_target (GtkSelectionData *selection_data)
{
  return selection_data ? selection_data->target : 0;
}
# endif /* not GTK_CHECK_VERSION(2,1,0) */
#endif /* not USE_GTK_EMULATE */


/******************************************************************************
* Signals                                                                     *
******************************************************************************/
#if ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,16,0)
GdkAtom
gtk_selection_data_get_selection (GtkSelectionData *selection_data)
{
  return selection_data->target;
}
#endif /* ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,16,0) */


/******************************************************************************
* GtkDialog                                                                   *
******************************************************************************/
#if ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,14,0)
GtkWidget *
gtk_dialog_get_action_area (GtkDialog *dialog)
{
  return GTK_DIALOG (dialog)->action_area;
}


GtkWidget *
gtk_dialog_get_content_area (GtkDialog *dialog)
{
  return GTK_DIALOG (dialog)->vbox;
}
#endif /* ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,14,0) */


/******************************************************************************
* GtkColorSelectionDialog                                                     *
******************************************************************************/
#if ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,14,0)
GtkWidget *
gtk_color_selection_dialog_get_color_selection (GtkColorSelectionDialog *colorsel)
{
  return GTK_COLOR_SELECTION_DIALOG (dialog)->colorsel;
}
#endif /* ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,14,0) */


/******************************************************************************
* GtkAdjustment                                                               *
******************************************************************************/
#if ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,14,0)
gdouble
gtk_adjustment_get_lower (GtkAdjustment *adjustment)
{
  return adjustment ? adjustment->lower : 0;
}


void
gtk_adjustment_set_lower (GtkAdjustment *adjustment,
                          gdouble        lower)
{
  if (adjustment)
    adjustment->lower = lower;
}


gdouble
gtk_adjustment_get_upper (GtkAdjustment *adjustment)
{
  return adjustment ? adjustment->upper : 0;
}


void
gtk_adjustment_set_upper (GtkAdjustment *adjustment,
                          gdouble        upper)
{
  if (adjustment)
    adjustment->upper = upper;
}


gdouble
gtk_adjustment_get_step_increment (GtkAdjustment *adjustment)
{
  return adjustment ? adjustment->step_increment : 0;
}


void
gtk_adjustment_set_step_increment (GtkAdjustment *adjustment,
                                   gdouble        step_increment)
{
  if (adjustment)
    adjustment->step_increment = step_increment;
}


gdouble
gtk_adjustment_get_page_increment (GtkAdjustment *adjustment)
{
  return adjustment ? adjustment->page_increment : 0;
}


void
gtk_adjustment_set_page_increment (GtkAdjustment *adjustment,
                                   gdouble        page_increment)
{
  if (adjustment)
    adjustment->page_increment = page_increment;
}


gdouble
gtk_adjustment_get_page_size (GtkAdjustment *adjustment)
{
  return adjustment ? adjustment->page_size : 0;
}


void
gtk_adjustment_set_page_size (GtkAdjustment *adjustment,
                              gdouble        page_size)
{
  if (adjustment)
    adjustment->page_size = page_size;
}
#endif /* ! defined (USE_GTK_EMULATE) && ! GTK_CHECK_VERSION(2,14,0) */


/******************************************************************************
* GtkWidget                                                                   *
******************************************************************************/
#ifndef USE_GTK_EMULATE
# if ! GTK_CHECK_VERSION(2,14,0)
GdkWindow *
gtk_widget_get_window (GtkWidget *widget)
{
  return widget->window;
}
# endif /* not GTK_CHECK_VERSION(2,14,0) */


# if ! GTK_CHECK_VERSION(2,18,0)
void
gtk_widget_get_allocation (GtkWidget     *widget,
                           GtkAllocation *allocation)
{
  *allocation = widget->allocation;
}


void
gtk_widget_set_allocation (GtkWidget           *widget,
                           const GtkAllocation *allocation)
{
  widget->allocation = *allocation;
}


GtkStateType
gtk_widget_get_state (GtkWidget *widget)
{
  return GTK_WIDGET_STATE (widget);
}


gboolean
gtk_widget_is_toplevel (GtkWidget *widget)
{
  return GTK_WIDGET_TOPLEVEL (widget);
}


gboolean
gtk_widget_get_has_window (GtkWidget *widget)
{
  return ! GTK_WIDGET_NO_WINDOW (widget);
}


void
gtk_widget_set_has_window (GtkWidget *widget,
                           gboolean   has_window)
{
  if (has_window)
    GTK_WIDGET_UNSET_FLAGS (widget, GTK_NO_WINDOW);
  else
    GTK_WIDGET_SET_FLAGS (widget, GTK_NO_WINDOW);
}
# endif /* not GTK_CHECK_VERSION(2,18,0) */


# if ! GTK_CHECK_VERSION(2,20,0)
gboolean
gtk_widget_get_realized (GtkWidget *widget)
{
  return GTK_WIDGET_REALIZED (widget);
}


void
gtk_widget_set_realized (GtkWidget *widget,
                         gboolean   realized)
{
  if (realized)
    GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  else
    GTK_WIDGET_UNSET_FLAGS (widget, GTK_REALIZED);
}


gboolean
gtk_widget_get_mapped (GtkWidget *widget)
{
  return GTK_WIDGET_MAPPED (widget);
}


void
gtk_widget_set_mapped (GtkWidget *widget,
                       gboolean   mapped)
{
  if (mapped)
    GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);
  else
    GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);
}
# endif /* not GTK_CHECK_VERSION(2,20,0) */


# if ! GTK_CHECK_VERSION(2,18,0)
gboolean
gtk_widget_get_visible (GtkWidget *widget)
{
  return GTK_WIDGET_VISIBLE (widget);
}


void
gtk_widget_set_visible (GtkWidget *widget,
                        gboolean   visible)
{
  if (visible)
    gtk_widget_show (widget);
  else
    gtk_widget_hide (widget);
}


gboolean
gtk_widget_is_drawable (GtkWidget *widget)
{
  return GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget);
}


gboolean
gtk_widget_get_sensitive (GtkWidget *widget)
{
  return GTK_WIDGET_SENSITIVE (widget);
}


gboolean gtk_widget_is_sensitive (GtkWidget *widget)
{
  return GTK_WIDGET_SENSITIVE (widget) && GTK_WIDGET_PARENT_SENSITIVE (widget);
}


gboolean
gtk_widget_get_can_focus (GtkWidget *widget)
{
  return GTK_WIDGET_CAN_FOCUS (widget);
}


void
gtk_widget_set_can_focus (GtkWidget *widget,
                          gboolean   can_focus)
{
  if (can_focus)
    GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_FOCUS);
  else
    GTK_WIDGET_UNSET_FLAGS (widget, GTK_CAN_FOCUS);
}


gboolean
gtk_widget_has_focus (GtkWidget *widget)
{
  return GTK_WIDGET_HAS_FOCUS (widget);
}


gboolean
gtk_widget_get_can_default (GtkWidget *widget)
{
  return GTK_WIDGET_CAN_DEFAULT (widget);
}


void
gtk_widget_set_can_default (GtkWidget *widget,
                            gboolean   can_default)
{
  if (can_default)
    GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_DEFAULT);
  else
    GTK_WIDGET_UNSET_FLAGS (widget, GTK_CAN_DEFAULT);
}


gboolean
gtk_widget_get_receives_default (GtkWidget *widget)
{
  return GTK_WIDGET_RECEIVES_DEFAULT (widget);
}


void
gtk_widget_set_receives_default (GtkWidget *widget,
                                 gboolean   receives_default)
{
  if (receives_default)
    GTK_WIDGET_SET_FLAGS (widget, GTK_RECEIVES_DEFAULT);
  else
    GTK_WIDGET_UNSET_FLAGS (widget, GTK_RECEIVES_DEFAULT);
}


gboolean
gtk_widget_has_default (GtkWidget *widget)
{
  return GTK_WIDGET_HAS_DEFAULT (widget);
}


gboolean
gtk_widget_has_grab (GtkWidget *widget)
{
  return GTK_WIDGET_HAS_GRAB (widget);
}
# endif /* not GTK_CHECK_VERSION(2,18,0) */


# if ! GTK_CHECK_VERSION(2,20,0)
gboolean
gtk_widget_has_rc_style (GtkWidget *widget)
{
  return GTK_WIDGET_RC_STYLE (widget);
}
# endif /* not GTK_CHECK_VERSION(2,20,0) */


# if ! GTK_CHECK_VERSION(2,18,0)
gboolean
gtk_widget_get_app_paintable (GtkWidget *widget)
{
  return GTK_WIDGET_APP_PAINTABLE (widget);
}


gboolean
gtk_widget_get_double_buffered (GtkWidget *widget)
{
  return GTK_WIDGET_DOUBLE_BUFFERED (widget);
}
# endif /* not GTK_CHECK_VERSION(2,18,0) */
#endif /* not USE_GTK_EMULATE */


/******************************************************************************
* Extended                                                                    *
******************************************************************************/
gchar *
g_strcpy (gchar       *dest,
          const gchar *src)
{
  return dest && src ? strcpy (dest, src) : dest;
}


gchar *
g_strncpy (gchar       *dest,
           const gchar *src,
           const gsize  n)
{
  return dest && src ? strncpy (dest, src, n) : dest;
}


gchar *
g_strcat (gchar       *dest,
          const gchar *src)
{
  return dest && src ? strcat (dest, src) : dest;
}


gchar *
g_strncat (gchar       *dest,
           const gchar *src,
           const gsize  n)
{
  return dest && src ? strncat (dest, src, n) : dest;
}


gint
g_strcmp (const gchar *s1,
          const gchar *s2)
{
  if (!s1 && !s2)
    return 0;
  else if (s1 && !s2)
    return G_MININT;
  else if (!s1 && s2)
    return G_MAXINT;
  return strcmp (s1, s2);
}


gint
g_strncmp (const gchar *s1,
           const gchar *s2,
           const gsize  n)
{
  if (!s1 && !s2)
    return 0;
  else if (s1 && !s2)
    return G_MININT;
  else if (!s1 && s2)
    return G_MAXINT;
  return strncmp (s1, s2, n);
}


gchar *
g_strchr (const gchar *s,
          const gint   c)
{
  return s ? strchr (s, c) : NULL;
}


gchar *
g_strrchr (const gchar *s,
           const gint   c)
{
  return s ? strrchr (s, c) : NULL;
}


gsize
g_strspn (const gchar *s,
          const gchar *accept)
{
  return s && accept ? strspn (s, accept) : 0;
}


gsize
g_strcspn (const gchar *s,
           const gchar *reject)
{
  return s && reject ? strcspn (s, reject) : 0;
}


gchar *
g_strpbrk (const gchar *s,
           const gchar *accept)
{
  return s && accept ? strpbrk (s, accept) : NULL;
}


gchar *
g_strstr (const gchar *haystack,
          const gchar *needle)
{
  return g_strstr_len (haystack, -1, needle);
}


gsize
g_strlen (const gchar *s)
{
  return s ? strlen (s) : 0;
}


gchar *
g_strtok (gchar       *str,
          const gchar *delim)
{
  return delim ? strtok (str, delim) : NULL;
}


glong
g_strtol (const gchar  *nptr,
          gchar       **endptr,
          gint          base)
{
  glong ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = strtol (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}


gulong
g_strtoul (const gchar  *nptr,
           gchar       **endptr,
           gint          base)
{
  gulong ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = strtoul (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}


gint
g_memcmp (gconstpointer s1,
          gconstpointer s2,
          const gsize   n)
{
  if (!s1 && !s2)
    return 0;
  else if (s1 && !s2)
    return G_MININT;
  else if (!s1 && s2)
    return G_MAXINT;
  return memcmp (s1, s2, n);
}


gpointer
g_memchr (gconstpointer s,
          const gint    c,
          const gsize   n)
{
  return s ? memchr (s, c, n) : NULL;
}


gpointer
g_memset (gpointer    s,
          const gint  c,
          const gsize n)
{
  return s ? memset (s, c, n) : s;
}


gint
g_asprintf (gchar       **string,
            const gchar  *format,
            ...)
{
  gsize len;
  va_list args;

  va_start (args, format);
  len = g_vasprintf (string, format, args);
  va_end (args);
  return len;
}
