/*
    orzcomm
    copyright (c) 1998-2012 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "orzcomm.h"
#include "misc/fileio.h"
#ifdef G_OS_WIN32
# include <tchar.h>
#endif /* G_OS_WIN32 */


enum
{
  RECV_SIGNAL,
  LAST_SIGNAL
};
#ifndef G_OS_WIN32
enum
{
  TARGET_STRING
};
#endif /* not G_OS_WIN32 */


static gint orz_comm_signals[LAST_SIGNAL] = {0};
#ifdef G_OS_WIN32
static GList *glist = NULL;
#endif /* G_OS_WIN32 */
# ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
# endif /* USE_THREAD */


/******************************************************************************
*                                                                             *
******************************************************************************/
G_DEFINE_TYPE (OrzComm, orz_comm, G_TYPE_OBJECT);
static void orz_comm_dispose (GObject *object);


static void
orz_comm_class_init (OrzCommClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->dispose = orz_comm_dispose;

  klass->recv = NULL;

  orz_comm_signals[RECV_SIGNAL]
        = g_signal_new ("recv",
                G_TYPE_FROM_CLASS (klass),
                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                G_STRUCT_OFFSET (OrzCommClass, recv),
                NULL, NULL,
                g_cclosure_marshal_VOID__POINTER,
                G_TYPE_NONE, 1,
                G_TYPE_POINTER);
}


/*  ja:新規作成
    RET,オブジェクト                                                        */
static void
orz_comm_init (OrzComm *comm)
{
  comm->handler_destroy  = 0;
  comm->window           = NULL;
#ifdef G_OS_WIN32
  comm->ddeInst          = 0;
  comm->hszService       = NULL;
  comm->hszTopic         = NULL;
  comm->hConv            = NULL;
#else /* not G_OS_WIN32 */
  comm->instance         = -1;
  comm->first            = TRUE;
  comm->arg              = NULL;
  comm->handler_get      = 0;
  comm->handler_received = 0;
  comm->atom0            = NULL;
  comm->atom1            = NULL;
  comm->atom2            = NULL;
#endif /* not G_OS_WIN32 */
}


static void
orz_comm_dispose (GObject *object)
{
  OrzComm *comm;

  comm = ORZ_COMM (object);
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
#ifdef G_OS_WIN32
  if (comm->ddeInst)
    {
      if (comm->hConv)
        DdeDisconnect (comm->hConv);
      DdeNameService (comm->ddeInst, comm->hszService, NULL, DNS_UNREGISTER);
      DdeFreeStringHandle (comm->ddeInst, comm->hszService);
      DdeFreeStringHandle (comm->ddeInst, comm->hszTopic);
      DdeUninitialize (comm->ddeInst);
    }
  glist = g_list_remove (glist, comm);
#else /* not G_OS_WIN32 */
  g_free (comm->arg);
  if (comm->handler_get)
    g_signal_handler_disconnect (G_OBJECT (comm->window), comm->handler_get);
  if (comm->handler_received)
    g_signal_handler_disconnect (G_OBJECT (comm->window),
                                                    comm->handler_received);
#endif /* not G_OS_WIN32 */
  if (comm->handler_destroy)
    g_signal_handler_disconnect (G_OBJECT (comm->window),
                                                        comm->handler_destroy);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */

  if (G_OBJECT_CLASS (orz_comm_parent_class)->dispose)
    G_OBJECT_CLASS (orz_comm_parent_class)->dispose (object);
}


#ifdef G_OS_WIN32
static HDDEDATA CALLBACK
DdemlCallback (UINT     uType,
               UINT     uFmt,
               HCONV    hConv,
               HSZ      hszTopic,
               HSZ      hszService,
               HDDEDATA hData,
               DWORD    dwData1,
               DWORD    dwData2)
{
  GList *gl;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
    {
      OrzComm *comm;

      comm = gl->data;
      if (!comm)
        break;
      switch (uType)
        {
          case XTYP_CONNECT:
            return (HDDEDATA)
                    (DdeCmpStringHandles (hszService, comm->hszService) == 0
                    && DdeCmpStringHandles (hszTopic, comm->hszTopic) == 0);
          case XTYP_POKE:
            if (DdeCmpStringHandles (hszTopic, comm->hszTopic) == 0)
              {
                gchar *text;

                text = DdeAccessData (hData, NULL);
                if (text)
                  {
                    gchar **argv;

                    argv = g_strsplit (text, "\n", 0);
                    g_signal_emit_by_name (G_OBJECT (comm), "recv", argv);
                    g_strfreev (argv);
                  }
                DdeFreeDataHandle (hData);
                return (HDDEDATA)DDE_FACK;
              }
        }
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return NULL;
}


#else /* not G_OS_WIN32 */
static void
orz_comm_selection_get (GtkWidget        *widget,
                        GtkSelectionData *data,
                        guint             info,
                        guint             time,
                        OrzComm          *comm)
{
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (data->selection == comm->atom0)
    {
      /* ja:他プロセスがプロセスの存在を確認しにきたとき
            このプロセスの存在を知らせる */
      gtk_selection_data_set_text (data, "Orz Comm Selection", -1);
    }
  else if (data->selection == comm->atom1)
    {
      /* ja:他プロセスが引数の受渡しをするので
            このプロセスは引数を受け取りにいく */
      gtk_selection_convert (comm->window, comm->atom2,
                                GDK_SELECTION_TYPE_STRING, GDK_CURRENT_TIME);
    }
  else if (data->selection == comm->atom2 && comm->arg)
    {
      /* ja:他プロセスが引数を取りにきたのでこのプロセスの引数を渡す */
      gtk_selection_data_set_text (data, comm->arg, -1);
      g_free (comm->arg);
      comm->arg = NULL;
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
}


static void
orz_comm_selection_received (GtkWidget        *widget,
                             GtkSelectionData *data,
                             guint             time,
                             OrzComm          *comm)
{
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (data->selection == comm->atom0)
    {
      /* ja:他にプロセスが存在するならば何かデータが返ってきている */
      comm->instance = MAX (data->length, 0);
    }
  else if (data->selection == comm->atom2 && data->length > 0)
    {
      gchar *utf8str, *compressed, **argv;

      /* ja:他にプロセスのコマンドラインを受け取ったとき */
      utf8str = (gchar *)gtk_selection_data_get_text (data);
      compressed = g_strcompress (utf8str);
      g_free (utf8str);
      argv = g_strsplit (compressed, "\n", 0);
      g_free (compressed);
      g_signal_emit_by_name (G_OBJECT (comm), "recv", argv);
      g_strfreev (argv);
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
}
#endif /* not G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:プロセス間通信関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    window,ウィジェット
    unique,ユニークな文字列
       RET,オブジェクト                                                     */
GObject *
orz_comm_new (GtkWidget   *window,
              const gchar *unique)
{
  gchar *unique0, *unique1, *unique2;
  OrzComm *comm;
#ifdef G_OS_WIN32
  LPTSTR lpszUnique0, lpszUnique1;
#else /* not G_OS_WIN32 */
  GtkTargetEntry targets_comm[] = {{"STRING", 0, TARGET_STRING}};
#endif /* not G_OS_WIN32 */

  if (!window || !unique)
    return NULL;
  comm = ORZ_COMM (g_object_new (ORZ_TYPE_COMM, NULL));
#ifdef G_OS_WIN32
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  glist = g_list_append (glist, comm);
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
#endif /* G_OS_WIN32 */
  comm->window = window;
  comm->handler_destroy = g_signal_connect_swapped (G_OBJECT (window),
                    "destroy", G_CALLBACK (g_object_unref), G_OBJECT (comm));
  unique0 = g_strconcat (unique, " 0", NULL);
  unique1 = g_strconcat (unique, " 1", NULL);
  unique2 = g_strconcat (unique, " 2", NULL);
#ifdef G_OS_WIN32
# ifdef UNICODE
  lpszUnique0 = g_utf8_to_utf16 (unique0,  -1, NULL, NULL, NULL);
  lpszUnique1 = g_utf8_to_utf16 (unique1, -1, NULL, NULL, NULL);
# else /* not UNICODE */
  lpszUnique0 = g_locale_from_utf8 (unique0, -1, NULL, NULL, NULL);
  lpszUnique1 = g_locale_from_utf8 (unique1, -1, NULL, NULL, NULL);
# endif /* not UNICODE */
#else /* not G_OS_WIN32 */
  comm->atom0 = gdk_atom_intern (unique0, FALSE);
  comm->atom1 = gdk_atom_intern (unique1, FALSE);
  comm->atom2 = gdk_atom_intern (unique2, FALSE);
#endif /* not G_OS_WIN32 */
  g_free (unique0);
  g_free (unique1);
  g_free (unique2);
#ifdef G_OS_WIN32
  if (DdeInitialize (&comm->ddeInst, DdemlCallback,
                            APPCLASS_STANDARD | CBF_FAIL_SELFCONNECTIONS, 0)
                                                            != DMLERR_NO_ERROR)
    {
      g_free (lpszUnique0);
      g_free (lpszUnique1);
      return G_OBJECT (comm);
    }
# ifdef UNICODE
  comm->hszService = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique0, CP_WINUNICODE);
  comm->hszTopic   = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique1, CP_WINUNICODE));
# else /* not UNICODE */
  comm->hszService = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique0, CP_WINANSI);
  comm->hszTopic   = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique1, CP_WINANSI);
# endif /* not UNICODE */
  g_free (lpszUnique0);
  g_free (lpszUnique1);
  if (!comm->hszService || !comm->hszTopic)
    {
      if (comm->hszService)
        {
          DdeFreeStringHandle (comm->ddeInst, comm->hszService);
          comm->hszService = NULL;
        }
      if (comm->hszTopic)
        {
          DdeFreeStringHandle (comm->ddeInst, comm->hszTopic);
          comm->hszTopic = NULL;
        }
      DdeUninitialize (comm->ddeInst);
      comm->ddeInst = 0;
      return G_OBJECT (comm);
    }
  DdeNameService (comm->ddeInst, comm->hszService, NULL, DNS_REGISTER);
  comm->hConv = DdeConnect (comm->ddeInst,
                                    comm->hszService, comm->hszTopic, NULL);
#else /* not G_OS_WIN32 */
  gtk_selection_add_targets (window, comm->atom0, targets_comm, 1);
  gtk_selection_add_targets (window, comm->atom1, targets_comm, 1);
  gtk_selection_add_targets (window, comm->atom2, targets_comm, 1);
  comm->handler_get = g_signal_connect (G_OBJECT (window),
                            "selection-get",
                            G_CALLBACK (orz_comm_selection_get), comm);
  comm->handler_received = g_signal_connect (G_OBJECT (window),
                            "selection-received",
                            G_CALLBACK (orz_comm_selection_received), comm);

  gtk_selection_convert (window, comm->atom0,
                                GDK_SELECTION_TYPE_STRING, GDK_CURRENT_TIME);
# ifdef USE_THREAD
  gdk_threads_enter ();
# endif /* USE_THREAD */
  while (comm->instance < 0)
    while (gtk_events_pending ())
      gtk_main_iteration ();
# ifdef USE_THREAD
  gdk_threads_leave ();
# endif /* USE_THREAD */
  if (comm->instance > 0)
    {
      /* ja:既に起動しているとき */
      comm->first = FALSE;
    }
  else
    {
      /* ja:初めて起動してたとき */
      comm->first = TRUE;
      gtk_selection_owner_set (window, comm->atom0, GDK_CURRENT_TIME);
      gtk_selection_owner_set (window, comm->atom1, GDK_CURRENT_TIME);
    }
#endif /* not G_OS_WIN32 */
  return G_OBJECT (comm);
}


/*  ja:プロセスが最初のインスタンスか確認する
    comm,オブジェクト
     RET,TRUE:最初のプロセス,FALSE:2つ目以降のプロセス                      */
gboolean
orz_comm_is_first (OrzComm *comm)
{
#ifdef G_OS_WIN32
  return comm ? !comm->hConv : TRUE;
#else /* not G_OS_WIN32 */
  return comm ? comm->first : TRUE;
#endif /* not G_OS_WIN32 */
}


/*  ja:引数を最初のプロセスに送信する
    comm,オブジェクト
    argc,引数の数
    argv,引数                                                               */
void
orz_comm_send (OrzComm     *comm,
               const gint   argc,
               gchar      **argv)
{
  orz_comm_send_with_files (comm, argc, argv, NULL);
}


/*  ja:引数を最初のプロセスに送信する(ファイルの変換あり)
     comm,オブジェクト
     argc,引数の数
     argv,引数
    files,TRUE:ファイル,FALSE:ファイルではない                              */
void
orz_comm_send_with_files (OrzComm         *comm,
                          const gint       argc,
                          gchar          **argv,
                          const gboolean  *files)
{
  gsize length = 0;
  gchar *arg = NULL;
  gint i;

#ifdef G_OS_WIN32
  if (!comm || !comm->hConv)
#else /* not G_OS_WIN32 */
  if (!comm || comm->first)
#endif /* not G_OS_WIN32 */
    return;
  for (i = 0; i < argc; i++)
    {
      gsize leng;

      if (files && files[i])
        {
          gchar *tmp;

          tmp = fileio_get_full_path (argv[i]);
          leng = g_strlen (tmp);
          arg = g_realloc (arg, (length + leng + 1) * sizeof (gchar));
          g_memmove (arg + length, tmp, leng * sizeof (gchar));
          g_free (tmp);
        }
      else
        {
          leng = g_strlen (argv[i]);
          arg = g_realloc (arg, (length + leng + 1) * sizeof (gchar));
          g_memmove (arg + length, argv[i], leng * sizeof (gchar));
        }
      length += leng;
      arg[length++] = '\n';
    }
#ifdef G_OS_WIN32
  if (length > 0)
    {
      HDDEDATA hData;

      arg[length - 1] = '\0';
      hData = DdeCreateDataHandle (comm->ddeInst, arg, length, 0,
                                                comm->hszTopic, CF_TEXT, 0);
      g_free (arg);
      if (hData)
        DdeClientTransaction ((LPBYTE)hData, 0xffffffff, comm->hConv,
                    comm->hszTopic, CF_TEXT, XTYP_POKE, TIMEOUT_ASYNC, NULL);
    }
#else /* not G_OS_WIN32 */
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  g_free (comm->arg);
  comm->arg = NULL;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  if (length > 0)
    {
      arg[length - 1] = '\0';
      comm->arg = g_strescape (arg, NULL);
      g_free (arg);
      gtk_selection_owner_set (comm->window, comm->atom2, GDK_CURRENT_TIME);
      gtk_selection_convert (comm->window, comm->atom1,
                                GDK_SELECTION_TYPE_STRING, GDK_CURRENT_TIME);
# ifdef USE_THREAD
      gdk_threads_enter ();
# endif /* USE_THREAD */
      while (comm->arg)
        while (gtk_events_pending ())
          gtk_main_iteration ();
# ifdef USE_THREAD
      gdk_threads_leave ();
# endif /* USE_THREAD */
    }
#endif /* not G_OS_WIN32 */
}
