/*
    Melody maid
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "edit.h"
#include "general.h"
#include "mm_list.h"
#include "misc/fileio.h"
#include "misc/misc.h"
#include "orz/orzcharset.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* ja:編集ダイアログ関数群                                                     *
*                                                                             *
******************************************************************************/
typedef struct _EditDialog
{
  gboolean ok_sensitive;
  GtkWidget *dialog, *entry, *charset, *list;
} EditDialog;


/* ja:参照ボタンが押された */
static void
edit_dialog_clicked (GtkWidget  *widget,
                     EditDialog *editdlg)
{
  gchar *file;
  GtkWidget *dialog;

#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_file_chooser_dialog_new (_("Browse"),
                    GTK_WINDOW (editdlg->dialog), GTK_FILE_CHOOSER_ACTION_OPEN,
                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                    GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                    NULL);
  file = g_filename_from_utf8 (gtk_entry_get_text (GTK_ENTRY (editdlg->entry)),
                                                        -1, NULL, NULL, NULL);
  if (g_file_test (file, G_FILE_TEST_EXISTS))
    {
      gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog), file);
    }
  else
    {
      gchar *dir, *path;

      dir = g_path_get_dirname (file);
      path = fileio_get_full_path (dir);
      g_free (dir);
      if (g_file_test (path, G_FILE_TEST_EXISTS))
        gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), path);
      g_free (path);
    }
  g_free (file);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      gchar *utf8str;

      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      gtk_entry_set_text (GTK_ENTRY (editdlg->entry), utf8str);
      g_free (utf8str);
      g_free (file);
    }
#else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog = gtk_file_selection_new (_("Browse"));
  file = g_filename_from_utf8 (gtk_entry_get_text (GTK_ENTRY (editdlg->entry)),
                                                        -1, NULL, NULL, NULL);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), file);
  g_free (file);
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (gtk_file_selection_get_filename
                        (GTK_FILE_SELECTION (dialog), -1, NULL, NULL, NULL);
      gtk_entry_set_text (GTK_ENTRY (editdlg->entry), utf8str);
      g_free (utf8str);
    }
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_widget_destroy (dialog);
}


/* ja:コンボボックス/エントリー/スピンボタンでリターンが押された */
static void
edit_dialog_activate (GtkWidget  *widget,
                      EditDialog *editdlg)
{
  if (editdlg->ok_sensitive)
    gtk_dialog_response (GTK_DIALOG (editdlg->dialog), GTK_RESPONSE_OK);
}


/* ja:チェックボタンが変更された */
#define edit_dialog_toggled edit_dialog_changed


/* ja:文字符号化方式が変更された */
#define edit_dialog_edited edit_dialog_changed


/* ja:コンボボックス/エントリー/スピンボタンが変更された */
static void
edit_dialog_changed (GtkWidget *widget,
                     GtkWidget *label)
{
  gboolean result = TRUE;
  EditDialog *editdlg;

  editdlg = g_object_get_data (G_OBJECT (label), "user_data");
  if (!gtk_label_get_use_markup (GTK_LABEL (label)))
    {
      gchar *str;

      str = g_strconcat ("<b><i>", gtk_label_get_label (GTK_LABEL (label)),
                                                            "</i></b>*", NULL);
      gtk_label_set_markup_with_mnemonic (GTK_LABEL (label), str);
      g_free (str);
    }
  if (editdlg->entry)
    {
      const gchar *file;
      gint i;

      file = gtk_entry_get_text (GTK_ENTRY (editdlg->entry));
      for (i = mm_list_length (MM_LIST (editdlg->list)) - 1; i >= 0; i--)
        if (mm_list_is_selected (MM_LIST (editdlg->list), i))
          break;
      result = mm_list_is_valid_file (MM_LIST (editdlg->list), i, file);
    }
  if (result)
    {
      gchar *charset;

      charset = orz_charset_get_charset (ORZ_CHARSET (editdlg->charset));
      result = charset[0] == '\0'
                    || orz_charset_is_valid (ORZ_CHARSET (editdlg->charset));
      g_free (charset);
    }
  editdlg->ok_sensitive = result;
  gtk_dialog_set_response_sensitive (GTK_DIALOG (editdlg->dialog),
                                                    GTK_RESPONSE_OK, result);
}


/*  ja:編集ダイアログ表示
    mmaid,ウインドウ情報                                                    */
void
edit_dialog (MmaidWindow *mmaid)
{
  gint i, c, s = 0, count;
  GList *glist_title = NULL, *glist_artist = NULL, *glist_album = NULL;
  GList *glist_comment = NULL, *glist_genre = NULL;
  GtkWidget *table, *hbox, *button, *check, *entry, *spin;
  GtkWidget *combo_title, *combo_artist, *combo_album;
  GtkWidget *combo_comment, *combo_genre;
  GtkWidget *label_file, *label_title, *label_artist, *label_album;
  GtkWidget *label_comment, *label_year, *label_genre, *label_charset;
  const ID3Tag *id3tag = NULL;
  EditDialog editdlg;
#if GTK_CHECK_VERSION(2,6,0)
  gchar *title;
  GtkStockItem stock_item;
#endif /* GTK_CHECK_VERSION(2,6,0) */

  editdlg.list = mmaid->list;
  /* ja:メインウインドウ */
#if GTK_CHECK_VERSION(2,6,0)
  gtk_stock_lookup (GTK_STOCK_EDIT, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
  editdlg.dialog = gtk_dialog_new_with_buttons
                            (title,
                            GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
#ifdef G_OS_WIN32
#else /* not G_OS_WIN32 */
#endif /* not G_OS_WIN32 */
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
                            NULL);
  g_free (title);
#else /* not GTK_CHECK_VERSION(2,6,0) */
  editdlg.dialog = gtk_dialog_new_with_buttons
                            (_("Edit"),
                            GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
#ifdef G_OS_WIN32
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
#else /* not G_OS_WIN32 */
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
#endif /* not G_OS_WIN32 */
                            NULL);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
  g_signal_connect (G_OBJECT (editdlg.dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
  /* ja:ラベル */
  label_title = gtk_label_new_with_mnemonic (_("_Title"));
  label_artist = gtk_label_new_with_mnemonic (_("_Artist"));
  label_album = gtk_label_new_with_mnemonic (_("A_lbum"));
  label_year = gtk_label_new_with_mnemonic (_("_Year"));
  label_comment = gtk_label_new_with_mnemonic (_("_Comment"));
  label_genre = gtk_label_new_with_mnemonic (_("_Genre"));
  label_charset = gtk_label_new_with_mnemonic (_("Character _Encoding"));
  g_object_set_data (G_OBJECT (label_title), "user_data", &editdlg);
  g_object_set_data (G_OBJECT (label_artist), "user_data", &editdlg);
  g_object_set_data (G_OBJECT (label_album), "user_data", &editdlg);
  g_object_set_data (G_OBJECT (label_year), "user_data", &editdlg);
  g_object_set_data (G_OBJECT (label_comment), "user_data", &editdlg);
  g_object_set_data (G_OBJECT (label_genre), "user_data", &editdlg);
  g_object_set_data (G_OBJECT (label_charset), "user_data", &editdlg);
  /* ja:コンボボックス(準備) */
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 0; i < count; i++)
    {
      id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
      if (id3tag->title && !g_list_find_custom (glist_title, id3tag->title,
                                                    (GCompareFunc)g_strcmp))
        glist_title = g_list_insert_sorted (glist_title,
                            g_strdup (id3tag->title), (GCompareFunc)g_strcmp);
      if (id3tag->artist && !g_list_find_custom (glist_artist, id3tag->artist,
                                                    (GCompareFunc)g_strcmp))
        glist_artist = g_list_insert_sorted (glist_artist,
                            g_strdup (id3tag->artist), (GCompareFunc)g_strcmp);
      if (id3tag->album && !g_list_find_custom (glist_album, id3tag->album,
                                                    (GCompareFunc)g_strcmp))
        glist_album = g_list_insert_sorted (glist_album,
                            g_strdup (id3tag->album), (GCompareFunc)g_strcmp);
      if (id3tag->comment && !g_list_find_custom (glist_comment,
                                    id3tag->comment, (GCompareFunc)g_strcmp))
        glist_comment = g_list_insert_sorted (glist_comment,
                        g_strdup (id3tag->comment), (GCompareFunc)g_strcmp);
    }
  for (i = 0; i < 256; i++)
    if (genre[0][i] && !g_list_find_custom (glist_genre, genre[0][i],
                                                    (GCompareFunc)g_strcmp))
      glist_genre = g_list_append (glist_genre, g_strdup (genre[0][i]));
  /* ja:選択の数を数える */
  for (i = 0; i < count; i++)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i))
      {
        s++;
        if (s > 1)
          break;
        id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
      }
  if (s <= 0)
    return;
  /* ja:コンボボックス */
#if GTK_CHECK_VERSION(2,4,0)
# if GTK_CHECK_VERSION(2,24,0)
  combo_title = gtk_combo_box_text_new_with_entry ();
  combo_artist = gtk_combo_box_text_new_with_entry ();
  combo_album = gtk_combo_box_text_new_with_entry ();
  combo_comment = gtk_combo_box_text_new_with_entry ();
  combo_genre = gtk_combo_box_text_new_with_entry ();
# else /* not GTK_CHECK_VERSION(2,24,0) */
  combo_title = gtk_combo_box_entry_new_text ();
  combo_artist = gtk_combo_box_entry_new_text ();
  combo_album = gtk_combo_box_entry_new_text ();
  combo_comment = gtk_combo_box_entry_new_text ();
  combo_genre = gtk_combo_box_entry_new_text ();
# endif /* not GTK_CHECK_VERSION(2,24,0) */
  while ((glist_title = g_list_first (glist_title)))
    {
# if GTK_CHECK_VERSION(2,24,0)
      gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (combo_title),
# else /* not GTK_CHECK_VERSION(2,24,0) */
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo_title),
# endif /* not GTK_CHECK_VERSION(2,24,0) */
                                                        glist_title->data);
      g_free (glist_title->data);
      glist_title = g_list_delete_link (glist_title, glist_title);
    }
  while ((glist_artist = g_list_first (glist_artist)))
    {
# if GTK_CHECK_VERSION(2,24,0)
      gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (combo_artist),
# else /* not GTK_CHECK_VERSION(2,24,0) */
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo_artist),
# endif /* not GTK_CHECK_VERSION(2,24,0) */
                                                        glist_artist->data);
      g_free (glist_artist->data);
      glist_artist = g_list_delete_link (glist_artist, glist_artist);
    }
  while ((glist_album = g_list_first (glist_album)))
    {
# if GTK_CHECK_VERSION(2,24,0)
      gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (combo_album),
# else /* not GTK_CHECK_VERSION(2,24,0) */
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo_album),
# endif /* not GTK_CHECK_VERSION(2,24,0) */
                                                        glist_album->data);
      g_free (glist_album->data);
      glist_album = g_list_delete_link (glist_album, glist_album);
    }
  while ((glist_comment = g_list_first (glist_comment)))
    {
# if GTK_CHECK_VERSION(2,24,0)
      gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (combo_comment),
# else /* not GTK_CHECK_VERSION(2,24,0) */
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo_comment),
# endif /* not GTK_CHECK_VERSION(2,24,0) */
                                                        glist_comment->data);
      g_free (glist_comment->data);
      glist_comment = g_list_delete_link (glist_comment, glist_comment);
    }
  while ((glist_genre = g_list_first (glist_genre)))
    {
# if GTK_CHECK_VERSION(2,24,0)
      gtk_combo_box_text_append_text (GTK_COMBO_BOX_TEXT (combo_genre),
# else /* not GTK_CHECK_VERSION(2,24,0) */
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo_genre),
# endif /* not GTK_CHECK_VERSION(2,24,0) */
                                                        glist_genre->data);
      g_free (glist_genre->data);
      glist_genre = g_list_delete_link (glist_genre, glist_genre);
    }
#else /* not GTK_CHECK_VERSION(2,4,0) */
  combo_title = gtk_combo_new ();
  combo_artist = gtk_combo_new ();
  combo_album = gtk_combo_new ();
  combo_comment = gtk_combo_new ();
  combo_genre = gtk_combo_new ();
  if (glist_title)
    {
      gtk_combo_set_popdown_strings (GTK_COMBO (combo_title), glist_title);
      g_list_foreach (glist_title, (GFunc)g_free, NULL);
      g_list_free (glist_title);
    }
  if (glist_artist)
    {
      gtk_combo_set_popdown_strings (GTK_COMBO (combo_artist), glist_artist);
      g_list_foreach (glist_artist, (GFunc)g_free, NULL);
      g_list_free (glist_artist);
    }
  if (glist_album)
    {
      gtk_combo_set_popdown_strings (GTK_COMBO (combo_album), glist_album);
      g_list_foreach (glist_album, (GFunc)g_free, NULL);
      g_list_free (glist_album);
    }
  if (glist_comment)
    {
      gtk_combo_set_popdown_strings (GTK_COMBO (combo_comment), glist_comment);
      g_list_foreach (glist_comment, (GFunc)g_free, NULL);
      g_list_free (glist_comment);
    }
  if (glist_genre)
    {
      gtk_combo_set_popdown_strings (GTK_COMBO (combo_genre), glist_genre);
      g_list_foreach (glist_genre, (GFunc)g_free, NULL);
      g_list_free (glist_genre);
    }
  gtk_combo_set_value_in_list (GTK_COMBO (combo_title), FALSE, FALSE);
  gtk_combo_set_value_in_list (GTK_COMBO (combo_artist), FALSE, FALSE);
  gtk_combo_set_value_in_list (GTK_COMBO (combo_album), FALSE, FALSE);
  gtk_combo_set_value_in_list (GTK_COMBO (combo_comment), FALSE, FALSE);
  gtk_combo_set_value_in_list (GTK_COMBO (combo_genre), FALSE, FALSE);
  gtk_combo_set_case_sensitive (GTK_COMBO (combo_title), TRUE);
  gtk_combo_set_case_sensitive (GTK_COMBO (combo_artist), TRUE);
  gtk_combo_set_case_sensitive (GTK_COMBO (combo_album), TRUE);
  gtk_combo_set_case_sensitive (GTK_COMBO (combo_comment), TRUE);
  gtk_combo_set_case_sensitive (GTK_COMBO (combo_genre), TRUE);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  /* ja:文字符号化方式 */
  editdlg.charset = orz_charset_new ();
  orz_charset_add_candidate (ORZ_CHARSET (editdlg.charset), charset_list);
  /* ja:コンボボックス&文字符号化方式 */
  if (s == 1)
    {
#if GTK_CHECK_VERSION(2,4,0)
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_title))),
                                    id3tag->title ? id3tag->title : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_artist))),
                                    id3tag->artist ? id3tag->artist : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_album))),
                                    id3tag->album ? id3tag->album : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_comment))),
                                    id3tag->comment ? id3tag->comment : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_genre))),
                                    id3tag->genre ? id3tag->genre : "");
#else /* not GTK_CHECK_VERSION(2,4,0) */
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_title)->entry),
                                    id3tag->title ? id3tag->title : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_artist)->entry),
                                    id3tag->artist ? id3tag->artist : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_album)->entry),
                                    id3tag->album ? id3tag->album : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_comment)->entry),
                                    id3tag->comment ? id3tag->comment : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_genre)->entry),
                                    id3tag->genre ? id3tag->genre : "");
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      orz_charset_set_charset (ORZ_CHARSET (editdlg.charset),
                                    id3tag->charset ? id3tag->charset : "");
    }
  else
    {
      gboolean t = TRUE, a = TRUE, l = TRUE, c = TRUE, g = TRUE, e = TRUE;

      for (i = 0; i < count && (t || a || l || c || g || e); i++)
        if (mm_list_is_selected (MM_LIST (mmaid->list), i))
          {
            const ID3Tag *id3tag0;

            id3tag0 = mm_list_get_nth (MM_LIST (mmaid->list), i);
            t = t && id3tag0->title
                        && g_strcmp (id3tag->title,   id3tag0->title) == 0;
            a = a && id3tag0->artist
                        && g_strcmp (id3tag->artist,  id3tag0->artist) == 0;
            l = l && id3tag0->album
                        && g_strcmp (id3tag->album,   id3tag0->album) == 0;
            c = c && id3tag0->comment
                        && g_strcmp (id3tag->comment, id3tag0->comment) == 0;
            g = g && id3tag0->genre
                        && g_strcmp (id3tag->genre,   id3tag0->genre) == 0;
            e = e && id3tag0->charset
                        && g_strcmp (id3tag->charset, id3tag0->charset) == 0;
          }
#if GTK_CHECK_VERSION(2,4,0)
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_title))),
                                                    t ? id3tag->title   : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_artist))),
                                                    a ? id3tag->artist  : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_album))),
                                                    l ? id3tag->album   : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_comment))),
                                                    c ? id3tag->comment : "");
      gtk_entry_set_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_genre))),
                                                    g ? id3tag->genre   : "");
#else /* not GTK_CHECK_VERSION(2,4,0) */
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_title)->entry),
                                                    t ? id3tag->title   : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_artist)->entry),
                                                    a ? id3tag->artist  : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_album)->entry),
                                                    l ? id3tag->album   : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_comment)->entry),
                                                    c ? id3tag->comment : "");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo_genre)->entry),
                                                    g ? id3tag->genre   : "");
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      orz_charset_set_charset (ORZ_CHARSET (editdlg.charset),
                                                    e ? id3tag->charset : "");
    }
#if GTK_CHECK_VERSION(2,4,0)
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_title))),
                    "activate", G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_artist))),
                    "activate", G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_album))),
                    "activate", G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_comment))),
                    "activate", G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_genre))),
                    "activate", G_CALLBACK (edit_dialog_activate), &editdlg);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_title)->entry), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_artist)->entry), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_album)->entry), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_comment)->entry), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_genre)->entry), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_signal_connect (G_OBJECT (editdlg.charset), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
#if GTK_CHECK_VERSION(2,4,0)
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_title))),
                    "changed", G_CALLBACK (edit_dialog_changed), label_title);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_artist))),
                    "changed", G_CALLBACK (edit_dialog_changed), label_artist);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_album))),
                    "changed", G_CALLBACK (edit_dialog_changed), label_album);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_comment))),
                    "changed", G_CALLBACK (edit_dialog_changed), label_comment);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (combo_genre))),
                    "changed", G_CALLBACK (edit_dialog_changed), label_genre);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_title)->entry), "changed",
                            G_CALLBACK (edit_dialog_changed), label_title);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_artist)->entry), "changed",
                            G_CALLBACK (edit_dialog_changed), label_artist);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_album)->entry), "changed",
                            G_CALLBACK (edit_dialog_changed), label_album);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_comment)->entry), "changed",
                            G_CALLBACK (edit_dialog_changed), label_comment);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo_genre)->entry), "changed",
                            G_CALLBACK (edit_dialog_changed), label_genre);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_signal_connect (G_OBJECT (editdlg.charset), "edited",
                            G_CALLBACK (edit_dialog_edited), label_charset);
  /* ja:エントリー */
  entry = gtk_entry_new ();
  if (s == 1 && id3tag->year)
    gtk_entry_set_text (GTK_ENTRY (entry), id3tag->year);
  g_signal_connect (G_OBJECT (entry), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (entry), "changed",
                                G_CALLBACK (edit_dialog_changed), label_year);
  /* ja:チェックボタン */
  check = gtk_check_button_new_with_mnemonic (_("T_rack"));
  g_object_set_data (G_OBJECT (gtk_bin_get_child (GTK_BIN (check))),
                                                    "user_data", mmaid->list);
  if (s == 1)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
                                0 <= id3tag->track && id3tag->track <= 255);
  g_signal_connect (G_OBJECT (check), "toggled",
        G_CALLBACK (edit_dialog_toggled), gtk_bin_get_child (GTK_BIN (check)));
  /* ja:スピンボタン */
  spin = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
     (s == 1 && 0 <= id3tag->track && id3tag->track <= 255 ? id3tag->track : 0,
                                                    0, 255, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  g_signal_connect (G_OBJECT (spin), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
  g_signal_connect (G_OBJECT (spin), "changed",
        G_CALLBACK (edit_dialog_changed), gtk_bin_get_child (GTK_BIN (check)));
  /* ja:ラベル */
#if GTK_CHECK_VERSION(2,4,0)
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_title),
                                gtk_bin_get_child (GTK_BIN (combo_title)));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_artist),
                                gtk_bin_get_child (GTK_BIN (combo_artist)));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_album),
                                gtk_bin_get_child (GTK_BIN (combo_album)));
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_title),
                                            GTK_COMBO (combo_title)->entry);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_artist),
                                            GTK_COMBO (combo_artist)->entry);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_album),
                                            GTK_COMBO (combo_album)->entry);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_year), entry);
#if GTK_CHECK_VERSION(2,4,0)
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_comment),
                                gtk_bin_get_child (GTK_BIN (combo_comment)));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_genre),
                                gtk_bin_get_child (GTK_BIN (combo_genre)));
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_comment),
                                            GTK_COMBO (combo_comment)->entry);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_genre),
                                            GTK_COMBO (combo_genre)->entry);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_charset), editdlg.charset);
  /* ja:ファイル */
  if (s == 1)
    {
      /* ja:ボタン */
      button = gtk_button_new_with_mnemonic (_("_Browse..."));
      g_signal_connect (G_OBJECT (button), "clicked",
                                G_CALLBACK (edit_dialog_clicked), &editdlg);
      /* ja:ラベル */
      label_file = gtk_label_new_with_mnemonic (_("_File"));
      g_object_set_data (G_OBJECT (label_file), "user_data", &editdlg);
      /* ja:エントリー */
      editdlg.entry = gtk_entry_new ();
      gtk_entry_set_text (GTK_ENTRY (editdlg.entry), id3tag->file);
      g_signal_connect (G_OBJECT (editdlg.entry), "activate",
                                G_CALLBACK (edit_dialog_activate), &editdlg);
      g_signal_connect (G_OBJECT (editdlg.entry), "changed",
                                G_CALLBACK (edit_dialog_changed), label_file);
      /* ja:ラベル */
      gtk_label_set_mnemonic_widget (GTK_LABEL (label_file), editdlg.entry);
    }
  else
    {
      button = editdlg.entry = label_file = NULL;
    }
  /* ja:フレームとボックス */
  table = gtk_table_new (s == 1 ? 7 : 6, 4, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), SPACING);
  gtk_table_set_row_spacings (GTK_TABLE (table), SPACING);
  gtk_table_set_col_spacings (GTK_TABLE (table), SPACING);
  i = 0;
  if (s == 1)
    {
      hbox = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_end (GTK_BOX (hbox), label_file, FALSE, FALSE, 0);
      gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
      hbox = gtk_hbox_new (FALSE, SPACING);
      gtk_box_pack_start (GTK_BOX (hbox), editdlg.entry, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
      gtk_table_attach_defaults (GTK_TABLE (table), hbox,   1, 4, i, i + 1);
      i++;
    }
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), label_title, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table), combo_title, 1, 4, i, i + 1);
  i++;
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), label_artist, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table), combo_artist, 1, 4, i, i + 1);
  i++;
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), label_album, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table), combo_album, 1, 4, i, i + 1);
  i++;
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), label_comment, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table), combo_comment, 1, 4, i, i + 1);
  i++;
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), label_year, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, i, i + 1);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), check, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 2, 3, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table), spin,   3, 4, i, i + 1);
  i++;
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), label_genre, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table), combo_genre, 1, 2, i, i + 1);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), label_charset, FALSE, FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 2, 3, i, i + 1,
                                                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (table),
                                            editdlg.charset, 3, 4, i, i + 1);
  gtk_container_add (GTK_CONTAINER
        (gtk_dialog_get_content_area (GTK_DIALOG (editdlg.dialog))), table);

  /* ja:表示 */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (editdlg.dialog),
                                                    GTK_RESPONSE_OK, FALSE);
  mm_list_get_cursor (MM_LIST (mmaid->list), NULL, &c);
  switch (c)
    {
      case MM_LIST_FILE:
        if (editdlg.entry)
          gtk_widget_grab_focus (editdlg.entry);
        break;
      case MM_LIST_TITLE:
#if GTK_CHECK_VERSION(2,4,0)
        gtk_widget_grab_focus (gtk_bin_get_child (GTK_BIN (combo_title)));
        break;
      case MM_LIST_ARTIST:
        gtk_widget_grab_focus (gtk_bin_get_child (GTK_BIN (combo_artist)));
        break;
      case MM_LIST_ALBUM:
        gtk_widget_grab_focus (gtk_bin_get_child (GTK_BIN (combo_album)));
        break;
#else /* not GTK_CHECK_VERSION(2,4,0) */
        gtk_widget_grab_focus (GTK_COMBO (combo_title)->entry); break;
      case MM_LIST_ARTIST:
        gtk_widget_grab_focus (GTK_COMBO (combo_artist)->entry); break;
      case MM_LIST_ALBUM:
        gtk_widget_grab_focus (GTK_COMBO (combo_album)->entry); break;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      case MM_LIST_YEAR:
        gtk_widget_grab_focus (entry); break;
      case MM_LIST_COMMENT:
#if GTK_CHECK_VERSION(2,4,0)
        gtk_widget_grab_focus (gtk_bin_get_child (GTK_BIN (combo_comment)));
        break;
#else /* not GTK_CHECK_VERSION(2,4,0) */
        gtk_widget_grab_focus (GTK_COMBO (combo_comment)->entry); break;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      case MM_LIST_TRACK:
        gtk_widget_grab_focus (check); break;
      case MM_LIST_GENRE:
#if GTK_CHECK_VERSION(2,4,0)
        gtk_widget_grab_focus (gtk_bin_get_child (GTK_BIN (combo_genre)));
        break;
#else /* not GTK_CHECK_VERSION(2,4,0) */
        gtk_widget_grab_focus (GTK_COMBO (combo_genre)->entry); break;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      case MM_LIST_CHARSET:
        gtk_widget_grab_focus (editdlg.charset); break;
    }
  gtk_widget_show_all (table);

  editdlg.ok_sensitive = FALSE;

  if (gtk_dialog_run (GTK_DIALOG (editdlg.dialog)) == GTK_RESPONSE_OK)
    {
      gboolean f, t, a, l, y, c, r, g, e;
      gchar *file = NULL, *charset;
      const gchar *title, *artist, *album, *year, *comment, *genre;
      gint i, count, track;

      if (editdlg.entry)
        {
          gchar *tmp;

          file = g_filename_from_utf8 (gtk_entry_get_text
                            (GTK_ENTRY (editdlg.entry)), -1, NULL, NULL, NULL);
          tmp = fileio_get_full_path (file);
          g_free (file);
          file = g_filename_to_utf8 (tmp, -1, NULL, NULL, NULL);
          g_free (tmp);
          f = file && gtk_label_get_use_markup (GTK_LABEL (label_file));
        }
      else
        {
          f = FALSE;
        }
#if GTK_CHECK_VERSION(2,4,0)
      title   = gtk_entry_get_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_title))));
      artist  = gtk_entry_get_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_artist))));
      album   = gtk_entry_get_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_album))));
#else /* not GTK_CHECK_VERSION(2,4,0) */
      title   = gtk_entry_get_text
                                (GTK_ENTRY (GTK_COMBO (combo_title)->entry));
      artist  = gtk_entry_get_text
                                (GTK_ENTRY (GTK_COMBO (combo_artist)->entry));
      album   = gtk_entry_get_text
                                (GTK_ENTRY (GTK_COMBO (combo_album)->entry));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      year    = gtk_entry_get_text (GTK_ENTRY (entry));
#if GTK_CHECK_VERSION(2,4,0)
      comment = gtk_entry_get_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_comment))));
#else /* not GTK_CHECK_VERSION(2,4,0) */
      comment = gtk_entry_get_text
                                (GTK_ENTRY (GTK_COMBO (combo_comment)->entry));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      track   = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check))
              ? gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin)) : -1;
#if GTK_CHECK_VERSION(2,4,0)
      genre   = gtk_entry_get_text
                    (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo_genre))));
#else /* not GTK_CHECK_VERSION(2,4,0) */
      genre   = gtk_entry_get_text
                                (GTK_ENTRY (GTK_COMBO (combo_genre)->entry));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      charset = orz_charset_get_charset (ORZ_CHARSET (editdlg.charset));
      t = gtk_label_get_use_markup (GTK_LABEL (label_title));
      a = gtk_label_get_use_markup (GTK_LABEL (label_artist));
      l = gtk_label_get_use_markup (GTK_LABEL (label_album));
      y = gtk_label_get_use_markup (GTK_LABEL (label_year));
      c = gtk_label_get_use_markup (GTK_LABEL (label_comment));
      r = gtk_label_get_use_markup
                            (GTK_LABEL (gtk_bin_get_child (GTK_BIN (check))));
      g = gtk_label_get_use_markup (GTK_LABEL (label_genre));
      e = gtk_label_get_use_markup (GTK_LABEL (label_charset));

      count = mm_list_length (MM_LIST (mmaid->list));
      for (i = 0; i < count; i++)
        if (mm_list_is_selected (MM_LIST (mmaid->list), i))
          {
              if (f)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_FILE,
                                                        g_strdup (file), -1);
              if (t)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_TITLE,
                          title[0] != '\0' ? g_strdup (title)   : NULL, -1);
              if (a)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_ARTIST,
                         artist[0] != '\0' ? g_strdup (artist)  : NULL, -1);
              if (l)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_ALBUM,
                          album[0] != '\0' ? g_strdup (album)   : NULL, -1);
              if (y)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_YEAR,
                           year[0] != '\0' ? g_strdup (year)    : NULL, -1);
              if (c)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_COMMENT,
                        comment[0] != '\0' ? g_strdup (comment) : NULL, -1);
              if (r)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_TRACK,
                                                                    track, -1);
              if (g)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_GENRE,
                          genre[0] != '\0' ? g_strdup (genre)   : NULL, -1);
              if (e)
                mm_list_change (MM_LIST (mmaid->list), i, MM_LIST_CHARSET,
                        charset[0] != '\0' ? g_strdup (charset) : NULL, -1);
          }
      if (e && charset[0] != '\0')
        charlist_renewal_all (charset);
      g_free (charset);
      g_free (file);
      orz_mdi_set_edited (ORZ_MDI (mdi),
                    orz_mdi_get_page_from_data (ORZ_MDI (mdi), mmaid), TRUE);
    }
  gtk_widget_destroy (editdlg.dialog);
}
