/*
    Melody maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "abort.h"
#include "file.h"
#include "general.h"
#include "mm_list.h"
#include "sigfile.h"
#include "sigmain.h"
#include "misc/fileio.h"
#include "misc/misc.h"


/******************************************************************************
*                                                                             *
* ja:ファイル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの履歴に加える
    file,ファイル名                                                         */
void
file_set_history (const gchar *file)
{
  gchar *label;
  gint i, count;
  GList *glist;
  GtkWidget *menu_shell, *menu_item;

  menu_shell = gtk_item_factory_get_widget (ifactory_menu, "<main>/File");
  glist = gtk_container_children (GTK_CONTAINER (menu_shell));
  count = g_list_length (glist);
  for (i = 0; i < count - MENUFILE - 1; i++)
    {
      menu_item = g_list_nth_data (glist, i + MENUFILE - 1);
      label = g_object_get_data (G_OBJECT (menu_item), "user_data");
      if (g_strfilecmp (label, file) == 0)
        {
          gtk_container_remove (GTK_CONTAINER (menu_shell), menu_item);
          break;
        }
    }
  g_list_free (glist);
  if (i >= history)
    {
      glist = gtk_container_children (GTK_CONTAINER (menu_shell));
      gtk_container_remove (GTK_CONTAINER (menu_shell),
                        g_list_nth_data (glist, MENUFILE + history + 1 - 3));
      g_list_free (glist);
    }
  if (count <= MENUFILE)
    {
      menu_item = gtk_menu_item_new ();
      gtk_menu_insert (GTK_MENU (menu_shell), menu_item, MENUFILE - 2);
        gtk_widget_show (menu_item);
    }
  label = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  menu_item = gtk_menu_item_new_with_label (label);
  g_free (label);
  label = g_strdup (file);
  g_signal_connect (G_OBJECT (menu_item), "activate",
                            G_CALLBACK (signal_activate_menu_history), label);
  g_signal_connect (G_OBJECT (menu_item), "destroy",
                            G_CALLBACK (signal_destroy_menu_history), label);
  g_object_set_data (G_OBJECT (menu_item), "user_data", label);
  gtk_menu_insert (GTK_MENU (menu_shell), menu_item, MENUFILE - 1);
  gtk_widget_show (menu_item);
}


/*  ja:ファイルを加える
    file,加えようとするファイルのフルパス
    same,フルパスに付属する数値
     RET,加えようとするファイルのラベル                                     */
gchar *
file_add_edit (const gchar *file, gint *same)
{
  gchar *label, *utf8str;
  const gchar *name0, *name1;
  gint i;
  GList *glist;
  GtkWidget *child;
  MmaidWindow *mmaid;

  *same=-1;
  /* ja:ファイル名を調べる */
  glist = gtk_container_children (GTK_CONTAINER (notebook));
  /* ja:同名同パスを調べる */
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      child = g_list_nth_data (glist, i);
      mmaid = g_object_get_data (G_OBJECT (child), "user_data");
      if (g_strfilecmp (file, mmaid->file) == 0)
        {
          if (mmaid->same < 0)
            {
              mmaid->same = 0;
              utf8str = g_filename_to_utf8 (mmaid->file, -1, NULL, NULL, NULL);
              label = g_strdup_printf ("%s:%d", utf8str, mmaid->same);
              g_free (utf8str);
              gtk_label_set_text (GTK_LABEL (mmaid->label), label);
              g_free (label);
            }
            if (*same <= mmaid->same)
              *same = mmaid->same + 1;
        }
    }
  if (*same >= 0)
    {
      /* ja:同名同パスがあるとき */
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      label = g_strdup_printf ("%s:%d", utf8str, *same);
      g_free (utf8str);
    }
  else
    {
      /* ja:同名同パスはないとき */
      /* ja:同名異パスを調べる */
      name0 = g_basename (file);
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          child = g_list_nth_data (glist, i);
          mmaid = g_object_get_data (G_OBJECT (child), "user_data");
          name1 = g_basename (mmaid->file);
          if (g_strfilecmp (name0, name1) == 0)
            {
              if (mmaid->same < 0)
                {
                  utf8str = g_filename_to_utf8 (mmaid->file, -1,
                                                            NULL, NULL, NULL);
                  gtk_label_set_text (GTK_LABEL (mmaid->label), utf8str);
                  g_free (utf8str);
                }
              break;
            }
        }
      label = g_filename_to_utf8 (i >= 0 ? file : name0, -1, NULL, NULL, NULL);
    }
  g_list_free (glist);
  return label;
}


/*  ja:ファイルを削除する
    file,削除しようとするファイルのフルパス                                 */
void
file_delete_edit (const gchar *file)
{
  gchar *utf8str;
  const gchar *name0,*name1;
  gint i,count=0;
  GList *glist;
  MmaidWindow *mmaid;
  GtkWidget *child;

  /* ja:ファイル名を調べる */
  glist = gtk_container_children (GTK_CONTAINER (notebook));
  /* ja:同名異パスを調べる */
  name0 = g_basename (file);
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      mmaid = g_object_get_data (g_list_nth_data (glist, i), "user_data");
      name1 = g_basename (mmaid->file);
      if (g_strfilecmp (name0, name1) == 0)
        count++;
    }
  if (count <= 2)
    {
      /* ja:同名異パスのファイルを通常表示に変更する */
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          child = g_list_nth_data (glist, i);
          mmaid = g_object_get_data (G_OBJECT (child), "user_data");
          name1 = g_basename (mmaid->file);
          if (g_strfilecmp (name0, name1) == 0)
            {
              utf8str = g_filename_to_utf8 (name1, -1, NULL, NULL, NULL);
              gtk_label_set_text (GTK_LABEL (mmaid->label), utf8str);
              g_free (utf8str);
            }
        }
    }
  else
    {
      /* ja:同名同パスを調べる */
      count = 0;
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          mmaid = g_object_get_data (g_list_nth_data (glist, i), "user_data");
          if (g_strfilecmp (file, mmaid->file) == 0)
            count++;
        }
      if (count <= 2)/* ja:同名同パスのファイルをフルパス表示に変更する */
        for (i = g_list_length (glist) - 1; i >= 0; i--)
          {
            child = g_list_nth_data (glist, i);
            mmaid = g_object_get_data (G_OBJECT (child), "user_data");
            if (g_strfilecmp (file,  mmaid->file) == 0)
              {
                mmaid->same = -1;
                utf8str = g_filename_to_utf8 (mmaid->file, -1,
                                                            NULL, NULL, NULL);
                gtk_label_set_text (GTK_LABEL (mmaid->label), utf8str);
                g_free (utf8str);
              }
          }
    }
}


/******************************************************************************
*                                                                             *
* ja:ファイル入力関数群                                                       *
*                                                                             *
******************************************************************************/
#define METAINFO_PATH   0
#define METAINFO_TITLE  1
#define METAINFO_ARTIST 2
#define METAINFO_ALBUM  3
#define METAINFO_GENRE  4

typedef struct MetaInfo_Tag
{
  gsize len[5];
  gchar *str[5];
} MetaInfo;


static const gchar irivdb_header0[] = {
  0x69, 0x52, 0x69, 0x76, 0x44, 0x42, 0x20, 0x56,
  0x65, 0x72, 0x20, 0x30, 0x2e, 0x31, 0x32, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x69, 0x52, 0x69, 0x76, 0x65, 0x72, 0x20, 0x69,
  0x48, 0x50, 0x2d, 0x31, 0x30, 0x30, 0x20, 0x44,
  0x42, 0x20, 0x46, 0x69, 0x6c, 0x65, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static const gchar irivdb_header1[] = {
  0x44, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x65, 0x64,
  0x20, 0x62, 0x79, 0x20, 0x69, 0x52, 0x69, 0x76,
  0x65, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


/*  ja:パスのキャラクターセットを変換する
       file,ファイル名
       leng,バイト数
    charset,キャラクターセット
        err,TRUE:エラーなし,FALSE:強行
        RET,変換されたファイル名,NULL:エラー                                */
static gchar *
file_path_convert (const gchar *file,
                   const gsize  leng,
                   const gchar *charset,
                   gboolean    *err)
{
  gchar *result = NULL, *tmp;
  gsize bytes_done = 0;

  if (err)
    *err = TRUE;
  if (!file || leng <= 0 || !charset)
    return NULL;
  while (bytes_done < leng)
    {
      if (file[bytes_done] == '\\')
        {
          tmp = result ? g_strconcat (result, G_DIR_SEPARATOR_S, NULL)
                       : g_strdup (G_DIR_SEPARATOR_S);
          bytes_done++;
        }
      else
        {
          gchar *str = NULL;
          gsize len = 1;

          for (len = 1;bytes_done + len <= leng; len++)
            {
              str = g_convert (file + bytes_done, len, "UTF-8", charset,
                                                            NULL, NULL, NULL);
              if (str)
                break;
            }
          if (str)
            {
              tmp = result ? g_strconcat (result, str, NULL) : g_strdup (str);
              bytes_done += len;
              g_free (str);
            }
          else
            {
              tmp = result ? g_strconcat (result, "?", NULL) : g_strdup ("?");
              bytes_done++;
              if (err)
                *err = FALSE;
            }
        }
      g_free (result);
      result = tmp;
    }
  return result;
}


/*  ja:ファイルの種類を取得する
    file,ファイル名
     RET,FILE_KIND_DB:db,FILE_KIND_TAG:ID3TAG,FILE_KIND_UNKNOWN:不明        */
guint
file_is_type (const gchar *file)
{
  guint8 header[0xa0c0];
  guint8 tag[128];
  FileIO *fio;

  fio = fileio_open (file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (fio == NULL)
    return FILE_KIND_ERROR;
  if (fileio_read (fio, header, sizeof (header)) == sizeof (header)
   && g_memcmp (irivdb_header0, header, sizeof (irivdb_header0)) == 0
   && g_memcmp (irivdb_header1, header + 0xa080, sizeof (irivdb_header1)) == 0)
    {
      fileio_close (fio);
      return FILE_KIND_DB;
    }
  if (fileio_seek (fio, -128, FILEIO_SEEK_END) == -1
                    || fileio_read (fio, tag, sizeof (tag)) != sizeof (tag))
    {
      fileio_close (fio);
      return FILE_KIND_UNKNOWN;
    }
  fileio_close (fio);
  return tag[0] == 'T' && tag[1] == 'A' && tag[2] == 'G'
    && (tag[126] == 0 || tag[126] == ' ') ? FILE_KIND_TAG : FILE_KIND_UNKNOWN;
}


/*  ja:mp3ファイルを開く
        file,ファイル名
    charlist,キャラクターセットリスト
         RET,タグ                                                           */
ID3Tag *
file_open_id3tag (const gchar *file,
                  const gchar *charlist)
{
  gchar **charset, *tmp;
  gchar tag[3], title[30], artist[30], album[30], year[4], comment[30];
  guint8 gnum;
  gint i;
  FileIO *fio;
  ID3Tag *id3tag;

  if (!file || !charlist)
    return NULL;
  fio = fileio_open (file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (fio == NULL)
    return NULL;
  id3tag = g_malloc0 (sizeof (ID3Tag));
  tmp = fileio_get_full_path (file);
  id3tag->file = g_filename_to_utf8 (tmp, -1, NULL, NULL, NULL);
  g_free (tmp);
  id3tag->track = -1;
  if (fileio_seek (fio, -128, FILEIO_SEEK_END) == -1
            || fileio_read (fio, tag,     sizeof (tag))     != sizeof (tag)
            || fileio_read (fio, title,   sizeof (title))   != sizeof (title)
            || fileio_read (fio, artist,  sizeof (artist))  != sizeof (artist)
            || fileio_read (fio, album,   sizeof (album))   != sizeof (album)
            || fileio_read (fio, year,    sizeof (year))    != sizeof (year)
            || fileio_read (fio, comment, sizeof (comment)) != sizeof (comment)
            || fileio_read (fio, &gnum,   sizeof (gnum))    != sizeof (gnum)
            || tag[0] != 'T' || tag[1] != 'A' || tag[2] != 'G')
    {
      FileIOTime mtime;

      fileio_close (fio);
      if (fileio_gettime (file, NULL, &mtime))
        {
          FileIOSysTime stime;

          fileio_getsystime (&mtime, &stime);
          id3tag->year = g_strdup_printf ("%d", stime.year);
        }
      return id3tag;
    }
  fileio_close (fio);
  if (comment[29] == 0)
    {
      id3tag->track = (guint8)comment[28];
      comment[28] = comment[29] = ' ';
    }
  /* ja:キャラクターセット変換 */
  charset = g_strsplit (charlist, ",", G_MAXINT);
  for (i = 0; charset[i]; i++)
    {
      id3tag->title   = g_convert (title,   sizeof (title),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->artist  = g_convert (artist,  sizeof (artist),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->album   = g_convert (album,   sizeof (album),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->year    = g_convert (year,    sizeof (year),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->comment = g_convert (comment, sizeof (comment),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      if (id3tag->title && id3tag->artist && id3tag->album && id3tag->year
                                                            && id3tag->comment)
        break;
    }
  if (!charset[i])
    {
      i = 0;
      id3tag->title   = g_convert (title,   sizeof (title),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->artist  = g_convert (artist,  sizeof (artist),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->album   = g_convert (album,   sizeof (album),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->year    = g_convert (year,    sizeof (year),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
      id3tag->comment = g_convert (comment, sizeof (comment),
                                        "UTF-8", charset[i], NULL, NULL, NULL);
    }
  id3tag->charset = g_strdup (charset[i]);
  g_strfreev (charset);
  if (id3tag->title)
    {
      g_strchomp (id3tag->title);
      if (id3tag->title[0] == '\0')
        {
          g_free (id3tag->title);
          id3tag->title = NULL;
        }
    }
  if (id3tag->artist)
    {
      g_strchomp (id3tag->artist);
      if (id3tag->artist[0] == '\0')
        {
          g_free (id3tag->artist);
          id3tag->artist = NULL;
        }
    }
  if (id3tag->album)
    {
      g_strchomp (id3tag->album);
      if (id3tag->album[0] == '\0')
        {
          g_free (id3tag->album);
          id3tag->album = NULL;
        }
    }
  if (id3tag->year)
    {
      g_strchomp (id3tag->year);
      if (id3tag->year[0] == '\0')
        {
          g_free (id3tag->year);
          id3tag->year = NULL;
        }
    }
  if (id3tag->comment)
    {
      g_strchomp (id3tag->comment);
      if (id3tag->comment[0] == '\0')
        {
          g_free (id3tag->comment);
          id3tag->comment = NULL;
        }
    }
  for (i = 0; i < 256; i++)
    if (genre[i][gnum])
      {
        id3tag->genre = g_strdup (genre[i][gnum]);
        break;
      }
  return id3tag;
}


/*  ja:dbファイルを開く
    mmaid,ウインドウ情報                                                    */
void
file_open_db (MmaidWindow *mmaid)
{
  gchar **charset, *dir, *tmp;
  guint8 header[0xa0c0];
  goffset offset = 0;
  gint i, j, entries = 0;
  FileIO *fio;
  MetaInfo mi_r[10240], mi_c[10240];
  GtkWidget *dialog;

  g_memset (mi_r, 0, sizeof (mi_r));
  /* ja:ダイアログボックス */
  userbreak = TRUE;
  dialog = abort_dialog (_("Reading"));
  gtk_grab_add (dialog);
  /* ja:読み込み */
  fio = fileio_open (mmaid->file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (fio == NULL
    || fileio_read (fio, header, sizeof (header)) != sizeof (header)
    || g_memcmp (irivdb_header0, header, sizeof (irivdb_header0)) != 0
    || g_memcmp (irivdb_header1, header + 0xa080, sizeof (irivdb_header1)) != 0
    || (entries = GINT32_FROM_LE (*(gint32 *)(header + 0x0040))) > 10240)
    entries = 0;/* ja:ヘッタにエラーがある */
  /* ja:ヘッタのPaddingをチェック */
  if (entries)
    for (i = 0x0042; i <= 0x007F; i++)
      if (header[i] != 0)
        {
          entries = 0;
          break;
        }
  /* ja:ヘッタのオフセットのない部分をチェック */
  if (entries)
    for (i = entries * sizeof (guint32) + 0x0080; i <= 0xa07f; i++)
      if (header[i] != 0)
        {
          entries = 0;
          break;
        }
  /* ja:エントリ読み込み */
  for (i = 0; userbreak && i < entries; i++)
    {
      guint16 len[5];

      while (gtk_events_pending ())
        gtk_main_iteration ();
      if (offset != GUINT32_FROM_LE (((guint32 *)(header + 0x0080))[i])
                    || fileio_read (fio, len, sizeof (len)) != sizeof (len))
        {
          entries = 0;
          break;
        }
      offset += sizeof (len);
      for (j = 0; j < 5; j++)
        {
          if ((mi_r[i].len[j] = GUINT16_FROM_LE (len[j])) <= 0
                        || !(mi_r[i].str[j] = g_malloc0 (mi_r[i].len[j]))
                        || fileio_read (fio, mi_r[i].str[j], mi_r[i].len[j])
                                                            != mi_r[i].len[j]
                        || (mi_r[i].str[j])[mi_r[i].len[j] - 1] != '\0')
            {
              entries = 0;
              break;
            }
          offset += mi_r[i].len[j];
        }
    }
  if (entries)
    entries = i;
  if (fio)
    fileio_close (fio);
  /* ja:キャラクターセット変換 */
  charset = g_strsplit (mmaid->charset, ",", G_MAXINT);
  g_free (mmaid->charset);
  mmaid->charset = NULL;
  for (i = 0; charset[i]; i++)
    {
      gint k;

      g_memset (mi_c, 0, sizeof (mi_c));
      for (j = 0; j < entries; j++)
        {
          gboolean err;
          gchar *str;

          while (gtk_events_pending ())
            gtk_main_iteration ();
          str = file_path_convert (mi_r[j].str[0], mi_r[j].len[0] - 1,
                                                            charset[i], &err);
          mi_c[j].str[0] = str;
          if (!str || !err)
            break;
          for (k = 1; k < 5; k++)
            {
              str = g_convert (mi_r[j].str[k], mi_r[j].len[k] - 1,
                                        "UTF-8", charset[i], NULL, NULL, NULL);
              if (!str)
                break;
              mi_c[j].str[k] = str;
            }
          if (k < 5)
            break;
        }
      if (j == entries)
        break;
      do
        for (k = 0; k < 5; k++)
          g_free (mi_c[j].str[k]);
      while (--j >= 0);
    }
  /* ja:キャラクターセット変換を強行する */
  if (!charset[i])
    {
      g_memset (mi_c, 0, sizeof (mi_c));
      for (i = 0; i < entries; i++)
        {
          gchar *str;

          while (gtk_events_pending ())
            gtk_main_iteration ();
          mi_c[i].str[0]
                    = file_path_convert (mi_r[i].str[0], mi_r[i].len[0] - 1,
                                                            charset[0], NULL);
          if (!mi_c[i].str[0])
            break;
          for (j = 1; j < 5; j++)
            {
              gchar *buffer;
              gsize bytes_read_prev, bytes_read;
              GError *error = NULL;

              bytes_read_prev = mi_r[i].len[j];
              buffer = g_memdup (mi_r[i].str[j], mi_r[i].len[j]);
              while ((g_clear_error (&error),
                        !(str = g_convert (buffer, mi_r[i].len[j] - 1,
                            "UTF-8", charset[0], &bytes_read, NULL, &error)))
                            && bytes_read_prev != bytes_read && error != NULL
                            && g_error_matches (error, G_CONVERT_ERROR,
                                            G_CONVERT_ERROR_ILLEGAL_SEQUENCE))
                {
                  while (gtk_events_pending ())
                    gtk_main_iteration ();
                  buffer[bytes_read] = '?';
                  bytes_read_prev = bytes_read;
                }
              g_free (buffer);
              g_clear_error (&error);
              mi_c[i].str[j] = str;
            }
        }
      i = 0;
    }
  mmaid->charset = g_strdup (charset[i]);
  g_strfreev (charset);
  for (i = 0; i < 10240; i++)
    for (j = 0; j < 5; j++)
      g_free (mi_r[i].str[j]);
  /* ja:リストに追加 */
  tmp = g_path_get_dirname (mmaid->file);
  dir = g_filename_to_utf8 (tmp, -1, NULL, NULL, NULL);
  g_free (tmp);
  for (i = 0; i < entries; i++)
    {
      ID3Tag *id3tag;

      id3tag = g_malloc0 (sizeof (ID3Tag));
      id3tag->file    = g_strconcat (dir, mi_c[i].str[METAINFO_PATH], NULL);
      g_free (mi_c[i].str[METAINFO_PATH]);
      id3tag->title   = mi_c[i].str[METAINFO_TITLE];
      id3tag->artist  = mi_c[i].str[METAINFO_ARTIST];
      id3tag->album   = mi_c[i].str[METAINFO_ALBUM];
      id3tag->track   = -1;
      id3tag->genre   = mi_c[i].str[METAINFO_GENRE];
      mm_list_append (MM_LIST (mmaid->list), id3tag);
    }
  g_free (dir);
  /* ja:終了処理 */
  gtk_grab_remove (dialog);
  gtk_widget_destroy (dialog);
}


/*  ja:ファイルを開く
    file,ファイル名
     RET,ウインドウ情報                                                     */
MmaidWindow *
file_open_edit(const gchar *file)
{
  gchar *text, *utf8str;
  MmaidWindow *mmaid;
  GtkWidget *menu_shell, *scroll, *hbox;
  static gint fcount = 0; /* ja:新規ファイルの数 */
#ifndef G_OS_WIN32
    GdkPixmap *pixmap;
    GdkBitmap *mask;
    GtkWidget *button, *frame;
    static gchar *xpm[]={
"8 7 2 1",
"   c None",
".  c #000000",
"..    ..",
" ..  .. ",
"  ....  ",
"   ..   ",
"  ....  ",
" ..  .. ",
"..    .."};
#endif /* not G_OS_WIN32 */

  mmaid = g_malloc0 (sizeof (MmaidWindow));
  mmaid->same = -1;
  /* ja:リスト */
  mmaid->list = mm_list_new ();
  mm_list_set_editable (MM_LIST (mmaid->list), TRUE);
  mm_list_set_editable_file (MM_LIST (mmaid->list), TRUE);
  if (!file)
    {
      gchar **charset;

      /* ja:新規 */
      mmaid->file = g_strdup_printf ("new%04d", fcount++ % 1000);
      mmaid->create = TRUE;
      charset = g_strsplit (charset_list, ",", G_MAXINT);
      mmaid->charset = g_strdup (charset[0]);
      g_strfreev (charset);
    }
  else
    {
      mmaid->file = fileio_get_full_path (file);
      mmaid->create = FALSE;
      mmaid->charset = g_strdup (charset_list);
      file_open_db (mmaid);
    }
  g_signal_connect (G_OBJECT (mmaid->list), "changed",
                                    G_CALLBACK (signal_changed), mmaid);
  g_signal_connect (G_OBJECT (mmaid->list), "clicked",
                                    G_CALLBACK (signal_clicked), mmaid);
  g_signal_connect (G_OBJECT (mmaid->list), "deleted",
                                    G_CALLBACK (signal_deleted), mmaid);
  g_signal_connect (G_OBJECT (mmaid->list), "edited",
                                    G_CALLBACK (signal_edited), mmaid);
  g_signal_connect (G_OBJECT (mmaid->list), "inserted",
                                    G_CALLBACK (signal_inserted), mmaid);
  g_signal_connect (G_OBJECT (mmaid->list), "moved",
                                    G_CALLBACK (signal_moved), mmaid);
  g_signal_connect (G_OBJECT (mmaid->list), "destroy",
                                    G_CALLBACK (signal_destroy_list), mmaid);

  /* ja:ファイル名を調べる */
  text = file_add_edit (mmaid->file, &mmaid->same);

  /* ja:スクロールウインドウ */
  scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scroll), mmaid->list);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  g_object_set_data (G_OBJECT (scroll), "user_data", mmaid);
  /* ja:メニュー */
  menu_shell = gtk_item_factory_get_widget (ifactory_menu, "<main>/Window");
  utf8str = g_filename_to_utf8 (mmaid->file, -1, NULL, NULL, NULL);
  mmaid->menu_item = gtk_menu_item_new_with_label (utf8str);
  g_free (utf8str);
  g_signal_connect (G_OBJECT (mmaid->menu_item), "activate",
                            G_CALLBACK (signal_activate_menu_window), scroll);
  gtk_menu_append (GTK_MENU (menu_shell), mmaid->menu_item);
  gtk_widget_show (mmaid->menu_item);
  /* ja:ボタン */
#ifndef G_OS_WIN32
  button = gtk_button_new ();
  frame = gtk_frame_new (NULL);
  gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
  pixmap = gdk_pixmap_create_from_xpm_d (window->window, &mask, NULL, xpm);
  gtk_container_add (GTK_CONTAINER (frame), gtk_pixmap_new (pixmap, mask));
  gtk_container_add (GTK_CONTAINER (button), frame);
  g_signal_connect (G_OBJECT (button), "clicked",
                                G_CALLBACK (signal_clicked_button), scroll);
#endif /* not G_OS_WIN32 */
  /* ja:ラベル */
  mmaid->label = gtk_label_new (text);
  g_free (text);
  /* ja:マーク */
  mmaid->mark = gtk_label_new ("*");
  /* ja:表示 */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), mmaid->label, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), mmaid->mark, FALSE, FALSE, 0);
#ifndef G_OS_WIN32
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
#endif /* not G_OS_WIN32 */
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), scroll, hbox);
  gtk_widget_show (hbox);
  gtk_widget_show (mmaid->label);
#ifndef G_OS_WIN32
  gtk_widget_show_all (button);
#endif /* not G_OS_WIN32 */
  gtk_widget_show_all (scroll);
  gtk_notebook_set_page (GTK_NOTEBOOK (notebook),
                    gtk_notebook_page_num (GTK_NOTEBOOK (notebook),scroll));
  gtk_widget_grab_focus (mmaid->list);

  /* ja:新規作成ではなく、ファイルが存在する時には履歴に加える */
  if (file && history > 0 && fileio_isfile (mmaid->file))
    file_set_history (mmaid->file);

  return mmaid;
}


/******************************************************************************
*                                                                             *
* ja:ファイル出力関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:mp3ファイルを保存する
      file,ファイル名
    id3tag,タグ
      gnum,ジャンル
       RET,エラーコード                                                     */
gint
file_save_id3tag (const gchar  *file,
                  const ID3Tag *id3tag,
                  const gint    gnum)
{
  gboolean has;
  gchar tag[3], *title, *artist, *album, *year, *comment, *tmp;
  guint8 buf[128];
  FileIO *fio;
  FileIOTime atime, mtime;

  buf[127] = gnum;
  buf[0] = 'T';
  buf[1] = 'A';
  buf[2] = 'G';
  g_memset (buf + 3, ' ', 124);
  title   = buf    +  3;
  artist  = title  + 30;
  album   = artist + 30;
  year    = album  + 30;
  comment = year   +  4;
  if (id3tag->title)
    {
      tmp = g_convert (id3tag->title, -1,
                                id3tag->charset, "UTF-8", NULL, NULL, NULL);
      if (!tmp)
        return FILE_SAVE_CHARSET_TITLE;
      if (g_strlen (tmp) > 30)
        {
          g_free (tmp);
          return FILE_SAVE_LENGTH_TITLE;
        }
      g_memmove (title, tmp, g_strlen (tmp));
      g_free (tmp);
    }
  if (id3tag->artist)
    {
      tmp = g_convert (id3tag->artist, -1,
                                id3tag->charset, "UTF-8", NULL, NULL, NULL);
      if (!tmp)
        return FILE_SAVE_CHARSET_ARTIST;
      if (g_strlen (tmp) > 30)
        {
          g_free (tmp);
          return FILE_SAVE_LENGTH_ARTIST;
        }
      g_memmove (artist, tmp, g_strlen (tmp));
      g_free (tmp);
    }
  if (id3tag->album)
    {
      tmp = g_convert (id3tag->album, -1,
                                id3tag->charset, "UTF-8", NULL, NULL, NULL);
      if (!tmp)
        return FILE_SAVE_CHARSET_ALBUM;
      if (g_strlen (tmp) > 30)
        {
          g_free (tmp);
          return FILE_SAVE_LENGTH_ALBUM;
        }
      g_memmove (album, tmp, g_strlen (tmp));
      g_free (tmp);
    }
  if (id3tag->year)
    {
      tmp = g_convert (id3tag->year, -1,
                                id3tag->charset, "UTF-8", NULL, NULL, NULL);
      if (!tmp)
        return FILE_SAVE_CHARSET_YEAR;
      if (g_strlen (tmp) > 4)
        {
          g_free (tmp);
          return FILE_SAVE_LENGTH_YEAR;
        }
      g_memmove (year, tmp, g_strlen (tmp));
      g_free (tmp);
    }
  if (id3tag->comment)
    {
      tmp = g_convert (id3tag->comment, -1,
                                id3tag->charset, "UTF-8", NULL, NULL, NULL);
      if (!tmp)
        return FILE_SAVE_CHARSET_COMMENT;
      if (g_strlen (tmp) > (0 <= id3tag->track && id3tag->track <= 255
                                                                    ? 28 : 30))
        {
          g_free (tmp);
          return FILE_SAVE_LENGTH_COMMENT;
        }
      g_memmove (comment, tmp, g_strlen (tmp));
      g_free (tmp);
    }
  if (0 <= id3tag->track && id3tag->track <= 255)
    {
      comment[28] = 0;
      comment[29] = id3tag->track;
    }
  if (!fileio_gettime (file, &atime, &mtime))
    return FILE_SAVE_ERRROR;
  fio = fileio_open (file, FILEIO_ACCESS_READ | FILEIO_ACCESS_WRITE,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING);
  if (fio == NULL)
    {
      fileio_settime (file, &atime, &mtime);
      return FILE_SAVE_ERRROR;
    }
  has = fileio_seek (fio, -128, FILEIO_SEEK_END) != -1
                        && fileio_read (fio, tag, sizeof (tag)) == sizeof (tag)
                        && tag[0] == 'T' && tag[1] == 'A' && tag[2] == 'G';
  if (fileio_seek (fio, has ? -128 : 0, FILEIO_SEEK_END) == -1
                    || fileio_write (fio, buf, sizeof (buf)) != sizeof (buf))
    {
      fileio_close (fio);
      fileio_settime (file, &atime, &mtime);
      return FILE_SAVE_ERRROR;
    }
  fileio_close (fio);
  if (!fileio_settime (file, &atime, &mtime))
    return FILE_SAVE_ERRROR;
  return FILE_SAVE_SUCCESS;
}


/*  ja:dbファイルを保存する
       file,ファイル名
      mmaid,ウインドウ情報
       root,ルート
    charset,キャラクターセット
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
file_save_db (const gchar *file,
              MmaidWindow *mmaid,
              const gchar *root,
              const gchar *charset)
{
  guint8 *buf;
  gint i, count, length;
  goffset offset = 0xa0c0;
  GtkWidget *dialog;
  FileIO *fio;

  /* ja:ダイアログボックス */
  userbreak = TRUE;
  dialog = abort_dialog (_("Writing"));
  gtk_grab_add (dialog);
  /* ja:変数 */
  count = MIN (mm_list_length (MM_LIST (mmaid->list)), 10240);
  length = g_strlen (root);
  buf = g_malloc0 ((gsize)offset);
  g_memmove (buf,          irivdb_header0, sizeof (irivdb_header0));
  g_memmove (buf + 0xa080, irivdb_header1, sizeof (irivdb_header1));
  *(guint16 *)(buf + 0x40) = GUINT16_TO_LE (count);
  /* ja:タグ */
  for (i = 0; userbreak && i < count; i++)
    {
      gchar *file = NULL, *title, *artist, *album, *genre, **ary, *tmp;
      gint j;
      const ID3Tag *id3tag;

      while (gtk_events_pending ())
        gtk_main_iteration ();
      id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
      ary = g_strsplit (id3tag->file + length, G_DIR_SEPARATOR_S, G_MAXINT);
      for (j = 0; ary[j]; j++)
        {
          tmp = g_convert (ary[j], -1, charset, "UTF-8", NULL, NULL, NULL);
          if (!tmp)
            {
              g_free (file);
              file = NULL;
              break;
            }
          file = file ? g_strconcat (file, "\\", tmp, NULL)
                      : g_strconcat (      "\\", tmp, NULL);
          g_free (tmp);
        }
      g_strfreev (ary);
      title  = id3tag->title  ? g_convert (id3tag->title,  -1,
                                        charset, "UTF-8", NULL, NULL, NULL)
                              : g_malloc0 (sizeof (gchar));
      artist = id3tag->artist ? g_convert (id3tag->artist, -1,
                                        charset, "UTF-8", NULL, NULL, NULL)
                              : g_malloc0 (sizeof (gchar));
      album  = id3tag->album  ? g_convert (id3tag->album,  -1,
                                        charset, "UTF-8", NULL, NULL, NULL)
                              : g_malloc0 (sizeof (gchar));
      genre  = id3tag->genre  ? g_convert (id3tag->genre,  -1,
                                        charset, "UTF-8", NULL, NULL, NULL)
                              : g_malloc0 (sizeof (gchar));
      if (file && title && artist && album && genre)
        {
          guint8 *p;
          guint16 len[5];
          gsize leng[5], size = sizeof (len);

          leng[0] = g_strlen (file)   + 1;
          leng[1] = g_strlen (title)  + 1;
          leng[2] = g_strlen (artist) + 1;
          leng[3] = g_strlen (album)  + 1;
          leng[4] = g_strlen (genre)  + 1;
          for (j = 0; j < 5; j++)
            {
              len[j] = GUINT16_TO_LE (leng[j]);
              size += leng[j];
            }
          ((guint32 *)(buf + 0x80))[i] = offset - 0xa0c0;
          buf = g_realloc (buf, (gsize)offset + size);
          p = buf + offset;
          offset += size;
          g_memmove (p, len,    sizeof (len)); p += sizeof (len);
          g_memmove (p, file,   leng[0]);      p += leng[0];
          g_memmove (p, title,  leng[1]);      p += leng[1];
          g_memmove (p, artist, leng[2]);      p += leng[2];
          g_memmove (p, album,  leng[3]);      p += leng[3];
          g_memmove (p, genre,  leng[4]);
        }
      else
        {
          gchar *text;

          text = g_strdup_printf (_("Conversion from character encoding"
                                " 'UTF-8' to '%s' is not supported"), charset);
          if (!file)
            {
              tmp = g_strdup_printf (_("%s\nFile : %s"), text, id3tag->file);
              g_free (text);
              text = tmp;
            }
          if (!title)
            {
              tmp = g_strdup_printf (_("%s\nTitle : %s"), text, id3tag->title);
              g_free (text);
              text = tmp;
            }
          if (!artist)
            {
              tmp = g_strdup_printf (_("%s\nArtist : %s"),
                                                        text, id3tag->artist);
              g_free (text);
              text = tmp;
            }
          if (!album)
            {
              tmp = g_strdup_printf (_("%s\nAlbum : %s"), text, id3tag->album);
              g_free (text);
              text = tmp;
            }
          if (!genre)
            {
              tmp = g_strdup_printf (_("%s\nGenre : %s"), text, id3tag->genre);
              g_free (text);
              text = tmp;
            }
          misc_message_box ("Melody maid", text, 0, _("OK"), NULL);
          g_free (text);
          userbreak = FALSE;
        }
      g_free (file);
      g_free (title);
      g_free (artist);
      g_free (album);
      g_free (genre);
    }
  fio = userbreak ? fileio_open (file,
                        FILEIO_ACCESS_READ | FILEIO_ACCESS_WRITE,
                        FILEIO_SHARE_READ, FILEIO_MODE_CREATE_ALWAYS) : NULL;
  if (fio == NULL)
    {
      misc_message_box ("Melody maid", _("Can not open file"),
                                                            0, _("OK"), NULL);
      userbreak = FALSE;
    }
  if (fio != NULL && fileio_write (fio, buf, (gsize)offset) != offset)
    {
      misc_message_box ("Melody maid", _("Can not write"), 0, _("OK"), NULL);
      userbreak = FALSE;
    }
  if (fio != NULL)
    fileio_close (fio);
  g_free (buf);
  /* ja:終了処理 */
  gtk_grab_remove (dialog);
  gtk_widget_destroy (dialog);
  return userbreak;
}
