/*
    comm
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "fileio.h"
#include "misc_comm.h"
#ifdef G_OS_WIN32
# include <tchar.h>
# include <windows.h>
#endif /* G_OS_WIN32 */


enum
{
  RECV_SIGNAL,
  LAST_SIGNAL
};
#ifndef G_OS_WIN32
enum
{
    TARGET_STRING
};
#endif /* not G_OS_WIN32 */


static void misc_comm_class_init (MiscCommClass *klass);
static void misc_comm_init       (MiscComm      *comm);
static void misc_comm_dispose    (GObject       *object);


static GObjectClass *parent_class = NULL;
static gint misc_comm_signals[LAST_SIGNAL] = {0};
#ifdef G_OS_WIN32
static GList *glist = NULL;
#endif /* G_OS_WIN32 */
# ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
# endif /* USE_THREAD */


/******************************************************************************
*                                                                             *
******************************************************************************/
GType
misc_comm_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (MiscCommClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)misc_comm_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (MiscComm),
        0,              /* n_preallocs */
        (GInstanceInitFunc)misc_comm_init,
      };

      type = g_type_register_static (G_TYPE_OBJECT, "MiscComm", &info, 0);
    }

  return type;
}


static void
misc_comm_class_init (MiscCommClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass *) klass;

  object_class->dispose = misc_comm_dispose;

  klass->recv = NULL;

  misc_comm_signals[RECV_SIGNAL]
        = g_signal_new ("recv",
                G_TYPE_FROM_CLASS (klass),
                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                G_STRUCT_OFFSET (MiscCommClass, recv),
                NULL, NULL,
                g_cclosure_marshal_VOID__POINTER,
                G_TYPE_NONE, 1,
                G_TYPE_POINTER);
}


/*  ja:新規作成
    RET,オブジェクト                                                        */
static void
misc_comm_init (MiscComm *comm)
{
  comm->handler_destroy  = 0;
  comm->window           = NULL;
#ifdef G_OS_WIN32
  comm->ddeInst          = 0;
  comm->hszService       = NULL;
  comm->hszTopic         = NULL;
  comm->hConv            = NULL;
#else /* not G_OS_WIN32 */
  comm->instance         = -1;
  comm->first            = TRUE;
  comm->arg              = NULL;
  comm->handler_get      = 0;
  comm->handler_received = 0;
  comm->atom0            = NULL;
  comm->atom1            = NULL;
  comm->atom2            = NULL;
#endif /* not G_OS_WIN32 */
}


static void
misc_comm_dispose (GObject *object)
{
  MiscComm *comm = MISC_COMM (object);

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
#ifdef G_OS_WIN32
  if (comm->ddeInst)
    {
      if (comm->hConv)
        DdeDisconnect (comm->hConv);
      DdeNameService (comm->ddeInst, comm->hszService, NULL, DNS_UNREGISTER);
      DdeFreeStringHandle (comm->ddeInst, comm->hszService);
      DdeFreeStringHandle (comm->ddeInst, comm->hszTopic);
      DdeUninitialize (comm->ddeInst);
    }
  glist = g_list_remove (glist, comm);
#else /* not G_OS_WIN32 */
  g_free (comm->arg);
  g_signal_handler_disconnect (G_OBJECT (comm->window), comm->handler_get);
  g_signal_handler_disconnect (G_OBJECT (comm->window),
                                                    comm->handler_received);
#endif /* not G_OS_WIN32 */
  g_signal_handler_disconnect (G_OBJECT (comm->window), comm->handler_destroy);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */

  if (G_OBJECT_CLASS (parent_class)->dispose)
    G_OBJECT_CLASS (parent_class)->dispose (object);
}


#ifdef G_OS_WIN32
static HDDEDATA CALLBACK
DdemlCallback (UINT     uType,
               UINT     uFmt,
               HCONV    hConv,
               HSZ      hszTopic,
               HSZ      hszService,
               HDDEDATA hData,
               DWORD    dwData1,
               DWORD    dwData2)
{
  guint i, length;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  length = g_list_length (glist);
  for (i = 0; i < length; i++)
    {
      MiscComm *comm;

      comm = g_list_nth_data (glist, i);
      if (!comm)
        break;
      switch (uType)
        {
          case XTYP_CONNECT:
            return (HDDEDATA)
                    (DdeCmpStringHandles (hszService, comm->hszService) == 0
                    && DdeCmpStringHandles (hszTopic, comm->hszTopic) == 0);
          case XTYP_POKE:
            if (DdeCmpStringHandles (hszTopic, comm->hszTopic) == 0)
              {
                gchar *text;

                text = DdeAccessData (hData, NULL);
                if (text)
                  {
                    gchar **argv;

                    argv = g_strsplit (text, "\n", G_MAXINT);
                    g_signal_emit_by_name (G_OBJECT (comm), "recv", argv);
                    g_strfreev (argv);
                  }
                DdeFreeDataHandle (hData);
                return (HDDEDATA)DDE_FACK;
              }
        }
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return NULL;
}


#else /* not G_OS_WIN32 */
static void
misc_comm_selection_get (GtkWidget        *widget,
                         GtkSelectionData *data,
                         guint             info,
                         guint             time,
                         MiscComm         *comm)
{
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (data->selection == comm->atom0)
    {
      /* ja:他プロセスがプロセスの存在を確認しにきたとき
            このプロセスの存在を知らせる */
      gchar *text;

      text = "Misc Comm Selection";
      gtk_selection_data_set_text (data, text, g_strlen (text));
    }
  else if (data->selection == comm->atom1)
    {
      /* ja:他プロセスが引数の受渡しをするので
            このプロセスは引数を受け取りにいく */
      gtk_selection_convert (comm->window, comm->atom2,
                                GDK_SELECTION_TYPE_STRING, GDK_CURRENT_TIME);
    }
  else if (data->selection == comm->atom2 && comm->arg)
    {
      /* ja:他プロセスが引数を取りにきたのでこのプロセスの引数を渡す */
      gtk_selection_data_set_text (data, comm->arg, g_strlen (comm->arg));
      g_free (comm->arg);
      comm->arg = NULL;
    }
}


static void
misc_comm_selection_received (GtkWidget        *widget,
                              GtkSelectionData *data,
                              guint             time,
                              MiscComm         *comm)
{
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (data->selection == comm->atom0)
    {
      /* ja:他にプロセスが存在するならば何かデータが返ってきている */
      comm->instance = MAX (data->length, 0);
    }
  else if (data->selection == comm->atom2 && data->length > 0)
    {
      gchar **argv;

      /* ja:他にプロセスのコマンドラインを受け取ったとき */
      argv = g_strsplit (data->data, "\n", G_MAXINT);
      g_signal_emit_by_name (G_OBJECT (comm), "recv", argv);
      g_strfreev (argv);
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
}
#endif /* not G_OS_WIN32 */


/*  ja:新規作成
    window,ウィジェット
    unique,ユニークな文字列
       RET,オブジェクト                                                     */
GObject*
misc_comm_new (GtkWidget   *window,
               const gchar *unique)
{
  gchar *unique0, *unique1, *unique2;
  MiscComm *comm;
#ifdef G_OS_WIN32
  LPTSTR lpszUnique0, lpszUnique1;
#else /* not G_OS_WIN32 */
  GtkTargetEntry targets_comm[] = {{"STRING", 0, TARGET_STRING}};
#endif /* not G_OS_WIN32 */

  if (!window || !unique)
    return NULL;
  comm = MISC_COMM (g_object_new (MISC_TYPE_COMM, NULL));
#ifdef G_OS_WIN32
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  glist = g_list_append (glist, comm);
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
#endif /* G_OS_WIN32 */
  comm->window = window;
  comm->handler_destroy = g_signal_connect_swapped (G_OBJECT (window),
                    "destroy", G_CALLBACK (g_object_unref), G_OBJECT (comm));
  unique0 = g_strconcat (unique, " 0", NULL);
  unique1 = g_strconcat (unique, " 1", NULL);
  unique2 = g_strconcat (unique, " 2", NULL);
#ifdef G_OS_WIN32
# ifdef UNICODE
  lpszUnique0 = g_utf8_to_utf16 (unique0,  -1, NULL, NULL, NULL);
  lpszUnique1 = g_utf8_to_utf16 (unique1, -1, NULL, NULL, NULL);
# else /* not UNICODE */
  lpszUnique0 = g_locale_from_utf8 (unique0, -1, NULL, NULL, NULL);
  lpszUnique1 = g_locale_from_utf8 (unique1, -1, NULL, NULL, NULL);
# endif /* not UNICODE */
#else /* not G_OS_WIN32 */
  comm->atom0 = gdk_atom_intern (unique0, FALSE);
  comm->atom1 = gdk_atom_intern (unique1, FALSE);
  comm->atom2 = gdk_atom_intern (unique2, FALSE);
#endif /* not G_OS_WIN32 */
  g_free (unique0);
  g_free (unique1);
  g_free (unique2);
#ifdef G_OS_WIN32
  if (DdeInitialize (&comm->ddeInst, DdemlCallback,
                            APPCLASS_STANDARD | CBF_FAIL_SELFCONNECTIONS, 0)
                                                            != DMLERR_NO_ERROR)
    {
      g_free (lpszUnique0);
      g_free (lpszUnique1);
      return G_OBJECT (comm);
    }
# ifdef UNICODE
  comm->hszService = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique0, CP_WINUNICODE);
  comm->hszTopic   = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique1, CP_WINUNICODE));
# else /* not UNICODE */
  comm->hszService = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique0, CP_WINANSI);
  comm->hszTopic   = DdeCreateStringHandle (comm->ddeInst,
                                                lpszUnique1, CP_WINANSI);
# endif /* not UNICODE */
  g_free (lpszUnique0);
  g_free (lpszUnique1);
  if (!comm->hszService || !comm->hszTopic)
    {
      if (comm->hszService)
        {
          DdeFreeStringHandle (comm->ddeInst, comm->hszService);
          comm->hszService = NULL;
        }
      if (comm->hszTopic)
        {
          DdeFreeStringHandle (comm->ddeInst, comm->hszTopic);
          comm->hszTopic = NULL;
        }
      DdeUninitialize (comm->ddeInst);
      comm->ddeInst = 0;
      return G_OBJECT (comm);
    }
  DdeNameService (comm->ddeInst, comm->hszService, NULL, DNS_REGISTER);
  comm->hConv = DdeConnect (comm->ddeInst,
                                    comm->hszService, comm->hszTopic, NULL);
#else /* not G_OS_WIN32 */
  gtk_selection_add_targets (window, comm->atom0, targets_comm, 1);
  gtk_selection_add_targets (window, comm->atom1, targets_comm, 1);
  gtk_selection_add_targets (window, comm->atom2, targets_comm, 1);
  comm->handler_get = g_signal_connect (G_OBJECT(window),
                            "selection-get",
                            G_CALLBACK (misc_comm_selection_get), comm);
  comm->handler_received = g_signal_connect (G_OBJECT (window),
                            "selection-received",
                            G_CALLBACK (misc_comm_selection_received), comm);

  gtk_selection_convert (window, comm->atom0,
                                GDK_SELECTION_TYPE_STRING, GDK_CURRENT_TIME);
# ifdef USE_THREAD
  gdk_threads_enter ();
# endif /* USE_THREAD */
  while (comm->instance < 0)
    while (gtk_events_pending ())
      gtk_main_iteration ();
# ifdef USE_THREAD
  gdk_threads_leave ();
# endif /* USE_THREAD */
  if (comm->instance > 0)
    {
      /* ja:既に起動しているとき */
      comm->first = FALSE;
    }
  else
    {
      /* ja:初めて起動してたとき */
      comm->first = TRUE;
      gtk_selection_owner_set (window, comm->atom0, GDK_CURRENT_TIME);
      gtk_selection_owner_set (window, comm->atom1, GDK_CURRENT_TIME);
    }
#endif /* not G_OS_WIN32 */
  return G_OBJECT (comm);
}


/*  ja:引数を最初のプロセスに送信する(ファイルの変換あり)
     comm,オブジェクト
     argc,引数の数
     argv,引数
    files,TRUE:ファイル,FALSE:ファイルではない                              */
void
misc_comm_send_with_files (MiscComm  *comm,
                           gint       argc,
                           gchar    **argv,
                           gboolean  *files)
{
  gsize length = 0;
  gchar *arg = NULL, *tmp;
  gint i;

#ifdef G_OS_WIN32
  if (!comm || !comm->hConv)
#else /* not G_OS_WIN32 */
  if (!comm || comm->first)
#endif /* not G_OS_WIN32 */
    return;
  for (i = 0; i < argc; i++)
    {
      gint leng;

      if (files && files[i])
        {
          tmp = fileio_get_full_path (argv[i]);
          leng = g_strlen (tmp) * sizeof (gchar);
          arg = g_realloc (arg, length + leng + sizeof (gchar));
          g_memmove (arg + length, tmp, leng);
          g_free (tmp);
        }
      else
        {
          leng = g_strlen (argv[i]) * sizeof (gchar);
          arg = g_realloc (arg, length + leng + sizeof (gchar));
          g_memmove (arg + length, argv[i], leng);
        }
      length += leng;
      arg[length++] = '\n';
    }
#ifdef G_OS_WIN32
  if (length > 0)
    {
      HDDEDATA hData;

      arg[length - 1] = '\0';
      hData = DdeCreateDataHandle (comm->ddeInst, arg, length, 0,
                                                comm->hszTopic, CF_TEXT, 0);
      if (hData)
        DdeClientTransaction ((LPBYTE)hData, 0xffffffff, comm->hConv,
                    comm->hszTopic, CF_TEXT, XTYP_POKE, TIMEOUT_ASYNC, NULL);
    }
#else /* not G_OS_WIN32 */
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  tmp = comm->arg;
  g_free (tmp);
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  if (length > 0)
    {
      arg[length - 1] = '\0';
      comm->arg = arg;
      gtk_selection_owner_set (comm->window, comm->atom2, GDK_CURRENT_TIME);
      gtk_selection_convert (comm->window, comm->atom1,
                                GDK_SELECTION_TYPE_STRING, GDK_CURRENT_TIME);
# ifdef USE_THREAD
      gdk_threads_enter ();
# endif /* USE_THREAD */
      while (comm->arg)
        while (gtk_events_pending ())
          gtk_main_iteration ();
# ifdef USE_THREAD
      gdk_threads_leave ();
# endif /* USE_THREAD */
    }
#endif /* not G_OS_WIN32 */
}
