/*
    Melody maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <gdk/gdkkeysyms.h>
#include "charlist.h"
#include "mm_list.h"


/******************************************************************************
*                                                                             *
* ja:キャラクターセットリスト関数群                                           *
*                                                                             *
******************************************************************************/
/*  キャラクターセットが正しいか判定する
    charset,キャラクターセット
        RET,TRUE:正しい,FALSE:不正                                          */
gboolean
charlist_is_valid (const gchar *charset)
{
    gboolean parenthesis = TRUE;
    gint i;
    GIConv converter;

    if (!charset || !g_ascii_isalnum (charset[0]))
        return FALSE;
    for (i = 1; charset[i] != '\0'; i++)
        if (charset[i] == '(' && parenthesis)
            parenthesis = FALSE;
        else if (charset[i] == ')' && !parenthesis)
            parenthesis = TRUE;
        else if (!g_ascii_isalnum (charset[i]) && charset[i] != '-'
                && charset[i] != '.' && charset[i] != ':' && charset[i] != '_')
            return FALSE;
    i--;
    if (!g_ascii_isalnum (charset[i]) && charset[i] != ')')
        return FALSE;
    converter = g_iconv_open (charset, "UTF-8");
    if (converter == (GIConv)-1 || g_iconv_close (converter) != 0)
        return FALSE;
    converter = g_iconv_open ("UTF-8", charset);
    return converter != (GIConv)-1 && g_iconv_close (converter) == 0;
}


/*  キャラクターセットリストを整形する
    charlist,キャラクターセットリスト
         RET,整形されたキャラクターセットリスト                             */
gchar *
charlist_strip (const gchar *charlist)
{
    gchar **charset, *tmp, *result = NULL;
    gint i, j;

    if (charlist != NULL)
      {
        charset = g_strsplit (charlist, ",", G_MAXINT);
        for (i = 0; charset[i] != NULL; i++)
          {
            g_strstrip (charset[i]);
            if (charlist_is_valid (charset[i]))
              {
                for (j = 0; j < i; j++)
                    if (g_strcasecmp (charset[i], charset[j]) == 0)
                        break;
                if (i <= j)
                  {
                    tmp = result == NULL ? g_strdup (charset[i])
                            : g_strconcat (result, ",", charset[i], NULL);
                    g_free (result);
                    result = tmp;
                  }
              }
          }
        g_strfreev (charset);
      }
    return result;
}


/*  キャラクターセットリストを置換する
    charlist0,キャラクターセットリスト
    charlist1,置換するキャラクターセットリスト
         RET,置換されたキャラクターセットリスト                             */
gchar *
charlist_rename (const gchar *charlist0, const gchar *charlist1)
{
    gchar **charset0, **charset1, *tmp, *result = NULL;
    gint i, j;

    if (charlist0 == NULL || charlist1 == NULL)
        return charlist0 != NULL ? g_strdup (charlist0) : NULL;
    charset0 = g_strsplit (charlist0, ",", G_MAXINT);
    charset1 = g_strsplit (charlist1, ",", G_MAXINT);
    for (i = 0; charset0[i] != NULL; i++)
      {
        for (j = 0; charset1[j] != NULL; j++)
            if (g_strcasecmp (charset0[i], charset1[j]) == 0)
                break;
        if (charset1[j] != NULL)
            tmp = result == NULL ? g_strdup (charset1[j])
                            : g_strconcat (result, ",", charset1[j], NULL);
        else
            tmp = result == NULL ? g_strdup (charset0[i])
                            : g_strconcat (result, ",", charset0[i], NULL);
        g_free (result);
        result = tmp;
      }
    return result;
}


/*  キャラクターセットリストを追加する
    charlist0,キャラクターセットリスト
    charlist1,追加するキャラクターセットリスト
          RET,追加されたキャラクターセットリスト                            */
gchar *
charlist_append (const gchar *charlist0, const gchar *charlist1)
{
    gchar **charset, *tmp, *result;
    gint i;

    if (charlist0 == NULL || charlist1 == NULL)
        return charlist0 != NULL ? g_strdup (charlist0)
                            : charlist1 != NULL ? g_strdup (charlist1) : NULL;
    tmp = charlist_strip (charlist1);
    result = charlist_rename (charlist0, tmp);
    g_free (tmp);
    charset = g_strsplit (charlist1, ",", G_MAXINT);
    for (i = 0; charset[i] != NULL; i++)
        if (g_strstr (result, charset[i]) == NULL)
          {
            tmp = g_strconcat (result, ",", charset[i], NULL);
            g_free (result);
            result = tmp;
          }
    g_strfreev (charset);
    return result;
}


/*  キャラクターセットを削除する
    charlist,キャラクターセットリスト
     charset,削除するキャラクターセット
         RET,削除されたキャラクターセットリスト                             */
gchar *
charlist_remove (const gchar *charlist, const gchar *charset)
{
    gchar **str_array, *tmp, *result = NULL;
    gint i;

    if (charlist == NULL || charset == NULL)
        return charlist != NULL ? g_strdup (charlist) : NULL;
    str_array = g_strsplit (charlist, ",", G_MAXINT);
    for (i = 0; str_array[i] != NULL; i++)
      {
        if (g_strcasecmp (str_array[i], charset) != 0)
          {
            tmp = result == NULL ? g_strdup (str_array[i])
                            : g_strconcat (result, ",", str_array[i], NULL);
            g_free (result);
            result = tmp;
          }
      }
    g_strfreev (str_array);
    return result;
}


/*  ja:キャラクターセットを置換する
    charlist,置換するキャラクターセット                                     */
void
charlist_renewal_all (const gchar *charlist)
{
  gchar *tmp;
  gint i;
  GList *glist;

  if (!charlist)
    return;
  if (charset_list != charlist)
    {
      tmp = charlist_append (charset_list, charlist);
      g_free (charset_list);
      charset_list = tmp;
    }
  glist = gtk_container_children (GTK_CONTAINER (notebook));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      MmaidWindow *mmaid;

      mmaid = g_object_get_data (G_OBJECT (g_list_nth_data (glist, i)),
                                                                "user_data");
      if (mmaid->charset != charlist)
        {
          tmp = charlist_rename (mmaid->charset, charlist);
          g_free (mmaid->charset);
          mmaid->charset = tmp;
        }
      mm_list_charset_renewal (MM_LIST (mmaid->list), charlist);
    }
  g_list_free (glist);
}


/******************************************************************************
*                                                                             *
* キャラクターセットリストダイアログ関数群                                    *
*                                                                             *
******************************************************************************/
static GtkListStore *store;
static GtkWidget *button0, *button1, *button2, *button3, *button4, *combo;


/* リストボックスの選択 */
static void
charlist_dialog_changed_tview (GtkTreeSelection *select, gpointer user_data)
{
    gint count, pos = -1;
    GtkTreeIter iter;

    for (count = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                &iter, NULL, count); count++)
        if (gtk_tree_selection_iter_is_selected (select, &iter))
            pos = count;
    if (pos >= 0)
      {
        gtk_widget_set_sensitive (button1, count > 1);
        gtk_widget_set_sensitive (button2, pos > 0);
        gtk_widget_set_sensitive (button3, pos < count - 1);
      }
    else
      {
        gtk_widget_set_sensitive (button1, FALSE);
        gtk_widget_set_sensitive (button2, FALSE);
        gtk_widget_set_sensitive (button3, FALSE);
      }
}


/* 新規 */
static void
charlist_dialog_clicked_new (GtkWidget *widget, GtkTreeSelection *select)
{
    const gchar *charset, *label;
    gint i;
    GList *glist;
    GtkTreeIter iter;

    /* 挿入 */
    charset = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (combo)->entry));
    for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
        if (gtk_tree_selection_iter_is_selected (select, &iter))
          {
            i++;
            break;
          }
    gtk_list_store_insert (store, &iter, i);
    gtk_list_store_set (store, &iter, 0, g_strdup (charset), -1);
    gtk_tree_selection_select_iter (select, &iter);
    /* コンボ */
    glist = gtk_container_children (GTK_CONTAINER (GTK_COMBO (combo)->list));
    for (i = g_list_length (glist) - 1; i >= 0; i--)
      {
        label = gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                                    (GTK_BIN (g_list_nth_data (glist, i)))));
        if (g_strcasecmp (label, charset) == 0)
            gtk_list_clear_items (GTK_LIST (GTK_COMBO (combo)->list),
                                                                    i, i + 1);
      }
    g_list_free (glist);
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo)->entry), "");
    /* OKボタン */
    gtk_widget_set_sensitive (button4, TRUE);
}


/* 削除 */
static void
charlist_dialog_clicked_remove (GtkWidget *widget, GtkTreeSelection *select)
{
    gchar *charset;
    const gchar *label;
    gint i;
    GList *glist;
    GtkTreeIter iter;
    GtkWidget *item;

    /* 削除 */
    for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
        if (gtk_tree_selection_iter_is_selected (select, &iter))
          {
            gtk_tree_model_get (GTK_TREE_MODEL (store),
                                                    &iter, 0, &charset, -1);
            gtk_list_store_remove (store, &iter);
            /* コンボ */
            glist = gtk_container_children
                                    (GTK_CONTAINER (GTK_COMBO (combo)->list));
            for (i = g_list_length (glist) - 1; i >= 0; i--)
              {
                label = gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                                    (GTK_BIN (g_list_nth_data (glist, i)))));
                if (g_strcasecmp (label, charset) <= 0)
                    break;
              }
            g_list_free (glist);
            item = gtk_list_item_new_with_label (charset);
            gtk_widget_show (item);
            gtk_list_insert_items (GTK_LIST (GTK_COMBO (combo)->list),
                                            g_list_append (NULL, item), i + 1);
            gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo)->entry), charset);
            g_free (charset);
          }
    /* OKボタン */
    gtk_widget_set_sensitive (button4, TRUE);
}


/* 上へ */
static void
charlist_dialog_clicked_up (GtkWidget *widget, GtkTreeSelection *select)
{
    gchar *charset0, *charset1;
    gint i;
    GtkTreeIter iter0, iter1;

    for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter0, NULL, i); i++)
        if (gtk_tree_selection_iter_is_selected (select, &iter0))
          {
            gtk_tree_model_get (GTK_TREE_MODEL (store),
                                                    &iter0, 0, &charset0, -1);
            gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter1, NULL, i - 1);
            gtk_tree_model_get (GTK_TREE_MODEL (store),
                                                    &iter1, 0, &charset1, -1);
            gtk_list_store_set (store, &iter0, 0, charset1, -1);
            gtk_list_store_set (store, &iter1, 0, charset0, -1);
            gtk_tree_selection_select_iter (select, &iter1);
            break;
          }
    /* OKボタン */
    gtk_widget_set_sensitive (button4, TRUE);
}


/* 下へ */
static void
charlist_dialog_clicked_down (GtkWidget *widget, GtkTreeSelection *select)
{
    gchar *charset0, *charset1;
    gint i;
    GtkTreeIter iter0, iter1;

    for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter0, NULL, i); i++)
        if (gtk_tree_selection_iter_is_selected (select, &iter0))
          {
            gtk_tree_model_get (GTK_TREE_MODEL (store),
                                                    &iter0, 0, &charset0, -1);
            gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter1, NULL, i + 1);
            gtk_tree_model_get (GTK_TREE_MODEL (store),
                                                    &iter1, 0, &charset1, -1);
            gtk_list_store_set (store, &iter0, 0, charset1, -1);
            gtk_list_store_set (store, &iter1, 0, charset0, -1);
            gtk_tree_selection_select_iter (select, &iter1);
            break;
          }
    /* OKボタン */
    gtk_widget_set_sensitive (button4, TRUE);
}


/* OKボタンが押された */
static void
charlist_dialog_clicked_ok (GtkWidget *widget, GtkWidget *dialog)
{
    gchar *charset, **result, *tmp;
    gint i;
    GtkTreeIter iter;

    result = g_object_get_data (G_OBJECT (dialog), "user_data");
    for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
      {
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 0, &charset, -1);
        tmp = *result == NULL ? g_strdup (charset)
                                : g_strconcat (*result, ",", charset, NULL);
        g_free (*result);
        *result = tmp;
      }
    gtk_widget_destroy (dialog);
}


/* リストボックスの破棄 */
static void
charlist_dialog_destroy (GtkObject *object, gpointer user_data)
{
    gchar *charset;
    GtkTreeIter iter;

    while (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                            &iter, NULL, 0))
      {
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 0, &charset, -1);
        gtk_list_store_remove (store, &iter);
        g_free (charset);
      }
}


/* エントリーでリターンが押された */
static void
charlist_dialog_activate (GtkWidget *widget, GtkTreeSelection *select)
{
    if (GTK_WIDGET_IS_SENSITIVE (button0))
        charlist_dialog_clicked_new (widget, select);
}


/* エントリーが変更された */
static void
charlist_dialog_changed_combo (GtkWidget *widget, gpointer user_data)
{
    gchar *charset;
    const gchar *text;
    gint i;
    GtkTreeIter iter;

    text = gtk_entry_get_text (GTK_ENTRY (widget));
    if (g_strlen (text) <= 0 || g_strchr (text, ',') != NULL)
      {
        gtk_widget_set_sensitive (button0, FALSE);
        return;
      }
    for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
      {
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 0, &charset, -1);
        if (g_strcasecmp (charset, text) == 0)
          {
            gtk_widget_set_sensitive (button0, FALSE);
            return;
          }
      }
    gtk_widget_set_sensitive (button0, charlist_is_valid (text));
}


/* ESCが押された */
static gboolean
charlist_dialog_key_press (GtkWidget *widget, GdkEventKey *event,
                                                            gpointer user_data)
{
    if (event->keyval == GDK_Escape)
        gtk_widget_destroy (widget);
    return FALSE;
}


gchar *
charlist_dialog (gchar *charlist)
{
    gchar **charset, *result = NULL;
    gint i;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    GtkTreeSelection *select;
    GtkTreeViewColumn *column;
    GtkWidget *dialog, *button5, *tview, *scroll, *hbox, *vbox;

    /* メインウインドウ */
    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (dialog), _("Character Encoding"));
    gtk_window_set_default_size (GTK_WINDOW (dialog), 320, 240);
    g_signal_connect_after (G_OBJECT(dialog), "key-press-event",
                                G_CALLBACK (charlist_dialog_key_press), NULL);
    g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
    g_object_set_data (G_OBJECT (dialog), "user_data", &result);
    /* リストボックス */
    store = gtk_list_store_new (1, G_TYPE_STRING);
    tview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tview), FALSE);
    g_signal_connect (G_OBJECT (tview), "destroy",
                                G_CALLBACK (charlist_dialog_destroy), NULL);
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes
            (_("Registered Character Encoding"), renderer, "text", 0, NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tview), column);
    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tview));
    g_signal_connect (G_OBJECT (select), "changed",
                            G_CALLBACK (charlist_dialog_changed_tview), NULL);
    charset = g_strsplit (charlist, ",", G_MAXINT);
    for (i = 0; charset[i] != NULL; i++)
      {
        gtk_list_store_append (store, &iter);
        gtk_list_store_set (store, &iter, 0, g_strdup (charset[i]), -1);
      }
    g_strfreev (charset);
    /* スクロールウインドウ */
    scroll=gtk_scrolled_window_new(NULL,NULL);
    gtk_container_add (GTK_CONTAINER (scroll), tview);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                        GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
    /* コンボボックス */
    combo = gtk_combo_new();
    gtk_combo_set_value_in_list (GTK_COMBO (combo), FALSE, FALSE);
    gtk_combo_set_case_sensitive (GTK_COMBO (combo), TRUE);
    g_signal_connect (G_OBJECT (GTK_COMBO (combo)->entry), "activate",
                                G_CALLBACK (charlist_dialog_activate), select);
    g_signal_connect (G_OBJECT (GTK_COMBO (combo)->entry), "changed",
                            G_CALLBACK (charlist_dialog_changed_combo), NULL);
    /* ボタン */
    button0 = gtk_button_new_with_mnemonic (_("_New"));
    button1 = gtk_button_new_with_mnemonic (_("De_lete"));
    button2 = gtk_button_new_with_mnemonic (_("_Up"));
    button3 = gtk_button_new_with_mnemonic (_("_Down"));
    button4 = gtk_button_new_with_label (_("OK"));
    button5 = gtk_button_new_with_label (_("Cancel"));
    g_signal_connect (G_OBJECT (button0), "clicked",
                        G_CALLBACK (charlist_dialog_clicked_new), select);
    g_signal_connect (G_OBJECT (button1), "clicked",
                        G_CALLBACK (charlist_dialog_clicked_remove), select);
    g_signal_connect (G_OBJECT (button2), "clicked",
                        G_CALLBACK (charlist_dialog_clicked_up), select);
    g_signal_connect (G_OBJECT (button3), "clicked",
                        G_CALLBACK (charlist_dialog_clicked_down), select);
    g_signal_connect (G_OBJECT (button4), "clicked",
                        G_CALLBACK (charlist_dialog_clicked_ok), dialog);
    g_signal_connect_swapped (G_OBJECT (button5), "clicked",
                        G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
    /* ボックス */
    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
    gtk_box_pack_start (GTK_BOX (vbox), scroll, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), combo, FALSE, FALSE, 0);
    hbox = gtk_hbox_new (FALSE, SPACING);
    gtk_box_pack_start (GTK_BOX (hbox), button0, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), button1, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), button2, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), button3, FALSE, FALSE, 0);
    gtk_box_pack_end (GTK_BOX (hbox), button5, FALSE, FALSE, 0);
    gtk_box_pack_end (GTK_BOX (hbox), button4, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (dialog), vbox);

    /* 表示 */
    charlist_dialog_changed_combo (GTK_COMBO (combo)->entry, NULL);
    gtk_widget_set_sensitive (button0, FALSE);
    gtk_widget_set_sensitive (button1, FALSE);
    gtk_widget_set_sensitive (button2, FALSE);
    gtk_widget_set_sensitive (button3, FALSE);
    gtk_widget_set_sensitive (button4, FALSE);
    gtk_widget_grab_focus (button5);

    gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);
    gtk_grab_add (dialog);
    gtk_widget_show_all (dialog);
    gtk_main ();

    return result;
}
