/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>

#include "mmapconfig.h"
#include "glarea.h"
#include "ww_mmap.h"

/**********************************/

static WwMmap *wwmmap = NULL;

void
color_set_mmap (WwMmap * mmap)
{
	wwmmap = mmap;
	g_object_ref (mmap);
}

/**********************************/

/* 数値地図の標高は１メートル単位なので０〜４０００ｍのデータをあらかじめ用意しておく */

typedef struct {
	guint8 r;
	guint8 g;
	guint8 b;
} MColor;

#define ARRAY_MAX 4000
static MColor alt_array[ARRAY_MAX];

/**********************************/

/* 上の alt_array を作るための標高と色のセット */
typedef struct {
	gint alt;					/* 標高 m */
	guint8 r;
	guint8 g;
	guint8 b;
} AltColor;
static AltColor altcolor[5];

/**********************************/

/* ダイアログで持ち回るためのデータ入れ */
typedef struct {
	AltColor altcolor_backup[5];

	GtkWidget *spin_alt0;
	GtkWidget *spin_alt4;

	GtkWidget *spin_ratio1;
	GtkWidget *spin_ratio2;
	GtkWidget *spin_ratio3;

	GtkWidget *label_alt1;
	GtkWidget *label_alt2;
	GtkWidget *label_alt3;

	GtkWidget *cbutton[5];

	GtkWidget *sample;
} DialogData;

#define SAMPLE_W 400
#define SAMPLE_H 50

/*********************************/

void
color_get_rgb_from_alt (gint alt, guint8 * r, guint8 * g, guint8 * b)
{
	*r = alt_array[alt].r;
	*g = alt_array[alt].g;
	*b = alt_array[alt].b;
}

/* rgba の各値から 32 ビットの色データを作る */
static guint32
color_to_rgba (guint8 r, guint8 g, guint8 b, gint a)
{
	guint32 pixel;

	pixel = r * 0x01000000 + g * 0x00010000 + b * 0x00000100 + a;
	/*
	   g_print ("color_to_rgba:%d %d %d\n",r,g,b);
	 */
	return pixel;
}

/* 高度を受け取って、それに対応する色のデータを返す */
static void
get_rgb_from_alt (gint alt, guint8 * r, guint8 * g, guint8 * b)
{
	AltColor *n1, *n2;			/* alt が入っている両端の altcolor */
	gdouble d_alt;
	gdouble x_alt;
	gdouble r_tmp, g_tmp, b_tmp;

	if (alt < altcolor[0].alt) {
		*r = altcolor[0].r;
		*g = altcolor[0].g;
		*b = altcolor[0].b;
		return;
	} else if (alt < altcolor[1].alt) {
		n1 = &(altcolor[0]);
		n2 = &(altcolor[1]);
	} else if (alt < altcolor[2].alt) {
		n1 = &(altcolor[1]);
		n2 = &(altcolor[2]);
	} else if (alt < altcolor[3].alt) {
		n1 = &(altcolor[2]);
		n2 = &(altcolor[3]);
	} else if (alt < altcolor[4].alt) {
		n1 = &(altcolor[3]);
		n2 = &(altcolor[4]);
	} else {
		*r = altcolor[4].r;
		*g = altcolor[4].g;
		*b = altcolor[4].b;
		return;
	}

	d_alt = n2->alt - n1->alt;
	x_alt = alt - n1->alt;

	r_tmp = ((gdouble) (n2->r - n1->r) * x_alt) / d_alt + n1->r;
	g_tmp = ((gdouble) (n2->g - n1->g) * x_alt) / d_alt + n1->g;
	b_tmp = ((gdouble) (n2->b - n1->b) * x_alt) / d_alt + n1->b;

	*r = (guint8) r_tmp;
	*g = (guint8) g_tmp;
	*b = (guint8) b_tmp;

	/*
	   g_print("get_rgb_from_alt:%d: %hd %hd %hd\n", alt, *r, *g, *b);
	 */
}

static void
calc_alt_array (void)
{
	gint i;

	for (i = 0; i < ARRAY_MAX; ++i) {
		MColor *c = &(alt_array[i]);

		get_rgb_from_alt (i, &(c->r), &(c->g), &(c->b));
	}
}

static void
change_color_sample (DialogData * dd)
{
	GdkPixbuf *pixbuf1;			/* １ラインだけの pixbuf */
	GdkPixbuf *pixbuf;
	gint i;

	DB (g_print ("change_color_sample\n"));

	calc_alt_array ();

	pixbuf = gtk_image_get_pixbuf (GTK_IMAGE (dd->sample));
	pixbuf1 = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 1, SAMPLE_H);

	for (i = 0; i < SAMPLE_W; ++i) {
		guint32 pixel;
		guint8 r, g, b;
		gint alt;

		alt = ARRAY_MAX * i / SAMPLE_W;
		get_rgb_from_alt (alt, &r, &g, &b);
		pixel = color_to_rgba (r, g, b, 0);
		gdk_pixbuf_fill (pixbuf1, pixel);
		gdk_pixbuf_copy_area (pixbuf1, 0, 0, 1, SAMPLE_H, pixbuf, i, 0);
	}

	g_object_unref (pixbuf1);
	gtk_image_set_from_pixbuf (GTK_IMAGE (dd->sample), pixbuf);
}

static void
set_altcolor_from_current_dialogdata (DialogData * dd)
{
	gint i;
	gdouble ratio1, ratio2, ratio3;
	gdouble alt0, alt4;

	ratio1 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio1));
	ratio2 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio2));
	ratio3 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio3));
	alt0 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_alt0));
	alt4 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_alt4));

	altcolor[0].alt = alt0;
	altcolor[1].alt = (alt4 - alt0) * ratio1 + alt0;
	altcolor[2].alt = (alt4 - alt0) * ratio2 + alt0;
	altcolor[3].alt = (alt4 - alt0) * ratio3 + alt0;
	altcolor[4].alt = alt4;

	for (i = 0; i < 5; ++i) {
		GdkColor color;

		gtk_color_button_get_color (GTK_COLOR_BUTTON (dd->cbutton[i]), &color);
		altcolor[i].r = color.red / 0xff;
		altcolor[i].g = color.green / 0xff;
		altcolor[i].b = color.blue / 0xff;
	}
}

static void
trybutton_clicked_cb (GtkButton * button, gpointer user_data)
{
	DialogData *dd = user_data;

	DB (g_print ("trybutton_clicked_cb\n"));

	set_altcolor_from_current_dialogdata (dd);

	ww_mmap_color_change_all (wwmmap);

	glarea_force_render ();
}

static GtkWidget *
set_frame (GtkWidget * widget)
{
	GtkWidget *frame;

	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (frame), widget);

	return frame;
}

static GtkWidget *
widget_color (DialogData * dd)
{
	GtkWidget *table;
	GtkWidget *label;

	table = gtk_table_new (4, 6, FALSE);

	label = gtk_label_new ("標高(m)");
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (label), 0, 1, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->spin_alt0, 1, 2, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (dd->label_alt1), 2, 3, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (dd->label_alt2), 3, 4, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (dd->label_alt3), 4, 5, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->spin_alt4, 5, 6, 0, 1);

	label = gtk_label_new ("比");
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (label), 0, 1, 1, 2);
	label = gtk_label_new ("0.00");
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (label), 1, 2, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->spin_ratio1, 2, 3, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->spin_ratio2, 3, 4, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->spin_ratio3, 4, 5, 1, 2);
	label = gtk_label_new ("1.00");
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (label), 5, 6, 1, 2);

	label = gtk_label_new ("色");
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (label), 0, 1, 2, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->cbutton[0], 1, 2, 2, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->cbutton[1], 2, 3, 2, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->cbutton[2], 3, 4, 2, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->cbutton[3], 4, 5, 2, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->cbutton[4], 5, 6, 2, 3);

	label = gtk_label_new ("サンプル\n0〜4000m");
	gtk_table_attach_defaults (GTK_TABLE (table), set_frame (label), 0, 1, 3, 4);
	gtk_table_attach_defaults (GTK_TABLE (table), dd->sample, 1, 6, 3, 4);

	return table;
}

static void
color_set_cb (GtkColorButton * widget, gpointer user_data)
{
	DialogData *dd = user_data;
	GdkColor color;
	guint32 pixel;

	gtk_color_button_get_color (widget, &color);
	pixel = color_to_rgba (color.red / 0xff, color.green / 0xff, color.blue / 0xff, 0);

	set_altcolor_from_current_dialogdata (dd);
	change_color_sample (dd);
}

static void
alt_value_changed_cb (GtkSpinButton * spinbutton, gpointer user_data)
{
	DialogData *dd = user_data;
	GtkWidget *spin = GTK_WIDGET (spinbutton);
	gdouble ratio1, ratio2, ratio3;
	gdouble alt0, alt4;
	gdouble alt1, alt2, alt3;
	gchar text[20];

	alt0 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_alt0));
	alt4 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_alt4));

	if (spin == dd->spin_alt0) {
		if (alt0 > alt4) {
			alt0 = alt4;
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_alt0), alt4);
		}
	} else if (spin == dd->spin_alt4) {
		if (alt0 > alt4) {
			alt4 = alt0;
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_alt4), alt0);
		}
	} else {
		g_print ("??? in alt_value_changed_cb\n");

		return;
	}

	ratio1 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio1));
	ratio2 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio2));
	ratio3 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio3));

	alt1 = (alt4 - alt0) * ratio1 + alt0;
	alt2 = (alt4 - alt0) * ratio2 + alt0;
	alt3 = (alt4 - alt0) * ratio3 + alt0;

	sprintf (text, "%.0f", alt1);
	gtk_label_set_text (GTK_LABEL (dd->label_alt1), text);
	sprintf (text, "%.0f", alt2);
	gtk_label_set_text (GTK_LABEL (dd->label_alt2), text);
	sprintf (text, "%.0f", alt3);
	gtk_label_set_text (GTK_LABEL (dd->label_alt3), text);

	set_altcolor_from_current_dialogdata (dd);
	change_color_sample (dd);
}

static void
ratio_value_changed_cb (GtkSpinButton * spinbutton, gpointer user_data)
{
	DialogData *dd = user_data;
	GtkWidget *spin = GTK_WIDGET (spinbutton);
	gchar text[20];
	gdouble ratio;
	gdouble alt;
	gdouble alt0, alt4;

	alt0 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_alt0));
	alt4 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_alt4));

	ratio = gtk_spin_button_get_value (spinbutton);
	alt = (alt4 - alt0) * ratio + alt0;
	sprintf (text, "%.0f", alt);

	if (spin == dd->spin_ratio1) {
		gdouble ratio2;

		ratio2 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio2));
		if (ratio2 < ratio) {
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_ratio2), ratio);
		}
		gtk_label_set_text (GTK_LABEL (dd->label_alt1), text);
	} else if (spin == dd->spin_ratio2) {
		gdouble ratio1, ratio3;

		ratio1 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio1));
		ratio3 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio3));
		if (ratio1 > ratio) {
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_ratio1), ratio);
		}
		if (ratio3 < ratio) {
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_ratio3), ratio);
		}

		gtk_label_set_text (GTK_LABEL (dd->label_alt2), text);
	} else if (spin == dd->spin_ratio3) {
		gdouble ratio2;

		ratio2 = gtk_spin_button_get_value (GTK_SPIN_BUTTON (dd->spin_ratio2));
		if (ratio2 > ratio) {
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_ratio2), ratio);
		}
		gtk_label_set_text (GTK_LABEL (dd->label_alt3), text);
	}

	set_altcolor_from_current_dialogdata (dd);
	change_color_sample (dd);
}

/* n:0-4 */
static gdouble
get_ratio_from_backup (DialogData * dd, gint n)
{
	gdouble ratio;
	gdouble delta;

	delta = dd->altcolor_backup[4].alt - dd->altcolor_backup[0].alt;
	ratio = (dd->altcolor_backup[n].alt - dd->altcolor_backup[0].alt) / delta;

	return ratio;
}

static void
dialog_data_init (DialogData * dd)
{
	gint i;
	GdkPixbuf *pixbuf;

	for (i = 0; i < 5; ++i) {
		dd->altcolor_backup[i] = altcolor[i];
	}

	dd->spin_alt0 = gtk_spin_button_new_with_range (0.0, 4000.0, 10.0);
	dd->spin_alt4 = gtk_spin_button_new_with_range (0.0, 4000.0, 10.0);
	gtk_entry_set_alignment (GTK_ENTRY (dd->spin_alt0), 0.5);
	gtk_entry_set_alignment (GTK_ENTRY (dd->spin_alt4), 0.5);
	g_signal_connect (dd->spin_alt0, "value_changed", G_CALLBACK (alt_value_changed_cb), dd);
	g_signal_connect (dd->spin_alt4, "value_changed", G_CALLBACK (alt_value_changed_cb), dd);

	dd->spin_ratio1 = gtk_spin_button_new_with_range (0.0, 1.0, 0.01);
	dd->spin_ratio2 = gtk_spin_button_new_with_range (0.0, 1.0, 0.01);
	dd->spin_ratio3 = gtk_spin_button_new_with_range (0.0, 1.0, 0.01);
	gtk_entry_set_alignment (GTK_ENTRY (dd->spin_ratio1), 0.5);
	gtk_entry_set_alignment (GTK_ENTRY (dd->spin_ratio2), 0.5);
	gtk_entry_set_alignment (GTK_ENTRY (dd->spin_ratio3), 0.5);
	g_signal_connect (dd->spin_ratio1, "value_changed", G_CALLBACK (ratio_value_changed_cb), dd);
	g_signal_connect (dd->spin_ratio2, "value_changed", G_CALLBACK (ratio_value_changed_cb), dd);
	g_signal_connect (dd->spin_ratio3, "value_changed", G_CALLBACK (ratio_value_changed_cb), dd);

	dd->label_alt1 = gtk_label_new ("");
	dd->label_alt2 = gtk_label_new ("");
	dd->label_alt3 = gtk_label_new ("");

	for (i = 0; i < 5; ++i) {
		dd->cbutton[i] = gtk_color_button_new ();
		g_signal_connect (dd->cbutton[i], "color_set", G_CALLBACK (color_set_cb), dd);
	}

	pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, SAMPLE_W, SAMPLE_H);
	dd->sample = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (pixbuf);
}

static void
dialog_realize_cb (GtkWidget * widget, gpointer user_data)
{
	DialogData *dd = user_data;
	gint i;

	DB (g_print ("dialog_realize_cb\n"));

	/* バックアップの値を使ってスピンボタンや色を初期化 */

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_alt0), dd->altcolor_backup[0].alt);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_alt4), dd->altcolor_backup[4].alt);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_ratio1), get_ratio_from_backup (dd, 1));
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_ratio2), get_ratio_from_backup (dd, 2));
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dd->spin_ratio3), get_ratio_from_backup (dd, 3));

	for (i = 0; i < 5; ++i) {
		GdkColor color;

		color.red = 0xff * dd->altcolor_backup[i].r;
		color.green = 0xff * dd->altcolor_backup[i].g;
		color.blue = 0xff * dd->altcolor_backup[i].b;

		gtk_color_button_set_color (GTK_COLOR_BUTTON (dd->cbutton[i]), &color);
	}

	set_altcolor_from_current_dialogdata (dd);
	change_color_sample (dd);
}

void
color_dialog_create (GtkWindow * parent)
{
	GtkWidget *dialog;
	GtkWidget *color;
	GtkWidget *trybutton;
	GtkWidget *vbox;
	gint response;
	DialogData *dd;

	dd = g_new (DialogData, 1);
	dialog_data_init (dd);

	color = widget_color (dd);

	trybutton = gtk_button_new_with_label ("試してみる");
	g_signal_connect (trybutton, "clicked", G_CALLBACK (trybutton_clicked_cb), dd);

	vbox = gtk_vbox_new (FALSE, 10);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
	gtk_box_pack_start (GTK_BOX (vbox), color, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), trybutton, FALSE, FALSE, 0);

	dialog = gtk_dialog_new_with_buttons ("色の設定", parent, GTK_DIALOG_DESTROY_WITH_PARENT,
										  GTK_STOCK_OK, GTK_RESPONSE_OK, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);
	g_signal_connect (dialog, "realize", G_CALLBACK (dialog_realize_cb), dd);


	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);
	gtk_widget_show_all (dialog);

	response = gtk_dialog_run (GTK_DIALOG (dialog));

	if (response == GTK_RESPONSE_OK) {
		set_altcolor_from_current_dialogdata (dd);

	} else {
		gint i;

		/* バックアップで復元 */
		for (i = 0; i < 5; ++i) {
			altcolor[i] = dd->altcolor_backup[i];
		}
	}
	calc_alt_array ();

	gtk_widget_destroy (dialog);
	g_free (dd);

	{
		gint i;
		gchar *alt[] = { "alt1", "alt2", "alt3", "alt4", "alt5" };
		gchar *pixel[] = { "pixel1", "pixel2", "pixel3", "pixel4", "pixel5" };

		for (i = 0; i < 5; ++i) {
			gchar buf[] = "00000000";
			guint32 color;

			color = color_to_rgba (altcolor[i].r, altcolor[i].g, altcolor[i].b, 0);
			sprintf (buf, "%x", color);

			mmap_config_set_int (alt[i], altcolor[i].alt);
			mmap_config_set_string (pixel[i], buf);
		}
	}

	ww_mmap_color_change_all (wwmmap);

	glarea_force_render ();
}

void
color_init (void)
{
	gchar *alt[] = { "alt1", "alt2", "alt3", "alt4", "alt5" };
	gchar *pixel[] = { "pixel1", "pixel2", "pixel3", "pixel4", "pixel5" };
	gint default_alt[] = { 0, 300, 600, 900, 1200 };
	gchar *default_color[] = { "ffffff00", "dffed300", "edffad00", "ffefad00", "fee6a300" };
	gint i;

	for (i = 0; i < 5; ++i) {
		gint r, g, b;
		gchar *tmp;

		tmp = mmap_config_get_string (pixel[i], NULL);

		if (tmp == NULL) {
			altcolor[i].alt = default_alt[i];
			sscanf (default_color[i], "%2x%2x%2x", &r, &g, &b);
		} else {
			altcolor[i].alt = mmap_config_get_int (alt[i], NULL);
			sscanf (tmp, "%2x%2x%2x", &r, &g, &b);
		}
		altcolor[i].r = r;
		altcolor[i].g = g;
		altcolor[i].b = b;

		g_free (tmp);

	}

	calc_alt_array ();
}
