/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtkgl.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glext.h>

#include "mmap.h"
#include "disk.h"
#include "mark.h"
#include "mark_dialog.h"
#include "camera.h"
#include "util.h"
#include "window.h"
#include "glarea.h"



static GSList *sl_mark = NULL;	/* 中身は、Mark */

/* MarkPoint 中の lon,lat,alt からそれ以外の値を計算しなおす。*/
static void
mark_calc_point (MarkPoint * mp)
{
	gdouble geta = 30.0; 
    /*
	g_print ("mark_calc_point:%.1f %.1f %.1f\n", mp->lon, mp->lat, mp->alt);
    */

	/* 標高に +30.0 しているのは、マークの球を地上から浮かすため */
	mmap_deg_to_xyz (mp->lon / 3600.0, mp->lat / 3600.0, mp->alt + geta, &(mp->x), &(mp->y), &(mp->z));
	mmap_deg_to_xyz (mp->lon / 3600.0, mp->lat / 3600.0, geta, &(mp->x0), &(mp->y0), &(mp->z0));

	DB (g_print ("mark_calc_point:%.1f %.1f %.1f\n", mp->lon, mp->lat, mp->alt));
	DB (g_print ("mark_calc_point:%.0f %.0f %.0f\n", mp->x, mp->y, mp->z));
	DB (g_print ("mark_calc_point:%s\n", mp->text));
}

/* no_mark 番目の Mark の no_point 番目の MarkPoint を削除する */
/* no_mark >=0  no_point >=0 */
void
mark_del_point (guint no_mark, guint no_point)
{
	Mark *mark;
	MarkPoint *point;

	DB (g_print ("mark_del_point:no_mark %d:no_point %d", no_mark, no_point));

	mark = g_slist_nth_data (sl_mark, no_mark);
	point = g_slist_nth_data (mark->sl_point, no_point);

	mark->sl_point = g_slist_remove (mark->sl_point, point);
	g_free (point->text);
	g_free (point);
}

/* no(>=0) 番目の Mark を削除する */
void
mark_del_mark (guint no_mark)
{
	GSList *sl;
	Mark *mark;

	DB (g_print ("mark_del_mark:no_mark %d", no_mark));

	mark = g_slist_nth_data (sl_mark, no_mark);
	sl_mark = g_slist_remove (sl_mark, mark);

	for (sl = mark->sl_point; sl != NULL; sl = sl->next) {
		MarkPoint *point = sl->data;

		g_free (point->text);
		g_free (point);
	}
	g_slist_free (mark->sl_point);
	mark->sl_point = NULL;

	g_free (mark->text);
	g_free (mark);
}

/* Mark 内の MarkPoint の数を返す。 */
guint
mark_get_point_n (guint no_mark)
{
	Mark *mark;
	guint n;

	mark = g_slist_nth_data (sl_mark, no_mark);
	n = g_slist_length (mark->sl_point);

	DB (g_print ("mark_get_point_n:%d\n", n));

	return n;
}

/* Mark の数を返す。 */
guint
mark_get_mark_n (void)
{
	guint n;
	n = g_slist_length (sl_mark);

	DB (g_print ("mark_get_mark_n:%d\n", n));

	return n;
}

/* slist_mark の最後に Mark を追加する。 */
/* 追加した n(>=0) を返す */
guint
mark_add_mark (void)
{
	Mark *mk;
	gchar no_name[] = "?????";
	guint n;

	mk = g_new (Mark, 1);
	mk->text = g_strdup (no_name);
	mk->sl_point = NULL;

	sl_mark = g_slist_append (sl_mark, mk);
	n = g_slist_length (sl_mark);

	return n - 1;
}

/* Mark の最後に MarkPoint を追加する。*/
/* 追加した n(>=0) を返す */
guint
mark_add_point (guint no_mark)
{
	Mark *mark;
	MarkPoint *point;
	gchar no_name[] = "";
	guint n;

	point = g_new (MarkPoint, 1);
	point->lon = 0.0;
	point->lat = 0.0;
	point->alt = 0.0;
	point->text = g_strdup (no_name);

	mark_calc_point (point);

	mark = g_slist_nth_data (sl_mark, no_mark);
	mark->sl_point = g_slist_append (mark->sl_point, point);

	n = g_slist_length (mark->sl_point);

	return n - 1;
}

void
mark_render (void)
{
	GSList *sl;
    gdouble h;

	DB (g_print ("mark_render\n"));

    h = camera_get_dist();

	for (sl = sl_mark; sl != NULL; sl = sl->next) {
		Mark *mark = sl->data;
		GSList *sl_p;

		/*
		g_print("mark:%s:%d:%p:\n",mark->text,mark->no,mark->slist_point);
		*/

		for (sl_p = mark->sl_point; sl_p != NULL; sl_p = sl_p->next) {
			MarkPoint *mp = sl_p->data;

			if (mp == NULL) {
				continue;
			}

            /*
			   g_print("point xyz      :%.0f %.0f %.0f\n",mp->x,mp->y,mp->z);
			   g_print("point x0y0z0   :%.0f %.0f %.0f\n",mp->x0,mp->y0,mp->z0);
			   g_print("point lonlatalt:%.0f %.0f %.0f\n",mp->lon,mp->lat,mp->alt);
			   g_print("point text     :%s:\n\n",mp->text);
            */

			glPushMatrix ();
			if (mmap_get_view_mode () == MMAP_VIEW_MODE_2D) {
				glTranslated (mp->x0, mp->y0, mp->z0);
				glColor3d (1.0, 0.0, 0.0);

				glDisable (GL_DEPTH_TEST);
				gdk_gl_draw_sphere (TRUE, h / 400.0, 12, 6);
				glEnable (GL_DEPTH_TEST);
			} else {
				glTranslated (mp->x, mp->y, mp->z);
				glColor3d (1.0, 0.0, 0.0);

				gdk_gl_draw_sphere (TRUE, h / 400.0, 12, 6);
			}
			glPopMatrix ();
		}
	}
}

MarkPoint *
mark_get_point (guint no_mark, guint no_point)
{
	Mark *mark;

	DB (g_print ("mark_get_point:no_mark:%d no_point:%d\n", no_mark, no_point));

	mark = g_slist_nth_data (sl_mark, no_mark);

	return g_slist_nth_data (mark->sl_point, no_point);
}

Mark *
mark_get_mark (guint no_mark)
{
	DB (g_print ("mark_get_mark:%d\n", no_mark));

	return g_slist_nth_data (sl_mark, no_mark);
}

void
mark_swap_point (guint no_mark, guint n1, guint n2)
{
	Mark *mark;
	GSList *sl1;
	GSList *sl2;
	MarkPoint *tmp;

	DB (g_print ("mark_swap_point:%d %d\n", n1, n2));

	mark = g_slist_nth_data (sl_mark, no_mark);

	sl1 = g_slist_nth (mark->sl_point, n1);
	sl2 = g_slist_nth (mark->sl_point, n2);

	tmp = sl1->data;
	sl1->data = sl2->data;
	sl2->data = tmp;
}

void
mark_swap_mark (guint n1, guint n2)
{
	GSList *sl1;
	GSList *sl2;
	Mark *tmp;

	DB (g_print ("mark_swap_mark:%d %d\n", n1, n2));

	sl1 = g_slist_nth (sl_mark, n1);
	sl2 = g_slist_nth (sl_mark, n2);

	tmp = sl1->data;
	sl1->data = sl2->data;
	sl2->data = tmp;
}

void
mark_set_mark (guint no_mark, const gchar * name)
{
	Mark *mark;

	mark = g_slist_nth_data (sl_mark, no_mark);
	g_free (mark->text);
	mark->text = g_strdup (name);
}

void
mark_set_point (guint no_mark, guint no_point, gdouble lon, gdouble lat, gdouble alt, const gchar * text)
{
	Mark *mark;
	MarkPoint *point;

	mark = g_slist_nth_data (sl_mark, no_mark);

	point = g_slist_nth_data (mark->sl_point, no_point);
	point->lon = lon;
	point->lat = lat;
	point->alt = alt;
	g_free (point->text);
	point->text = g_strdup (text);

	mark_calc_point (point);
}

/* Mark の 中の slist_point を埋める */
static GSList *
analyze_token (Mark * mark, gchar * point[])
{
	guint i;
	GSList *sl = NULL;

	DB (g_print ("analyze_token\n"));

	for (i = 0; point[i] != NULL; ++i) {
		MarkPoint *mp;
		gint deg1, min1;
		gint deg2, min2;
		gdouble sec1, sec2, alt;
		gchar text[1024] = "";	/* この制限は問題？ */

		mp = g_new0 (MarkPoint, 1);

		sscanf (point[i], "%d:%d:%le:%d:%d:%le:%le:%s", &deg1, &min1, &sec1, &deg2, &min2, &sec2, &alt, text);
		DB (g_print ("%s\n", point[i]));
		DB (g_print ("%d:%d:%.2f:  :%d:%d:%.2f:   %.2f\n", deg1, min1, sec1, deg2, min2, sec2, alt));
		DB (g_print ("text:%s:\n", text));

		mp->lon = deg1 * 3600.0 + min1 * 60.0 + sec1;
		mp->lat = deg2 * 3600.0 + min2 * 60.0 + sec2;
		mp->alt = alt;
		if (text[0] == '\0') {
			mp->text = g_strdup ("");
		} else {
			mp->text = g_strdup (text);
		}

		mark_calc_point (mp);

		sl = g_slist_append (sl, mp);
	}
	return sl;
}

void
mark_read (void)
{
	gchar *line = NULL;
	GIOChannel *ch = NULL;
	GError *err = NULL;
	GIOStatus status;
	gchar *fullpath;
	guint i = 0;

	fullpath = g_strconcat (mmap_dir, "/mark.dat", NULL);
	if (g_file_test (fullpath, G_FILE_TEST_EXISTS) == FALSE) {
		g_free (fullpath);
		return;					/* 最初の起動時 */
	}
	ch = disk_channel_open (fullpath, READ_LOCAL);
	g_free (fullpath);

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		gchar **token = NULL;
		Mark *mark = NULL;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("mark_read:error:g_io_channel_read_line:%s:%s\n", line, err->message);
			exit (-1);
		}

		line = g_strchomp (line);
		if (line[0] == '#' || line[0] == '\0') {
			g_free (line);
			status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
			continue;
		}

		token = g_strsplit (line, "\t", 0);

		mark = g_new (Mark, 1);
		mark->text = g_strdup (token[0]);
		mark->sl_point = NULL;

		mark->sl_point = analyze_token (mark, &(token[1]));	/* ２つ目以降だけを渡す */

		sl_mark = g_slist_append (sl_mark, mark);

		g_strfreev (token);
		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
		++i;
	}

	g_io_channel_unref (ch);
}

void
mark_write (void)
{
	gchar *mark_dat;
	gchar *mark_dat_bak;
	GIOChannel *ch = NULL;
	GError *err = NULL;
	GSList *sl;
#ifdef DEBUG_MARK
	g_print ("mark_write\n");
#endif

	mark_dat = g_strconcat (mmap_dir, "/mark.dat", NULL);
	mark_dat_bak = g_strconcat (mmap_dir, "/mark.dat.bak", NULL);
	if (g_file_test (mark_dat, G_FILE_TEST_IS_REGULAR | G_FILE_TEST_EXISTS)) {
		rename (mark_dat, mark_dat_bak);
	}
	ch = disk_channel_open (mark_dat, WRITE_LOCAL);
	g_free (mark_dat_bak);
	g_free (mark_dat);

	g_io_channel_write_chars (ch, "#mark file for MMap\n", -1, NULL, &err);
	if (err != NULL) {
		g_print ("mark_write:error:g_io_channel_write:%s\n", err->message);
		exit (1);
	}

	for (sl = sl_mark; sl != NULL; sl = sl->next) {
		Mark *mk = sl->data;
		GSList *slp;

		g_io_channel_write_chars (ch, mk->text, -1, NULL, &err);

		for (slp = mk->sl_point; slp != NULL; slp = slp->next) {
			MarkPoint *mp = slp->data;
			gchar *buf;
			gint deg, min;
			gdouble sec;

			util_sec_to_deg_min_sec (mp->lon, &deg, &min, &sec);
			buf = g_strdup_printf ("\t%d:%d:%.1f", deg, min, sec);
			g_io_channel_write_chars (ch, buf, -1, NULL, &err);
			g_free (buf);

			util_sec_to_deg_min_sec (mp->lat, &deg, &min, &sec);
			buf = g_strdup_printf (":%d:%d:%.1f", deg, min, sec);
			g_io_channel_write_chars (ch, buf, -1, NULL, &err);
			g_free (buf);

			buf = g_strdup_printf (":%.1f:%s", mp->alt, mp->text);
			g_io_channel_write_chars (ch, buf, -1, NULL, &err);
			g_free (buf);
		}
		g_io_channel_write_chars (ch, "\n", -1, NULL, &err);
	}
	g_io_channel_unref (ch);
}

/*************************************************************/

/*********** マークの記録 ***************/

static guint mark_no = -1;		/* マークを追加している Mark の番号 */

static void
dialog_response_cb (GtkDialog * dialog, gint arg1, gpointer user_data)
{
	g_print ("dialog_response_cb\n");

	gtk_widget_destroy (GTK_WIDGET (dialog));

	window_menu_mark_sensitive (TRUE);
	glarea_tilt_lock (FALSE);

	mark_no = -1;

	window_set_mark_submenu ();
}

static void
mark_clicked_cb (GtkButton * button, gpointer user_data)
{
	guint point_no;
	gdouble lon, lat, alt;		/* 度 */

	point_no = mark_add_point (mark_no);

	lon = camera_get_lon () * 3600.0;
	lat = camera_get_lat () * 3600.0;
	/* マーク地点の標高データを取るようにしないとマークが埋もれてしまう。要修正。 */
	/*
	   alt = camera_get_lon();
	 */
	alt = 0.0;
	mark_set_point (mark_no, point_no, lon, lat, alt, "");
}

void
mark_start (GtkWindow * parent)
{
	GtkWidget *dialog;
	GtkWidget *button;

	button = gtk_button_new_with_label (_("Mark"));
	g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (mark_clicked_cb), NULL);

	dialog = gtk_dialog_new_with_buttons (_("Recording"), parent,
										  GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_STOP, GTK_RESPONSE_CLOSE, NULL);
	g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (dialog_response_cb), NULL);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), button);
	gtk_window_set_default_size (GTK_WINDOW (dialog), 100, 100);
	gtk_widget_show_all (dialog);

	window_menu_mark_sensitive (FALSE);
	glarea_tilt_lock (TRUE);

	mark_no = mark_add_mark ();
}

/**************************/

static void
mark_activate_cb(GtkAction *action, gpointer   user_data)
{
	Mark *mark = (Mark *)user_data;
	MarkPoint *mp;

	if (mark->sl_point == NULL) {
		return;
	}
	mp = mark->sl_point->data;

	camera_set_position (mp->lon / 3600.0, mp->lat / 3600.0);

	glarea_force_render ();
}

static gint uniq_n = 0;



void
mark_append_menu (GtkUIManager * ui_manager, GtkActionGroup *action_group, guint ui_id)
{
	GSList *sl;

    for (sl = sl_mark; sl != NULL; sl = sl->next) {
		Mark *mark = sl->data;
        GtkAction *action;
        gchar *name;

		if (mark->sl_point == NULL) {
			continue;
		}

        name = g_strdup_printf("uniq_id_%d", uniq_n);
        ++uniq_n;

		action = gtk_action_new(name, mark->text, NULL, NULL);
		g_signal_connect (G_OBJECT (action), "activate", G_CALLBACK (mark_activate_cb), mark);
        
        gtk_action_group_add_action (action_group, action);
        g_object_unref (action);

        gtk_ui_manager_add_ui (ui_manager, ui_id, "/menubar/mark", name, name, GTK_UI_MANAGER_MENUITEM, FALSE);

        g_free(name);
	}
}
