/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <stdlib.h>

#include "mmap.h"

typedef struct {
	gchar *name;
	gchar *value;
} MmapConfig;

static GSList *sl_config = NULL;	/* 中身は MmapConfig */

static void
mmap_config_print_all (void)
{
	GSList *sl;

	for (sl = sl_config; sl != NULL; sl = sl->next) {
		MmapConfig *wc = sl->data;

		g_print ("mmap_config_print_all:(%s, %s)\n", wc->name, wc->value);
	}
}

static const gchar *
mmap_config_get_value (const gchar * name, gboolean *exist)
{
	GSList *sl;

	for (sl = sl_config; sl != NULL; sl = sl->next) {
		MmapConfig *wc = sl->data;

		if (g_strcasecmp (wc->name, name) == 0) {
            if(exist != NULL){
                *exist = TRUE;
            }
			return wc->value;
		}
	}

    if(exist != NULL){
        *exist = FALSE;
    }

	return NULL;
}

static void
mmap_config_set_value (const gchar * name, const gchar * value)
{
	GSList *sl;
	MmapConfig *wc;

	/* 既にあるなら値を書き換える */
	for (sl = sl_config; sl != NULL; sl = sl->next) {
		MmapConfig *wc = sl->data;

		if (g_strcasecmp (wc->name, name) == 0) {
			g_free (wc->value);
			wc->value = g_strdup (value);

			return;
		}
	}

	/* ないなら作る */
	wc = g_new (MmapConfig, 1);
	wc->name = g_strdup (name);
	wc->value = g_strdup (value);

	sl_config = g_slist_append (sl_config, wc);
}

gboolean
mmap_config_get_bool (const gchar * name, gboolean *exist)
{
	const gchar *bool_char;

	bool_char = mmap_config_get_value (name, exist);
	if (bool_char == NULL) {
		return FALSE;
	} else {
		if (g_strcasecmp (bool_char, "true") == 0) {
			return TRUE;
		} else {
			return FALSE;
		}
	}
}

void
mmap_config_set_bool (const gchar * name, gboolean bool)
{
	gchar true_char[] = "true";
	gchar false_char[] = "false";

	if (bool == TRUE) {
		mmap_config_set_value (name, true_char);
	} else {
		mmap_config_set_value (name, false_char);
	}
}

gint
mmap_config_get_int (const gchar * name, gboolean *exist)
{
	const gchar *int_char;
	gint num;

	int_char = mmap_config_get_value (name, exist);
	if (int_char == NULL) {
		num = 0;
	} else {
		sscanf (int_char, "%d", &num);
	}

	return num;
}

void
mmap_config_set_int (const gchar * name, gint num)
{
	gchar *int_char;

	int_char = g_strdup_printf ("%d", num);

	mmap_config_set_value (name, int_char);

	g_free (int_char);
}

gdouble
mmap_config_get_double (const gchar * name, gboolean *exist)
{
	const gchar *double_char;
	gdouble num;

	double_char = mmap_config_get_value (name, exist);
	if (double_char == NULL) {
		num = 0.0;
	} else {
		sscanf (double_char, "%lf", &num);
	}

	return num;
}

void
mmap_config_set_double (const gchar * name, gdouble num)
{
	gchar *double_char;

	double_char = g_strdup_printf ("%lf", num);

	mmap_config_set_value (name, double_char);

	g_free (double_char);
}

gchar *
mmap_config_get_string (const gchar * name, gboolean *exist)
{
	const gchar *string_char;

	string_char = mmap_config_get_value (name, exist);
	if (string_char == NULL) {
		return NULL;
	} else {
		return g_strdup (string_char);
	}
}

void
mmap_config_set_string (const gchar * name, const gchar * string)
{
	mmap_config_set_value (name, string);
}

void
mmap_config_read (void)
{
	gchar *config_file;
	gchar *line = NULL;
	GIOChannel *ch = NULL;
	GError *err = NULL;
	GIOStatus status;
	const gchar *codeset;		/* free してはダメ */


	config_file = g_strconcat (mmap_dir, "mmap.config2", NULL);

	g_print ("mmap_config_read:%s\n", config_file);

	ch = g_io_channel_new_file (config_file, "r", &err);
	if (err != NULL) {
		g_print ("mmap_config_read:%s\n", err->message);
		return;
	}

	g_get_charset (&codeset);
	status = g_io_channel_set_encoding (ch, codeset, &err);
	if (err != NULL) {
		g_print ("error:g_io_channel_set_encoding:%s\n", err->message);
		exit (-1);
	}

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {

		line = g_strstrip (line);
		if (line[0] == '#' || line[0] == '\0') {
			;
		} else {
            gchar **token;

            token = g_strsplit (line, "::", 0);
			mmap_config_set_value (token[0], token[1]);

            /*
            g_print("#################### line:%s:\n", line);
            g_print("#################### name:%s: value:%s:\n", token[0], token[1]);
            */

            g_strfreev(token);
		}

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_free (config_file);

	g_io_channel_unref (ch);

    /*
	mmap_config_print_all ();
    */
}

void
mmap_config_write (void)
{
	gchar *config_file;
	gchar *config_file_bak;
	GIOChannel *ch = NULL;
	GError *err = NULL;
	GIOStatus status;
	const gchar *codeset;		/* free してはダメ */
	GSList *sl;

	g_print ("write_config\n");

    /*
	mmap_config_print_all ();
    */

	config_file = g_strconcat (mmap_dir, "mmap.config2", NULL);
	config_file_bak = g_strconcat (mmap_dir, "mmap.config.bak", NULL);
	if (g_file_test (config_file, G_FILE_TEST_IS_REGULAR | G_FILE_TEST_EXISTS)) {
		rename (config_file, config_file_bak);
	}

	ch = g_io_channel_new_file (config_file, "w", &err);
	if (err != NULL) {
		g_print ("error:g_io_channel_new_file:%s:%s\n", config_file, err->message);
		return;
	}

	g_get_charset (&codeset);
	status = g_io_channel_set_encoding (ch, codeset, &err);
	if (err != NULL) {
		g_print ("error:g_io_channel_set_encoding:%s\n", err->message);
		exit (-1);
	}

	g_io_channel_write_chars (ch, "#config file for MMap+\n", -1, NULL, &err);
	g_io_channel_write_chars (ch, "\n", -1, NULL, &err);

	for (sl = sl_config; sl != NULL; sl = sl->next) {
		MmapConfig *wc = sl->data;
		gchar *buf;

		buf = g_strconcat (wc->name, "::", wc->value, "\n", NULL);
		g_io_channel_write_chars (ch, buf, -1, NULL, &err);

		g_free (buf);
	}

	g_io_channel_unref (ch);

	g_free (config_file);
	g_free (config_file_bak);
}
