/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*
#define DB(x) (x)
*/
#define DB(x)

#include <stdlib.h>

#include <gtk/gtk.h>

#include "xmlutil.h"

#include "ww_data.h"
#include "ww_layerset.h"
#include "ww_image.h"
#include "ww_quadtile.h"
#include "ww_mmap.h"
#include "ww_placename.h"
#include "ww_terrainaccessor.h"
#include "ww_world.h"

/***********************************/

/* 要修正 各クラスに作らせる方がいいかも */

typedef struct {
	gchar *text;
	GType type;
} WwType;

static GSList *sl_wwtype = NULL;

static WwType *
wwtype_create (const gchar * text, GType type)
{
	WwType *wt;

	wt = g_new (WwType, 1);
	wt->text = g_strdup (text);
	wt->type = type;

	return wt;
}

static gboolean
wwtype_search (const gchar * text, GType * type)
{
	GSList *sl;

	for (sl = sl_wwtype; sl != NULL; sl = sl->next) {
		WwType *wt = sl->data;

		if (g_ascii_strcasecmp (wt->text, text) == 0) {
			*type = wt->type;
			return TRUE;
		}
	}

	return FALSE;
}

void
xmlutil_type_init (void)
{
	WwType *wt;

	wt = wwtype_create ("LayerSet", WW_TYPE_LAYERSET);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("ImageLayer", WW_TYPE_IMAGE);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("ChildLayerSet", WW_TYPE_LAYERSET);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("QuadTileSet", WW_TYPE_QUADTILE);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("Mmap", WW_TYPE_MMAP);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("TiledPlacenameSet", WW_TYPE_PLACENAME);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("TerrainAccessor", WW_TYPE_TERRAINACCESSOR);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("HigherResolutionSubsets", WW_TYPE_TERRAINACCESSOR);
	sl_wwtype = g_slist_append (sl_wwtype, wt);

	wt = wwtype_create ("World", WW_TYPE_WORLD);
	sl_wwtype = g_slist_append (sl_wwtype, wt);
}


/***********************************/

struct _XmlData {
	/* 全部スタックのように使う。element だけ頻繁に更新される。 */
	GSList *sl_state;			/* 中身は gint * */
	GSList *sl_element;			/* 中身は gchar * */
	GSList *sl_object;			/* 中身は gpointer */

	gpointer result;
};

void
debug_print_xmldata (XmlData * xmldata, const gchar * text)
{
	gint n0, n1, n2;
	n0 = g_slist_length (xmldata->sl_state);
	n1 = g_slist_length (xmldata->sl_element);
	n2 = g_slist_length (xmldata->sl_object);

	g_print ("debug_print_xmldata:s:%d  e:%d  o:%d :::: %s\n", n0, n1, n2, text);
}

/***********************************/
void
xmlutil_push_object (XmlData * xmldata, gint state, gpointer object, const gchar * element)
{
	gint *state_p = g_new (gint, 1);
	*(state_p) = state;

	xmldata->sl_state = g_slist_prepend (xmldata->sl_state, state_p);
	xmldata->sl_object = g_slist_prepend (xmldata->sl_object, object);
	xmldata->sl_element = g_slist_prepend (xmldata->sl_element, g_strdup (element));

	DB (debug_print_xmldata (xmldata, "push_object"));
}

void
xmlutil_push_element (XmlData * xmldata, const gchar * element)
{
	xmldata->sl_element = g_slist_prepend (xmldata->sl_element, g_strdup (element));

	DB (debug_print_xmldata (xmldata, "push_object"));
}

gpointer
xmlutil_pop_object (XmlData * xmldata)
{
	gint *state;
	gpointer object;
	gchar *element;

	state = g_slist_nth_data (xmldata->sl_state, 0);
	object = g_slist_nth_data (xmldata->sl_object, 0);
	element = g_slist_nth_data (xmldata->sl_element, 0);
	xmldata->sl_state = g_slist_remove (xmldata->sl_state, state);
	xmldata->sl_object = g_slist_remove (xmldata->sl_object, object);
	xmldata->sl_element = g_slist_remove (xmldata->sl_element, element);

	xmldata->result = object;

	g_free (state);
	g_free (element);

	DB (debug_print_xmldata (xmldata, "push_object"));

	return object;
}

void
xmlutil_pop_element (XmlData * xmldata)
{
	gchar *element;

	element = g_slist_nth_data (xmldata->sl_element, 0);
	xmldata->sl_element = g_slist_remove (xmldata->sl_element, element);

	g_free (element);

	DB (debug_print_xmldata (xmldata, "push_object"));
}

gint
xmlutil_get_state (XmlData * xmldata)
{
	/*
	   g_print("xmldata:%p:   sl_state:%p:\n", xmldata, xmldata->sl_state);
	 */

	gint *state = g_slist_nth_data (xmldata->sl_state, 0);

	return *state;
}

const gchar *
xmlutil_get_element0 (XmlData * xmldata)
{
	gchar *element = g_slist_nth_data (xmldata->sl_element, 0);

	return element;
}

const gchar *
xmlutil_get_element1 (XmlData * xmldata)
{
	gchar *element = g_slist_nth_data (xmldata->sl_element, 1);

	return element;
}

gpointer
xmlutil_get_object (XmlData * xmldata)
{
	gpointer object = g_slist_nth_data (xmldata->sl_object, 0);

	return object;
}

gpointer
xmlutil_get_result (XmlData * xmldata)
{
	return xmldata->result;
}

void
xmlutil_xmldata_free (XmlData * xmldata)
{
	g_free (xmldata);
}

#include <string.h>
#include "disk.h"

XmlData *
xmlutil_prepare (const gchar * filename, GMarkupParser * parser)
{
	XmlData *xmldata;
	gchar *text = NULL;
	gsize length;
	GError *err = NULL;
	GMarkupParseContext *context;

	xmldata = g_new0 (XmlData, 1);
	xmldata->sl_state = NULL;
	xmldata->sl_object = NULL;
	xmldata->sl_element = NULL;
	xmldata->result = NULL;

	/*
	   {
	   GIOChannel *ch;
	   GIOStatus status;
	   gchar *line = NULL;
	   GError *err = NULL;

	   context = g_markup_parse_context_new (parser, 0, xmldata, NULL);

	   ch = disk_channel_open (filename, READ_SJIS);
	   status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	   while (status != G_IO_STATUS_EOF) {
	   if (status == G_IO_STATUS_ERROR) {
	   g_print ("error:g_io_channel_read_line:%s\n", err->message);
	   exit (-1);
	   }

	   g_print ("%s:\n", line);
	   g_strstrip (line);
	   g_print ("%s:\n", line);

	   if (!g_markup_parse_context_parse (context, line, strlen (line), NULL)) {
	   g_print ("error:g_markup_parse_context_parse\n");

	   g_markup_parse_context_free (context);
	   exit (1);
	   }


	   g_free (line);
	   status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	   }
	   }
	 */

	if (!g_file_get_contents (filename, &text, &length, &err)) {
		g_print ("xmlutil_prepare:error:%s:%s\n", filename, err->message);
		g_error_free (err);
		exit (1);
	}

	context = g_markup_parse_context_new (parser, 0, xmldata, NULL);

	if (!g_markup_parse_context_parse (context, text, length, NULL)) {
		g_print ("error:g_markup_parse_context_parse\n");

		g_markup_parse_context_free (context);
		exit (1);
	}

	if (!g_markup_parse_context_end_parse (context, NULL)) {
		g_print ("error:g_markup_parse_context_end_parse\n");

		g_markup_parse_context_free (context);
		exit (1);
	}

	g_free (text);
	g_markup_parse_context_free (context);

	return xmldata;;
}

/***************************************************/

static void
start_element_handler (GMarkupParseContext * context, const gchar * element_name,
					   const gchar ** attribute_names, const gchar ** attribute_values,
					   gpointer user_data, GError ** err)
{
	XmlData *xmldata = user_data;
	GType gtype;

	DB (g_print ("ELEMENT '%s'\n", element_name));

	if (wwtype_search (element_name, &gtype) == TRUE) {
		WwData *parent = (WwData *) xmlutil_get_object (xmldata);
		WwData *obj = g_object_new (gtype, NULL);

		ww_data_set_attribute (obj, attribute_names, attribute_values);
		if (parent != NULL) {
			ww_data_set_parent (obj, parent);
		}
		xmlutil_push_object (xmldata, 0, obj, element_name);

	} else {
		xmlutil_push_element (xmldata, element_name);
	}
}

static void
end_element_handler (GMarkupParseContext * context, const gchar * element_name, gpointer user_data, GError ** err)
{
	XmlData *xmldata = user_data;
	GType gtype;

	DB (g_print ("ELEMENT end (world)'%s'\n", element_name));

	if (wwtype_search (element_name, &gtype) == TRUE) {
		WwData *child = (WwData *) xmlutil_pop_object (xmldata);
		WwData *parent = (WwData *) xmlutil_get_object (xmldata);

        /***************************************************************/
        /********** 2006/5/7 現在、WMS へのアクセス方法を用意してないので WMS を利用する Quadtile を捨てる.
         * *****************************************************/
        if(WW_IS_QUADTILE(child) == TRUE){
            WwObject *obj = WW_OBJECT(child);

            if(g_ascii_strcasecmp(obj->name, "OnEarth 15m Global Mosaic, pseudocolor") == 0 ||
                g_ascii_strcasecmp(obj->name, "OnEarth 15m Global Mosaic, visual") == 0 ||
                g_ascii_strcasecmp(obj->name, "GlobeXplorer WMS Evaluation") == 0){

		        xmlutil_pop_element (xmldata);
                return;
            }

        }
        if(WW_IS_LAYERSET(child) == TRUE){
            WwObject *obj = WW_OBJECT(child);

            if(g_ascii_strcasecmp(obj->name, "Evaluation Data") == 0){

		        xmlutil_pop_element (xmldata);
                return;
            }
        }
        /***************************************************************/


		if (parent != NULL) {
			ww_data_set_child (parent, child);
		}

	} else {
		xmlutil_pop_element (xmldata);
	}
}

static void
text_handler (GMarkupParseContext * context, const gchar * text, gsize text_len, gpointer user_data, GError ** err)
{
	XmlData *xmldata = user_data;
	gchar *tmp;
	const gchar *element0;
	const gchar *element1;
	WwData *obj;

	tmp = g_strndup (text, text_len);
	tmp = g_strstrip (tmp);

	if (strlen (tmp) == 0) {	/* スペースだけの行が頻繁に出るのでとばす。出る理由はわからない。 */
		g_free (tmp);
		return;
	}

	DB (g_print ("TEXT(world) '%s' :%p:\n", text, xmldata));

	element0 = xmlutil_get_element0 (xmldata);
	element1 = xmlutil_get_element1 (xmldata);

	obj = (WwData *) xmlutil_get_object (xmldata);
	ww_data_set_element (obj, element0, element1, text);

	g_free (tmp);
}

static void
passthrough_handler (GMarkupParseContext * context,
					 const gchar * passthrough_text, gsize text_len, gpointer user_data, GError ** err)
{
	DB (g_print ("PASS '%.*s'\n", (int) text_len, passthrough_text));
}

static void
error_handler (GMarkupParseContext * context, GError * err, gpointer user_data)
{
	DB (g_print ("ERR %s\n", err->message));
}

static GMarkupParser parser = {
	start_element_handler,
	end_element_handler,
	text_handler,
	passthrough_handler,
	error_handler
};

/*********************************************************/

gpointer
xmlutil_data_build (const gchar * path_xml)
{
	g_print ("xmlutil_data_build:%s\n", path_xml);

	gpointer data;
	XmlData *xmldata;

	xmldata = xmlutil_prepare (path_xml, &parser);
	data = xmlutil_get_result (xmldata);
	xmlutil_xmldata_free (xmldata);

	return data;
}
