/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <stdlib.h>

#include "config.h"

#include "mmap.h"
#include "ww_data.h"
#include "ww_object.h"
#include "ww_image.h"
#include "ww_texture.h"
#include "ww_mesh.h"
#include "util.h"
#include "ww_layerset.h"
#include "camera.h"
#include "glarea.h"
#include "mesh.h"
#include "disk.h"

typedef struct _WwImagePrivate WwImagePrivate;
struct _WwImagePrivate {
	gboolean dispose_has_run;

	/*****/

	gdouble distance_above_surface;

	gdouble north;
	gdouble south;
	gdouble west;
	gdouble east;

	gchar *texture_path;
	guint8 opacity;
	gboolean terrain_mapped;

	/* ΣĤϻ㤬ʤΤǤ狼ʤ */
	guint8 transparency_color_blue;
	guint8 transparency_color_green;
	guint8 transparency_color_red;
	gchar *legend_image_path;

	/*****/


	WwTexture *texture;
	WwMesh *mesh;
};

#define WW_IMAGE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_IMAGE, WwImagePrivate))

static WwObjectClass *parent_class = NULL;

static void ww_image_class_init (WwImageClass * klass);
static void ww_image_init (WwImage * object);
static void ww_image_dispose (GObject * object);
static void ww_image_finalize (GObject * object);
static GObject *ww_image_constructor (GType type, guint n_props, GObjectConstructParam * props);
static void ww_image_interface_init (gpointer g_iface, gpointer iface_data);

/*****************************************************************************/

static void ww_image_set_attribute (WwData * data, const gchar ** name, const gchar ** value);
static void ww_image_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value);
static void ww_image_set_parent (WwData * data, WwData * parent);
static void ww_image_set_child (WwData * data, WwData * child);

static void ww_image_update (WwObject * obj);
static void ww_image_render (WwObject * obj);
static void ww_image_set_on (WwObject * obj, gboolean is_on);
static void ww_image_debug_print (WwObject * obj);

/*****************************************************************************/

GType
ww_image_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwImageClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_image_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwImage),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_image_init
		};

		static const GInterfaceInfo ww_data_info = {
			(GInterfaceInitFunc) ww_image_interface_init,	/* interface_init */
			NULL,				/* interface_finalize */
			NULL				/* interface_data */
		};

		type = g_type_register_static (WW_TYPE_OBJECT, "WwImage", &info, 0);

		g_type_add_interface_static (type, WW_TYPE_DATA, &ww_data_info);

        /*
		g_print ("ww_image_get_type\n");
        */
	}

	return type;
}

static void
ww_image_class_init (WwImageClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	WwObjectClass *wwobject_class = WW_OBJECT_CLASS (klass);

	/*
	   g_print ("ww_image_class_init:klass:%p:\n", klass);
	 */

	g_type_class_add_private (klass, sizeof (WwImagePrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_image_constructor;
	object_class->dispose = ww_image_dispose;
	object_class->finalize = ww_image_finalize;

	wwobject_class->update = ww_image_update;
	wwobject_class->render = ww_image_render;
	wwobject_class->set_on = ww_image_set_on;
	wwobject_class->debug_print = ww_image_debug_print;
}

static void
ww_image_init (WwImage * self)
{
	WwImagePrivate *priv = WW_IMAGE_GET_PRIVATE (self);

	/*
	   g_print ("ww_image_init:self:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;
	priv->distance_above_surface = 0.0;
	priv->north = 0.0;
	priv->south = 0.0;
	priv->west = 0.0;
	priv->east = 0.0;
	priv->texture_path = NULL;
	priv->opacity = 0;
	priv->terrain_mapped = FALSE;
	priv->transparency_color_blue = 0;
	priv->transparency_color_green = 0;
	priv->transparency_color_red = 0;
	priv->legend_image_path = NULL;


	priv->texture = NULL;
	priv->mesh = NULL;
}

static void
ww_image_dispose (GObject * obj)
{
	WwImage *self = WW_IMAGE (obj);
	WwImagePrivate *priv = WW_IMAGE_GET_PRIVATE (self);

	/*
	   g_print ("dispose\n");
	 */

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	g_object_unref (priv->texture);
	priv->texture = NULL;

	g_object_unref (priv->mesh);
	priv->mesh = NULL;

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_image_finalize (GObject * obj)
{
	WwImage *self = WW_IMAGE (obj);
	WwImagePrivate *priv = WW_IMAGE_GET_PRIVATE (self);

	/*
	   g_print ("finalize\n");
	 */

	g_free (priv->texture_path);
	g_free (priv->legend_image_path);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_image_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);

	/*
	   g_print ("constructor\n");
	 */

	object = object_class->constructor (type, n_props, props);

	return object;
}

static void
ww_image_interface_init (gpointer g_iface, gpointer iface_data)
{
	WwDataInterface *iface = (WwDataInterface *) g_iface;

	iface->set_attribute = ww_image_set_attribute;
	iface->set_element = ww_image_set_element;
	iface->set_parent = ww_image_set_parent;
	iface->set_child = ww_image_set_child;
}

WwObject *
ww_image_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_IMAGE, NULL);

	return WW_OBJECT (object);
}

/********************************************************************/

static void
image_change_texture_cb (WwTexture * texture, gpointer data)
{
	WwImagePrivate *priv;
	WwImage *image;

	g_return_if_fail (WW_IS_TEXTURE (texture));
	g_return_if_fail (WW_IS_IMAGE (data));

	image = WW_IMAGE (data);

	priv = WW_IMAGE_GET_PRIVATE (image);

	g_object_unref (image);


	priv->texture = texture;
}

static void
ww_image_update (WwObject * obj)
{
	WwImagePrivate *priv = WW_IMAGE_GET_PRIVATE (obj);

	if (obj->is_on == FALSE) {
		return;
	}
	/*
	   g_print ("ww_image_update:%s\n", obj->name);
	 */

	if (camera_check_overlap (priv->north, priv->south, priv->west, priv->east) == FALSE) {
		g_print ("overlap:FALSE\n");
		obj->visible = FALSE;
		return;
	}
	/*
	   g_print ("overlap:TRUE\n");
	 */

	if (priv->texture == NULL) {
		gchar *path;
		gchar *path_dds;
		gchar *path_jpg;
		gint w, h;
		WwTexture *texture = NULL;

		path = ww_object_get_path (obj);
		if (g_strrstr (priv->texture_path, "land_shallow_topo_2048") != NULL) {
			path_dds = g_strconcat (mmap_dir, priv->texture_path, NULL);
			path_jpg = g_strconcat (mmap_dir_pkg, "/", "Data/Earth/", "land_shallow_topo_2048.jpg", NULL);
			w = 2048;
			h = 1024;
		} else {
			path_dds = NULL;
			path_jpg = g_strconcat (mmap_dir, priv->texture_path, NULL);
			w = 512;
			h = 512;
		}

		g_print ("ww_image_update:priv->texture_path:%s\n", priv->texture_path);
		g_print ("ww_image_update:path_dds:%s  path_jpg:%s\n", path_dds, path_jpg);

		if (glarea_is_support_compress () == TRUE) {
			if (filetest (path_dds) == TRUE) {
				texture = ww_texture_new_dds (path_dds, w, h);
			} else if (filetest (path_jpg) == TRUE) {
				texture = ww_texture_new_jpg (path_jpg, path_dds);
			}

		} else {
			texture = ww_texture_new_jpg (path_jpg, path_dds);
		}
		g_object_ref (obj);
		g_signal_connect (G_OBJECT (texture), "initialized", G_CALLBACK (image_change_texture_cb), obj);
		mmap_idle_add_single (ww_texture_gen, G_OBJECT (texture));

		g_free (path);
		g_free (path_dds);
		g_free (path_jpg);

		glarea_force_render ();
	}

	if (priv->mesh == NULL) {
		gint split = 64;
		priv->mesh = ww_mesh_new_simple (priv->east, priv->west, priv->north, priv->south,
										 split, split / 2, priv->distance_above_surface, priv->terrain_mapped);
	}
	obj->visible = TRUE;
}

static void
ww_image_render (WwObject * self)
{
	WwImagePrivate *priv = WW_IMAGE_GET_PRIVATE (self);

	if (self->is_on == FALSE || self->visible == FALSE) {
		return;
	}
	/*
	   g_print ("ww_image_render:%s\n", self->name);
	 */
	if (ww_texture_bind (priv->texture) == TRUE) {
		ww_mesh_render (priv->mesh, WW_MESH_RENDER_TYPE_TEXTURE, FALSE, 0);
	} else {
		;
	}
}

static void
ww_image_set_on (WwObject * obj, gboolean is_on)
{
	WwObject *obj_parent = obj->wwparent;

	g_print ("ww_image_set_on:name:%s:\n", obj->name);

	if (is_on == TRUE && WW_IS_LAYERSET (obj_parent)) {
		/* is_on == TRUE λΤߤǤʤȡ롼פ */
		g_print ("err3\n");
		ww_layerset_check_show_only_one_layer (WW_LAYERSET (obj_parent));
	}

	parent_class->set_on (obj, is_on);
}

static void
ww_image_set_attribute (WwData * data, const gchar ** name, const gchar ** value)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);
	/*
	   g_print ("ww_image_set_attribute:o:%p\n", obj);
	 */

	/* Ѥ attribute Ϥʤ */

	parent_wwdata_iface->set_attribute (data, name, value);
}

static void
ww_image_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value)
{
	WwImage *obj = WW_IMAGE (data);
	WwImagePrivate *priv = WW_IMAGE_GET_PRIVATE (obj);
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);
	/*
	   g_print ("ww_image_set_element:o:%p e0:%s  e1:%s  value:%s\n", obj, element0, element1, value);
	 */

	if (g_ascii_strcasecmp (element0, "DistanceAboveSurface") == 0) {
		priv->distance_above_surface = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "North") == 0) {
		priv->north = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "South") == 0) {
		priv->south = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "West") == 0) {
		priv->west = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "East") == 0) {
		priv->east = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "TexturePath") == 0) {
		gchar *tmp = g_strdup (value);
		util_separator_dos_to_unix (tmp);
		priv->texture_path = tmp;

	} else if (g_ascii_strcasecmp (element0, "Opacity") == 0) {
		priv->opacity = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "TerrainMapped") == 0) {
		priv->terrain_mapped = util_char_to_boolean (value);

	} else {
		;
	}

	parent_wwdata_iface->set_element (data, element0, element1, value);
}

static void
ww_image_set_parent (WwData * data, WwData * parent)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);

	/* ƤΤޤޤǤ */
	parent_wwdata_iface->set_parent (data, parent);
}

static void
ww_image_set_child (WwData * data, WwData * child)
{
	/* ʤˤ⤷ʤ */
}

static void
ww_image_debug_print (WwObject * obj)
{
	WwImagePrivate *priv = WW_IMAGE_GET_PRIVATE (obj);

	g_print ("ww_image_debug_print##########\n");

	parent_class->debug_print (obj);

	g_print ("\t distance_above_surface:%.2f\n", priv->distance_above_surface);
	g_print ("\t north:%.2f\n", priv->north);
	g_print ("\t south:%.2f\n", priv->south);
	g_print ("\t west:%.2f\n", priv->west);
	g_print ("\t east:%.2f\n", priv->east);
	g_print ("\t texture_path:%s\n", priv->texture_path);
	g_print ("\t opacity:%d\n", priv->opacity);
	util_print_bool ("\t terrain_mapped", priv->terrain_mapped);
	g_print ("\t transparency_color_blue:%d\n", priv->transparency_color_blue);
	g_print ("\t transparency_color_green:%d\n", priv->transparency_color_green);
	g_print ("\t transparency_color_green:%d\n", priv->transparency_color_red);
	g_print ("\t legend_image_path:%s\n", priv->legend_image_path);
}
