/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>

#include <math.h>
#include <stdlib.h>
#include <GL/gl.h>

#include "config.h"

#include "mmap.h"
#include "gsi.h"
#include "ww_mmap.h"
#include "ww_smap.h"
#include "ww_texture.h"
#include "mesh.h"
#include "camera.h"
#include "util.h"
#include "glarea.h"
#include "mmapthread.h"
#include "disk.h"
#include "color.h"
#include "mmapdata.h"
#include "light.h"

typedef struct _WwSmapPrivate WwSmapPrivate;

struct _WwSmapPrivate {
	gboolean dispose_has_run;

	gboolean need_flag;

	gchar *name;				/* 例 52356755 */

	gint nx, ny;				/* 縦横の分割数 */

	gint nv;					/* 頂点数 */
	gint ni;					/* インデックス数 */

	/* 奇数点上にしか標高でーたがないので各地図は (1,1)-(451,301) の４分 */
	/* -1 - 225、225 - 451 でとるので両端の２秒がダブっているのは、左右を同じサイズ(226)にするため */
	gint x0, y0;				/* 地図の左下の座標（秒） */
	gint x1, y1;				/* 地図の右上の座標（秒） */

	gdouble *v_deg;				/* vertex 経度、緯度、標高 */
	gdouble *v_xyz;				/* vertex(単位:ｍ) */
	gdouble *n;					/* normal */
	gdouble *c;					/* color */
	guint t_name;				/* テクスチャー座標の buffer_object 名 */
	const gdouble *t;			/* テクスチャー座標 */
	guint index_name;			/* index の buffer_object 名 */
	const guint *index;			/* index */


	gchar *path_dds;
	gchar *path_png;
	gchar *uri;
	gboolean initialized;
	gchar *data;
	gint size;
	WwTexture *texture;

	WwMmap *parent;

	gboolean is_detached;		/* 描画対象から外されたら（ww_mmap から外されたら）TRUE */
};

#define WW_SMAP_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_SMAP, WwSmapPrivate))

static GObjectClass *parent_class = NULL;

static void ww_smap_class_init (WwSmapClass * klass);
static void ww_smap_init (WwSmap * object);
static void ww_smap_finalize (GObject * object);
static void ww_smap_dispose (GObject * object);
static GObject *ww_smap_constructor (GType type, guint n_props, GObjectConstructParam * props);

/*****************************************************************************/


/*****************************************************************************/

GType
ww_smap_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwSmapClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_smap_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwSmap),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_smap_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwSmap", &info, 0);

        /*
		g_print ("ww_smap_get_type\n");
        */
	}

	return type;
}

static void
ww_smap_class_init (WwSmapClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_print ("ww_smap_class_init:c:%p:\n", klass);

	g_type_class_add_private (klass, sizeof (WwSmapPrivate));

	parent_class = g_type_class_peek_parent (klass);

	object_class->constructor = ww_smap_constructor;
	object_class->finalize = ww_smap_finalize;
	object_class->dispose = ww_smap_dispose;
}

static void
ww_smap_init (WwSmap * self)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (self);

	/*
	   g_print ("ww_smap_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;
	priv->need_flag = TRUE;
	priv->name = NULL;
	priv->nx = (W_SECOND + 1) / 2;
	priv->ny = H_SECOND / 2;
	priv->nv = (priv->nx + 1) * (priv->ny + 1);
	priv->ni = ((priv->nx + 1) * 2 + 2) * priv->ny;

	priv->x0 = 0;
	priv->x1 = 0;
	priv->y0 = 0;
	priv->y1 = 0;

	priv->v_deg = NULL;
	priv->v_xyz = NULL;
	priv->n = NULL;
	priv->c = NULL;
	priv->t_name = mesh_mmap_t_st_buffer_name ();
	priv->t = mesh_mmap_t_st_init ();
	priv->index_name = mesh_mmap_index_buffer_name ();
	priv->index = mesh_mmap_i_init ();

	priv->path_dds = NULL;
	priv->path_png = NULL;
	priv->uri = NULL;
	priv->initialized = FALSE;
	priv->data = NULL;
	priv->size = 0;
	priv->texture = NULL;

	priv->parent = NULL;

	priv->is_detached = FALSE;
}

static void
ww_smap_dispose (GObject * obj)
{
	WwSmap *self = WW_SMAP (obj);
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (self);

	/*
	   g_print ("ww_smap_dispose:x:%d y:%d level:%d\n", priv->x, priv->y, priv->level);
	 */

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	g_object_unref (priv->parent);

	if (priv->texture != NULL) {
		g_object_unref (priv->texture);
	}

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_smap_finalize (GObject * obj)
{
	WwSmap *self = WW_SMAP (obj);
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (self);

	/*
	   g_print ("ww_smap_finalize\n");
	 */
	g_free (priv->name);
	g_free (priv->v_deg);
	g_free (priv->v_xyz);
	g_free (priv->n);
	g_free (priv->c);

	g_free (priv->path_dds);
	g_free (priv->path_png);
	g_free (priv->uri);
	g_free (priv->data);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_smap_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);
	/*
	   g_print ("ww_smap_constructor\n");
	 */
	object = object_class->constructor (type, n_props, props);

	return object;
}

WwSmap *
ww_smap_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_SMAP, NULL);
	/*
	   g_print ("ww_smap_new:o:%p\n", object);
	 */
	return WW_SMAP (object);
}

/********************************************************************/

static void
smap_change_texture_cb (WwTexture * texture, gpointer data)
{
	WwSmapPrivate *priv;
	WwSmap *smap;

	g_return_if_fail (WW_IS_TEXTURE (texture));
	g_return_if_fail (WW_IS_SMAP (data));

	smap = WW_SMAP (data);

	priv = WW_SMAP_GET_PRIVATE (smap);

	priv->texture = texture;
	priv->initialized = TRUE;

	g_object_unref (smap);

	glarea_force_update_and_render ();
}


static void
ww_smap_texture_process (GObject * obj)
{
	WwSmap *smap = WW_SMAP (obj);
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
    WwTexture *texture;

	GdkPixbuf *pixbuf;
	GdkPixbuf *pixbuf_scale;
	gint texture_size = glarea_get_max_texture_size ();
	gboolean left;

	g_print ("ww_smap_texture_process:%s\n", priv->name);

	if (priv->is_detached == TRUE) {
		g_print ("ww_smap_texture_process:detacched:%s\n", priv->name);
		return;
	}

	if (priv->data == NULL) {
		return;
	}

	if (priv->name[7] == '0') {
		left = TRUE;
	} else {
		left = FALSE;
	}

	disk_save_bin (priv->path_png, priv->data, priv->size);
	pixbuf = util_bin_to_pixbuf (priv->data, priv->size);
	pixbuf_scale = util_pixbuf_scale_down (pixbuf, left, texture_size);

	texture = ww_texture_new_pixbuf (pixbuf_scale, priv->path_dds);

	g_object_ref (obj);
	g_signal_connect (G_OBJECT (texture), "initialized", G_CALLBACK (smap_change_texture_cb), obj);
	mmap_idle_add_single (ww_texture_gen, G_OBJECT (texture));

	g_object_unref (pixbuf);
	g_object_unref (pixbuf_scale);
	g_free (priv->data);
	priv->data = NULL;

}

static void
ww_smap_texture_download (GObject * obj)
{
	WwSmap *smap = WW_SMAP (obj);
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	gint size;

	if (priv->is_detached == TRUE) {
		g_print ("ww_smap_texture_download:detached:%s\n", priv->name);
		return;
	}

	g_print ("ww_smap_texture_download:%s:\n", priv->uri);
	priv->data = mmapthread_download_get (priv->uri, &size);
	priv->size = size;

	mmapthread_add_power (ww_smap_texture_process, obj);
}


/* 経度方向は、-1 - 225 と 225 - 451 でとり
 * 緯度方向は、+1 - 151 と 151 - 301 でとる */
WwSmap *
ww_smap_new_with_name (WwMmap * parent, gint lon_sec, gint lat_sec, const gchar * name)
{
	WwSmap *smap;
	WwSmapPrivate *priv;
	gchar *base = "http://cyberjapan.jp/watchizu_data/";
	gchar area[5];
	gchar *path_v;
	gchar *path_n;
	gint dummy;
	gdouble multi;

	smap = ww_smap_new ();
	priv = WW_SMAP_GET_PRIVATE (smap);
	priv->parent = parent;
	g_object_ref (parent);

	priv->name = g_strdup (name);

	if (name[7] == '0') {
		priv->x0 = lon_sec - 1;
	} else {
		priv->x0 = lon_sec;
	}

	priv->x1 = priv->x0 + W_SECOND + 1;
	priv->y0 = lat_sec + 1;
	priv->y1 = priv->y0 + H_SECOND;

	/*
	   g_print ("ww_smap_new_with_name:%d %d %s\n", lon_sec, lat_sec, name);
	   g_print ("ww_smap_new_with_name:%d %d %d %d\n", priv->x0, priv->y0, priv->x1, priv->y1);
	 */

	path_v = g_strconcat (mmap_dir_gsi_vn, priv->name, ".v", NULL);
	path_n = g_strconcat (mmap_dir_gsi_vn, priv->name, ".n", NULL);

	/*
	   multi = mmap_get_vertical_exaggeration ();
	 */
	multi = 1.0;

	if (filetest (path_v) == TRUE && filetest (path_n) == TRUE) {
		/* v と n のキャッシュがあったのでそれを使う */
		priv->v_deg = (gdouble *) disk_load_bin (path_v, &dummy);
		priv->v_xyz = mesh_mmap_v_xyz_init (priv->v_deg, priv->nx, priv->ny, multi);
		priv->n = (gdouble *) disk_load_bin (path_n, &dummy);
		priv->c = mesh_mmap_c_init (priv->nx, priv->ny);
		mesh_mmap_c_calc (priv->c, priv->v_deg, priv->nv);
	} else {
		priv->v_deg = mesh_mmap_v_deg_init (priv->x0, priv->y0, priv->nx, priv->ny);
		priv->v_xyz = mesh_mmap_v_xyz_init (priv->v_deg, priv->nx, priv->ny, multi);
		priv->n = mesh_mmap_n_init (priv->v_xyz, priv->nx, priv->ny);
		priv->c = mesh_mmap_c_init (priv->nx, priv->ny);

		mmapdata_prepare (smap);
	}
	g_free (path_v);
	g_free (path_n);

	sscanf (name, "%4s", area);;
	priv->path_dds = g_strconcat (mmap_dir_gsi_tex, name, ".tex", NULL);
	priv->path_png = g_strconcat (mmap_dir_gsi_tex, name, ".png", NULL);
	priv->uri = g_strconcat (base, area, "/", name, ".png", NULL);

	/*
	   g_print ("ww_smap_new_with_name:path_dds:%s\n", priv->path_dds);
	   g_print ("ww_smap_new_with_name:path_png:%s\n", priv->path_png);
	   g_print ("ww_smap_new_with_name:uri:%s\n", priv->uri);
	 */

	priv->texture = NULL;
	priv->initialized = FALSE;

	if (filetest (priv->path_dds) == TRUE) {
		WwTexture *texture;
		gint max_texture_size = glarea_get_max_texture_size ();

		texture = ww_texture_new_dds (priv->path_dds, max_texture_size, max_texture_size);

		g_object_ref (smap);
		g_signal_connect (G_OBJECT (texture), "initialized", G_CALLBACK (smap_change_texture_cb), smap);
		mmap_idle_add_single (ww_texture_gen, G_OBJECT (texture));

	} else if (filetest (priv->path_png) == TRUE) {
		WwTexture *texture;
		GdkPixbuf *pixbuf;
		GdkPixbuf *pixbuf_scale;
		gint texture_size = glarea_get_max_texture_size ();
		gboolean left;

		if (priv->name[7] == '0') {
			left = TRUE;
		} else {
			left = FALSE;
		}

		pixbuf = disk_load_jpg (priv->path_png);
		pixbuf_scale = util_pixbuf_scale_down (pixbuf, left, texture_size);

		texture = ww_texture_new_pixbuf (pixbuf_scale, priv->path_dds);

		g_object_ref (smap);
		g_signal_connect (G_OBJECT (texture), "initialized", G_CALLBACK (smap_change_texture_cb), smap);
		mmap_idle_add_single (ww_texture_gen, G_OBJECT (texture));

		g_object_unref (pixbuf);
		g_object_unref (pixbuf_scale);

	} else {
		mmapthread_add_download2 (priv->uri, ww_smap_texture_download, G_OBJECT (smap));
	}

	return smap;
}

void
ww_smap_clear_flag (WwSmap * smap)
{
	WwSmapPrivate *priv;

	if (smap == NULL) {
		return;
	}

	priv = WW_SMAP_GET_PRIVATE (smap);

	priv->need_flag = FALSE;
}

void
ww_smap_update (WwSmap * smap)
{
	/*
	   WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	 */

	;
}


void
ww_smap_render (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);

	/*
	   g_print("ww_smap_render:%s\n", priv->name);
	 */

	if (priv->texture != NULL) {
		ww_texture_bind (priv->texture);

#ifdef GL_VERSION_1_5
		if (glarea_is_support_vbo () == TRUE) {
			glBindBuffer (GL_ARRAY_BUFFER, priv->t_name);
			glTexCoordPointer (2, GL_DOUBLE, 0, priv->t);
			glBindBuffer (GL_ARRAY_BUFFER, 0);
		} else {
			glTexCoordPointer (2, GL_DOUBLE, 0, priv->t);
		}
#else
		glTexCoordPointer (2, GL_DOUBLE, 0, priv->t);
#endif
	}

	if (ww_texture_is_initialized (priv->texture) == TRUE) {
		/* テクスチャーを表示 */

		light_on ();

		glEnableClientState (GL_COLOR_ARRAY);
		glEnableClientState (GL_NORMAL_ARRAY);

		/*
		   glColor3d (1.0, 1.0, 1.0);
		 */
		glVertexPointer (3, GL_DOUBLE, 0, priv->v_xyz);
		glColorPointer (3, GL_DOUBLE, 0, priv->c);
		glNormalPointer (GL_DOUBLE, 0, priv->n);

#ifdef GL_VERSION_1_5
		if (glarea_is_support_vbo () == TRUE) {
			glBindBuffer (GL_ELEMENT_ARRAY_BUFFER, priv->index_name);
			glDrawElements (GL_TRIANGLE_STRIP, priv->ni, GL_UNSIGNED_INT, priv->index);
			glBindBuffer (GL_ELEMENT_ARRAY_BUFFER, 0);
		} else {
			glDrawElements (GL_TRIANGLE_STRIP, priv->ni, GL_UNSIGNED_INT, priv->index);
		}
#else
		glDrawElements (GL_TRIANGLE_STRIP, priv->ni, GL_UNSIGNED_INT, priv->index);
#endif

#if 0
		glDisable (GL_TEXTURE_2D);
		glDisable (GL_DEPTH_TEST);
		/*
		   glVertexPointer (3, GL_DOUBLE, 0, priv->v_xyz);
		   glDrawElements (GL_LINE_STRIP, priv->ni, GL_UNSIGNED_INT, priv->index);
		 */
		glColor3d (1.0, 0.0, 0.0);
		glLineWidth (2.0);
		glBegin (GL_LINE_LOOP);
		glVertex3dv (priv->v_xyz);
		glVertex3dv (priv->v_xyz + 3 * priv->nx);
		glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
		glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
		glEnd ();

		glLineWidth (1.0);
		glBegin (GL_LINE_STRIP);
		glVertex3dv (priv->v_xyz);
		glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
		glVertex3dv (priv->v_xyz + 3 * priv->nx);
		glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
		glEnd ();

		glLineWidth (1.0);
		glEnable (GL_DEPTH_TEST);
		glEnable (GL_TEXTURE_2D);
#endif

		glDisableClientState (GL_COLOR_ARRAY);
		glDisableClientState (GL_NORMAL_ARRAY);

		light_off ();

	} else {
		glDisable (GL_TEXTURE_2D);
		glColor3d (1.0, 0.0, 0.0);
		glLineWidth (2.0);
		glBegin (GL_LINE_LOOP);
		glVertex3dv (priv->v_xyz);
		glVertex3dv (priv->v_xyz + 3 * priv->nx);
		glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
		glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
		glEnd ();
		/*
		   glLineWidth (1.0);
		   glBegin (GL_LINE_STRIP);
		   glVertex3dv (priv->v_xyz);
		   glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
		   glVertex3dv (priv->v_xyz + 3 * priv->nx);
		   glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
		   glEnd ();
		 */
		glLineWidth (1.0);
		glEnable (GL_TEXTURE_2D);
	}
}

gboolean
ww_smap_get_need_flag (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	return priv->need_flag;
}

void
ww_smap_set_need_flag (WwSmap * smap, gboolean need_flag)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	priv->need_flag = need_flag;
}

const gchar *
ww_smap_get_name (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	return (const gchar *) (priv->name);
}

/* 色を変えたい時に呼ぶ。標高データが無い時には無意味 */
void
ww_smap_color_set (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);

	g_print ("ww_smap_color_set\n");

	mesh_mmap_c_calc (priv->c, priv->v_deg, priv->nv);

}

void
ww_smap_get_rect (WwSmap * smap, gdouble * x0, gdouble * y0, gdouble * x1, gdouble * y1)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);

	*x0 = priv->x0;
	*y0 = priv->y0;
	*x1 = priv->x1;
	*y1 = priv->y1;
}


/* ３点を反時計周りに見て手前が正 */
static void
housen (gdouble * v0, gdouble * v1, gdouble * v2, gdouble * n)
{
	/* 1-2 */
	gdouble ax = *(v1 + 0) - *(v0 + 0);
	gdouble ay = *(v1 + 1) - *(v0 + 1);
	gdouble az = *(v1 + 2) - *(v0 + 2);
	/* 1-3 */
	gdouble bx = *(v2 + 0) - *(v0 + 0);
	gdouble by = *(v2 + 1) - *(v0 + 1);
	gdouble bz = *(v2 + 2) - *(v0 + 2);

	gdouble l;

	/*
	   g_print("%.0f %.0f %.0f   %.0f %.0f %.0f\n", ax,ay,az,bx,by,bz);
	 */

	*(n + 0) = ay * bz - az * by;
	*(n + 1) = az * bx - ax * bz;
	*(n + 2) = ax * by - ay * bx;

	l = sqrtf (*(n + 0) * *(n + 0) + *(n + 1) * *(n + 1) + *(n + 2) * *(n + 2));
	*(n + 0) /= l;
	*(n + 1) /= l;
	*(n + 2) /= l;

	/*
	   g_print("n:%.2f %.2f %.2f\n", *(n+0), *(n+1), *(n+2));
	 */
}

/* lon, lat: 単位（度） */
static void
set_p (gdouble lon, gdouble lat, gdouble * p)
{
	gdouble x, y, z;
	gdouble alt;

	alt = mmapdata_get_alt (lon, lat);

	mmap_deg_to_xyz (lon, lat, alt, &x, &y, &z);

	p[0] = x;
	p[1] = y;
	p[2] = z;
}

static void
n_calc (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	gint i, j;
	gdouble *v_deg = priv->v_deg;
	gdouble *v_xyz = priv->v_xyz;
	gdouble *n = priv->n;
	gint w = priv->nx + 1;		/* 横の頂点数 */
	gint h = priv->ny + 1;		/* 縦の頂点数 */

	g_print ("n_calc:\n");

	for (j = 0; j < h; ++j) {
		for (i = 0; i < w; ++i) {
			gint k, k0;
			gdouble p[7][3];
			gdouble lon, lat;
			gdouble l;
			gdouble n0, n1, n2;

			k0 = ((i + 0) + (j + 0) * w) * 3;	/* 中心 */
			p[0][0] = *(v_xyz + k0 + 0);
			p[0][1] = *(v_xyz + k0 + 1);
			p[0][2] = *(v_xyz + k0 + 2);

			if (j == 0) {		/* 下 */
				lon = *(v_deg + k0 + 0) + 0.0 / 3600.0;
				lat = *(v_deg + k0 + 1) - 2.0 / 3600.0;
				set_p (lon, lat, p[1]);
			} else {
				k = ((i + 0) + (j - 1) * w) * 3;
				p[1][0] = *(v_xyz + k + 0);
				p[1][1] = *(v_xyz + k + 1);
				p[1][2] = *(v_xyz + k + 2);
			}

			if (j == 0 || i == w - 1) {	/* 右下 */
				lon = *(v_deg + k0 + 0) + 2.0 / 3600.0;
				lat = *(v_deg + k0 + 1) - 2.0 / 3600.0;
				set_p (lon, lat, p[2]);
			} else {
				k = ((i + 1) + (j - 1) * w) * 3;
				p[2][0] = *(v_xyz + k + 0);
				p[2][1] = *(v_xyz + k + 1);
				p[2][2] = *(v_xyz + k + 2);
			}

			if (i == w - 1) {	/* 右 */
				lon = *(v_deg + k0 + 0) + 2.0 / 3600.0;
				lat = *(v_deg + k0 + 1) + 0.0 / 3600.0;
				set_p (lon, lat, p[3]);
			} else {
				k = ((i + 1) + (j + 0) * w) * 3;
				p[3][0] = *(v_xyz + k + 0);
				p[3][1] = *(v_xyz + k + 1);
				p[3][2] = *(v_xyz + k + 2);
			}

			if (j == h - 1) {	/* 上 */
				lon = *(v_deg + k0 + 0) + 0.0 / 3600.0;
				lat = *(v_deg + k0 + 1) + 2.0 / 3600.0;
				set_p (lon, lat, p[4]);
			} else {
				k = ((i + 0) + (j + 1) * w) * 3;
				p[4][0] = *(v_xyz + k + 0);
				p[4][1] = *(v_xyz + k + 1);
				p[4][2] = *(v_xyz + k + 2);
			}

			if (j == h - 1 || i == 0) {	/* 左上 */
				lon = *(v_deg + k0 + 0) - 2.0 / 3600.0;
				lat = *(v_deg + k0 + 1) + 2.0 / 3600.0;
				set_p (lon, lat, p[5]);
			} else {
				k = ((i - 1) + (j + 1) * w) * 3;
				p[5][0] = *(v_xyz + k + 0);
				p[5][1] = *(v_xyz + k + 1);
				p[5][2] = *(v_xyz + k + 2);
			}

			if (i == 0) {		/* 左 */
				lon = *(v_deg + k0 + 0) - 2.0 / 3600.0;
				lat = *(v_deg + k0 + 1) + 0.0 / 3600.0;
				set_p (lon, lat, p[6]);
			} else {
				k = ((i - 1) + (j + 0) * w) * 3;
				p[6][0] = *(v_xyz + k + 0);
				p[6][1] = *(v_xyz + k + 1);
				p[6][2] = *(v_xyz + k + 2);
			}

			n0 = n1 = n2 = 0.0;
			{
				gint a;
				gdouble n_tmp[6][3];

				housen (p[0], p[1], p[2], n_tmp[0]);
				housen (p[0], p[2], p[3], n_tmp[1]);
				housen (p[0], p[3], p[4], n_tmp[2]);
				housen (p[0], p[4], p[5], n_tmp[3]);
				housen (p[0], p[5], p[6], n_tmp[4]);
				housen (p[0], p[6], p[1], n_tmp[5]);

				for (a = 0; a < 6; ++a) {
					n0 += n_tmp[a][0];
					n1 += n_tmp[a][1];
					n2 += n_tmp[a][2];
				}
			}

			l = sqrt (n0 * n0 + n1 * n1 + n2 * n2);

			*(n + k0 + 0) = n0 / l;
			*(n + k0 + 1) = n1 / l;
			*(n + k0 + 2) = n2 / l;

			/*
			   g_print("n:%.2f %.2f %.2f\n", *(n+k0+0), *(n+k0+1), *(n+k0+2));
			 */
		}
	}
}

/* 標高データが準備できた後に呼ばれ、標高データ付きデータを作成しなおすとともに保存もする */
void
ww_smap_vcn_calc (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	gchar *path_v;
	gchar *path_n;
	gint i;
	gdouble multi = mmap_get_vertical_exaggeration ();

	g_print ("ww_smap_vcn_calc:\n");

	for (i = 0; i < priv->nv; ++i) {
		gint k;
		gdouble lon, lat, alt;
		gdouble x, y, z;
		guint8 r, g, b;

		k = i * 3;

		lon = *(priv->v_deg + k + 0);
		lat = *(priv->v_deg + k + 1);

		alt = mmapdata_get_alt (lon, lat);

		*(priv->v_deg + k + 2) = alt;

		mmap_deg_to_xyz (lon, lat, alt, &x, &y, &z);
		*(priv->v_xyz + k + 0) = x;
		*(priv->v_xyz + k + 1) = y;
		*(priv->v_xyz + k + 2) = z;

		if (alt < 0.0) {
			/* 海の色は青 */
			*(priv->c + k + 0) = 0.6;
			*(priv->c + k + 1) = 0.7;
			*(priv->c + k + 2) = 1.0;
		} else {
			color_get_rgb_from_alt ((gint) alt, &r, &g, &b);
			/*
			   g_print("ww_smap_vcn_calc:%hd %hd %hd\n", r, g, b);
			 */
			*(priv->c + k + 0) = (gdouble) r / 0xff;
			*(priv->c + k + 1) = (gdouble) g / 0xff;
			*(priv->c + k + 2) = (gdouble) b / 0xff;
		}
		/*
		   g_print("ww_smap_vcn_calc:alt:%.2f c:%.2f %.2f %.2f\n", alt, *(priv->c + k + 0), *(priv->c + k + 1), *(priv->c + k + 2));
		 */
	}
	n_calc (smap);

	path_v = g_strconcat (mmap_dir_gsi_vn, priv->name, ".v", NULL);
	path_n = g_strconcat (mmap_dir_gsi_vn, priv->name, ".n", NULL);

	disk_save_bin (path_v, priv->v_deg, 3 * priv->nv * sizeof (gdouble));
	disk_save_bin (path_n, priv->n, 3 * priv->nv * sizeof (gdouble));

	g_free (path_v);
	g_free (path_n);

	glarea_force_update_and_render ();
}

void
ww_smap_is_detached_from_render_list (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);

	priv->is_detached = TRUE;

}

void
ww_smap_change_vertical_exaggeration (WwSmap * smap)
{
	WwSmapPrivate *priv = WW_SMAP_GET_PRIVATE (smap);
	gdouble multi;

	multi = mmap_get_vertical_exaggeration ();

	mesh_mmap_v_xyz_change (priv->v_deg, priv->v_xyz, priv->nx, priv->ny, multi);
}
