/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*
#define DB(x) (x)
*/
#define DB(x)

#include <math.h>

#include "config.h"

#include "ww_world.h"
#include "mmap.h"
#include "util.h"
#include "xmlutil.h"
#include "ww_object.h"
#include "layermanager.h"


typedef struct _WwWorldPrivate WwWorldPrivate;
struct _WwWorldPrivate {
	gboolean dispose_has_run;

	WwTerrainaccessor *terrain_accessor;

	GSList *sl_layers;

	/* attribute */
	gchar *name;
	gdouble equatorial_radius;
	gchar *layer_directory;
};

#define WW_WORLD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_WORLD, WwWorldPrivate))

static GObjectClass *parent_class = NULL;

static void ww_world_class_init (WwWorldClass * klass);
static void ww_world_init (WwWorld * object);
static void ww_world_finalize (GObject * object);
static void ww_world_dispose (GObject * object);
static GObject *ww_world_constructor (GType type, guint n_props, GObjectConstructParam * props);
static void ww_world_interface_init (gpointer g_iface, gpointer iface_data);

/*****************************************************************************/

static void ww_world_set_attribute (WwData * data, const gchar ** name, const gchar ** value);
static void ww_world_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value);
static void ww_world_set_parent (WwData * data, WwData * parent);
static void ww_world_set_child (WwData * data, WwData * child);

/*****************************************************************************/

GType
ww_world_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwWorldClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_world_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwWorld),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_world_init
		};

		static const GInterfaceInfo ww_data_info = {
			(GInterfaceInitFunc) ww_world_interface_init,	/* interface_init */
			NULL,				/* interface_finalize */
			NULL				/* interface_data */
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwWorld", &info, 0);

		g_type_add_interface_static (type, WW_TYPE_DATA, &ww_data_info);
	}

	return type;
}

static void
ww_world_class_init (WwWorldClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	/*
	   g_print ("ww_world_class_init:c:%p:\n", klass);
	 */
	g_type_class_add_private (klass, sizeof (WwWorldPrivate));

	parent_class = g_type_class_peek_parent (klass);

	object_class->constructor = ww_world_constructor;
	object_class->finalize = ww_world_finalize;
	object_class->dispose = ww_world_dispose;
}

static void
ww_world_init (WwWorld * self)
{
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (self);
	/*
	   g_print ("ww_world_init:o:%p:\n", self);
	 */
	priv->dispose_has_run = FALSE;

	priv->terrain_accessor = NULL;
	priv->sl_layers = NULL;

	/* attribute */
	priv->name = NULL;
	priv->equatorial_radius = 0.0;
	priv->layer_directory = NULL;;
}

static void
ww_world_dispose (GObject * obj)
{
	WwWorld *self = WW_WORLD (obj);
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (self);
	GSList *sl;
	/*
	   g_print ("dispose\n");
	 */
	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	g_object_unref (priv->terrain_accessor);
	priv->terrain_accessor = NULL;
	for (sl = priv->sl_layers; sl != NULL; sl = sl->next) {
		WwObject *layer = sl->data;

		g_object_unref (layer);
	}

	g_slist_free (priv->sl_layers);
	priv->sl_layers = NULL;

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_world_finalize (GObject * obj)
{
	WwWorld *self = WW_WORLD (obj);
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (self);

	/*
	   g_print ("finalize\n");
	 */
	g_free (priv->name);
	g_free (priv->layer_directory);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_world_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);

	/*
	   g_print ("ww_world_constructor\n");
	 */

	object = object_class->constructor (type, n_props, props);

	return object;
}

static void
ww_world_interface_init (gpointer g_iface, gpointer iface_data)
{
	WwDataInterface *iface = (WwDataInterface *) g_iface;
	iface->set_attribute = ww_world_set_attribute;
	iface->set_element = ww_world_set_element;
	iface->set_parent = ww_world_set_parent;
	iface->set_child = ww_world_set_child;
}

WwWorld *
ww_world_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_WORLD, NULL);
	/*
	   g_print ("ww_world_new:o:%p\n", object);
	 */
	return WW_WORLD (object);
}

/********************************************************************/

static void
ww_world_set_attribute (WwData * data, const gchar ** name, const gchar ** value)
{
	WwWorld *obj = WW_WORLD (data);
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (obj);
	gint i;
	/*
	   g_print("ww_world_set_attribute:o:%p\n", obj);
	 */
	for (i = 0; name[i] != NULL; ++i) {
		DB (g_print ("ww_world_set_attribute:name:%s: value:%s:\n", name[i], value[i]));

		if (g_ascii_strcasecmp (name[i], "Name") == 0) {
			priv->name = g_strdup (value[i]);

		} else if (g_ascii_strcasecmp (name[i], "EquatorialRadius") == 0) {
			priv->equatorial_radius = util_char_to_double (value[i]);

		} else if (g_ascii_strcasecmp (name[i], "LayerDirectory") == 0) {
			gchar *tmp;

			tmp = g_strdup (value[i]);
			util_separator_dos_to_unix (tmp);
			priv->layer_directory = tmp;

		} else {
			g_print ("world_new_with_attribute:error:%s:%s:\n", name[i], value[i]);
		}
	}
}

static void
ww_world_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value)
{
	/*
	   g_print("ww_world_set_element:o:%p  e0:%s e1:%s value:%s\n", obj, element0, element1, value);
	 */
	/* なし */
}

static void
ww_world_set_parent (WwData * data, WwData * parent)
{
	/*
	   WwWorld *obj = WW_WORLD (data);
	   WwWorld *obj_parent = WW_WORLD (parent);

	   obj->wwparent = obj_parent;
	   g_object_ref (obj_parent);
	 */
}

static void
ww_world_set_child (WwData * data, WwData * child)
{
	WwWorld *obj = WW_WORLD (data);
	WwTerrainaccessor *obj_child = WW_TERRAINACCESSOR (child);
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (obj);

	priv->terrain_accessor = obj_child;
}

/******************************/

void
ww_world_debug_print (WwWorld * obj)
{
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (obj);


	g_print ("world:name\t:%s:\n", priv->name);
	g_print ("world:equatorial_radius\t:%.2f:\n", priv->equatorial_radius);
	g_print ("world:layer_directory\t:%s:\n", priv->layer_directory);
	if (priv->terrain_accessor != NULL) {
		ww_terrainaccessor_debug_print (priv->terrain_accessor);
	}
}


void
ww_world_update (WwWorld * world)
{
	GSList *sl;
	WwWorldPrivate *priv;

	g_return_if_fail (world != NULL);

	priv = WW_WORLD_GET_PRIVATE (world);

	for (sl = priv->sl_layers; sl != NULL; sl = sl->next) {
		WwObject *wwo = (WwObject *) sl->data;

		ww_object_update (wwo);
	}
}

void
ww_world_render (WwWorld * world)
{
	GSList *sl;
	WwWorldPrivate *priv;

	g_return_if_fail (world != NULL);

	priv = WW_WORLD_GET_PRIVATE (world);

	for (sl = priv->sl_layers; sl != NULL; sl = sl->next) {
		WwObject *wwo = (WwObject *) sl->data;

		ww_object_render (wwo);
	}
}

gdouble
ww_world_get_equatorial_radius (WwWorld * world)
{
	WwWorldPrivate *priv;

	/*
	   g_return_val_if_fail (world != NULL, 0.0);
	 */
	if (world == NULL) {
		return 0.0;
	}

	priv = WW_WORLD_GET_PRIVATE (world);

	return priv->equatorial_radius;
}

/* 赤道断面をＸＹ平面とする。東経０がＸ軸、東経９０がＹ軸。地軸がＺ軸で、北極側がプラス。 */
/* 経度(deg)、緯度(deg)、標高(m)を受け取り、ＸＹＺの座標(m)を得る */
void
ww_world_deg_to_xyz (WwWorld * world, gdouble lon, gdouble lat, gdouble alt, gdouble * x, gdouble * y, gdouble * z)
{
	WwWorldPrivate *priv;
	gdouble r;					/* 楕円体をある緯度で切った時にできる円の半径(m) */
	gdouble radius;

	DB (g_print ("ww_world_deg_to_xyz:%.1f %.1f %.1f\n", lon, lat, alt));

	if (world == NULL) {
		*x = 0.0;
		*y = 0.0;
		*z = 0.0;
		return;
	}

	priv = WW_WORLD_GET_PRIVATE (world);

	radius = priv->equatorial_radius;

	r = (radius + alt) * cos (2.0 * G_PI * lat / 360.0);
	*x = r * cos (2.0 * G_PI * lon / 360.0);
	*y = r * sin (2.0 * G_PI * lon / 360.0);
	*z = (radius + alt) * sin (2.0 * G_PI * lat / 360.0);

	DB (g_print ("ww_world_deg_to_xyz:x:%.1f y:%.1f z:%.1f\n", *x, *y, *z));
}

WwTerrainaccessor *
ww_world_get_terrainaccessor (WwWorld * world, gdouble x0, gdouble y0, gdouble x1, gdouble y1)
{
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (world);

	return ww_terrainaccessor_select (priv->terrain_accessor, x0, y0, x1, y1);
}

const gchar *
ww_world_get_name (WwWorld * world)
{
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (world);

	return (const gchar *) priv->name;
}

static void
ww_world_create_layermanager_menu (WwWorld * world)
{
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (world);
	GSList *sl;

	layermanager_clear ();

	for (sl = priv->sl_layers; sl != NULL; sl = sl->next) {
		WwObject *obj = (WwObject *) sl->data;

		ww_object_set_layermanager_menu (obj, NULL);
	}
}

void
ww_world_create_layer (WwWorld * world)
{
	WwWorldPrivate *priv = WW_WORLD_GET_PRIVATE (world);
	gchar *files_earth[] = { "Gsi.xml", "Base.xml", "Tiled.xml", "Images.xml", "Placenames.xml" };
	gchar *files_moon[] = { "Images.xml", "Placenames.xml" };
	gchar **files;
	gint i, n;

	g_print ("ww_world_create_layer\n");

	if (g_strcasecmp (priv->name, "Earth") == 0) {
		files = files_earth;
		n = sizeof files_earth / sizeof (gchar *);
	} else {
		files = files_moon;
		n = sizeof files_moon / sizeof (gchar *);
	}

	for (i = 0; i < n; ++i) {
		gchar *path_xml;
		WwObject *wwo;

		path_xml = g_strconcat (mmap_dir_pkg_config, "/", priv->name, "/", files[i], NULL);
		wwo = (WwObject *) xmlutil_data_build (path_xml);
		priv->sl_layers = g_slist_append (priv->sl_layers, wwo);
		g_free (path_xml);
	}

	/*
	   world_debug_print ();
	 */

	ww_world_create_layermanager_menu (world);
}
