/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "mmap.h"
#include "disk.h"
#include "net.h"
#include "mmapdata.h"
#include "ww_smap.h"
#include "camera.h"

/********************* 標高データ検索用 *****************/

typedef struct {
	gchar *path;				/* pref/11111.lzh */
	gdouble x;					/* 経度 緯度 */
	gdouble y;
	gint nx;					/* 経度 緯度方向のデータ数×２ */
	gint ny;
	/* gint nall; *//* 標高以外のデータの点数 */
	gint nalt;					/* 標高のデータの点数 */
	/* gchar *pref; */
	gchar *city;
} Alt;

static GSList *sl_alt = NULL;

static void
mmapdata_create_alt_db (void)
{
	gchar *fullpath;
	gchar *line = NULL;
	GIOChannel *ch = NULL;
	GError *err = NULL;
	GIOStatus status;

	fullpath = g_strconcat (mmap_dir_pkg, "/alt.dat", NULL);
	ch = disk_channel_open (fullpath, READ_UTF8);
	g_free (fullpath);

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Alt *alt;
		gchar path[30];
		gint64 x64, y64;
		gdouble x, y;
		gint nx, ny;
		gint nall, nalt;
		gchar pref[30];
		gchar city[150];

		if (status == G_IO_STATUS_ERROR) {
			g_print ("error:g_io_channel_read_line:%s:%s\n", line, err->message);
			exit (-1);
		}
		if (line[0] == '#') {	/* コメント行をとばす */
			continue;
		}

		/* 何？このコメント。意味不明。(2005/11/30) */
		/* 最後の市が空白で区切られていて最初しかとれないが、使ってないのでこのまま行く。 */
		/* 経度緯度は９桁の場合もある */
		sscanf (line, "%s %Ld %Ld %d %d %d %d %s %s", path, &x64, &y64, &nx, &ny, &nall, &nalt, pref, city);

		x = (gdouble) x64 / 10000, 0;
		y = (gdouble) y64 / 10000, 0;

		DB (g_print ("%s\n", line));
		DB (g_print ("%s:%.0f %.0f %d %d %d %d:%s:%s:\n", path, x, y, nx, ny, nall, nalt, pref, city));

		alt = g_new (Alt, 1);
		alt->path = g_strdup (path);
		alt->x = x;
		alt->y = y;
		alt->nx = nx / 10000;
		alt->ny = ny / 10000;
		/* alt->nall = nall; */
		alt->nalt = nalt;
		/* alt->pref = g_strdup (pref); */
		alt->city = g_strdup (city);

		sl_alt = g_slist_prepend (sl_alt, alt);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);

	g_io_channel_unref (ch);
}

/* 左下と右上の経度緯度（秒）をもらって、その範囲と重なる市のデータのポインタを GSList で返す */
static GSList *
mmapdata_lookup_alt_citylist (gdouble x1, gdouble y1, gdouble x2, gdouble y2)
{
	GSList *sl;
	GSList *rlist = NULL;

	DB (g_print ("mmapdata_lookup_alt_citylist:%.0f %.0f %.0f %.0f\n", x1, y1, x2, y2));

	for (sl = sl_alt; sl != NULL; sl = sl->next) {
		Alt *alt = sl->data;
		gdouble xx1, yy1, xx2, yy2;	/* 市の左下と右上の経度緯度 */

		xx1 = alt->x;
		yy1 = alt->y;
		xx2 = xx1 + alt->nx;
		yy2 = yy1 + alt->ny;

		/* ４つの辺それぞれに重なりがあるかどうか見ている */
		if ((yy1 >= y1 && yy1 <= y2 && xx1 <= x2 && xx2 >= x1) ||
			(yy2 >= y1 && yy2 <= y2 && xx1 <= x2 && xx2 >= x1) ||
			(xx1 >= x1 && xx1 <= x2 && yy1 <= y2 && yy2 >= y1) || (xx2 >= x1 && xx2 <= x2 && yy1 <= y2 && yy2 >= y1) ||
			/* すっぽり入ってるパターン */
			(xx1 <= x1 && xx2 >= x2 && yy1 <= y1 && yy2 >= y2)) {

			rlist = g_slist_prepend (rlist, alt);	/* 重なってる */

			DB (g_print ("%.0f %.0f %.0f %.0f %s %s\n", xx1, yy1, xx2, yy2, alt->path, alt->city));
		} else {
			;					/* 重なってない */
		}
	}

	return rlist;
}

/***************************************************************/

static GThreadPool *pool_mmapdata = NULL;

typedef struct {
	gint key;
	gint z;
} KeyVal;

typedef struct {
	/* 下の２つの実体は data.c が持っている。解放してはダメ。 */
	const gchar *path;			/* 例：pref/11111.lzh */
	const gchar *city;

	/*
	 * 以下のいくつかのデータは Alt にも含まれるが、Alt は国土地理院のデータが
	 * 修正される前の、誤差を含んだ値かもしれない。
	 * よって、ダウンロードしたデータから作りなおす。
	 */
	gdouble x0;					/* 左下の経度緯度(秒)(１の桁が偶数でない場合あり) */
	gdouble y0;
	gdouble x1;					/* 右上の経度緯度 */
	gdouble y1;

	gint xmax;					/* 経度方向のオフセット最大値(奇数のみデータあり) */
	gint ymax;

	gint zmin;					/* 最高標高と最低標高 */
	gint zmax;

	KeyVal *keyval;

	GHashTable *hash;			/* 各データは KeyVal */

	gint ref_count;				/* このデータを必要としている地図の数 */
} City;

static GSList *sl_city = NULL;

/* *******.slm のファイルから左下の基準点の経度緯度を読み込む */
static void
read_data_slm (const gchar * path_slm, gdouble * x, gdouble * y)
{
	gchar *line = NULL;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	gint64 x64, y64;

	DB (g_print ("read_data_slm:%s\n", name));

	ch = disk_channel_open (path_slm, READ_SJIS);

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	if (status == G_IO_STATUS_ERROR) {
		g_print ("error:g_io_channel_read_line:%s:%s\n", line, err->message);
		exit (-1);
	}
	line = g_strchomp (line);

	/* slm の基準点は、低緯度のところでは９桁になる */
	sscanf (line, "%Ld,%Ld", &x64, &y64);
	*x = (gdouble) x64 / 10000.0;
	*y = (gdouble) y64 / 10000.0;

	g_io_channel_unref (ch);
	g_free (line);
}


/* 経度緯度の下４桁だけを使い、経度を１００００倍した物と緯度を足してキーとする */
/* ちなみに３桁だけ使うようにしたら、約５０ｋｍ毎に同じキーを持つ値ができてしまったのでボツ。
 * 一つの市や町で５００ｋｍ以上あるところがあれば、断崖になって出てしまう */
static gint
create_hash_key_from_keidoido (gint keido, gint ido)
{
	return (keido % 10000) * 10000 + (ido % 10000);
}

/* 11111 の名前をもらって、そのデータをメモリー中に読み込む */
static void
create_hash_data (City * city, const gchar * slm, const gchar * sal, const gchar * slp)
{
	gint i;
	gchar *line_sal = NULL;
	GIOChannel *ch_sal;
	GIOStatus status_sal;
	GError *err = NULL;
	gdouble x0, y0;				/* 基準点の経度と緯度 */
	gint x0l, y0l;				/* x0 y0 の int 版。必ず偶数にする */
	gint zmin = 10000;
	gint zmax = 0;
	gint xmax = 0;
	gint ymax = 0;

	gchar *slp_data;
	gint slp_size;
	gint num_line;
	gchar *p_slp;
	KeyVal *p_keyval;

	DB (g_print ("create_hash_data:%s\n", city->path));

	slp_data = disk_load_bin (slp, &slp_size);
	p_slp = slp_data;
	num_line = slp_size / 17;	/* １行１７文字決め打ち */

	city->keyval = g_new (KeyVal, num_line);
	p_keyval = city->keyval;

	read_data_slm (slm, &x0, &y0);

	/* slm の基準点は、偶数も奇数も小数もある。経度が奇数で緯度が偶数とかもある(43343)。
	 * mh.slp のオフセットにも偶数と奇数と小数がある。これも経度が偶数で緯度が奇数がある。
	 * 基準点が奇数の時は、オフセットが偶数で
	 * 基準点が偶数の時は、オフセットが奇数となる。
	 * 小数の場合も同じで基準とオフセットの和が奇数になるようにしてある。
	 *
	 * 基準点が小数の場合、オフセットは 0.**** と１以下からはじまっている。
	 *
	 * 基準が奇数の時でも、オフセットの最小値にゼロはなく１。
	 *
	 * まとめると、
	 * 基準点はいろいろな値をとるが、標高値を持つのは奇数の点だけだ
	 * ということ。
	 *
	 * 基準点が整数の時には、オフセットも整数で、足すと必ず奇数になる。
	 * 基準点が小数の時には、オフセットも小数だが、それぞれ小数点以下を
	 * 切り捨ててしまっているので、足すと偶数になり、切捨て分の１を足すのが正解。
	 */
	x0l = (gint) x0;
	y0l = (gint) y0;
	if (x0 - x0l > 0) {
		++x0l;
	}
	if (y0 - y0l > 0) {
		++y0l;
	}
	DB (g_print ("x0:%.2f y0:%.2f    x0l:%d y0l:%d\n", x0, y0, x0l, y0l));

	ch_sal = disk_channel_open (sal, READ_SJIS);

	status_sal = g_io_channel_read_line (ch_sal, &line_sal, NULL, NULL, &err);
	for (i = 0; i < num_line; ++i, ++p_keyval, p_slp += 17) {
		gint x_offset = 0;
		gint y_offset = 0;
		gint x, y;
		gint z;

		if (status_sal == G_IO_STATUS_ERROR) {
			g_print ("error:sal read:%s:%s\n", line_sal, err->message);
			exit (1);
		}

		/* 小数点以下は切捨てる */
		x_offset += (p_slp[0] - '0') * 1000;
		x_offset += (p_slp[1] - '0') * 100;
		x_offset += (p_slp[2] - '0') * 10;
		x_offset += (p_slp[3] - '0');

		y_offset += (p_slp[8] - '0') * 1000;
		y_offset += (p_slp[9] - '0') * 100;
		y_offset += (p_slp[10] - '0') * 10;
		y_offset += (p_slp[11] - '0');

		/*
		   DB(g_print ("x0l:%d y0l:%d x_off:%d y_off:%d\n", x0l,y0l,x_offset,y_offset));
		 */
		x = x0l + x_offset;
		y = y0l + y_offset;

		sscanf (line_sal + 20, "%d", &z);

		if (xmax < x_offset)
			xmax = x_offset;
		if (ymax < y_offset)
			ymax = y_offset;
		if (zmax < z)
			zmax = z;
		if (zmin > z)
			zmin = z;

		p_keyval->key = create_hash_key_from_keidoido (x, y);
		p_keyval->z = z;
		g_hash_table_insert (city->hash, &(p_keyval->key), p_keyval);
		/*
		   g_print ("to hash:%d %d %d\n", x,y,z);
		 */
		g_free (line_sal);
		status_sal = g_io_channel_read_line (ch_sal, &line_sal, NULL, NULL, &err);
	}
	g_free (line_sal);

	g_free (slp_data);

	city->x0 = x0;
	city->y0 = y0;
	city->x1 = x0 + xmax;
	city->y1 = y0 + ymax;
	city->xmax = xmax;
	city->ymax = ymax;
	city->zmin = zmin;
	city->zmax = zmax;

	g_io_channel_unref (ch_sal);
}

static void
free_city (City * city)
{
	g_hash_table_destroy (city->hash);
	g_free (city->keyval);
	g_free (city);
}

/* path: aichi/11111 */
static void
lzh_extract (const gchar * lzh)
{
	gchar *argv[] = { "lha", "xti", NULL, NULL };
	GError *err = NULL;
	gchar *current_dir;

	/* 国土地理院のデータが全国で統一されてない。具体的には、市ごとの圧縮データが
	 * ディレクトリ付きのものと、そうでないものがある。ということで、中身を全部解凍することにした。
	 * データ修正後はなおってる可能性があるが、このままで行く。
	 */

	argv[2] = (gchar *) lzh;
	/*
	   printf ("%s %s %s\n", argv[0], argv[1], argv[2]);
	 */
	current_dir = g_get_current_dir ();
	chdir (mmap_dir_gsi_vn);
	g_spawn_sync (NULL, argv, NULL, G_SPAWN_SEARCH_PATH | G_SPAWN_STDOUT_TO_DEV_NULL, NULL, NULL, NULL, NULL, NULL, &err);
	if (err != NULL) {
		g_print ("spawn error::lzh_extract:%s:\n", err->message);
	}
	chdir (current_dir);

	g_free (current_dir);
}

/* 数値データの読み込み用。読み込んだ後の展開までする。 */
/* path:"aichi/11111" */
static void
get_lzh (City * city, const gchar * slm, const gchar * sal, const gchar * slp, const gchar * lzh)
{
	DB (g_print ("loader_get_data:uri:%s\n", city->path));

	if (filetest (slm) && filetest (sal) && filetest (slp)) {
		/* 展開後のデータがあったので何もしない */

		g_print ("loader_get_data:slm..:%s\n", slm);
		;
	} else if (filetest (lzh) == TRUE) {
		/* 展開前のデータがあったので展開する */

		g_print ("loader_get_data:lzh:%s\n", lzh);
		lzh_extract (lzh);
	} else {
		/* キャッシュに何もなかったのでダウンロード後に展開する */
		const gchar *uri_l = "http://sdf.gsi.go.jp/data25k/";
		gchar *uri;
		void *data;
		gint bytes;

		uri = g_strconcat (uri_l, city->path, ".lzh", NULL);	/* aichi/11111 */

		data = net_download (uri, &bytes);
		if (data == NULL) {
			g_print ("error: net_download return NULL:%s\n", city->path);
			exit (1);
		}

		DB (g_print ("data:%s:%d\n", city->path, bytes));

		/* キャッシュに入れとく */
		disk_save_bin (lzh, data, bytes);

		g_free (data);
		g_free (uri);
		lzh_extract (lzh);
	}

}

/* 現在表示している位置から遠いかをしらべる。遠いなら TRUE */
static gboolean
is_far_enough (City * city)
{
	gdouble x0, y0;				/* 現在範囲(左下) */
	gdouble x1, y1;				/* 現在範囲(右上) */
	gdouble far_enough = 100.0;	/* 200秒(5km位はあるか？)以上遠いと十分遠いと判断する */
	/* マップ単位でしか比較してないので 100 にしても 150 とか 225 でしかかわらない */

	camera_get_view_narrow (&x0, &y0, &x1, &y1);
	x0 *= 3600.0;
	y0 *= 3600.0;
	x1 *= 3600.0;
	y1 *= 3600.0;

	if (x0 - city->x1 > far_enough || city->x0 - x1 > far_enough || y0 - city->y1 > far_enough || city->y0 - y1 > far_enough) {

		/*
		   g_print ("is_far_enough:TRUE\n");
		 */
		return TRUE;
	} else {
		/*
		   g_print ("is_far_enough:FALSE\n");
		 */
		return FALSE;
	}
}

/* 以後不要かつ、現在値から十分遠い市のデータを解放する */
static void
clean_slist_city (void)
{
	GSList *s;
	GSList *next;
	GSList *prev;

	DB (g_print ("#########################clean start#\n"));

	for (s = sl_city, prev = NULL; s != NULL; s = next) {
		City *city = s->data;

		next = s->next;

		DB (g_print ("city:::::ref_count:%d::", city->ref_count));

		if (city->ref_count == 0 && is_far_enough (city) == TRUE) {
			DB (g_print ("######    remove:%s:%s\n", city->path, city->city));

			if (prev == NULL) {
				sl_city = next;
			} else {
				prev->next = next;
			}
			/* GSLise は専用アロケータ使ってるようで g_free ではだめ？ */
			g_slist_free_1 (s);

			free_city (city);
		} else {
			DB (g_print ("######not remove:%s:%s\n", city->path, city->city));

			prev = s;
		}
	}
	DB (g_print ("#########################clean stop#\n"));
}



/* 名前を受け取り、そのデータが既にあればそれを、なければ NULL。path は pref/11111.lzh */
static City *
get_city_from_path (const gchar * path)
{
	GSList *sl;

	DB (g_print ("get_city_from_path:%s\n", path));

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		City *city = sl->data;

		/*
		   g_print ("get_city_from_path:city->path:%s\n", city->path);
		 */

		if (g_ascii_strcasecmp (path, city->path) == 0) {
			/*
			   g_print ("get_city_from_path:return:city->path:%s\n", city->path);
			 */
			return city;
		}
	}
	return NULL;
}

/* 既にメモリー中に持っているものは、ref_count を +1 し、
 * 持ってないものはダウンロードしてメモリー中に用意する。
 */
void
check_citylist (GSList * citylist)
{
	GSList *sl;

	g_print ("check_citylist\n");
	DB (g_print ("check_citylist\n"));

	for (sl = citylist; sl != NULL; sl = sl->next) {
		Alt *alt = sl->data;
		City *city;

		city = get_city_from_path (alt->path);

		if (city == NULL) {
			gchar *name_top;
			gchar *slm;
			gchar *sal;
			gchar *slp;
			gchar *lzh;

			g_print ("check_citylist:%s\n", alt->path);
			name_top = g_strrstr (alt->path, "/");
			++name_top;
			slm = g_strconcat (mmap_dir_gsi_vn, name_top, ".slm", NULL);
			sal = g_strconcat (mmap_dir_gsi_vn, name_top, "mh.sal", NULL);
			slp = g_strconcat (mmap_dir_gsi_vn, name_top, "mh.slp", NULL);
			lzh = g_strconcat (mmap_dir_gsi_vn, name_top, ".lzh", NULL);

			city = g_new (City, 1);
			city->path = alt->path;
			city->city = alt->city;
			city->keyval = NULL;
			city->hash = g_hash_table_new (g_int_hash, g_int_equal);
			city->ref_count = 1;

			sl_city = g_slist_prepend (sl_city, city);

			DB (g_print ("city:::::prepend:%s:%s\n", city->path, city->city));

			/* キャッシュに展開された状態を作る */
			get_lzh (city, slm, sal, slp, lzh);

			create_hash_data (city, slm, sal, slp);

			g_free (slm);
			g_free (sal);
			g_free (slp);
			g_free (lzh);

		} else {
			city->ref_count += 1;
		}
	}
}

void
ref_count_down (GSList * citylist)
{
	GSList *sl;

	DB (g_print ("ref_count_down\n"));

	for (sl = citylist; sl != NULL; sl = sl->next) {
		Alt *alt = sl->data;
		City *city;

		city = get_city_from_path (alt->path);

		if (city == NULL) {
			g_print ("error:ref_count_down:city not found:%s\n", alt->path);
		} else {
			city->ref_count -= 1;
		}
	}
}



static gpointer
mmapdata_thread_func (gpointer data, gpointer user_data)
{
	WwSmap *smap = data;
	GSList *sl_all;				/* 地図に必要な市の全リスト */
	gdouble x0, y0;
	gdouble x1, y1;

	g_print ("mmapdata_thread_func:\n");

	/* この地図に必要な市のリストを取得 */
	ww_smap_get_rect (smap, &x0, &y0, &x1, &y1);

	g_print ("mmapdata_thread_func:%.5f %.5f  %.5f %.5f\n", x0, y0, x1, y1);

	sl_all = mmapdata_lookup_alt_citylist (x0, y0, x1, y1);

	/* 既にメモリー中にあるものは ref_count を +1 し、ないものはダウンロードしてメモリー中に用意。 */
	check_citylist (sl_all);

	ww_smap_vcn_calc (smap);

	ref_count_down (sl_all);
	g_slist_free (sl_all);		/* リストの中身は開放してはいけない */

	clean_slist_city ();		/* ref_count が０で十分遠いデータを削除 */

	g_object_unref (smap);

	return NULL;
}

void
mmapdata_prepare (WwSmap * smap)
{
	GError *err = NULL;

	g_object_ref (smap);

	g_thread_pool_push (pool_mmapdata, (gpointer) smap, &err);
	if (err != NULL) {
		g_print ("error!:g_thread_pool_push:%s\n", err->message);
		exit (1);
	}
}

void
mmapdata_init (void)
{
	GError *err = NULL;

	mmapdata_create_alt_db ();

	pool_mmapdata = g_thread_pool_new ((GFunc) mmapdata_thread_func, NULL, 1, TRUE, &err);
	if (err != NULL) {
		g_print ("error:mmapdata_init:g_thread_pool_new:%s\n", err->message);
		exit (-1);
	}
}

/* lon,lat の単位は度 */
gdouble
mmapdata_get_alt (gdouble lon, gdouble lat)
{
	GSList *sl;
	gint key;
	gint x, y;
	gdouble alt = -1.0;

	/* lon, lat ともに整数で奇数のはずだがキャスト時に偶数になることがあるので 0.01 足してある */
	x = (gint) (lon * 3600.0 + 0.01);
	y = (gint) (lat * 3600.0 + 0.01);

	/*
	   g_print ("mmapdata_get_alt:x:%d y:%d :::::: ", x, y);
	 */

	key = create_hash_key_from_keidoido (x, y);

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		City *city = sl->data;
		KeyVal *keyval;

		keyval = g_hash_table_lookup (city->hash, &key);
		if (keyval == NULL) {
			;
		} else {
			alt = keyval->z;
		}
	}

	/*
	   g_print ("alt:%.2f\n", alt);
	 */

	return alt;
}
