/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*
*/
#define DB(x)
#define DB_MORE(x)

#include <glib.h>
#include <glib/gprintf.h>
#include <stdio.h>				/* sscanf() */
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "data.h"
#include "gsi.h"

static gdouble M_2_PI_360 = 2.0 * G_PI / 360.0;

/**************************************************************************/
/* 楕円体の定数 */
static gdouble GRS_A = 6378137.0;	/* 長半径(m)         6,378,137.00m */
static gdouble GRS_B = 6356752.31;	/* 短半径(m)(a と e から計算)  6,356,752.31m */
/* static gdouble GRS_E = 1.0 / 298.257222101;*//* 扁平率 (e=(a-b)/a) */

/* GRS80 の楕円体の赤道断面をＸＹ平面とする。東経０がＸ軸、東経９０がＹ軸。地軸がＺ軸で、北極側がプラス。 */

/* 受け取る緯度は、単位が度！！他と違うので注意！！
 * 原点を通りＸ軸との角度ａの直線と楕円の交点ＸＹを求める。ただし、Ｘ>0 Ｙ>0。
 * ａが緯度、Ｘが楕円体を緯度ａで切った断面の円の半径、Ｙが MMap の地球座標の Ｚ。 */
static void
xy_kouten (gdouble ido_deg, gdouble alt, gdouble * x, gdouble * y)
{
	gdouble a = GRS_A + alt;
	gdouble b = GRS_B + alt;
	gdouble t;
	gdouble tmp;
	/*
	   t = tan (2 * G_PI * ido_deg / 360.0);
	 */
	t = tan (M_2_PI_360 * ido_deg);

	tmp = sqrt (1.0 / (a * a) + (t * t) / (b * b));

	*x = 1.0 / tmp;
	*y = t * (*x);
}

/* 経度(秒)、緯度(秒)、標高(m)を受け取り、ＸＹＺの座標(m)を得る */
void
gsi_keidoido_to_xyz (gdouble keido, gdouble ido, gdouble alt, gdouble * x, gdouble * y, gdouble * z)
{
	gdouble r;					/* 楕円体をある緯度で切った時にできる円の半径(m) */

	/* 単位を秒から度へかえる */
	keido /= 3600.0;
	ido /= 3600.0;

	DB (g_print ("gsi_keidoido_to_xyz:%.1f %.1f %.1f\n", keido, ido, alt));

	xy_kouten (ido, alt, &r, z);

	/*
	 *x = r * cos (2.0 * G_PI * keido / 360.0);
	 *y = r * sin (2.0 * G_PI * keido / 360.0);
	 */
	*x = r * cos (M_2_PI_360 * keido);
	*y = r * sin (M_2_PI_360 * keido);

	DB (g_print ("gsi_keidoido_to_xyz:x:%.1f y:%.1f z:%.1f\n", *x, *y, *z));
}

/***************************************/
/*
 * データの基本形は３つ。
 * (1) AA(名前){中身}     名前付きデータで他から参照できる
 * (2) AA{中身}           名前なしデータ
 * (3) AA(名前)           他のデータへの参照
 *
 * 名前も中身もデータで、名前は(2)のデータを１つだけ持ち、中身は(1)(2)(3)のデータ
 * を複数持つ。名前には以下の２種類がある。
 * ・ID{AA****}     (1) の名前として使われる
 * ・IR{AA****}     (3) の名前として使われる
 *
 * データは１件１行で書かれている。
 */

/* { か ( を指すアドレスをもらって、その括弧に対応する閉じ括弧までの文字列を data にいれて返す。
 * 返り値は閉じ括弧の次のアドレスを返す。 */
static const gchar *
get_char (const gchar * text, gchar ** data)
{
	gchar kakko;
	gchar *p;
	gint nest = 0;
	/*
	   DB(g_print("get_char:%s:\n",text));
	 */
	if (text[0] == '(') {
		kakko = ')';
	} else if (text[0] == '{') {
		kakko = '}';
	} else {
		DB (g_print ("err:get_char:%s:\n", text));
		exit (1);
	}

	p = (gchar *) (text + 1);
	while (*p != '\0') {
		{
			guint8 c;

			c = (guint8) (*p);

			/* 次の文字が SJIS の２バイト文字ならスキップ */
			if ((c >= 0x81 && c <= 0x9f) || (c >= 0xe0 && c <= 0xfc)) {
				p += 2;
				continue;
			}
		}

		if (*p == text[0]) {
			++nest;
		} else if (*p == kakko) {
			if (nest == 0) {
				gint n;

				n = p - text - 1;
				*data = g_strndup (text + 1, n);

				return p + 1;
			} else {
				--nest;
			}
		}
		++p;
	}

	return p;
}

/* text はデータの先頭アドレスを指し、終端が '\0' である必要がある。返り値は次のデータの先頭アドレス。 */
const gchar *
gsi25k_get_data (const gchar * text, Data25k * data25k)
{
	gchar *name;
	gchar *value;
	const gchar *next;

	DB (g_print ("gsi25k_get_data:%s\n", text));

	data25k->id[0] = text[0];
	data25k->id[1] = text[1];
	data25k->id[2] = '\0';

	if (text[2] == '{') {
		DB (g_print ("without_name\n"));
		name = NULL;
		next = get_char (&(text[2]), &value);

	} else if (text[2] == '(' && text[4] == 'D') {
		DB (g_print ("with_name\n"));
		next = get_char (&(text[2]), &name);
		next = get_char (next, &value);

	} else if (text[2] == '(' && text[4] == 'R') {
		DB (g_print ("ref\n"));
		next = get_char (&(text[2]), &name);
		value = NULL;

	} else {
		g_print ("err:gsi25k_get_data:%s:\n", text);

		exit (1);
	}

	data25k->name = name;
	data25k->value = value;

	DB (g_print ("data25k_get_data:id   :%s:\n", data25k->id));
	DB (g_print ("data25k_get_data:name :%s:\n", data25k->name));
	DB (g_print ("data25k_get_data:value:%s:\n", data25k->value));

	return next;
}

/********************************************************/

/* インデックスのカンマ区切りの文字列を受けとり、座標の配列を作って返す。 */
void
gsi25k_print_curve (const gchar * curve, Slp * slp, GIOChannel * ch_edge, GIOChannel * ch_node, gint multi)
{
	gint n = 0;
	gint i;
	const gchar *p;
	gchar buf[10];

	/* 数える。curve は ６桁の数字のカンマ区切りで終端 '\0' */
	n = (strlen (curve) + 1) / 7;
	DB (g_print ("get_curve:n:%d:\n", n));

	g_sprintf (buf, "%d,", multi);
	g_io_channel_write_chars (ch_edge, buf, -1, NULL, NULL);

	for (i = 0, p = curve; i < n; ++i, p += 7) {
		gint index = 0;

		index += (p[0] - '0') * 100000;
		index += (p[1] - '0') * 10000;
		index += (p[2] - '0') * 1000;
		index += (p[3] - '0') * 100;
		index += (p[4] - '0') * 10;
		index += (p[5] - '0') * 1;

		--index;				/* 国土地理院のインデックスは１から、MSMap は０から */
		/* g_print("%s", slp->v[index]); */

		/* エッジの書き出し */
		g_io_channel_write_chars (ch_edge, slp->v[index], -1, NULL, NULL);
		if (i != n - 1) {
			g_io_channel_write_chars (ch_edge, ",", -1, NULL, NULL);
		}

		if (i == 0 || i == n - 1) {
			g_io_channel_write_chars (ch_node, slp->v[index], -1, NULL, NULL);
			g_io_channel_write_chars (ch_node, "\n", -1, NULL, NULL);
		}
	}
	g_io_channel_write_chars (ch_edge, "\n", -1, NULL, NULL);
}
