/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>

#include <math.h>
#include <stdlib.h>
#include <GL/gl.h>

#include "config.h"
#include "mmap.h"
#include "simplemath.h"
#include "ww_quadtile.h"
#include "ww_quadchild.h"
#include "ww_texturetile.h"
#include "ww_textureloader.h"
#include "ww_meshst.h"
#include "ww_mesh.h"
#include "ww_meshloader.h"
#include "camera.h"
#include "util.h"
#include "glarea.h"
#include "mmapthread.h"
#include "ww_terrainaccessor.h"

typedef enum {
	WW_QUADCHILD_STATE_INIT,	/* 子レベルに引き継いで自分自身は空の状態 */
	WW_QUADCHILD_STATE_TEMP,	/* 親から引き継いだテクスチャー番号を使っている状態 */
	WW_QUADCHILD_STATE_ORIG0,	/* 本来の自分のレベルのテクスチャー番号を持っている状態 */
	WW_QUADCHILD_STATE_ORIG1,	/* 一つ上のレベルのテクスチャー番号を持っている状態 */
} WwQuadchildState;

typedef enum {
	WW_QUADCHILD_AREA0,			/* 視点中心からタイル幅の 2.5倍の距離まで。詳細なテクスチャーを使う。*/
	WW_QUADCHILD_AREA1,			/* 視点中心からタイル幅の 6倍の距離まで。テクスチャーのレベルを一つ下げる */
	WW_QUADCHILD_AREA2,			/* さらに外。視野の外 */
} WwQuadchildArea;

typedef struct _WwQuadchildPrivate WwQuadchildPrivate;

struct _WwQuadchildPrivate {
	gboolean dispose_has_run;

	WwQuadchildState state;

	gint x;
	gint y;
	gint level;
	gdouble tile_deg;

	WwQuadchild *parent;
	WwQuadchild *north_west;
	WwQuadchild *north_east;
	WwQuadchild *south_west;
	WwQuadchild *south_east;

	gchar *path_dds;
	gchar *path_jpg;
	gchar *uri;
	WwTexture *texture;
	WwTextureloader *textureloader;
	WwMeshst *meshst;
	WwMesh *mesh;
	WwMeshloader *meshloader;
	GMutex *mutex_texture_and_mesh;

	gboolean need;

	WwQuadtile *quadtile;

	WwTerrainaccessor *ta;
};

#define WW_QUADCHILD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_QUADCHILD, WwQuadchildPrivate))

static GObjectClass *parent_class = NULL;

static void ww_quadchild_class_init (WwQuadchildClass * klass);
static void ww_quadchild_init (WwQuadchild * object);
static void ww_quadchild_finalize (GObject * object);
static void ww_quadchild_dispose (GObject * object);
static GObject *ww_quadchild_constructor (GType type, guint n_props, GObjectConstructParam * props);

static gint SPLIT = 40;

/*****************************************************************************/

GType
ww_quadchild_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwQuadchildClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_quadchild_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwQuadchild),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_quadchild_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwQuadchild", &info, 0);
		/*
		   g_print ("ww_quadchild_get_type\n");
		 */
	}

	return type;
}

static void
ww_quadchild_class_init (WwQuadchildClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	/*
	   g_print ("ww_quadchild_class_init:c:%p:\n", klass);
	 */
	g_type_class_add_private (klass, sizeof (WwQuadchildPrivate));

	parent_class = g_type_class_peek_parent (klass);

	object_class->constructor = ww_quadchild_constructor;
	object_class->finalize = ww_quadchild_finalize;
	object_class->dispose = ww_quadchild_dispose;
}

static void
ww_quadchild_init (WwQuadchild * self)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (self);
	/*  
	   g_print ("ww_quadchild_init:o:%p:\n", self);
	 */
	priv->dispose_has_run = FALSE;

	priv->state = WW_QUADCHILD_STATE_INIT;
	priv->x = 0;
	priv->y = 0;
	priv->level = 0;
	priv->tile_deg = 0.0;
	priv->parent = NULL;
	priv->north_west = NULL;
	priv->north_east = NULL;
	priv->south_west = NULL;
	priv->south_east = NULL;
	priv->quadtile = NULL;

	priv->path_dds = NULL;
	priv->path_jpg = NULL;
	priv->uri = NULL;
	priv->texture = NULL;
	priv->textureloader = NULL;
	priv->meshst = NULL;
	priv->mesh = NULL;
	priv->meshloader = NULL;
	priv->mutex_texture_and_mesh = g_mutex_new ();

	priv->need = TRUE;

	priv->ta = NULL;
}

static void
ww_quadchild_dispose (GObject * obj)
{
	WwQuadchild *self = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (self);
    /*
	g_print ("ww_quadchild_dispose:x:%d y:%d level:%d\n", priv->x, priv->y, priv->level);
    */

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

/*
	if (priv->parent != NULL) {
		g_object_unref (priv->parent);
	}
    */

	if (priv->north_west != NULL) {
		g_object_unref (priv->north_west);
	}
	if (priv->north_east != NULL) {
		g_object_unref (priv->north_east);
	}
	if (priv->south_west != NULL) {
		g_object_unref (priv->south_west);
	}
	if (priv->south_east != NULL) {
		g_object_unref (priv->south_east);
	}

	g_object_unref (priv->quadtile);

	if (priv->texture != NULL) {
		g_object_unref (priv->texture);
	}
	if (priv->textureloader != NULL) {
		g_object_unref (priv->textureloader);
	}
	if (priv->meshst != NULL) {
		g_object_unref (priv->meshst);
	}
	if (priv->mesh != NULL) {
		g_object_unref (priv->mesh);
	}
	if (priv->meshloader != NULL) {
		g_object_unref (priv->meshloader);
	}

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_quadchild_finalize (GObject * obj)
{
	WwQuadchild *self = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (self);

	/*
	   g_print ("ww_quadchild_finalize\n");
	 */

	g_free (priv->path_dds);
	g_free (priv->path_jpg);
	g_free (priv->uri);

	g_mutex_free (priv->mutex_texture_and_mesh);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_quadchild_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);
	/*
	   g_print ("ww_quadchild_constructor\n");
	 */
	object = object_class->constructor (type, n_props, props);

	return object;
}

WwQuadchild *
ww_quadchild_new (void)
{
	GObject *object = g_object_new (WW_TYPE_QUADCHILD, NULL);
	/*
	   g_print ("ww_quadchild_new:o:%p\n", object);
	 */
	return WW_QUADCHILD (object);
}

/********************************************************************/

static void
get_coordinate (WwQuadchild * child, gdouble * west, gdouble * south, gdouble * east, gdouble * north)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	*west = (priv->x + 0) * priv->tile_deg - 180.0;
	*east = (priv->x + 1) * priv->tile_deg - 180.0;
	*south = (priv->y + 0) * priv->tile_deg - 90.0;
	*north = (priv->y + 1) * priv->tile_deg - 90.0;

	/*
	g_print("get_coordinate:l:%d  x:%d y:%d  tile:%.2f %.2f %.2f %.2f %.2f\n", 
                                priv->level, priv->x, priv->y, priv->tile_deg, *west, *east, *south, *north);
	*/
}

static void
create_texture_sub (WwQuadchild * child, WwQuadchild * parent)
{
	WwQuadchildPrivate *priv_c = WW_QUADCHILD_GET_PRIVATE (child);
	WwQuadchildPrivate *priv_p;
	WwTextureloader *loader;
	gdouble x0, y0, x1, y1;

    if(parent == NULL){
	    get_coordinate (child, &x0, &y0, &x1, &y1);
	    priv_p = WW_QUADCHILD_GET_PRIVATE (child);
    }else{
	    get_coordinate (parent, &x0, &y0, &x1, &y1);
	    priv_p = WW_QUADCHILD_GET_PRIVATE (parent);
    }

    /*
	g_print ("create_texture_sub:%.2f %.2f %.2f %.2f\n", x0, y0, x1, y1);
    */

	loader = ww_textureloader_lookup (priv_p->uri);
	if (loader == NULL) {
		WwTexturetile *texturetile = ww_texturetile_new ();

		ww_texturetile_set_coordinate (texturetile, x0, y0, x1, y1);

        if(priv_c->textureloader != NULL){
            g_object_unref(priv_c->textureloader);
        }
		priv_c->textureloader = ww_textureloader_new ();
		ww_textureloader_set_source (priv_c->textureloader, priv_p->path_dds, priv_p->path_jpg, priv_p->uri, 512, 512);
		ww_textureloader_set_texture (priv_c->textureloader, WW_TEXTURE (texturetile));
		ww_textureloader_start (priv_c->textureloader);

	} else {
		g_object_ref (loader);
		priv_c->textureloader = loader;
	}
}

static WwMeshst *
create_meshst_sub_copy (WwTexture * texture, WwQuadchild * child)
{
	WwMeshst *st = NULL;

	gdouble tx0, ty0, tx1, ty1;
	gdouble cx0, cy0, cx1, cy1;

	ww_texturetile_get_coordinate (WW_TEXTURETILE (texture), &tx0, &ty0, &tx1, &ty1);
	get_coordinate (child, &cx0, &cy0, &cx1, &cy1);

    /*
	g_print ("create_meshst_sub_copy:t::%.2f %.2f %.2f %.2f\n", tx0, ty0, tx1, ty1);
	g_print ("                      :c::%.2f %.2f %.2f %.2f\n", cx0, cy0, cx1, cy1);
    */

	st = ww_meshst_new_complex (SPLIT, SPLIT, tx0, ty0, tx1, ty1, cx0, cy0, cx1, cy1);

	return st;
}

static WwMesh *
create_mesh_sub_copy (WwQuadchild * qc_mesh, WwQuadchild * child)
{
	WwQuadchildPrivate *priv_qc_mesh = WW_QUADCHILD_GET_PRIVATE (qc_mesh);
	WwQuadtile *tile = priv_qc_mesh->quadtile;
	gdouble above_surface = ww_quadtile_get_above_surface (tile);
	gboolean terrain_mapped = ww_quadtile_get_terrain_mapped (tile);

	WwMesh *mesh = NULL;

	gdouble mx0, my0, mx1, my1;
	gdouble cx0, cy0, cx1, cy1;

	get_coordinate (qc_mesh, &mx0, &my0, &mx1, &my1);
	get_coordinate (child, &cx0, &cy0, &cx1, &cy1);

    /*
	g_print ("create_mesh_sub_copy:m::%.2f %.2f %.2f %.2f\n", mx0, my0, mx1, my1);
	g_print ("                    :c::%.2f %.2f %.2f %.2f\n", cx0, cy0, cx1, cy1);
    */

	mesh = ww_mesh_new_copy (mx0, my0, mx1, my1, cx0, cy0, cx1, cy1, SPLIT, SPLIT, above_surface, terrain_mapped,
						  priv_qc_mesh->mesh);

	return mesh;
}

static WwMeshst *
create_meshst_sub (WwQuadchild * child, WwQuadchild *parent)
{
	gdouble px0, py0, px1, py1;
	gdouble cx0, cy0, cx1, cy1;
	WwMeshst *meshst = NULL;

    /*
	g_print ("create_meshst_sub:L:%d x:%d y:%d\n", priv_c->level, priv_c->x, priv_c->y);
    */

	get_coordinate (child, &cx0, &cy0, &cx1, &cy1);
    if(parent == NULL){
        px0 = cx0;
        py0 = cy0;
        px1 = cx1;
        py1 = cy1;
    }else{
	    get_coordinate (parent, &px0, &py0, &px1, &py1);
    }

	meshst = ww_meshst_new_complex (SPLIT, SPLIT, px0, py0, px1, py1, cx0, cy0, cx1, cy1);

    return meshst;
}

static void
create_mesh_sub (WwQuadchild * child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	WwQuadtile *tile = priv->quadtile;
	gdouble x0, y0, x1, y1;
	gdouble above_surface = ww_quadtile_get_above_surface (tile);
	gboolean terrain_mapped = ww_quadtile_get_terrain_mapped (tile);

    /*
	g_print ("create_mesh_sub:L:%d x:%d y:%d\n", priv->level, priv->x, priv->y);
    */

	get_coordinate (child, &x0, &y0, &x1, &y1);

	if (priv->mesh == NULL) {
		WwMesh *mesh = NULL;

		mesh = ww_mesh_new_with_terrain (x0, y0, x1, y1, SPLIT, SPLIT, above_surface, NULL);
		g_mutex_lock (priv->mutex_texture_and_mesh);
		{
			priv->mesh = mesh;
		}
		g_mutex_unlock (priv->mutex_texture_and_mesh);

	}

	if (terrain_mapped == TRUE) {
		priv->meshloader = ww_meshloader_new ();
		ww_meshloader_set_source (priv->meshloader, SPLIT, SPLIT, x0, y0, x1, y1, above_surface, priv->ta);
		ww_meshloader_start (priv->meshloader);
	}
}

WwQuadchild *
ww_quadchild_new_simple (WwQuadtile * tile, WwQuadchild * parent, gint x, gint y, gint level, gdouble tile_deg)
{
	WwQuadchild *child;
	WwQuadchildPrivate *priv;
	gchar *path;
	const gchar *server_url = ww_quadtile_get_server_url (tile);
	const gchar *data_set_name = ww_quadtile_get_data_set_name (tile);
	const gchar *world_name = mmap_get_world_name ();
	const gchar *ext = ww_quadtile_get_image_file_extension (tile);
	/*
	   g_print("ww_quadchild_new_simple:x:%d y:%d level:%d\n", x, y, level);
	 */
	child = ww_quadchild_new ();
	priv = WW_QUADCHILD_GET_PRIVATE (child);
	priv->state = WW_QUADCHILD_STATE_INIT;
	priv->x = x;
	priv->y = y;
	priv->level = level;
	priv->tile_deg = tile_deg;
	priv->quadtile = tile;
	g_object_ref (tile);

	priv->parent = parent;
	/*
	   if (parent != NULL) {
	   g_object_ref (parent);
	   }
	 */

	path = ww_object_get_path (WW_OBJECT (tile));
	priv->path_dds = g_strdup_printf ("%s%s/%s%d/%04d/%04d_%04d.dds", mmap_dir_cache, world_name, path, level, y, y, x);
	priv->path_jpg = g_strdup_printf ("%s%s/%s%d/%04d/%04d_%04d.%s", mmap_dir_cache, world_name, path, level, y, y, x, ext);
	priv->uri = g_strdup_printf ("%s?T=%s&L=%d&X=%d&Y=%d", server_url, data_set_name, level, x, y);
	/*
	   g_print ("ww_quadchild_new_simple:path_dds:%s\n", priv->path_dds);
	 */

	priv->ta = mmap_get_terrainaccessor (x * tile_deg, y * tile_deg, (x + 1) * tile_deg, (y + 1) * tile_deg);

	g_free (path);

	return child;
}

void
ww_quadchild_clear_flag (WwQuadchild * child)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);

	priv->need = FALSE;

	ww_quadchild_clear_flag (priv->north_west);
	ww_quadchild_clear_flag (priv->north_east);
	ww_quadchild_clear_flag (priv->south_west);
	ww_quadchild_clear_flag (priv->south_east);
}


static gboolean
is_in_view (gint x, gint y, gdouble tile_deg)
{
	gint x0, y0;
	gint x1, y1;
	gdouble north, south, west, east;

	camera_get_view_wide (&west, &south, &east, &north);
	util_tile_get_view (west, south, east, north, tile_deg, &x0, &y0, &x1, &y1);
	/*
	   g_print("is_in_view:tile_deg:%.2f (%d,%d)::x %d-%d  y %d-%d\n", tile_deg, x,y,x0,x1,y0,y1);
	 */

	if (x0 < x1) {
		if (x >= x0 && x <= x1 && y >= y0 && y <= y1) {
			return TRUE;
		} else {
			return FALSE;
		}
	} else {
		if ((x >= x0 || x <= x1) && y >= y0 && y <= y1) {
			return TRUE;
		} else {
			return FALSE;
		}
	}
}

void ww_quadchild_set_need_flag (WwQuadchild * child, gboolean need_flag);

static void
child_flag_set (WwQuadchild * self, gboolean value)
{
	WwQuadchildPrivate *priv;

	if (self == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (self);

	priv->need = value;

	ww_quadchild_set_need_flag (priv->north_west, value);
	ww_quadchild_set_need_flag (priv->north_east, value);
	ww_quadchild_set_need_flag (priv->south_west, value);
	ww_quadchild_set_need_flag (priv->south_east, value);
}

static WwQuadchild *
get_upper_level_texture_and_mesh (WwQuadchild * qc)
{
	WwQuadchildPrivate *priv;

	if (qc == NULL) {
		return NULL;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (qc);

    /*
	g_print ("get_upper_level_texture_and_mesh:%d\n", priv->level);
    */

	if (priv->texture != NULL && ww_texture_is_ok (priv->texture) == TRUE) {
		return qc;
	} else {
		return get_upper_level_texture_and_mesh (priv->parent);
	}
}

/* child で渡された子レベルについて次の処理をする。
 * 範囲内なら update()。その際、子レベルが未作成なら作成する。
 * 範囲外なら、その下の必要フラグを FALSE にする。
 */
static WwQuadchild *
check_child_level (WwQuadchild * child, gint x, gint y, WwQuadchild * parent, gint level)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (parent);
	gint next_level = priv->level + 1;
	gdouble next_tile_deg = priv->tile_deg / 2.0;

	if (is_in_view (x, y, next_tile_deg) == TRUE) {
		if (child == NULL) {
			child = ww_quadchild_new_simple (priv->quadtile, parent, x, y, next_level, next_tile_deg);
		}
		ww_quadchild_update (child, level);

	} else {
		child_flag_set (child, FALSE);
	}

	return child;
}

static WwQuadchildArea
check_area (WwQuadchild * child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	gdouble lon_camera, lat_camera;
	gdouble lon_qc, lat_qc;
	gdouble l;

	lon_camera = camera_get_lon ();
	lat_camera = camera_get_lat ();

	lon_qc = (priv->x + 0.5) * priv->tile_deg - 180.0;
	lat_qc = (priv->y + 0.5) * priv->tile_deg - 90.0;

    /*
	g_print ("check_area:x %d y:%d     tiledeg:%.2f\n", priv->x, priv->y, priv->tile_deg);
	g_print ("check_area:camera:%.2f %.2f   tile:%.2f %.2f\n", lon_camera, lat_camera, lon_qc, lat_qc);
    */
	l = distance_degree (lon_camera, lat_camera, lon_qc, lat_qc);

	if (l > priv->tile_deg * 6.0) {
        /*
		g_print ("check_area: area2 :l:%.2f > 6 * %.2f\n", l, priv->tile_deg);
        */

		return WW_QUADCHILD_AREA2;

	} else if (l > priv->tile_deg * 2.5) {
        /*
		g_print ("check_area: area1 :l:%.2f > 3 * %.2f\n", l, priv->tile_deg);
        */

		return WW_QUADCHILD_AREA1;

	} else {
        /*
		g_print ("check_area: area0 :l:%.2f < 3 * %.2f\n", l, priv->tile_deg);
        */

		return WW_QUADCHILD_AREA0;
	}
}

void
check_textureloader(WwQuadchild *child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	if (priv->textureloader != NULL) {
	    WwTexture *texture;

		texture = ww_textureloader_get_texture (priv->textureloader);
		if (texture != NULL) {
			g_object_ref (texture);
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				if (priv->texture != NULL) {
					g_object_unref (priv->texture);
				}
				priv->texture = texture;

				g_object_unref (priv->textureloader);
				priv->textureloader = NULL;

                if(priv->state == WW_QUADCHILD_STATE_ORIG0){
				    ww_meshst_change_simple (priv->meshst);
                }else{
                    g_object_unref(priv->meshst);
                    priv->meshst = create_meshst_sub (child, priv->parent);
                }

			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);

			glarea_force_update_and_render ();
		}
	}
}

void
check_meshloader(WwQuadchild *child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	if (priv->meshloader != NULL) {
		WwMesh *mesh;

		mesh = ww_meshloader_get_mesh (priv->meshloader);
		if (mesh != NULL) {
			g_object_ref (mesh);
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				if (priv->mesh != NULL) {
					g_object_unref (priv->mesh);
				}
				priv->mesh = mesh;

				g_object_unref (priv->meshloader);
				priv->meshloader = NULL;
			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);

			glarea_force_update_and_render ();
		}
	}
}


/*
#define DB_UP(x) (x)
*/
#define DB_UP(x)

/* 範囲に入ってないと quadchild は作られないので、ここにくるということは、少なくとも範囲に入っている。
 * 必要なレベルのみ texture と mesh を持つ。
 */

void
ww_quadchild_update (WwQuadchild * child, gint level)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	/*
	   g_print ("ww_quiadchild_update:l_target:%d   l:%d x:%d y:%d  ", level, priv->level, priv->x, priv->y);
	   if (priv->state == WW_QUADCHILD_STATE_INIT) {
	   g_print ("INIT\n");
	   } else if (priv->state == WW_QUADCHILD_STATE_TEMP) {
	   g_print ("TEMP\n");
	   } else if (priv->state == WW_QUADCHILD_STATE_ORIG0) {
	   g_print ("ORIG0\n");
	   } else if (priv->state == WW_QUADCHILD_STATE_ORIG1) {
	   g_print ("ORIG1\n");
	   } else {
	   g_print ("????\n");
	   }
	 */

    /* ここにくるということは範囲内なので必要 */
	priv->need = TRUE;

	if (priv->level < level) {
		/* 目標のレベルより上なら、下のレベル作ったり、範囲外なら必要フラグを落としたりしながら潜っていく。
		 * 潜っていく途中でテクスチャーやメッシュは作らない。目的のレベルの update で上に拾いにくる。 */
		gint x2 = 2 * priv->x;
		gint y2 = 2 * priv->y;

		priv->state = WW_QUADCHILD_STATE_INIT;

		priv->north_west = check_child_level (priv->north_west, x2 + 0, y2 + 1, child, level);
		priv->north_east = check_child_level (priv->north_east, x2 + 1, y2 + 1, child, level);
		priv->south_west = check_child_level (priv->south_west, x2 + 0, y2 + 0, child, level);
		priv->south_east = check_child_level (priv->south_east, x2 + 1, y2 + 0, child, level);

	} else {
		WwQuadchildArea area = check_area (child);

		if (area == WW_QUADCHILD_AREA0) {
			if (priv->state == WW_QUADCHILD_STATE_ORIG0) {
                check_textureloader(child);
                check_meshloader(child);

				/* texture と mesh のどちらかが完成していなかったら、子レベルを消さないように必要フラグを付ける */
				if (priv->texture == NULL || priv->mesh == NULL) {
					child_flag_set (child, TRUE);
				}

			} else if (priv->state == WW_QUADCHILD_STATE_ORIG1) {
                check_textureloader(child);
                check_meshloader(child);

				create_texture_sub (child, NULL);

				glarea_force_update_and_render ();

				priv->state = WW_QUADCHILD_STATE_ORIG0;

			} else if (priv->state == WW_QUADCHILD_STATE_TEMP) {
				create_texture_sub (child, NULL);
				create_mesh_sub (child);

				glarea_force_update_and_render ();

				priv->state = WW_QUADCHILD_STATE_ORIG0;

			} else if (priv->state == WW_QUADCHILD_STATE_INIT) {
				WwQuadchild *qc_mesh;

				/* mesh と texture を持つ quadchild。ここの mesh を使う。ここの texture はさらに上の階層のものの可能性もある。*/
				qc_mesh = get_upper_level_texture_and_mesh (priv->parent);

				child_flag_set (child, TRUE);

				if (qc_mesh == NULL) {
					create_texture_sub (child, NULL);

					g_mutex_lock (priv->mutex_texture_and_mesh);
					{
                        if(priv->meshst != NULL){
                            g_object_unref(priv->meshst);
                        }
					    priv->meshst = create_meshst_sub (child, NULL);
					}
					g_mutex_unlock (priv->mutex_texture_and_mesh);

					create_mesh_sub (child);

					priv->state = WW_QUADCHILD_STATE_ORIG0;

				} else {
					WwQuadchildPrivate *priv_qc_mesh = WW_QUADCHILD_GET_PRIVATE (qc_mesh);

					g_mutex_lock (priv->mutex_texture_and_mesh);
					{
						priv->texture = ww_texture_copy (priv_qc_mesh->texture);
						priv->meshst = create_meshst_sub_copy (priv_qc_mesh->texture, child);
						priv->mesh = create_mesh_sub_copy (qc_mesh, child);
					}
					g_mutex_unlock (priv->mutex_texture_and_mesh);

					priv->state = WW_QUADCHILD_STATE_TEMP;
				}
				glarea_force_update_and_render ();
			}

		} else if (area == WW_QUADCHILD_AREA1) {
			if (priv->state == WW_QUADCHILD_STATE_ORIG0) {
                check_textureloader(child);
                check_meshloader(child);

				create_texture_sub (child, priv->parent);

				glarea_force_update_and_render ();

				priv->state = WW_QUADCHILD_STATE_ORIG1;

			}else if (priv->state == WW_QUADCHILD_STATE_ORIG1) {
                check_textureloader(child);
                check_meshloader(child);

				/* texture と mesh のどちらかが完成していなかったら、子レベルを消さないように必要フラグを付ける */
				if (priv->texture == NULL || priv->mesh == NULL) {
					child_flag_set (child, TRUE);
				}

			}else if (priv->state == WW_QUADCHILD_STATE_TEMP) {
				create_texture_sub (child, priv->parent);
				create_mesh_sub (child);

				glarea_force_update_and_render ();

				priv->state = WW_QUADCHILD_STATE_ORIG1;

			}else if (priv->state == WW_QUADCHILD_STATE_INIT) {
				WwQuadchild *qc_mesh;

				/* mesh と texture を持つ quadchild。ここの mesh を使う。ここの texture はさらに上の階層のものの可能性もある。*/
				qc_mesh = get_upper_level_texture_and_mesh (priv->parent);

				child_flag_set (child, TRUE);

				if (qc_mesh == NULL) {
					create_texture_sub (child, priv->parent);
					g_mutex_lock (priv->mutex_texture_and_mesh);
					{
                        if(priv->meshst != NULL){
                            g_object_unref(priv->meshst);
                        }
					    priv->meshst = create_meshst_sub (child, NULL);
					}
					g_mutex_unlock (priv->mutex_texture_and_mesh);
					create_mesh_sub (child);

					priv->state = WW_QUADCHILD_STATE_ORIG1;

				} else {
					WwQuadchildPrivate *priv_qc_mesh = WW_QUADCHILD_GET_PRIVATE (qc_mesh);

					g_mutex_lock (priv->mutex_texture_and_mesh);
					{
						priv->texture = ww_texture_copy (priv_qc_mesh->texture);
						priv->meshst = create_meshst_sub_copy (priv_qc_mesh->texture, child);
						priv->mesh = create_mesh_sub_copy (qc_mesh, child);
					}
					g_mutex_unlock (priv->mutex_texture_and_mesh);

					priv->state = WW_QUADCHILD_STATE_TEMP;
				}
				glarea_force_update_and_render ();
            }

		} else if (area == WW_QUADCHILD_AREA2) {
			priv->state = WW_QUADCHILD_STATE_INIT;
		}
	}
}

gboolean
ww_quadchild_get_need_flag (WwQuadchild * child)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return FALSE;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);

	return priv->need;
}

void
ww_quadchild_set_need_flag (WwQuadchild * child, gboolean need)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);

	priv->need = need;
}

/* 必要フラグが落ちているものを unref する */
void
ww_quadchild_remove_child (WwQuadchild * child, gint level)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);
	/*
	   g_print ("ww_quadchild_remove_child:target_level:%d l:%d x:%d y:%d\n", level, priv->level, priv->x, priv->y);
	 */

	if (priv->level < level) {
		/* 目標レベルより浅いところは、全て必要。しかし、texture や mesh はいらない。 */
		/*
		   g_print ("ww_quadchild_remove_child:remove texture & mesh\n");
		 */
		g_mutex_lock (priv->mutex_texture_and_mesh);
		{
			if (priv->texture != NULL) {
				g_object_unref (priv->texture);
				priv->texture = NULL;
			}
			if (priv->textureloader != NULL) {
				g_object_unref (priv->textureloader);
				priv->textureloader = NULL;
			}
			if (priv->meshst != NULL) {
				g_object_unref (priv->meshst);
				priv->meshst = NULL;
			}
			if (priv->mesh != NULL) {
				g_object_unref (priv->mesh);
				priv->mesh = NULL;
			}
			if (priv->meshloader != NULL) {
				g_object_unref (priv->meshloader);
				priv->meshloader = NULL;
			}
		}
		g_mutex_unlock (priv->mutex_texture_and_mesh);
	} else if (priv->level == level) {
		/* 目標レベルそのものは、範囲外のものの texture と mesh を削除。 */
        if(priv->state == WW_QUADCHILD_STATE_INIT){
		    g_mutex_lock (priv->mutex_texture_and_mesh);
		    {
			    if (priv->textureloader != NULL) {
				    g_object_unref (priv->textureloader);
				    priv->textureloader = NULL;
			    }
			    if (priv->meshloader != NULL) {
				    g_object_unref (priv->meshloader);
				    priv->meshloader = NULL;
			    }
				if (priv->texture != NULL) {
					g_object_unref (priv->texture);
					priv->texture = NULL;
				}
				if (priv->meshst != NULL) {
					g_object_unref (priv->meshst);
					priv->meshst = NULL;
				}
				if (priv->mesh != NULL) {
					g_object_unref (priv->mesh);
					priv->mesh = NULL;
				}
		    }
		    g_mutex_unlock (priv->mutex_texture_and_mesh);

        }
	} else {
		/* textureloader と meshloader は破棄 */
		g_mutex_lock (priv->mutex_texture_and_mesh);
		{
			if (priv->textureloader != NULL) {
				g_object_unref (priv->textureloader);
				priv->textureloader = NULL;
			}
			if (priv->meshloader != NULL) {
				g_object_unref (priv->meshloader);
				priv->meshloader = NULL;
			}
		}
		g_mutex_unlock (priv->mutex_texture_and_mesh);

		/* 目標レベルより深いところは、それ自身が最下層な場合だけ、texture や mesh を保持。 */
		if (priv->north_west == NULL && priv->north_east == NULL && priv->south_west == NULL && priv->south_east == NULL) {
			;
		} else {
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				if (priv->texture != NULL) {
					g_object_unref (priv->texture);
					priv->texture = NULL;
				}
				if (priv->meshst != NULL) {
					g_object_unref (priv->meshst);
					priv->meshst = NULL;
				}
				if (priv->mesh != NULL) {
					g_object_unref (priv->mesh);
					priv->mesh = NULL;
				}
			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);
		}
	}

	if (priv->north_west != NULL) {
		if (ww_quadchild_get_need_flag (priv->north_west) == FALSE) {
			g_object_unref (priv->north_west);
			priv->north_west = NULL;
		} else {
			ww_quadchild_remove_child (priv->north_west, level);
		}
	}

	if (priv->north_east != NULL) {
		if (ww_quadchild_get_need_flag (priv->north_east) == FALSE) {
			g_object_unref (priv->north_east);
			priv->north_east = NULL;
		} else {
			ww_quadchild_remove_child (priv->north_east, level);
		}
	}

	if (priv->south_west != NULL) {
		if (ww_quadchild_get_need_flag (priv->south_west) == FALSE) {
			g_object_unref (priv->south_west);
			priv->south_west = NULL;
		} else {
			ww_quadchild_remove_child (priv->south_west, level);
		}
	}

	if (priv->south_east != NULL) {
		if (ww_quadchild_get_need_flag (priv->south_east) == FALSE) {
			g_object_unref (priv->south_east);
			priv->south_east = NULL;
		} else {
			ww_quadchild_remove_child (priv->south_east, level);
		}
	}
}

/*
#define DB_RE(x) (x)
*/
#define DB_RE(x)

void
ww_quadchild_render (WwQuadchild * child)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);

	DB_RE (g_print ("ww_quadchild_render:L:%d x:%d y:%d\n", priv->level, priv->x, priv->y));

	g_mutex_lock (priv->mutex_texture_and_mesh);
	{
		if (priv->texture == NULL || priv->mesh == NULL) {
			;
		} else {
			gboolean textured;
			gboolean is_in_progress = TRUE;
            gint area;

			textured = ww_texture_bind (priv->texture);

			if (priv->state == WW_QUADCHILD_STATE_ORIG0) {
				is_in_progress = FALSE;
			} else if (priv->state == WW_QUADCHILD_STATE_ORIG1) {
				is_in_progress = FALSE;
			} else if (priv->state == WW_QUADCHILD_STATE_TEMP) {
				is_in_progress = TRUE;
			}

            if(priv->state == WW_QUADCHILD_STATE_ORIG0){
                area = 0;
            }else if(priv->state == WW_QUADCHILD_STATE_ORIG1){
                area = 1;
            }else if(priv->state == WW_QUADCHILD_STATE_TEMP){
                area = 2;
            }else{
                area = 3;
            }

			if (textured == TRUE) {
				ww_meshst_bind (priv->meshst);
				ww_mesh_render (priv->mesh, WW_MESH_RENDER_TYPE_TEXTURE, is_in_progress, area);
			} else {
				ww_mesh_render (priv->mesh, WW_MESH_RENDER_TYPE_BOX, is_in_progress, area);
			}
		}
	}
	g_mutex_unlock (priv->mutex_texture_and_mesh);

	ww_quadchild_render (priv->north_west);
	ww_quadchild_render (priv->north_east);
	ww_quadchild_render (priv->south_west);
	ww_quadchild_render (priv->south_east);
}

gboolean
ww_quadchild_check_xy (WwQuadchild * child, gint x, gint y)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	if (priv->x == x && priv->y == y) {
		return TRUE;
	} else {
		return FALSE;
	}
}

void
ww_quadchild_change_vertical_exaggeration (WwQuadchild * child)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);
	/*
	   g_print ("ww_quadchild_change_vertical_exaggeration:\n");
	 */
	ww_mesh_change_vertical_exaggeration (priv->mesh);

	ww_quadchild_change_vertical_exaggeration (priv->north_west);
	ww_quadchild_change_vertical_exaggeration (priv->north_east);
	ww_quadchild_change_vertical_exaggeration (priv->south_west);
	ww_quadchild_change_vertical_exaggeration (priv->south_east);
}
