/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.sound.midi.*;

/**
 * Standard MIDI ファイルを整理するためのクラス. チャンネルごとにメッセージを分ける.
 * @author nazo
 */
public class MidiResolver {
    private Sequence midiSeq;
    private Track [] tracks;
    /** ticks for quota note */
    private int reso;
    /** mil sec for quota note */
    private int tempo=500000;
    private ArrayList<MidiEvent> [] channelList;

    public MidiResolver(File f){
        int i,j,ch;
        MidiEvent me;
        MidiMessage mm;
        ShortMessage sm;
        Track t;
        try {
            midiSeq=MidiSystem.getSequence(f);
            tracks=midiSeq.getTracks();
            reso=midiSeq.getResolution();
            channelList=new ArrayList[16];
            for(i=0;i<tracks.length;i++){
                t=tracks[i];
                for(j=0;j<t.size();j++){
                    me=t.get(j);
                    if(me==null)break;
                    if((mm=me.getMessage())==null)continue;
                    if(mm instanceof ShortMessage){
                        sm=((ShortMessage)mm);
                        ch=sm.getChannel();
                        if(channelList[ch]==null){
                            channelList[ch]=new ArrayList<MidiEvent>();
                        }
                        channelList[ch].add(me);
                    } else if(mm instanceof MetaMessage){
                        MetaMessage mem=((MetaMessage)mm);
                        if(mem.getType()==0x51){
                            byte []tempB=mem.getData();
                            tempo=(((int)tempB[0]&0xff)<<16)
                                    +(((int)tempB[1]&0xff)<<8)
                                    +((int)tempB[2]&0xff);
                        }
                    }
                }
            }
        } catch (InvalidMidiDataException ex) {
            Logger.getLogger(MidiResolver.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(MidiResolver.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    public MidiEvent [] getChannelMessage(int ch){
        if(channelList[ch]==null)
            return null;
        return channelList[ch].toArray(new MidiEvent[channelList[ch].size()]);
    }

    /**
     * @return the midiSeq
     */
    public Sequence getMidiSeq() {
        return midiSeq;
    }

    /**
     * tics per beat.
     * @return the reso
     */
    public int getReso() {
        return reso;
    }

    /**
     * mili second per beat
     * @return the tempo
     */
    public int getTemp() {
        return tempo;
    }

    public void add(MidiResolver mr){
        Track t=tracks[0];
        long length=midiSeq.getTickLength();
        if(length%reso >0){
            length=(length/reso+1)*reso;
        }

        MidiEvent [] mms;
        int ch;
        for(ch=0;ch<16;ch++){
            mms=mr.getChannelMessage(ch);
            if(mms!=null){
                if(channelList[ch]==null){
                    channelList[ch]=new ArrayList<MidiEvent>();                    
                }
                for(MidiEvent me:mms){
                    me.setTick(me.getTick()+length);
                    t.add(me);
                    channelList[ch].add(me);
                }
            }
        }
    }

    /**
     * シーケンスの出力. Starndard MIDI File Format 0 でシーケンスを書き出します.
     * Output Sequence. 
     * @param os 書き込み先
     * @throws IOException 書き込みエラー
     */
    public void write(OutputStream os) throws IOException{
        MidiSystem.write(midiSeq, 0, os);
    }

}
