/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Map;
import java.util.Properties;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.Box;
import jp.sourceforge.mmd.midiMotion.MidiResolver;
import jp.sourceforge.mmd.motion.swing.AboutDialog;
import jp.sourceforge.mmd.motion.swing.VMD2CSV;
import jp.sourceforge.mmd.motion.swing.MmdDialog;

/**
 * Midi to MMD Motion swing GUI main frame. MID2Motion の
 * 一番根本のウィンドウフレーム.
 * @author nazo
 */
public class MidiMotionFrame extends javax.swing.JFrame {
    protected MidiResolver midiR;
    protected File midiDir;
    protected Properties prop;
    protected MotionPane [] mps;
    protected float fps;

    /**
     * Creates new form MidiMotionFrame
     */
    public MidiMotionFrame() {
        try {
            FileInputStream fis=new FileInputStream("./midiMotion.properties");
            prop=new Properties();
            prop.load(fis);
            fis.close();
            for(String s:prop.stringPropertyNames()){
                System.setProperty(s, prop.getProperty(s));
            }
            String s=prop.getProperty("midiMotion.midiDir");
            if(s==null){s="./";}
            midiDir=new File(s);
            MmdDialog.loadDefault();
        } catch (FileNotFoundException ex) {
            midiDir=new File("./");
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this,"IOエラー:設定ファイルの読み込みに失敗\n"+ex.getLocalizedMessage());
        }
        mps=new MotionPane[]{
            new DrumMotionPane(),
            new BassMotionPane(),
            new KeyboardMotionPane(),
            new SampleMotionPane()};
        initComponents();
    }

    private void setFPS(){
        fps=Float.parseFloat(textFPS.getText());
        if(midiR!=null)
            textFPB.setText(String.valueOf(fps*midiR.getTemp()/1000000)+" frame/beet");
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        tabPane = new javax.swing.JTabbedPane();
        javax.swing.JLabel jLabel1 = new javax.swing.JLabel();
        midiFile = new javax.swing.JTextField();
        javax.swing.JLabel jLabel2 = new javax.swing.JLabel();
        textFPS = new javax.swing.JTextField();
        textLength = new javax.swing.JTextField();
        textTemp = new javax.swing.JTextField();
        textBeat = new javax.swing.JTextField();
        textFPB = new javax.swing.JTextField();
        menuBar = new javax.swing.JMenuBar();
        menuFile = new javax.swing.JMenu();
        menuOpen = new javax.swing.JMenuItem();
        menuClose = new javax.swing.JMenuItem();
        menuTools = new javax.swing.JMenu();
        menuVMD2CSV = new javax.swing.JMenuItem();
        menuConcat = new javax.swing.JMenuItem();
        menuHelp = new javax.swing.JMenu();
        menuAbout = new javax.swing.JMenuItem();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        setTitle("MIDI 2 Motion");
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });
        getContentPane().setLayout(new java.awt.GridBagLayout());

        tabPane.setEnabled(false);
        for(MotionPane mp:mps){
            tabPane.add(mp,mp.getName());
        }
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 0.1;
        gridBagConstraints.weighty = 0.1;
        getContentPane().add(tabPane, gridBagConstraints);

        jLabel1.setText("midi");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        getContentPane().add(jLabel1, gridBagConstraints);

        midiFile.setEditable(false);
        midiFile.setText("まだ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        getContentPane().add(midiFile, gridBagConstraints);

        jLabel2.setText("fps");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        getContentPane().add(jLabel2, gridBagConstraints);

        textFPS.setText("30");
        textFPS.addFocusListener(new java.awt.event.FocusAdapter() {
            public void focusLost(java.awt.event.FocusEvent evt) {
                textFPSFocusLost(evt);
            }
        });
        textFPS.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                textFPSActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        getContentPane().add(textFPS, gridBagConstraints);

        textLength.setEditable(false);
        textLength.setText("秒");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        getContentPane().add(textLength, gridBagConstraints);

        textTemp.setEditable(false);
        textTemp.setText("bpm");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        getContentPane().add(textTemp, gridBagConstraints);

        textBeat.setEditable(false);
        textBeat.setText("beat");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        getContentPane().add(textBeat, gridBagConstraints);

        textFPB.setEditable(false);
        textFPB.setText("frame/beat");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        getContentPane().add(textFPB, gridBagConstraints);

        menuFile.setText("File");

        menuOpen.setText("open midi");
        menuOpen.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuOpenActionPerformed(evt);
            }
        });
        menuFile.add(menuOpen);

        menuClose.setText("close");
        menuClose.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuCloseActionPerformed(evt);
            }
        });
        menuFile.add(menuClose);

        menuBar.add(menuFile);

        menuTools.setText("Tools");

        menuVMD2CSV.setText("VMD2CSV");
        menuVMD2CSV.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuVMD2CSVActionPerformed(evt);
            }
        });
        menuTools.add(menuVMD2CSV);

        menuConcat.setText("MIDI Concatenate");
        menuConcat.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuConcatActionPerformed(evt);
            }
        });
        menuTools.add(menuConcat);

        menuBar.add(menuTools);

        menuHelp.setText("help");

        menuAbout.setText("about");
        menuAbout.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuAboutActionPerformed(evt);
            }
        });
        menuHelp.add(menuAbout);

        menuBar.add(Box.createHorizontalGlue());

        menuBar.add(menuHelp);

        setJMenuBar(menuBar);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void menuOpenActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuOpenActionPerformed
        JFileChooser jfc=new JFileChooser(midiDir);
        jfc.setFileFilter(new FileNameExtensionFilter("MIDI", "mid"));
        if(jfc.showOpenDialog(this)!=JFileChooser.APPROVE_OPTION)
            return;
        File f=jfc.getSelectedFile();
        midiDir=f.getParentFile();
        midiR=new MidiResolver(f);
        if(midiR!=null){
            midiFile.setText(f.getName());
            textLength.setText(((double)midiR.getMidiSeq().getMicrosecondLength()/1000000)+"秒");
            textTemp.setText(((double)60000000/midiR.getTemp())+" bpm");
            textBeat.setText(((double)midiR.getMidiSeq().getTickLength()/midiR.getReso())+" beats");
            setFPS();
            tabPane.setEnabled(true);
            for(MotionPane mp:mps){
                mp.setMidiResolver(midiR);
            }
        }
    }//GEN-LAST:event_menuOpenActionPerformed

    private void menuCloseActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuCloseActionPerformed
        confirmClose();
    }//GEN-LAST:event_menuCloseActionPerformed

    private void confirmClose(){
        int i=JOptionPane.showConfirmDialog(this,"終了していいですか","MIDI2Motion",JOptionPane.YES_NO_OPTION);
        if(i!=JOptionPane.OK_OPTION){
            return;
        }
        FileOutputStream fos;
        Properties p=new Properties();
        MmdDialog.saveDefault();

        String key;
        for(Map.Entry<Object,Object> e:System.getProperties().entrySet()){
            key=(String)e.getKey();
            if(key.startsWith("midiMotion.") || key.startsWith("MMDMotion.")){
                p.setProperty(key, (String)e.getValue());
            }
        }
        try {
            p.setProperty("midiMotion.midiDir", midiDir.getCanonicalPath());
            fos = new FileOutputStream("midiMotion.properties");
            p.store(fos, "MidiMotion Setting File");
            fos.close();
        } catch (FileNotFoundException ex) {
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this,"IOエラー:設定ファイルの書き込みに失敗\n"+ex.getLocalizedMessage());
        }
        setVisible(false);
        dispose();
    }

    private void menuVMD2CSVActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuVMD2CSVActionPerformed
        VMD2CSV v2c=new VMD2CSV();
        v2c.setVisible(true);
    }//GEN-LAST:event_menuVMD2CSVActionPerformed

    private void textFPSActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_textFPSActionPerformed
        setFPS();
    }//GEN-LAST:event_textFPSActionPerformed

    private void textFPSFocusLost(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_textFPSFocusLost
        setFPS();
    }//GEN-LAST:event_textFPSFocusLost

    private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
        confirmClose();
    }//GEN-LAST:event_formWindowClosing

    private void menuAboutActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuAboutActionPerformed
         new AboutDialog(this,true,this.getClass(),
                 "MIDI to Motion は Standard MIDI File (.mid) から "
                         + "MMDモーションを生成する JavaライブラリーのGUIです。"
         ).show("MIDI to Motion");
    }//GEN-LAST:event_menuAboutActionPerformed

    private void menuConcatActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuConcatActionPerformed
        new MidiConcat().setVisible(true);
    }//GEN-LAST:event_menuConcatActionPerformed

    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {
        /* Set the Nimbus look and feel */
        //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
        /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
         */
        try {
            for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
                if ("Nimbus".equals(info.getName())) {
                    javax.swing.UIManager.setLookAndFeel(info.getClassName());
                    break;
                }
            }
        } catch (ClassNotFoundException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (InstantiationException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (IllegalAccessException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (javax.swing.UnsupportedLookAndFeelException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        }
        //</editor-fold>

        /* Create and display the form */
        java.awt.EventQueue.invokeLater(new Runnable() {
            public void run() {
                new MidiMotionFrame().setVisible(true);
            }
        });
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JMenuItem menuAbout;
    private javax.swing.JMenuBar menuBar;
    private javax.swing.JMenuItem menuClose;
    private javax.swing.JMenuItem menuConcat;
    private javax.swing.JMenu menuFile;
    private javax.swing.JMenu menuHelp;
    private javax.swing.JMenuItem menuOpen;
    private javax.swing.JMenu menuTools;
    private javax.swing.JMenuItem menuVMD2CSV;
    private javax.swing.JTextField midiFile;
    private javax.swing.JTabbedPane tabPane;
    private javax.swing.JTextField textBeat;
    private javax.swing.JTextField textFPB;
    private javax.swing.JTextField textFPS;
    private javax.swing.JTextField textLength;
    private javax.swing.JTextField textTemp;
    // End of variables declaration//GEN-END:variables
}
