/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;
import java.util.ArrayList;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.pmx.BoneFlags;
import jp.sourceforge.mmd.motion.BonePose;
import jp.sourceforge.mmd.motion.CsvSpliter;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;

/**
 * MMD ボーンの記録と操作.
 * @author nazo
 */
public class Bone {

    /**
     * Bone の回転のみのリンクなどを記録する内部クラス
     */
    final protected class BoneLink{
        final private Bone child;
        final private double linkedT;
        final private double linkedR;

        protected BoneLink(Bone c, double translation, double rotation) {
            child=c;linkedT=translation;linkedR=rotation;
        }
    }

    /**
     * 属するモデル
     */
    protected Model model;

    /**
     * ボーン名
     */
    protected String name=null;

    /**
     * ボーン種類. {@link BoneFlags}であわらわされるボーン種類.
     * MMDMotion ではほとんど無視.
     */
    protected short flags;
    
    /**
     * ボーンID. 一応記録している.
     * MMDMotion ではほとんど無視.
     */
    protected int id;

    /**
     * 親ボーン名.
     */
    private String parent=null;

    /**
     * リンク親ボーン名. (package access)
     */
    private String linkParent=null;
    
    /**
     * 子リスト
     */
    private ArrayList<Bone> children=new ArrayList<Bone>();

    /**
     * リンク子リスト
     */
    private ArrayList<BoneLink> linkChildren=new ArrayList<BoneLink>();

    /**
     * グローバル原点. (package access) (0,0,0)で初期化済み. MMD座標系での初期位置.親に影響されて動く. 
     */
    Vector3D gv=new Vector3D(0,0,0);

    /**
     * グローバル回転行列. 単位行列で初期化済み. 親に影響される.
     */
    private Matrix g_mr=new Matrix();

    /**
     * 初期並進位置. (0,0,0)で初期化済み. 
     */
    private Vector3D v=new Vector3D();

    /**
     * 回転行列. 単位行列で初期化済み.
     */
    private Matrix mr=new Matrix();

    /**
     * 初期回転行列. 単位行列で初期化済み.
     */
    private Matrix ini_mr=new Matrix();

    /**
     * 回転制限軸. なければ {@code null}.
     */
    protected Vector3D limitRot=null;

    /**
     * 変更フラグ. false: 変更されてない.
     */
    protected boolean changed=false;

    /**
     * コンストラクター. 必ず{@link Model Model}に所属すること.
     * {@code null} はだめ.
     * @param m 属するモデル
     */
    public Bone(Model m){
        model=m;
    }

    /**
     * ボーン名の設定.
     * @param name ボーン名.
     */
    public void setName(String name){
        this.name=name;
    }

    /**
     * ボーン名の取得.
     * @return ボーン名.
     */
    public String getName(){
        return name;
    }

    /**
     * ボーンIDの取得. MMDMotion ではまず使わない. 番号.
     * @return ID 番号
     */
    public int getId(){
        return id;
    }

    /**
     * ボーン種類の取得. {@code jp.sfjp.mikutoga.pmx.BoneFlag} と組み合わせて使う.
     * MMDMotion では基本的に無視している情報.
     * @return BoneFlag のshort値
     */
    public short getFlags(){
        return flags;
    }

    /**
     * hash を返す. HashMap など用.
     * @return モデル名+"/"+ボーン名のハッシュ.
     */
    @Override
    public int hashCode(){
        if(model==null)return 0;
        if(name==null)return 0;
        return (model.getName()+"/"+name).hashCode();
    }

    /**
     * 同じかどうかを比較する.
     * @return 同じなら true.
     */
    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Bone other = (Bone) obj;
        if (this.model != other.model && (this.model == null || this.model.getName().compareTo(other.model.getName())!=0 )) {
            return false;
        }
        return !((this.name == null) ? (other.name != null) : this.name.compareTo(other.name)==0);
    }

    /**
     * 親ボーン名の取得.
     * @return 親ボーン名. 無いときは{@code null}.
     */
    public String getParent(){
        return parent;
    }

    /**
     * リンク親ボーン名の取得
     * @return 親ボーン名. 無いときは{@code null}.
     */
    public String getLinkParent() {
        return linkParent;
    }

    /**
     * 子を登録する.
     * 同じモデルに属してなくてもいい.
     * @param child 子になるボーン.
     * @return {@code this} が帰る. 子になるボーンの親ボーン名が設定される.
     */
    public Bone addChild(Bone child){
        child.parent=name;
        children.add(child);
        return this;
    }

    /**
     * 子を取り除く
     * @param child 取り除く子ボーン.
     * @return {@code this} が帰る. 失敗(子が登録されてない)と{@code null} が帰る.
     */
    public Bone removeChild(Bone child){
        int i=children.indexOf(child);
        if(i>=0){
            children.remove(i);
            child.parent=null;
            return this;
        }
        return null;
    }

    /**
     * リンク先を登録する. リンク子にも親名が登録される.
     * @param child リンク子
     * @param translation 並進リンク度
     * @param rotation 回転リンク度
     * @return {@code this}.
     */
    public Bone addLinkChild(Bone child, double translation, double rotation){
        child.linkParent=name;
        linkChildren.add(new BoneLink(child,translation,rotation));
        return this;
    }

    /**
     * モデル初期ローカル座標の設定. package access method.
     * @param lx ローカルx軸
     * @param lz ローカルz軸
     */
    void setInicialLocalCoordinate(Vector3D lx,Vector3D lz) {
        ini_mr =new Matrix(lx, lz);
        mr=new Matrix(ini_mr);
    }
    
    /**
     * グローバルな位置の取得.
     * position in global
     * @return グローバルベクトル.
     */
    public Vector3D getPos(){
        Vector3D g=g_mr.times(v).add(gv);
        return g;
    }

    /**
     * グローバルな回転の取得.
     * coordinate in global
     * @return グローバルから見た回転行列.
     */
    public Matrix getGMatrix(){
        return g_mr.times(mr);
    }
    
    /**
     * ローカルな位置の取得.
     * position in local
     * @return local vector
     */
    public Vector3D getLPos(){
        return new Vector3D(v);
    }

    /**
     * ローカルな回転の取得.
     * coordinate in local
     * @return ローカルな回転行列.
     */
    public Matrix getMatrix(){
        return new Matrix(mr);
    }

    /**
     * 回転制限軸を取得する.
     * @return null だったら、回転に制限はない。帰ってくるVector3Dは回転軸。
     */
    public Vector3D getLimitRot() {
        return limitRot;
    }

    /**
     * 変更があったかどうか
     * @return trueなら変更あり。
     */
    public boolean getChanged(){
        return changed;
    }

    /**
     * 変更フラグを取り消す。
     */
    public void resetChanged(){
        changed=false;
    }

    /**
     * ローカルな位置ベクトルを設定する.
     * @param lv 目標のローカルベクトル.
     */
    public void translationTo(Vector3D lv){
        Vector3D gtv=g_mr.times(lv.sub(v));
        for(Bone c:children){
            c.translationG(gtv);
        }
        for(BoneLink l:linkChildren){
            if(l.linkedT!=0)
                l.child.translationG(gtv.times(l.linkedT));
        }
        v=new Vector3D(lv);
        changed=true;
    }

    /**
     * delta translation in global.
     * @param dv 変位ベクトル
     */
    private void translationG(Vector3D dv){
        for(Bone c:children){
            c.translationG(dv);
        }
        gv=gv.add(dv);
    }

    /**
     * ローカル軸 x (グローバルベクトル)
     * @return global vector
     */ 
    public Vector3D getLx(){
        return g_mr.times(mr.getLx());
    }

    /**
     * ローカル軸 y (グローバルベクトル)
     * @return global vector
     */ 
    public Vector3D getLy(){
        return g_mr.times(mr.getLy());
    }

    /**
     * ローカル軸 z (グローバルベクトル)
     * @return global vector
     */ 
    public Vector3D getLz(){
        return g_mr.times(mr.getLz());
    }

    /**
     * グローバルベクトルで回転.
     * @param axis_sin グローバルベクトルで長さがsine . 0ベクトルなら何もせず戻る
     * @param co cosine 成分 sine^2+cosine^2 == 1 であることを期待
     */
    public void rotate(Vector3D axis_sin,double co){
        double sine=axis_sin.norm();
        if(sine==0)return;

        Vector3D lsin=g_mr.inverse().times(axis_sin);
        mr.rotate(lsin,co);
        Vector3D center=getPos();
        for(Bone c:children){
            c.rotateChild(axis_sin, co,center);
        }
        for(BoneLink l:linkChildren){
            if(l.linkedR!=0){
                double angle=Math.atan2(co,sine)*l.linkedR;
                double []r = axis_sin.times(Math.sin(angle)/sine).toDouble();
                l.child.rotateL(Matrix.rotationQ(r[0], r[1], r[2], Math.cos(angle)));
            }
        }
        changed=true;
    }

    /**
     * 親から呼び出される回転.
     * @param axis_sine グローバルベクトルで長さがsine . 0ベクトルなら何もせず戻る.
     * @param co cosine 成分 sine^2+cosine^2 == 1 であることを期待.
     * @param center 親のグローバル座標
     */
    private void rotateChild(Vector3D axis_sin,double co,Vector3D center){
        Vector3D ta=gv.sub(center);
        Vector3D ra=ta.rotate_vector(axis_sin, co);
        gv=ra.add(center);

        g_mr.rotate(axis_sin,co);
        for(Bone c:children){
            c.rotateChild(axis_sin, co,center);
        }
    }

    /**
     * 回転行列を与えて回転させる.
     * @param dmr デルタ回転行列 in local.
     */
    public void rotate(Matrix dmr){
        mr=dmr.times(mr);
        Matrix gr=g_mr.times(dmr).times(g_mr.inverse());
        Vector3D center=getPos();
        for(Bone c:children){
            c.rotateChild(gr,center);
        }
        for(BoneLink l:linkChildren){
            if(l.linkedR!=0){
                l.child.rotateL(dmr.power(l.linkedR));
            }
        }
        changed=true;
    }

    /**
     * 親から呼び出される回転.
     * @param dmr デルタ回転行列 in global.
     * @param center 回転中心. 大抵は親のボーン位置.
     */
    private void rotateChild(Matrix dmr,Vector3D center){
        Vector3D ta=gv.sub(center);
        Vector3D ra=dmr.times(ta);
        gv=ra.add(center);

        g_mr=dmr.times(g_mr);
        for(Bone c:children){
            c.rotateChild(dmr,center);
        }
    }

    /**
     * ボーンモーフから呼び出される並進と回転. (package access)
     * @param dv デルタ並進 in global
     * @param dmr デルタ回転行列 in global
     */
    void moveMorph(Vector3D dv,Matrix dmr){
        for(Bone c:children){
            c.translationG(dv);
        }
        gv=gv.add(dv);
        Matrix gdmr=g_mr.times(dmr.times(g_mr.inverse()));
        g_mr=gdmr.times(g_mr);// global rotation
        Vector3D center=gv;
        for(Bone c:children){
            c.rotateChild(gdmr,center);
        }
    }

    /**
     * リンク親から呼び出される回転.
     * @param dmr デルタ回転行列 in local.
     */
    private void rotateL(Matrix dmr){
        ini_mr=dmr.times(ini_mr);
        mr=dmr.times(mr);
        for(Bone c:children){
            c.rotateChild(dmr,getPos());
        }
    }

    /**
     * 狙い通りのローカル角度を設定する. setPose などに使われる。
     * @param m ローカル角度
     */
    public void rotateTo(Matrix m){
        Matrix dmr=m.times(ini_mr).times(mr.inverse());
        rotate(dmr);
    }

    /**
     * 狙い通りのグローバル角度に為るようにローカル角度を設定する.
     * @param gm ローカル角度
     */
    public void rotateToG(Matrix gm){
        Matrix dmr=g_mr.inverse().times(gm).times(mr.inverse());
        rotate(dmr);
    }

    /**
     * ボーンにポーズを適用する.
     * @param p 適用するポーズ.
     */
    public void setPose(BonePose p){
        if(p.nameOfBone.compareTo(name)!=0){
            return;
        }
        if(!mr.equals(p.mr)){
            rotateTo(p.mr);
        }
        if(!v.equals(p.v)){
            translationTo(p.v);
        }
    }

    /**
     * ボーンのポーズを取得する。
     * local pose
     * @return local Pose
     */
    public BonePose getPose(){
        BonePose p=new BonePose();
        p.nameOfBone=name;
        p.v=v;
        p.mr=mr.times(ini_mr.inverse());
        return p;
    }

    /**
     * ターゲットマーカー用ポーズを取得する.
     * good for global target marker.
     * 
     * @return Pose for global target marker. nameOfBone="原点1"
     */
    public BonePose getCoordinate(){
        BonePose p=new BonePose();
        p.nameOfBone="原点1";
        p.v=g_mr.times(v).add(gv);
        p.mr=g_mr.times(mr);
        return p;
    }

    /**
     * 文字列化。デバグ用
     * @return name: V:ローカル位置 mr:ローカル回転行列 ini_mr:初期設定回転行列
     */
    @Override
    public String toString(){
        String ret=name+":\n";
        ret+="\tV:"+v+"\n";
        ret+="\tM:"+mr+"\n";
        ret+="\tinit_M:"+ini_mr+"\n";
        double []rv=mr.angles();
        ret+="\tA:"+rv[0]+","+rv[1]+","+rv[2]+"\n";
        return ret;
    }

    /**
     * CSV 1行から Bone を設定. Ver. 1.3 以降はあまり整備されてないので、使わないほうがいい.
     * @param m 属するモデル
     * @param line １行文字列
     * @return CSVから生成された Bone
     * @throws MmdFormatException MMDモデル用のCSVとして異常なとき.
     */
    static public Bone fromCSV(Model m,String line) throws MmdFormatException {
        Bone ret;
        String [] p=CsvSpliter.split(line);
        Vector3D v;

        if(p.length<39)
            return null;
        if(p[0].compareTo("Bone")!=0){
            throw new MmdFormatException("not Bone line");
        }
        ret=new Bone(m);
        ret.name=p[1];

        try{
            ret.gv=new Vector3D(Double.parseDouble(p[5]),
                    Double.parseDouble(p[6]),
                    Double.parseDouble(p[7])
            );
        }catch(NumberFormatException ex){
            throw new MmdFormatException("Number error in "+ret.name+"'s corrdinate.");
        }

        ret.flags=(short)(
                (p[8].contentEquals("1")?BoneFlags.ROTATE.encode():0)
                +(p[9].contentEquals("1")?BoneFlags.MOVE.encode():0)
                +(p[10].contentEquals("1")?BoneFlags.IK.encode():0)
                +(p[11].contentEquals("1")?BoneFlags.VISIBLE.encode():0)
                +(p[12].contentEquals("1")?BoneFlags.OP.encode():0)
                +(p[14].contentEquals("1")?BoneFlags.OFFSET.encode():0)
                +(p[19].contentEquals("1")?BoneFlags.LINK_FLAG.encode():0)
                +(p[20].contentEquals("1")?BoneFlags.ROTATE_LINK.encode():0)
                +(p[21].contentEquals("1")?BoneFlags.MOVE_LINK.encode():0)
                +(p[24].contentEquals("1")?BoneFlags.AXIS_ROTATE.encode():0)
                +(p[28].contentEquals("1")?BoneFlags.LOCAL_AXIS.encode():0));

        if(p[13].length()!=0){
            ret.model.get(p[13]).addChild(ret);
        }

        if(p[23].length()!=0){
            double rate=Double.parseDouble(p[22]);
            ret.model.get(p[23]).addLinkChild(ret,
                    ((p[21].startsWith("1"))?rate:0),
                    ((p[20].startsWith("1"))?rate:0));
        }

        if(p[24].equals("1")){// rotation limit
            ret.limitRot=new Vector3D(Double.parseDouble(p[25]),
                    Double.parseDouble(p[26]),
                    Double.parseDouble(p[27]));
        }

        if(p[28].equals("1")){
            Vector3D lx,lz;
            try{
                lx=new Vector3D(Double.parseDouble(p[29]),
                        Double.parseDouble(p[30]),
                        Double.parseDouble(p[31]));
                lz=new Vector3D(-Double.parseDouble(p[32]),
                        -Double.parseDouble(p[33]),
                        -Double.parseDouble(p[34]));
            }catch(NumberFormatException ex){
                System.err.println("Number error in "+ret.name+"'s matrix.");
                return ret;
            }
            ret.setInicialLocalCoordinate(lx, lz);
        }
        return ret;
    }
}
