/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion;

import jp.sourceforge.mmd.midiMotion.midi.MidiResolver;
import java.util.ArrayList;
import jp.sourceforge.mmd.motion.Motion;

/**
 * MotionBuilder のスーパークラス.
 * Standard MIDI File から MMD のモーションを生成する Class ので共通の機能などを
 * 提供する.
 * @author nazo
 */
public abstract class MotionBuilder {

    /**
     * Frame per Second.
     * 秒間コマ数
     */
    protected float fps = 30;

    /**
     * 出力対象のモーション
     */
    protected Motion motion;

    /**
     * 過程報告用のProgressLisner のリスト
     */
    protected ArrayList<ProgressLisner> pl;

    /**
     * 対象とするMidi信号を保持するオブジェクト
     */
    protected MidiResolver mm;

    protected MotionBuilder() {
        motion=new Motion();
        pl=new ArrayList<ProgressLisner>();
    }

    protected MotionBuilder(Motion init) {
        motion=init;
    }

    /**
     * MMD での frame / 秒を決める. 初期値は 30 (fps).
     * @param fps frame per second.
     */
    public void setFPS(float fps){
        this.fps=fps;
    }

    /**
     * 設定FPS を読み取る.
     * @return frame per second.
     */
    public float getFPS(){
        return fps;
    }

    /**
     * モーションを得る. {@link #loadMessages() }前では、空のモーション.
     * @return モーション
     */
    public Motion getMotion(){
        return motion;
    }

    /**
     * モーションをリセットする.
     */
    public void resetMotion(){
        motion=new Motion(motion.getModelName());
    }
    
    /**
     * ProgressListenner の追加
     * @param l 追加される ProgressListenner.
     */
    public void addProgressListener(ProgressLisner l){
        pl.add(l);
    }

    /**
     * ProgressListenner の除外. 処理が終わったときに呼ぶものだが
     * {@link ProgressLisner#progressEnd() } から呼ぶとエラーになるので注意.
     * {@link #loadMessages() } 後に呼ぶといい.
     * @param l 除外される ProgressListenner.
     */
    public void removeProgressListener(ProgressLisner l){
        pl.remove(l);
    }

    /**
     * 全ての{@link #pl} に進捗を報告.
     * @param frame 現在の処理フレーム番
     */
    protected void reportProgress(int frame){
        for(ProgressLisner l:pl){
            l.progressMid(frame);
        }
    }

    /**
     * 全ての{@link #pl} に処理開始を報告.
     */
    protected void reportProgressStart(){
        for(ProgressLisner l:pl){
            l.progressStart((int)
                        ((double)mm.getMidiSeq().getMicrosecondLength()*fps/1000000));
        }
    }

    /**
     * 全ての{@link #pl} に処理終了を報告.
     */
    protected void reportProgressEnd() {
        for(ProgressLisner l:pl){
            l.progressEnd();
        }
    }

    /**
     * MIDI tics time を MMD frame 番に変換する. 小数点以下切捨て.
     * @param tics midi tics time
     * @return frame 番 in MMD
     */
    protected int ticsToFrame(long tics){
        return (int)(((double)tics)*mm.getTemp()*fps
                    /(mm.getReso()*1000000));
    }

    /**
     * {@link MidiResolver} を設定する. 将来的には{@link #loadMessages() } に統合されるかも.
     * @param midiR 読み取り対象のMidiResolver
     */
    public void setMidiResolver(MidiResolver midiR){
       mm=midiR;
    }

    /**
     * MIDI Message を読み取り、モーションを作成する.
     */
    public abstract void loadMessages();
}
