# mmon
#
# Copyright (c) 2007 sanpo
#
# This program is free software.
# You can redistribute it and/or modify it under the terms of the GPL.

require 'gtk2'
require 'rsvg2'

class Icon
    attr_accessor :name, :exec1, :exec2

    def initialize(item, w, h, icon_path = '')
        @item = item
        @name = '??'
        @icon_path = ''
        @exec1 = ''
        @exec2 = ''

        @x = 0
        @y = 0
        @w = w
        @h = h

        @entered = false
        @pressed = false
        @clicked = false

        set_icon_path(icon_path)
    end

    def set_position(x, y)
        # puts "position #{x} #{y}"
        @x = x
        @y = y
    end

    def get_position
        [x, y]
    end

    def set_size(w, h)
        if @w == w && @h == h
            return
        else
            @w = w
            @h = h
            
            create_image()
        end
    end

    def get_size
        [@w, @h]
    end

    def press(button)
        @pressed = true
    end

    def release(button)
        @pressed = false
        @clicked = true

        case button
        when 1 then execute(@exec1)
        when 2 then execute(@exec2)
        end

        GLib::Timeout.add(1 * 1000) do  # 1 sec
            #puts 'timeout'
            @clicked = false
            @item.redraw_all()
            false
        end
    end

    def leave
        @entered = false
        @pressed = false
    end

    def move(pressed)
        @entered = true
        @pressed = pressed
    end

    def execute(exec)
        return if exec == ''
        return if exec.nil?

        argv = exec.split(/\s/)
        #p argv
        begin
            GLib::Spawn.async(nil, argv, nil, GLib::Spawn::SEARCH_PATH)
        rescue
            p $!
        ensure

        end
    end

    def render(cc)
        cc.save
        cc.translate(@x, @y)

        if @clicked || @pressed || @entered
            if @clicked
                cc.set_source_rgba(0.8, 0.5, 0.3, 0.3)
            elsif @pressed
                cc.set_source_rgba(0.8, 0.8, 0.3, 0.3)
            elsif @entered
                cc.set_source_rgba(0.8, 0.8, 0.6, 0.3)
            end
            cc.rectangle(-2, -2, @w + 4, @h + 4)
            cc.fill
        end

        if @pixbuf != nil
            cc.set_source_pixbuf(@pixbuf, 0, 0)
            cc.paint
        elsif @surface != nil
            cc.set_source(@surface, 0, 0)
            cc.paint
        end

        cc.restore
    end

    def read_element(e)
        e_name = e.elements['Name']
        @name = e_name.text unless e_name.nil?

        e_exec1 = e.elements['Exec1']
        @exec1 = e_exec1.text unless e_exec1.nil?
        @exec1 = '' if @exec1.nil?

        e_exec2 = e.elements['Exec2']
        @exec2 = e_exec2.text unless e_exec2.nil?
        @exec2 = '' if @exec2.nil?

        e_path = e.elements['IconPath']
        icon_path = e_path.text e_path.nil?

        set_icon_path(icon_path)
    end

    def write_element
        e_name = REXML::Element.new('Name')
        e_name.text = @name

        e_exec1 = REXML::Element.new('Exec1')
        e_exec1.text = @exec1

        e_exec2 = REXML::Element.new('Exec2')
        e_exec2.text = @exec2

        e_iconpath = REXML::Element.new('IconPath')
        e_iconpath.text = @icon_path

        e = REXML::Element.new(self.class.to_s)
        e.add_element(e_name)
        e.add_element(e_exec1)
        e.add_element(e_exec2)
        e.add_element(e_iconpath)

        return e
    end

    def set_icon_path(path)
        if path.nil?
            @icon_path = ''
        elsif @icon_path == path
            return 
        else
            @icon_path = path
        end

        create_image()
    end

    def icon_path
        @icon_path
    end

    private

    def create_image()
        @pixbuf = nil
        @surface = nil
        
        case File.extname(@icon_path) 
        #when '.desktop' 

        when '.png', '.jpg', '.xpm'
            buf = Gdk::Pixbuf.new(@icon_path)
            if buf.width != @w || buf.height != @h
                @pixbuf = buf.scale(@w, @h)
            else
                @pixbuf = buf
            end
        when '.svg' 
            handle = RSVG::Handle.new_from_file(@icon_path)
            scale_x =  @w.to_f / handle.width
            scale_y =  @h.to_f / handle.height
            puts "w:#{handle.width} h:#{handle.height}"
            @surface = Cairo::ImageSurface.new(Cairo::FORMAT_ARGB32, @w, @h)
            cc = Cairo::Context.new(@surface)
            cc.scale(scale_x, scale_y)
            cc.render_rsvg_handle(handle)
        else
            puts "Icon :: default icon..."
            @surface = Cairo::ImageSurface.new(Cairo::FORMAT_ARGB32, @w, @h)
            cc = Cairo::Context.new(@surface)
            cc.operator = Cairo::OPERATOR_SOURCE
            cc.set_source_rgba(0.2, 0.2, 0.2, 1.0)
            cc.paint
        end

        @item.redraw_all()
    end
end

class IconSettingWidget < Gtk::VBox
    def initialize(icon = nil)
        super()

        label_name = Gtk::Label.new('Name:')
        label_exec1 = Gtk::Label.new('Command1:')
        label_exec2 = Gtk::Label.new('Command2:')
        label_icon = Gtk::Label.new('Icon:')

        label_name.set_width_request(80)
        label_exec1.set_width_request(80)
        label_exec2.set_width_request(80)
        label_icon.set_width_request(80)

        label_name.set_alignment(1.0, 0.5)
        label_exec1.set_alignment(1.0, 0.5)
        label_exec2.set_alignment(1.0, 0.5)
        label_icon.set_alignment(1.0, 0.5)

        @entry_name = Gtk::Entry.new
        @entry_exec1 = Gtk::Entry.new
        @entry_exec2 = Gtk::Entry.new

        @button_icon = Gtk::FileChooserButton.new('Icon', Gtk::FileChooser::Action::OPEN)
        @button_icon.signal_connect('selection-changed') do |w| 
            #puts 'button selection_changed'

            if w.filename != nil && @icon != nil
               @icon.set_icon_path(w.filename)
            end
        end

        h0 = Gtk::HBox.new
        h0.pack_start(label_name, false, false, 5)
        h0.pack_start(@entry_name, true, true, 5)

        h1 = Gtk::HBox.new
        h1.pack_start(label_exec1, false, false, 5)
        h1.pack_start(@entry_exec1, true, true, 5)

        h2 = Gtk::HBox.new
        h2.pack_start(label_exec2, false, false, 5)
        h2.pack_start(@entry_exec2, true, true, 5)

        h3 = Gtk::HBox.new
        h3.pack_start(label_icon, false, false, 5)
        h3.pack_start(@button_icon, true, true, 5)

        pack_start(h0, false, false, 10)
        pack_start(h1, false, false, 10)
        pack_start(h2, false, false, 10)
        pack_start(h3, false, false, 10)
        signal_connect('destroy') { |w| sync_setting() }

        self.icon = icon
    end
    
    def icon=(icon)
        return if @icon == icon

        sync_setting()

        @icon = icon
        if @icon.nil?
            @entry_name.text = '' 
            @entry_exec1.text = ''
            @entry_exec2.text = ''
            @button_icon.set_filename('/')
            self.sensitive = false
        else
            @entry_name.text = @icon.name
            @entry_exec1.text = @icon.exec1
            @entry_exec2.text = @icon.exec2
            if @icon.icon_path == ''
                @button_icon.set_filename('/')
            else
                @button_icon.set_filename(@icon.icon_path)
            end
            self.sensitive = true
        end
    end

    def sync_setting
        return if @icon.nil?

        puts 'sync_setting'
        p @button_icon.filename

        @icon.name = @entry_name.text
        @icon.exec1 = @entry_exec1.text
        @icon.exec2 = @entry_exec2.text
        @icon.set_icon_path(@button_icon.filename)

        puts 'sync_setting end'
    end
end
