#!/usr/bin/ruby -Ku
#
# mmon
#
# Copyright (c) 2007 sanpo
#
# This program is free software.
# You can redistribute it and/or modify it under the terms of the GPL.

require 'gtk2'
require 'rexml/document'

class Cairo::Context
    @@use_alpha = false

    alias :orig_set_source_rgba :set_source_rgba

    def set_source_rgba(ar)
        set_source_rgba(ar[0], ar[1], ar[2], ar[3])
    end

    def set_source_rgba(r, g, b, a)
        #puts 'set_source_rgba'
        if @@use_alpha
            orig_set_source_rgba(r, g, b, a)
        else
            orig_set_source_rgba(r, g, b, 1.0)
        end
    end

    def self.use_alpha=(bool)
        #puts 'use_alpha ' + bool.to_s
        @@use_alpha = bool
    end
end

class Item < Gtk::Window
    attr_reader :item_name

    def initialize(parent = nil)
        super(Gtk::Window::TOPLEVEL)

        @parent = parent
        # サイズが必要な時に毎回 window から取ってたが、リサイズしてから window サイズが変更に
        # なるまでの間にサイズを取ると古いサイズが帰るのでメモることにする。
        @width = 72
        @height = 72

#        self.resizable = false
#        self.set_type_hint(Gdk::Window::TypeHint::DOCK)
        self.add_events(Gdk::Event::Mask::BUTTON_PRESS_MASK)
        self.app_paintable = true
        self.decorated = false
        self.skip_pager_hint = true
        self.skip_taskbar_hint = true

        self.set_item_size(@width, @height)

		signal_connect("destroy")           { |w, e| destroy_handler(w, e) }
		signal_connect("expose-event")          { |w, e| expose_handler(w, e) }
        signal_connect("realize")               { |w| realize_handler(w) }
        signal_connect("configure-event")       { |w, e| configure_handler(w, e) }
        signal_connect("button-press-event")    { |w, e| button_press_handler(e) }

        screen = self.screen
        colormap = screen.rgba_colormap
        if colormap.nil?
            #puts 'colormap is nil'
            self.colormap = screen.rgb_colormap
            Cairo::Context.use_alpha = false
        else
            #puts 'colormap is not nil'
            self.colormap = colormap
            Cairo::Context.use_alpha = true
        end

        @interval = 1  # sec
        @timeout_id = 0

        @size_changed = false

        @use_bg = false
        @cc_bg = nil

        @use_fg = false
        @cc_fg = nil
    end

    # debug 用
    def force_rgb
        self.colormap = self.screen.rgb_colormap
        Cairo::Context.use_alpha = false
    end

    def destroy_handler(widget, event)
        #puts 'item destroy_handler'
        GLib::Source.remove(@timeout_id) unless @timeout_id.zero?
        Gtk::main_quit if @parent.nil?
    end

    def expose_handler(widget, event)
        e_x = event.area.x
        e_y = event.area.y
        e_w = event.area.width
        e_h = event.area.height

        #puts "expose #{e_x} #{e_y} #{e_w} #{e_h}"

        cc = self.window.create_cairo_context
        cc.rectangle(e_x, e_y, e_w, e_h)
        cc.clip

        cc.operator = Cairo::OPERATOR_CLEAR
        cc.paint

        if @use_bg
            cc.operator = Cairo::OPERATOR_SOURCE
            cc.set_source(@cc_bg.target, 0, 0)
            cc.paint
        end

        w, h = self.window.size
        cc.operator = Cairo::OPERATOR_OVER
        render(cc, w, h)

        if @use_fg
            cc.operator = Cairo::OPERATOR_OVER
            cc.set_source(@cc_fg.target, 0, 0)
            cc.paint
        end

        return true
    end

    def button_press_handler(event)
        #puts "button_press #{event.button}"

        case event.button
        when 3 
            show_menu(event)
            return true
        end

        return false
    end

    def show_menu(event)
        x = event.x
        y = event.y

        menuitem_setting = Gtk::MenuItem.new('Settings')
        menuitem_setting.signal_connect('activate') { show_dialog(x, y) }

        menuitem_quit = Gtk::MenuItem.new('Quit \'' + get_item_name() + '\'') 
        menuitem_quit.signal_connect('activate') { self.destroy() }

        menu = Gtk::Menu.new
        menu.append(menuitem_setting)
        menu.append(menuitem_quit)
        @parent.add_menu(menu) unless @parent.nil?
        menu.show_all
        menu.popup(nil, nil, event.button, event.time)
    end

    def get_item_name
        if self.class.const_defined?(:NAME)
            return self.class.const_get(:NAME)
        else
            return self.class.to_s
        end
    end

    def show_dialog(x, y)
        mmon = @parent.setting_widget unless @parent.nil?

        title = Gtk::Label.new
        title.set_markup('<big><b>' + get_item_name() + '</b></big>')

        w = setting_widget(x, y)

        vbox = Gtk::VBox.new
        vbox.pack_start(title, false)
        vbox.pack_start(Gtk::HSeparator.new, false)
        vbox.pack_start(w, true) unless w.nil?

        notebook = Gtk::Notebook.new
        notebook.append_page(mmon, Gtk::Label.new('Mmon')) unless @parent.nil?
        notebook.append_page(vbox, Gtk::Label.new(get_item_name())) unless w.nil?

        notebook.show_all

        n = notebook.page_num(vbox)
        notebook.page = n

        dialog = Gtk::Dialog.new('Settings', nil, Gtk::Dialog::MODAL, 
                [Gtk::Stock::CLOSE, Gtk::Dialog::ResponseType::CLOSE])
        dialog.vbox.pack_start(notebook)
        dialog.set_default_size(400, 400)
        dialog.run
        dialog.destroy
    end

    def realize_handler(widget)
        #puts 'realize'

        unless @interval.zero?
            prepare()
            @timeout_id = GLib::Timeout.add(@interval * 1000) { update() }
        end
    end

    def configure_handler(widget, event)
        x = event.x
        y = event.y
        w = event.width
        h = event.height
        #puts "configure #{self.class.to_s} (x,y)=(#{x} #{y}) :w:#{w} h:#{h}"
    end

    def update()
        return if @interval == 0.0

        if prepare() || @size_changed
            #puts 'update interval:' + @interval.to_s
            @size_changed = false
            self.queue_draw()
        end

        return true
    end

    def item_size
        #return self.window.size
        return @width, @height
    end

    def set_item_size(w, h)
        puts self.class.to_s + " set_item_size w:#{w} h:#{h}"
        @width = w
        @height = h
        set_shape_mask(w, h)
        create_bg(w, h) if @use_bg
        create_fg(w, h) if @use_fg
        @size_changed = true
        self.resize(w, h)
    end

    def redraw_all
        w, h = item_size()
        rect = Gdk::Rectangle.new(0, 0, w, h)
        self.window.invalidate(rect, false)
    end

    def redraw(x, y, w, h)
        puts "redraw (#{x}, #{y}) #{w} #{h}"
        rect = Gdk::Rectangle.new(x, y, w, h)
        self.window.invalidate(rect, false)
    end

    def set_shape_mask(w, h)
        bitmap = Gdk::Pixmap.new(nil, w, h, 1)

        cc = bitmap.create_cairo_context
        cc.operator = Cairo::OPERATOR_CLEAR
        cc.paint
        cc.operator = Cairo::OPERATOR_SOURCE

        render_mask(cc, w, h)

        self.shape_combine_mask(bitmap, 0, 0)
    end

    def create_bg(w, h)
        @cc_bg = create_surface_context(w, h)
        render_bg(@cc_bg, w, h)
    end

    def create_fg(w, h)
        @cc_fg = create_surface_context(w, h)
        render_fg(@cc_fg, w, h)
    end

    def create_surface_context(w, h)
        surface = Cairo::ImageSurface.new(Cairo::FORMAT_ARGB32, w, h)

        cc = Cairo::Context.new(surface)
        cc.operator = Cairo::OPERATOR_CLEAR
        cc.paint
        cc.operator = Cairo::OPERATOR_SOURCE

        return cc
    end

    def read_element(e)
        x = e.elements['x'].text.to_i
        y = e.elements['y'].text.to_i
        
        self.move(x, y)

        read_element_local(e)
    end

    def write_element
        x, y = self.position

        e = REXML::Element.new(self.class.to_s)
        
        e_x = REXML::Element.new('x')
        e_x.text = x.to_s

        e_y = REXML::Element.new('y')
        e_y.text = y.to_s

        e.add_element(e_x)
        e.add_element(e_y)

        write_element_local(e)

        return e
    end

    private

    def prepare
        true
    end

    def render_bg(cc, w, h)
        cc.set_source_rgba(0.0, 1.0, 0.0, 0.8)
        cc.paint
    end

    def render_fg(cc, w, h)
        cc.set_source_rgba(0.0, 0.0, 1.0, 0.8)
        cc.paint
    end

    def render(cc, w, h)
        cc.set_source_rgba(1.0, 0.0, 0.0, 0.8)
        cc.paint
    end

    def render_mask(cc, w, h)
        cc.operator = Cairo::OPERATOR_SOURCE
        cc.paint
    end

    def read_element_local(e)
    end
    
    def write_element_local(e)
    end

    def setting_widget(x, y)
        return nil
    end
end
    
if __FILE__ == $0
    item = Item.new
    item.show

    Gtk::main()
end
