#!/usr/bin/ruby -Ku
#
# mmon
#
# Copyright (c) 2007 sanpo
#
# This program is free software.
# You can redistribute it and/or modify it under the terms of the GPL.

require 'mmon/item'
require 'mmon/util'
require 'mmon/network'
require 'mmon/component/trend_graph'

class ItemNet < Item
    NAME = 'Network'

    @@color_in = Color.new(0.0, 1.0, 1.0, 0.8)
    @@color_out = Color.new(0.0, 0.8, 0.8, 0.8)

    def initialize(mmon)
        super(mmon)

        @interval = 0.5
        @use_bg = true
        @use_fg = true
        w = 72
        h = 32
        @width = w
        @height = h

        @trend_in = TrendGraph.new(w - 2 * 2, h / 2  - 3)
        @trend_in.position(2, 2)
        @trend_in.color(@@color_in)

        @trend_out = TrendGraph.new(w - 2 * 2, h / 2  - 3)
        @trend_out.position(2, h / 2 + 1)
        @trend_out.color(@@color_out)

        @in_max = 512000
        @out_max = 64000

        @net = Network.new(@in_max, @out_max)
    end

    def render_mask(cc, w, h)
        Util.simple_bg2(cc, w, h)
        Util.simple_fg2(cc, w, h)
    end

    def render_bg(cc, w, h)
        Util.simple_bg2(cc, w, h)
    end

    def render_fg(cc, w, h)
        Util.simple_fg2(cc, w, h)
    end

    def render(cc, w, h)
        @trend_in.render(cc)
        @trend_out.render(cc)
    end

    private

    def prepare
        @net.read

        @trend_in.value(@net.in)
        @trend_out.value(@net.out)

        return true
    end

    def read_element_localvalue(e)
        e_in_max = e.elements['in_max']
        @in_max = e_in_max.text.to_i unless e_in_max.nil?
        e_out_max = e.elements['out_max']
        @out_max = e_out_max.text.to_i unless e_out_max.nil?

        @net.in_max = @in_max
        @net.out_max = @out_max
    end

    def write_element_local(e)
        e_in = REXML::Element.new('in_max')
        e_in.text = @in_max.to_s
        e_out = REXML::Element.new('out_max')
        e_out.text = @out_max.to_s

        e.add_element(e_in)
        e.add_element(e_out)
    end

    def setting_widget(x, y)
        label_title = Gtk::Label.new('bandwidth (KByte/s)')

        label_in = Gtk::Label.new('in')
        label_out = Gtk::Label.new('out')

        spin_in = Gtk::SpinButton.new(1.0, 12500.0, 1.0) 
        spin_out = Gtk::SpinButton.new(1.0, 12500.0, 1.0) 

        spin_in.signal_connect('value-changed') do |w|
            @in_max = w.value.to_i * 1000
            @net.in_max = @in_max
        end
        spin_out.signal_connect('value-changed') do |w|
            @out_max = w.value.to_i * 1000
            @net.out_max = @out_max
        end

        spin_in.set_value(@in_max / 1000)
        spin_out.set_value(@out_max / 1000)

        table = Gtk::Table.new(2, 2)
        table.attach(label_in, 0, 1, 0, 1)
        table.attach(label_out, 0, 1, 1, 2)

        table.attach(spin_in, 1, 2, 0, 1)
        table.attach(spin_out, 1, 2, 1, 2)


        vbox = Gtk::VBox.new
        vbox.pack_start(label_title, false, false, 10)
        vbox.pack_start(table, false, false, 10)

        return vbox
    end
end

if __FILE__ == $0
    require 'mmon/appmmon'

    app = AppMmon.new
    item = ItemNet.new(app)
    app.add_item(item)

    Gtk::main()
end
