/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: mod_fast_bbs.cpp 1812 2006-10-18 14:17:06Z svn $
 *****************************************************************************/

/**
 * @file
 * @brief Apache モジュールのエントリ関数群．
 */

// Apache の config.h を先に処理させるため，httpd.h のインクルードはこの位置
#include "httpd.h"

#include "Environment.h"

#include "http_main.h"
#include "http_config.h"
#include "http_request.h"

#include "apr_pools.h"
#include "apr_strings.h"

#include "mod_fast_bbs.h"
#include "fast_bbs_handler.h"

#include "ApacheResponse.h"
#include "ApacheLogger.h"
#include "BBSConfig.h"
#include "Auxiliary.h"
#include "Macro.h"
#include "SourceInfo.h"

#define TEMPLATE_INSTANTIATION
#define RESPONSE_TYPE_APACHE
#include "fast_bbs_handler.cpp"

SOURCE_INFO_ADD("$Id: mod_fast_bbs.cpp 1812 2006-10-18 14:17:06Z svn $");

static const char *HANDLER_NAME = "fast_bbs";

#define DIRECTIVE_ENTRY(directive, member)                              \
    AP_INIT_TAKE1(BBSConfig::directive.param,                           \
                  reinterpret_cast<const char*(*)()>(set_ ## member),   \
                  NULL, RSRC_CONF, BBSConfig::directive.desc)

#define AS_BCONFIG(pointer)        reinterpret_cast<BBSConfig *>(pointer)

#define DIRECTIVE_HANDLER(member, value)                                \
    static const char *set_ ## member(cmd_parms *parms, void *mconfig,  \
                                      char *arg)                        \
    {                                                                   \
        BBSConfig *config;                                              \
                                                                        \
        config = AS_BCONFIG(ap_get_module_config(parms->server->module_config, \
                                                 &fast_bbs_module));    \
        config->member = value;                                         \
                                                                        \
        return NULL;                                                    \
    }
#define TEMPLATE_DIRECTIVE_HANDLER(type, tmpl_file_path)                \
    static const char *set_ ## type(cmd_parms *parms, void *mconfig,    \
                                    char *arg)                          \
    {                                                                   \
        BBSConfig *config;                                              \
                                                                        \
        config = AS_BCONFIG(ap_get_module_config(parms->server->module_config, \
                                                 &fast_bbs_module));    \
        config->set_template(BBSConfig::type, tmpl_file_path);          \
                                                                        \
        return NULL;                                                    \
    }

BBSConfig *get_config(request_rec *r)
{
    return AS_BCONFIG(ap_get_module_config(r->server->module_config,
                                           &fast_bbs_module));
}

static apr_status_t finalize_bbs_config(void *data)
{
    BBSConfig *config;

    config = AS_BCONFIG(data);
    config->finalize();

    return APR_SUCCESS;
}

static void *fast_bbs_create_server_config(apr_pool_t *p, server_rec *s)
{
    BBSConfig *config;

    APR_PCALLOC(config, BBSConfig *, s->process->pool, sizeof(BBSConfig));

    new(config) BBSConfig(s->process->pool);

    return config;
}

static bool get_apache_debug_mode()
{
    const char **defs;

    defs = AS_CONST_CHAR_P(ap_server_config_defines->elts);

    for (int i = 0; i < ap_server_config_defines->nelts; i++) {
        // -X オプション付きで起動されたときは DEBUG が定義されている
        if (strcmp(defs[i], "DEBUG") == 0) {
            return true;
        }
    }

    return false;
}

static int fast_bbs_post_config_server(apr_pool_t *p, server_rec *s)
{
    BBSConfig *config;

    config = AS_BCONFIG(ap_get_module_config(s->module_config,
                                             &fast_bbs_module));
    try {
        config->init();
        config->is_debug_mode = get_apache_debug_mode();

        apr_pool_cleanup_register(p, config, finalize_bbs_config,
                                  apr_pool_cleanup_null);
    } catch(const char *message) {
        logger.error(__FILE__, __LINE__, s,
                     "(%s) %s", "BBSConfig::init", message);
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    return OK;
}

static int fast_bbs_post_config(apr_pool_t *pconf, apr_pool_t *plog,
                                apr_pool_t *ptemp, server_rec *s)
{
    void *user_data;
    apr_status_t status;

    apr_pool_userdata_get(&user_data, PACKAGE_NAME, s->process->pool);

    // 2 回呼ばれるので，1 回目は何もせずに return する
    if (user_data == NULL) {
        apr_pool_userdata_set(reinterpret_cast<const void *>(1), PACKAGE_NAME,
                              apr_pool_cleanup_null, s->process->pool);

        return OK;
    }

    do {
        status = fast_bbs_post_config_server(pconf, s);
        if (status != OK) {
            return status;
        }
    } while ((s = s->next) != NULL);

#ifndef WIN32
    // Windows でこれをするとサービス名が変わってしまうらしい
    ap_add_version_component(pconf, PACKAGE_NAME "/" PACKAGE_VERSION);
#endif

    logger.info(__FILE__, __LINE__, s, "SUCCESS: %s", "post config");

    return OK;
}

static void fast_bbs_child_init_server(apr_pool_t *pool, server_rec *server)
{
    BBSConfig *config;

    config = AS_BCONFIG(ap_get_module_config(server->module_config,
                                             &fast_bbs_module));
    try {
        config->child_init();
    } catch(const char *message) {
        logger.error(__FILE__, __LINE__, server,
                     "(%s) %s: %s", "BBSConfig::child_init", message,
                     get_last_error_message(pool));
    }
}

static void fast_bbs_child_init(apr_pool_t *p, server_rec *s)
{
    do {
        fast_bbs_child_init_server(p, s);
    } while ((s = s->next) != NULL);

    logger.info(__FILE__, __LINE__, s, "SUCCESS: %s", "child init");
}

static int fast_bbs_handler(request_rec *r)
{
    BBSConfig *config;

    if (strcmp(r->handler, HANDLER_NAME)) {
        return DECLINED;
    }

    if (UNLIKELY(!ap_is_initial_req(r))) {
        return DECLINED;
    }

    config = get_config(r);

    try {
        return fast_bbs_command_handler<ApacheResponse>(r, config,
                                                        r->path_info);
    } catch(const char *message) {
        logger.error(__FILE__, __LINE__, r, "Exception: %s", message);

        return HTTP_INTERNAL_SERVER_ERROR;
    }
}

static void fast_bbs_register_hooks(apr_pool_t *pool)
{
    ap_hook_post_config(fast_bbs_post_config, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_child_init(fast_bbs_child_init, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_handler(fast_bbs_handler, NULL, NULL, APR_HOOK_MIDDLE);
}

DIRECTIVE_HANDLER(base_url, apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(data_dir_path, apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(file_dir_path, apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(temp_dir_path, apr_pstrdup(parms->pool, arg));

TEMPLATE_DIRECTIVE_HANDLER(INDEX_VIEW, apr_pstrdup(parms->pool, arg));
TEMPLATE_DIRECTIVE_HANDLER(COMMENT_VIEW, apr_pstrdup(parms->pool, arg));
TEMPLATE_DIRECTIVE_HANDLER(THREAD_VIEW, apr_pstrdup(parms->pool, arg));
TEMPLATE_DIRECTIVE_HANDLER(POST_THREAD_VIEW, apr_pstrdup(parms->pool, arg));

static const command_rec fast_bbs_cmds[] = {
    DIRECTIVE_ENTRY(BASE_URL, base_url),
    DIRECTIVE_ENTRY(DATA_DIRECTORY, data_dir_path),
    DIRECTIVE_ENTRY(FILE_DIRECTORY, file_dir_path),
    DIRECTIVE_ENTRY(TEMP_DIRECTORY, temp_dir_path),

    DIRECTIVE_ENTRY(INDEX_VIEW_TEMPLATE, INDEX_VIEW),
    DIRECTIVE_ENTRY(COMMENT_VIEW_TEMPLATE, COMMENT_VIEW),
    DIRECTIVE_ENTRY(THREAD_VIEW_TEMPLATE, THREAD_VIEW),
    DIRECTIVE_ENTRY(POST_THREAD_VIEW_TEMPLATE, POST_THREAD_VIEW),
    { NULL }
};

extern "C" {
module AP_MODULE_DECLARE_DATA fast_bbs_module = {
    STANDARD20_MODULE_STUFF,
    NULL,
    NULL,
    fast_bbs_create_server_config,
    NULL,
    fast_bbs_cmds,
    fast_bbs_register_hooks
};
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:

