/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: fast_bbs_util.h 1898 2006-10-26 17:46:14Z svn $
 *****************************************************************************/

#ifndef FAST_BBS_UTIL_H
#define FAST_BBS_UTIL_H

#include "Environment.h"

#include "apr_tables.h"

#include "BBSCommentIterator.h"
#include "PostFlowController.h"
#include "ImageFile.h"
#include "RFC1867Parser.h"
#include "Auxiliary.h"
#include "Message.h"

class BBSConfig;

static const char POST_THREAD_ID_PARAM[]            = "thread_id";
static const char POST_SUBJECT_PARAM[]              = "subject";
static const char POST_IMAGE_PARAM[]                = "image";
static const char POST_NAME_PARAM[]                 = "name";
static const char POST_EMAIL_PARAM[]                = "email";
static const char POST_MESSAGE_PARAM[]              = "message";
static const char POST_TRIP_SEPARATOR               = '#';
static const char POST_SAGE_SPECIFIER[]             = "sage";

static const char ARG_SEPARATE_STR[]                = "/";
static const char ARG_COMMENT_RANGE_SEPARATE_CHAR   = ',';
static const char ARG_COMMENT_RANGE_INTERVAL_CHAR   = '-';
static const char ARG_COMMENT_RANGE_LAST_STR[]      = "l";
static const char ARG_ANCHOR_STR[]                  = "#";

static const char FILE_EXT_SEPARATOR[]              = ".";

const char *get_word(apr_pool_t *pool, const char **input,
                     const char delimiter);

apr_size_t get_range_count(const char *ranges_arg);

void get_thread_id(apr_pool_t *pool, const char *arg, apr_size_t *thread_id);

void get_thread_param(apr_pool_t *pool, const char *arg, apr_size_t *thread_id,
                      BBSCommentIterator::range_t **ranges,
                      apr_size_t *range_count);

template<class R>
void get_post_base_param(apr_pool_t *pool,
                         apr_array_header_t *content_array,
                         const char **name, const char **trip,
                         const char **email, const char **message,
                         apr_time_t *time, bool *is_age)
{
    typename RFC1867Parser<R>::content_t *name_content;
    typename RFC1867Parser<R>::content_t *email_content;
    typename RFC1867Parser<R>::content_t *message_content;

    name_content = RFC1867Parser<R>::get_content(content_array,
                                                 POST_NAME_PARAM);
    email_content = RFC1867Parser<R>::get_content(content_array,
                                                  POST_EMAIL_PARAM);
    message_content = RFC1867Parser<R>::get_content(content_array,
                                                    POST_MESSAGE_PARAM);

    if ((name_content == NULL) ||
        (name_content->type != RFC1867Parser<R>::TEXT) ||
        (email_content == NULL) ||
        (email_content->type != RFC1867Parser<R>::TEXT) ||
        (message_content == NULL) ||
        (message_content->type != RFC1867Parser<R>::TEXT)) {
        THROW(MESSAGE_POST_DATA_INVALID);
    }

    if (strchr(name_content->text, POST_TRIP_SEPARATOR) != NULL) {
        *trip = name_content->text;
        *name = get_word(pool, trip, POST_TRIP_SEPARATOR);
    } else {
        *name = name_content->text;
        *trip = "";
    }

    *email = email_content->text;
    *message = message_content->text;
    *time = apr_time_now();

    *is_age = (strncmp(*email, POST_SAGE_SPECIFIER,
                       LITERAL_STRLEN(POST_SAGE_SPECIFIER)) != 0);
}

template<class R>
void get_post_comment_param(apr_pool_t *pool,
                            apr_array_header_t *content_array,
                            apr_size_t *thread_id,
                            const char **name, const char **trip,
                            const char **email, const char **message,
                            apr_time_t *time, bool *is_age)
{
    typename RFC1867Parser<R>::content_t *thread_id_content;

    thread_id_content = RFC1867Parser<R>::get_content(content_array,
                                                      POST_THREAD_ID_PARAM);

    if ((thread_id_content == NULL) ||
        (thread_id_content->type != RFC1867Parser<R>::TEXT)) {
        THROW(MESSAGE_POST_DATA_INVALID);
    }

    *thread_id = atosize(thread_id_content->text);

    get_post_base_param<R>(pool, content_array, name, trip, email, message,
                           time, is_age);
}

void get_thumbnail_size(ImageFile *image_file,
                        apr_uint16_t *width, apr_uint16_t *height);

template<class R>
void get_image_param(apr_pool_t *pool,
                     typename RFC1867Parser<R>::content_t *image_content,
                     BBSThread::image_t *image, const char **image_temp_path)
{
    strncpy(image->ext, get_mime_image_ext(image_content->file.mime),
            sizeof(image->ext));
    image->size = image_content->file.size;

    ImageFile image_file(pool, image_content->file.temp_path);

    get_thumbnail_size(&image_file,
                       &(image->thumbnail.width), &(image->thumbnail.height));

    *image_temp_path = image_content->file.temp_path;
}

template<class R>
void get_post_thread_param(apr_pool_t *pool,
                           apr_array_header_t *content_array,
                           const char **subject, BBSThread::image_t *image,
                           const char **image_temp_path,
                           const char **name, const char **trip,
                           const char **email, const char **message,
                           apr_time_t *time, bool *is_age)
{
    typename RFC1867Parser<R>::content_t *subject_content;
    typename RFC1867Parser<R>::content_t *image_content;

    subject_content = RFC1867Parser<R>::get_content(content_array,
                                                    POST_SUBJECT_PARAM);
    image_content = RFC1867Parser<R>::get_content(content_array,
                                                  POST_IMAGE_PARAM);

    if ((subject_content == NULL) ||
        (subject_content->type != RFC1867Parser<R>::TEXT) ||
        (image_content == NULL) ||
        (image_content->type != RFC1867Parser<R>::FILE)) {
        THROW(MESSAGE_POST_DATA_INVALID);
    }

    *subject = subject_content->text;

    if (image_content->file.size != 0) {
        get_image_param<R>(pool, image_content, image, image_temp_path);
    }

    get_post_base_param<R>(pool, content_array, name, trip, email, message,
                           time, is_age);
}

void create_thumbnail(apr_pool_t *pool, const char *file_dir_path,
                      apr_size_t thread_id, BBSThread::image_t *image,
                      const char *image_temp_path);

void commit_image_file(apr_pool_t *pool, const char *file_dir_path,
                       apr_size_t thread_id, BBSThread::image_t *image,
                       const char *image_temp_path);

bool can_post(BBSConfig *config, const char *url);
void regist_post(BBSConfig *config, const char *url);

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:

