/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: PostDataChecker.cpp 1884 2006-10-22 10:36:58Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <cstring>

#include "PostDataChecker.h"
#include "Auxiliary.h"
#include "Message.h"
#include "Macro.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: PostDataChecker.cpp 1884 2006-10-22 10:36:58Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
void PostDataChecker::validate_thread(const char *subject,
                                      BBSThread::image_t *image)
{
    if (strlen(subject) == 0) {
        THROW(MESSAGE_POST_SUBJECT_EMPTY);
    } else if (strlen(subject) > (TRD_MAX_SUBJECT_SIZE-1)) {
        THROW(MESSAGE_POST_SUBJECT_TOO_LONG);
    }
}

void PostDataChecker::validate_comment(const char *name, const char *trip,
                                       const char *email, const char *message,
                                       bool is_age,
                                       BBSThread::bbs_comment_flag_t *flag)
{
    if (strlen(name) == 0) {
        THROW(MESSAGE_POST_NAME_EMPTY);
    } else if (strlen(name) > (TRD_MAX_NAME_SIZE-1)) {
        THROW(MESSAGE_POST_NAME_TOO_LONG);
    } else if (is_contain_html(name)) {
        THROW(MESSAGE_POST_NAME_HTML_FORBIDDEN);
    }

    if (strlen(trip) > (TRD_MAX_TRIP_SIZE-1)) {
        THROW(MESSAGE_POST_TRIP_TOO_LONG);
    } else if (is_contain_html(trip)) {
        THROW(MESSAGE_POST_TRIP_HTML_FORBIDDEN);
    }

    if (strlen(email) > (TRD_MAX_EMAIL_SIZE-1)) {
        THROW(MESSAGE_POST_EMAIL_TOO_LONG);
    } else if (is_contain_html(email)) {
        THROW(MESSAGE_POST_EMAIL_HTML_FORBIDDEN);
    }

    if (strlen(message) == 0) {
        THROW(MESSAGE_POST_MESSAGE_EMPTY);
    } else if (strlen(message) > (TRD_MAX_MESSAGE_SIZE-1)) {
        THROW(MESSAGE_POST_MESSAGE_TOO_LONG);
    }

    flag->is_age                = is_age ? 1 : 0;
    flag->is_need_url_link      = is_contain_url_link(message) ? 1 : 0;
    flag->is_need_comment_link  = is_contain_comment_link(message) ? 1 : 0;
    flag->is_need_html_escape   = is_contain_html(message) ? 1 : 0;
}

bool PostDataChecker::is_contain_url_link(const char *str)
{
    static const char *URL_PREFIX[]         = { "https://", "http://" };
    static const char URL_PREFIX_COMMON[]   = "http";
    static const apr_size_t URL_PREFIX_COMMON_LENGHT =
        LITERAL_STRLEN(URL_PREFIX_COMMON);

    const char *url_link;

    url_link = str;
    while ((url_link = strstr(url_link, URL_PREFIX_COMMON)) != NULL) {
        if ((strncmp(url_link, URL_PREFIX[0],
                     LITERAL_STRLEN(URL_PREFIX[0])) == 0) ||
            (strncmp(url_link, URL_PREFIX[1],
                     LITERAL_STRLEN(URL_PREFIX[1])) == 0)) {
            return true;
        }

        url_link += URL_PREFIX_COMMON_LENGHT;
    }

    return false;
}

bool PostDataChecker::is_contain_comment_link(const char *str)
{
    const char *comment_link;
    apr_size_t prefix_length;

    comment_link = str;
    prefix_length = strlen(BBSThread::COMMENT_LINK_PREFIX);
    while ((comment_link =
            strstr(comment_link, BBSThread::COMMENT_LINK_PREFIX)) != NULL) {
        comment_link += prefix_length;

        if (*comment_link == '\0') {
            return false;
        }

        if (isdigit(static_cast<unsigned char>(*comment_link))) {
            return true;
        }
    }

    return false;
}

bool PostDataChecker::is_contain_html(const char *str)
{
    static const char HTML_CHAR[] = { '<', '>', '&' };

    apr_size_t length;

    length = strlen(str);
    for (apr_size_t i = 0; i < ARRAY_SIZE_OF(HTML_CHAR); i++) {
        if (strnchr(str, length, HTML_CHAR[i]) != NULL) {
            return true;
        }
    }

    return false;
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
