/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: fast_bbs_handler.cpp 1887 2006-10-22 18:49:12Z svn $
 *****************************************************************************/

#ifndef TEMPLATE_INSTANTIATION
#include "Environment.h"
#endif

#include <cstring>
#include <memory>

#include "apr_strings.h"

#include "fast_bbs_handler.h"
#include "fast_bbs_util.h"

#include "RequestResponseImpl.h"

#include "BBSThread.h"
#include "BBSThreadIO.h"
#include "BBSCommentIterator.h"
#include "BBSThreadList.h"
#include "BBSTemplate.h"
#include "TemplateVariableCreator.h"
#include "BBSCommentVariableCreator.h"
#include "BBSThreadVariableCreator.h"
#include "BBSThreadManager.h"
#include "CleanPointer.h"
#include "Message.h"
#include "Macro.h"
#include "Auxiliary.h"
#include "SourceInfo.h"

#ifndef TEMPLATE_INSTANTIATION
SOURCE_INFO_ADD("$Id: fast_bbs_handler.cpp 1887 2006-10-22 18:49:12Z svn $");
#endif

#define IS_REQUEST_COMMAND(arg, command) \
    (strncmp(arg, command, LITERAL_STRLEN(command)) == 0)

static const char BASE_URL_VAR_NAME[]           = "BASE_URL";
static const char THREAD_ID_VAR_NAME[]          = "THREAD_ID";
static const char SUBJECT_VAR_NAME[]            = "SUBJECT";
static const char COMMENT_COUNT_VAR_NAME[]      = "COMMENT_COUNT";
static const char COMMENT_LIST_VAR_NAME[]       = "COMMENT_LIST";

static const char IMAGE_EXT_VAR_NAME[]          = "IMAGE_EXT";
static const char IMAGE_SIZE_VAR_NAME[]         = "IMAGE_SIZE";
static const char THUMBNAIL_WIDTH_VAR_NAME[]    = "THUMBNAIL_WIDTH";
static const char THUMBNAIL_HEIGHT_VAR_NAME[]   = "THUMBNAIL_HEIGHT";

static const char THREAD_INFO_LIST_VAR_NAME[]   = "THREAD_INFO_LIST";
static const char THREAD_LIST_VAR_NAME[    ]    = "THREAD_LIST";
static const char THREAD_LIST_SIZE_VAR_NAME[]   = "THREAD_LIST_SIZE";


static const char INDEX_COMMAND[]               = "index";
static const char COMMENT_COMMAND[]             = "comment";
static const char THREAD_COMMAND[]              = "thread";
static const char RSS_COMMAND[]                 = "rss";
static const char FILE_COMMAND[]                = "file";
static const char POST_COMMENT_COMMAND[]        = "post_comment";
static const char POST_THREAD_COMMAND[]         = "post_thread";
static const char POST_THREAD_INPUT_COMMAND[]   = "post_thread_input";


#ifdef DEBUG
static const char INFO_COMMAND[]                = "info";
#endif

template<class Response>
static int redirect(typename Response::Handle *r, typename Response::Writer& o,
                    const char *url)
{
    // POST した後にページを簡単にリロードできるようにするため，HTML で
    // リダイレクトさせる．

    Response::set_content_type(r, "text/html");

    o.write("<?xml version=\"1.0\" encoding=\"EUC-JP\"?>\n");
    o.write("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" "
             "\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n");
    o.write("<html xmlns=\"http://www.w3.org/1999/xhtml\" "
             "xml:lang=\"ja\" >\n");
    o.write(" <head>\n");
    o.write("  <meta http-equiv=\"content-type\" "
             "content=\"application/xhtml+xml\" />\n");
    o.write("  <meta http-equiv=\"refresh\" content=\"0;url=");
    o.write(url);
    o.write("\" />\n");
    o.write("  <title>" PACKAGE_NAME "</title>\n");
    o.write(" </head>\n");
    o.write(" <body />\n");
    o.write("</html>\n");

    o.finish();

    return OK;
}

template<class Response>
static int error(typename Response::Handle *r,
                 typename Response::Writer& o,
                 BBSConfig *config, const char *message)
{
    Response::set_content_type(r, "text/html");

    o.write("<div class=\"warning\">ERROR: ");
    o.write(message);
    o.write("</div>\n");
    o.finish();

    return OK;
}

template<class Response>
static int error(typename Response::Handle *r,
                 BBSConfig *config, const char *message)
{
    typename Response::Writer o(r);

    return error<Response>(r, o, config, message);
}

template<class Response>
static int error_with_log(typename Response::Handle *r,
                          typename Response::Writer& o,
                          BBSConfig *config, const char *message)
{
    logger.info(__FILE__, __LINE__, r, "ERROR: %s (%s)",
                message, get_last_error_message(r->pool));

    return error<Response>(r, o, config, message);
}

template<class Response>
static int index(typename Response::Handle *r, BBSConfig *config,
                 const char *arg)
{
    BBSTemplate *tmpl;
    BBSCommentIterator *bcomment_iter;
    BBSThread::info_t info_list[VIW_THREAD_INFO_INDEX_COUNT];
    apr_size_t info_list_size;

    Response::set_content_type(r, "text/html");
#ifndef DEBUG
    Response::set_modified_time(r, config->get_thread_list()->get_mtime());
    Response::set_last_modified(r, config->get_thread_list()->get_mtime());

    int status;
    if ((status = Response::is_meets_condition(r)) != OK) {
        return status;
    }
#endif
    if (Response::is_head_method(r)) {
        return OK;
    }

    typename Response::Writer o(r);
    try {
        APR_PALLOC(bcomment_iter, BBSCommentIterator *, r->pool,
                   BBSCommentIterator::MEMORY_SIZE *
                   VIW_THREAD_INFO_INDEX_COUNT);

        info_list_size = VIW_ABSTRACT_THREAD_COUNT;
        config->get_thread_list()->get_info_list(info_list, &info_list_size);

        tmpl = config->get_template(BBSConfig::INDEX_VIEW);

        TemplateVariableCreator var_creator(r->pool, tmpl->get_ids());

        var_creator.create(BASE_URL_VAR_NAME, config->base_url);

        var_creator.create
            (THREAD_INFO_LIST_VAR_NAME,
             tmpl->get_thread_var_creator()->create_info
             (r->pool, info_list, info_list_size));

        var_creator.create
            (THREAD_LIST_VAR_NAME,
             tmpl->get_thread_var_creator()->create
             (r->pool, tmpl->get_comment_var_creator(),
              bcomment_iter, config->get_thread_manager(),
              info_list, info_list_size, VIW_ABSTRACT_COMMENT_COUNT));

        try {
            var_creator.create(THREAD_LIST_SIZE_VAR_NAME, info_list_size);

            TemplateExecutor<typename Response::Writer> executor(r->pool, o);

            executor.exec(tmpl->get_node_tree(), var_creator.get_variables(),
                          tmpl->get_key_count());
        } catch(const char *) {
            // 後始末
            for (apr_size_t i = 0; i < info_list_size; i++) {
                bcomment_iter->~BBSCommentIterator();
                bcomment_iter =
                    AS_BCOMMENT_ITER(AS_CHAR(bcomment_iter) +
                                     BBSCommentIterator::MEMORY_SIZE);
            }

            throw;
        }
    } catch(const char *message) {
        return error_with_log<Response>(r, o, config, message);
    }
    o.finish();

    // 後始末
    for (apr_size_t i = 0; i < info_list_size; i++) {
        bcomment_iter->~BBSCommentIterator();
        bcomment_iter =
            AS_BCOMMENT_ITER(AS_CHAR(bcomment_iter) +
                             BBSCommentIterator::MEMORY_SIZE);
    }

    return OK;
}

template<class Response>
static int comment(typename Response::Handle *r, BBSConfig *config,
                   const char *arg)
{
    apr_size_t thread_id;
    BBSCommentIterator::range_t *ranges;
    apr_size_t range_count;
    BBSCommentIterator *bcomment_iter;
    BBSTemplate *tmpl;

    Response::set_content_type(r, "text/html");

    get_thread_param(r->pool, arg, &thread_id, &ranges, &range_count);

    typename Response::Writer o(r);
    try {
        APR_PALLOC(bcomment_iter, BBSCommentIterator *, r->pool,
                   BBSCommentIterator::MEMORY_SIZE);

        config->get_thread_manager()->get_thread_by_id(r->pool, thread_id,
                                                       bcomment_iter,
                                                       ranges, range_count);
        CleanPointer<BBSCommentIterator> clean_ptr(bcomment_iter);

#ifndef DEBUG
        Response::set_modified_time(r, bcomment_iter->get_mtime());
        Response::set_last_modified(r, bcomment_iter->get_mtime());

        int status;
        if ((status = Response::is_meets_condition(r)) != OK) {
            return status;
        }
#endif
        if (Response::is_head_method(r)) {
            return OK;
        }

        tmpl = config->get_template(BBSConfig::COMMENT_VIEW);

        TemplateVariableCreator var_creator(r->pool, tmpl->get_ids());

        var_creator.create(BASE_URL_VAR_NAME, config->base_url);
        var_creator.create(THREAD_ID_VAR_NAME, bcomment_iter->get_id());
        var_creator.create(SUBJECT_VAR_NAME, bcomment_iter->get_subject());

        var_creator.create(IMAGE_SIZE_VAR_NAME,
                           bcomment_iter->get_image_size());
        if (bcomment_iter->get_image_size() != 0) {
            var_creator.create(IMAGE_EXT_VAR_NAME,
                               bcomment_iter->get_image_ext());
            var_creator.create(THUMBNAIL_WIDTH_VAR_NAME,
                               bcomment_iter->get_thumbnail_width());
            var_creator.create(THUMBNAIL_HEIGHT_VAR_NAME,
                               bcomment_iter->get_thumbnail_height());
        }

        var_creator.create(COMMENT_COUNT_VAR_NAME,
                           bcomment_iter->get_comment_count());
        var_creator.create
            (COMMENT_LIST_VAR_NAME,
             tmpl->get_comment_var_creator()->create(r->pool,
                                                     bcomment_iter));

        TemplateExecutor<typename Response::Writer> executor(r->pool, o);

        executor.exec(tmpl->get_node_tree(), var_creator.get_variables(),
                      tmpl->get_key_count());

    } catch(const char *message) {
        return error_with_log<Response>(r, o, config, message);
    }
    o.finish();

    return OK;
}

template<class Response>
static int thread(typename Response::Handle *r, BBSConfig *config,
                  const char *arg)
{
    BBSTemplate *tmpl;

    Response::set_content_type(r, "text/html");
#ifndef DEBUG
    Response::set_modified_time(r, config->get_thread_list()->get_mtime());
    Response::set_last_modified(r, config->get_thread_list()->get_mtime());

    int status;
    if ((status = Response::is_meets_condition(r)) != OK) {
        return status;
    }
#endif
    if (Response::is_head_method(r)) {
        return OK;
    }

    typename Response::Writer o(r);
    try {
        tmpl = config->get_template(BBSConfig::THREAD_VIEW);

        TemplateVariableCreator var_creator(r->pool, tmpl->get_ids());

        var_creator.create(BASE_URL_VAR_NAME, config->base_url);

        var_creator.create
            (THREAD_INFO_LIST_VAR_NAME,
             tmpl->get_thread_var_creator()->create_info
             (r->pool, config->get_thread_list()));

        TemplateExecutor<typename Response::Writer> executor(r->pool, o);

        executor.exec(tmpl->get_node_tree(), var_creator.get_variables(),
                      tmpl->get_key_count());
    } catch(const char *message) {
        return error_with_log<Response>(r, o, config, message);
    }
    o.finish();

    return OK;
}

template<class Response>
static int file(typename Response::Handle *r, BBSConfig *config,
                const char *arg)
{
    apr_size_t thread_id;
    const char *file_name;
    const char *thread_id_str;
    const char *file_path;
    apr_off_t file_size;
    apr_time_t file_mtime;

    file_name = get_word(r->pool, &arg, ARG_SEPARATE_STR[0]);
    thread_id_str = apr_pstrdup(r->pool, file_name);
    thread_id = static_cast<apr_size_t>
        (apr_atoi64(get_word(r->pool, &thread_id_str, FILE_EXT_SEPARATOR[0])));

    try {
        file_path = BBSThreadIO::get_file_path(r->pool, config->file_dir_path,
                                               thread_id, file_name);

        File file(r->pool, file_path);
        file.open(APR_READ|APR_BINARY|APR_SENDFILE_ENABLED);
        file_size = file.get_size();
        file_mtime = file.get_mtime();

        Response::set_content_type(r, get_image_mime(file_name));

#ifndef DEBUG
        Response::set_modified_time(r, file_mtime);
        Response::set_last_modified(r, file_mtime);

        int status;
        if ((status = Response::is_meets_condition(r)) != OK) {
            return status;
        }
#endif
        if (Response::is_head_method(r)) {
            return OK;
        }

        Response::Writer::sendfile(r, file.get_handle(), file_size);
    } catch(const char *message) {
        return error<Response>(r, config, message);
    }

    return OK;
}

template<class Response>
static int rss(typename Response::Handle *r, BBSConfig *config,
               const char *arg)
{
    apr_size_t thread_id;
    BBSCommentIterator::range_t ranges[1];
    apr_size_t range_count;
    BBSCommentIterator *bcomment_iter;

    Response::set_content_type
        (r, BBSCommentRss<typename Response::Writer>::CONTENT_TYPE);

    if (Response::is_head_method(r)) {
        return OK;
    }

    get_thread_id(r->pool, arg, &thread_id);

    typename Response::Writer o(r);
    try {
        APR_PALLOC(bcomment_iter, BBSCommentIterator *, r->pool,
                   BBSCommentIterator::MEMORY_SIZE);

        ranges[0].start_no = BBSCommentIterator::RANGE_FROM_LAST;
        ranges[0].stop_no = VIW_RSS_COMMENT_COUNT;
        range_count = 1;

        config->get_thread_manager()->get_thread_by_id(r->pool, thread_id,
                                                       bcomment_iter,
                                                       ranges, range_count);
        CleanPointer<BBSCommentIterator> clean_ptr(bcomment_iter);

#ifndef DEBUG
        Response::set_modified_time(r, bcomment_iter->get_mtime());
        Response::set_last_modified(r, bcomment_iter->get_mtime());

        int status;
        if ((status = Response::is_meets_condition(r)) != OK) {
            return status;
        }
#endif
        if (Response::is_head_method(r)) {
            return OK;
        }

        BBSCommentRss<typename Response::Writer> comment_rss(r->pool, o);
        comment_rss.print(r->pool, config->base_url, bcomment_iter);
    } catch(const char *message) {
        return error_with_log<Response>(r, o, config, message);
    }
    o.finish();

    return OK;
}

template<class Response>
static int post_comment(typename Response::Handle *r, BBSConfig *config,
                        const char *arg)
{
    int status;
    apr_array_header_t *content_array;

    if (!Response::is_post_method(r)) {
        return HTTP_BAD_REQUEST;
    }

    if ((status = Response::prepare_post_read(r)) != OK) {
        return status;
    }

    if (!can_post(config, Response::get_remote_ip(r))) {
        return error<Response>(r, config, MESSAGE_POST_INTERVAL_TOO_SHORT);
    }

    typename Response::Writer o(r);
    try {
        apr_size_t thread_id;
        const char *name;
        const char *trip;
        const char *email;
        const char *message;
        apr_time_t time;
        bool is_age;
        RequestReader::progress_t progress;

        typename Response::Reader i(&progress, r);
        RFC1867Parser<typename Response::Reader> parser
            (r->pool, i, config->temp_dir_path,
             TRD_MAX_MESSAGE_SIZE, 0, PST_MAX_ITEM_COUNT);

        content_array = parser.parse(Response::get_content_type(r),
                                     Response::get_content_size(r));

        get_post_comment_param<typename Response::Reader>
            (r->pool, content_array,
             &thread_id, &name, &trip, &email, &message, &time, &is_age);
        config->get_thread_manager()->add_comment
            (thread_id, name, trip, email, message, time, is_age);

        regist_post(config, Response::get_remote_ip(r));

        return redirect<Response>
            (r, o,
             apr_pstrcat
             (r->pool,
              config->base_url,
              ARG_SEPARATE_STR, COMMENT_COMMAND, ARG_SEPARATE_STR,
              apr_ltoa(r->pool, static_cast<long>(thread_id)),
              ARG_SEPARATE_STR, ARG_COMMENT_RANGE_LAST_STR,
              APR_STRINGIFY(VIW_LATEST_COMMENT_COUNT),
              ARG_ANCHOR_STR, POST_COMMENT_COMMAND,
              NULL));
    } catch(const char *message) {
        return error_with_log<Response>(r, o, config, message);
    }
    o.finish();

    return OK;
}

template<class Response>
static int post_thread(typename Response::Handle *r, BBSConfig *config,
                       const char *arg)
{
    int status;
    apr_array_header_t *content_array;
    const char *image_temp_path;

    if (!Response::is_post_method(r)) {
        return HTTP_BAD_REQUEST;
    }

    if ((status = Response::prepare_post_read(r)) != OK) {
        return status;
    }

    typename Response::Writer o(r);
    image_temp_path = NULL;
    try {
        apr_size_t thread_id;
        const char *subject;
        const char *name;
        const char *trip;
        const char *email;
        const char *message;
        apr_time_t time;
        bool is_age;
        RequestReader::progress_t progress;
        BBSThread::image_t image;

        typename Response::Reader i(&progress, r);
        RFC1867Parser<typename Response::Reader>
            parser(r->pool, i, config->temp_dir_path,
                   TRD_MAX_MESSAGE_SIZE, TRD_MAX_IMAGE_SIZE,
                   PST_MAX_ITEM_COUNT);

        content_array = parser.parse(Response::get_content_type(r),
                                     Response::get_content_size(r));

        get_post_thread_param<typename Response::Reader>
            (r->pool, content_array,
             &subject, &image, &image_temp_path,
             &name, &trip, &email, &message, &time, &is_age);

        thread_id = config->get_thread_manager()->create_thread
            (subject, &image, name, trip, email, message, time, is_age);

        if (image.size != 0) {
            commit_image_file(r->pool, config->file_dir_path, thread_id,
                              &image, image_temp_path);
        }

        return redirect<Response>
            (r, o,
             apr_pstrcat
             (r->pool,
              config->base_url,
              ARG_SEPARATE_STR, COMMENT_COMMAND, ARG_SEPARATE_STR,
              apr_ltoa(r->pool, static_cast<long>(thread_id)),
              ARG_SEPARATE_STR, ARG_COMMENT_RANGE_LAST_STR,
              APR_STRINGIFY(VIW_LATEST_COMMENT_COUNT),
              ARG_ANCHOR_STR, POST_COMMENT_COMMAND,
              NULL));
    } catch(const char *message) {
        if (image_temp_path != NULL) {
            apr_file_remove(image_temp_path, r->pool);
        }
        return error_with_log<Response>(r, o, config, message);
    }
    o.finish();

    return OK;
}

template<class Response>
static int post_thread_input(typename Response::Handle *r, BBSConfig *config,
                       const char *arg)
{
    BBSTemplate *tmpl;

    Response::set_content_type(r, "text/html");

    if (Response::is_head_method(r)) {
        return OK;
    }

    typename Response::Writer o(r);
    try {
        tmpl = config->get_template(BBSConfig::POST_THREAD_VIEW);

        TemplateVariableCreator var_creator(r->pool, tmpl->get_ids());

        var_creator.create(BASE_URL_VAR_NAME, config->base_url);

        TemplateExecutor<typename Response::Writer> executor(r->pool, o);

        executor.exec(tmpl->get_node_tree(), var_creator.get_variables(),
                      tmpl->get_key_count());
    } catch(const char *message) {
        return error_with_log<Response>(r, o, config, message);
    }
    o.finish();

    return OK;
}

template<class Response>
static int info(typename Response::Handle *r, BBSConfig *config)
{
    Response::set_content_type(r, "text/plain");

    if (Response::is_head_method(r)) {
        return OK;
    }

    typename Response::Writer o(r);

    o.write("* BBSConfig\n");
    o.write(config->to_string(r->pool));
    o.finish();

    return OK;
}

template<class Response>
int fast_bbs_command_handler(typename Response::Handle *r, BBSConfig *config,
                             const char *arg)
{
    const char *command;
    const char *request_uri;

    if (UNLIKELY(strlen(arg) > HDL_MAX_PATH_INFO_SIZE)) {
        THROW(MESSAGE_HNDL_PATH_INFO_TOO_LONG);
    }

    if (UNLIKELY(*arg == '\0')) {
        request_uri = Response::get_request_uri(r);
#ifdef DEBUG
        if (UNLIKELY(*(request_uri + strlen(request_uri) - 1) ==
                     ARG_SEPARATE_STR[0])) {
            return error<Response>(r, config,
                                   MESSAGE_ENVIRONMENT_LOCATION_DIR_EXIST);
        }
#endif
        Response::set_location(r,
                               apr_pstrcat(r->pool, request_uri, "/", NULL));

        return HTTP_TEMPORARY_REDIRECT;
    } else {
        arg++;
    }

    command = get_word(r->pool, &arg, ARG_SEPARATE_STR[0]);

    if ((*command == '\0') || IS_REQUEST_COMMAND(command, INDEX_COMMAND)) {
        return index<Response>(r, config, arg);
    } else if (IS_REQUEST_COMMAND(command, COMMENT_COMMAND)) {
        return comment<Response>(r, config, arg);
    } else if (IS_REQUEST_COMMAND(command, THREAD_COMMAND)) {
        return thread<Response>(r, config, arg);
    } else if (IS_REQUEST_COMMAND(command, RSS_COMMAND)) {
        return rss<Response>(r, config, arg);
    } else if (IS_REQUEST_COMMAND(command, FILE_COMMAND)) {
        return file<Response>(r, config, arg);
    } else if (IS_REQUEST_COMMAND(command, POST_COMMENT_COMMAND)) {
        return post_comment<Response>(r, config, arg);
    } else if (IS_REQUEST_COMMAND(command, POST_THREAD_INPUT_COMMAND)) {
        return post_thread_input<Response>(r, config, arg);
    } else if (IS_REQUEST_COMMAND(command, POST_THREAD_COMMAND)) {
        return post_thread<Response>(r, config, arg);
#ifdef DEBUG
    } else if (IS_REQUEST_COMMAND(command, INFO_COMMAND)) {
        return info<Response>(r, config);
#endif
    } else {
        return error<Response>(r, config, MESSAGE_HANDLER_COMMAND_INVALID);
    }

    return OK;
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
